// Copyright (c) 2002 The Regents of the University of California.
// All rights reserved.
//
// Redistribution and use in source and binary forms, with or without
// modification, are permitted provided that the following conditions
// are met:
//   1. Redistributions of source code must retain the above copyright
//      notice, this list of conditions, and the following disclaimer.
//   2. Redistributions in binary form must reproduce the above
//      copyright notice, this list of conditions, and the following
//      disclaimer in the documentation and/or other materials provided
//      with the distribution.
//   3. Redistributions must acknowledge that this software was
//      originally developed by the UCSF Computer Graphics Laboratory
//      under support by the NIH National Center for Research Resources,
//      grant P41-RR01081.
//
// THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDER "AS IS" AND ANY
// EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
// IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR
// PURPOSE ARE DISCLAIMED.  IN NO EVENT SHALL THE REGENTS BE LIABLE
// FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
// CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT
// OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR
// BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY,
// WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE
// OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE,
// EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
//
// $Id: VideoControl.cpp,v 1.3 2002/02/07 22:49:15 gregc Exp $

#include "VideoControl.h"
#include <vector>
#include <algorithm>
#include <string.h>
#include <iostream> /* for debugging */

VideoControl::VideoControl(const X11Gui *x11gui): gui(x11gui), screen(0),
	useCombination(false)
{
	int major, minor;
	if (!XSGIvcQueryVersion(gui->display(), &major, &minor))
		throw std::runtime_error("Unable to change video format: missing " XSGIVC_PROTOCOL_NAME " extension");
	screen = DefaultScreen(gui->display());
	if (!XSGIvcQueryVideoScreenInfo(gui->display(), screen, &sinfo))
		throw std::runtime_error("Unable to get screen information");
#ifdef DEBUG
	std::cerr << "num channels:" << sinfo.numChannels << '\n';
	std::cerr << "flags:" << sinfo.flags << '\n';
	std::cerr << "lockOp:" << sinfo.lockOp << '\n';
	std::cerr << "names:" << sinfo.graphicsType << '\n';
#endif
#if 0
	useCombination = (sinfo.flags & XSGIVC_SIFFormatCombination);
#else
	// prefer per-channel
	useCombination = (sinfo.flags & XSGIVC_SIFFormatPerChannel) == 0;
#endif
	if (!useCombination)
		sinfo.numChannels = 1;	// we only support one channel

	setup();
#ifdef DEBUG
	std::cerr << "originalVF: " << originalVF << '\n';
	std::cerr << "stereoVF: " << stereoVF << '\n';
#endif
	if (stereoVF.empty())
		throw std::runtime_error("unable to find suitable stereo video format");
}

VideoControl::~VideoControl()
{
}

void
VideoControl::setStereo(bool stereo)
{
	if (originalVF == stereoVF)
		return;
	std::string *vf = stereo ? &stereoVF : &originalVF;
	if (useCombination)
		XSGIvcLoadVideoFormatCombination(gui->display(), screen,
								vf->c_str());
	else
		XSGIvcLoadVideoFormatByName(gui->display(), screen, 0,
								vf->c_str());
}

#if 0
class VFIHasName {
	const char *name;
public:
	typedef XSGIvcVideoFormatInfo VFI;
	VFIHasName(const char *n): name(n) {}
	bool operator()(const VFI &a) const {
		return strcmp(name, a.name) == 0;
	}
};
#endif

// findVideoFormats returns true if the original format is in stereo

void
VideoControl::setup()
{
	// fetch original channel information
	typedef std::vector<CI *> CIVec;
	CIVec cinfo(sinfo.numChannels);
	for (int i = 0; i != sinfo.numChannels; ++i)
		XSGIvcQueryChannelInfo(gui->display(), screen, i, &cinfo[i]);

	// ignore unused channels (assume channels are used from zero upwards)
	while (sinfo.numChannels > 1) {
		if (cinfo[sinfo.numChannels - 1]->active)
			break;
		--sinfo.numChannels;
	}

	if (!useCombination) {
		// find current video format
		VFI *originalVFI = &cinfo[0]->vfinfo;
		originalVF = originalVFI->name;

		// search for stereo video format
		if (originalVFI->formatFlags & XSGIVC_VFIStereo) {
			stereoVF = originalVF;
			return;
		}
		int count = 0;
		VFI tmp;
		memset(&tmp, 0, sizeof tmp);
		VFI *info = XSGIvcListVideoFormats(gui->display(),
				screen, 0, /* non-null needed! */ &tmp, 0,
				false, MAX, &count);
		if (info == 0)
			throw std::runtime_error("unable to list video formats");
		VFI *stereoVFI = NULL;
		for (int i = 0; i != count; ++i) {
			VFI *vfi = &info[i];
			if ((vfi->formatFlags & XSGIVC_VFIStereo) == 0
			|| vfi->width > originalVFI->width
			|| vfi->height > originalVFI->height)
				continue;
			if (stereoVFI == NULL) {
				stereoVFI = vfi;
				continue;
			}
			if (vfi->width >= stereoVFI->width
			&& vfi->height >= stereoVFI->height
			&& vfi->verticalRetraceRate >= stereoVFI->verticalRetraceRate)
				stereoVFI = vfi;
		}
		if (stereoVFI != NULL)
			stereoVF = stereoVFI->name;
		XFree(info);
		return;
	}

	// use the (more complicated) video combinations
	int count;
	char **combinations = XSGIvcListVideoFormatCombinations(gui->display(),
						screen, "*", MAX, &count);
	if (combinations == NULL)
		throw std::runtime_error("unable to list video combinations");
	std::vector<VFI *> combInfo(count);
	std::vector<int> combInfoCount(count);
	for (int i = 0; i != count; ++i)
		combInfo[i] = XSGIvcListVideoFormatsInCombination(gui->display(),
			screen, combinations[i], &combInfoCount[i]);
	// find current video format
	for (int i = 0; i != count; ++i) {
		if (combInfoCount[i] != sinfo.numChannels)
			continue;
		for (int c = 0; c != sinfo.numChannels; ++c)
			if (strcmp(cinfo[c]->vfinfo.name, combInfo[i][c].name) != 0)
				goto next;
		originalVF = combinations[i];
		break;
	next:;
	}
	if (originalVF.empty())
		throw std::runtime_error("unable to find original video combination");
	// find stereo video format
	// note: by using combinations, we are insuring that all channels
	// share the same video rate.
	if (cinfo[0]->vfinfo.formatFlags & XSGIVC_VFIStereo) {
		stereoVF = originalVF;
		return;
	}
	VFI *stereoVFIs;
	for (int i = 0; i != count; ++i) {
		if (combInfoCount[i] != sinfo.numChannels)
			continue;
		VFI *vfis = combInfo[i];
		if ((vfis[0].formatFlags & XSGIVC_VFIStereo) == 0)
			continue;
		for (int c = 0; c != sinfo.numChannels; ++c) {
			if (vfis[c].width > cinfo[c]->vfinfo.width
			|| vfis[c].height > cinfo[c]->vfinfo.height)
				goto nextCombination;
		}
		if (stereoVFIs == NULL) {
			stereoVFIs = vfis;
			stereoVF = combinations[i];
			continue;
		}
		if (vfis[0].width >= stereoVFIs[0].width
		&& vfis[0].height >= stereoVFIs[0].height
		&& vfis[0].verticalRetraceRate >= stereoVFIs[0].verticalRetraceRate) {
			stereoVFIs = vfis;
			stereoVF = combinations[i];
		}
		continue;
	nextCombination:;
	}
	// cleanup
	for (int i = 0; i != count; ++i)
		XFree(combInfo[i]);
	XFree(combinations);
}
