// Copyright (c) 2002 The Regents of the University of California.
// All rights reserved.
//
// Redistribution and use in source and binary forms, with or without
// modification, are permitted provided that the following conditions
// are met:
//   1. Redistributions of source code must retain the above copyright
//      notice, this list of conditions, and the following disclaimer.
//   2. Redistributions in binary form must reproduce the above
//      copyright notice, this list of conditions, and the following
//      disclaimer in the documentation and/or other materials provided
//      with the distribution.
//   3. Redistributions must acknowledge that this software was
//      originally developed by the UCSF Computer Graphics Laboratory
//      under support by the NIH National Center for Research Resources,
//      grant P41-RR01081.
//
// THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDER "AS IS" AND ANY
// EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
// IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR
// PURPOSE ARE DISCLAIMED.  IN NO EVENT SHALL THE REGENTS BE LIABLE
// FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
// CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT
// OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR
// BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY,
// WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE
// OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE,
// EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
//
// $Id: X11Gui.cpp,v 1.6 2002/02/07 22:49:15 gregc Exp $

#include "X11Gui.h"
#include <X11/Xlib.h>
#include <X11/Xatom.h>
#include <stdlib.h> /* for system() */
#include <sys/types.h>
#include <sys/socket.h>
#include <netinet/in.h>
#include <net/if.h>
#include <sys/ioctl.h>
#include <iostream> /* for debugging */
#include <sstream>

X11Gui::X11Gui(const char *displayName, const char *applicationName,
	bool mustBeLocal): appName(applicationName), property(None)
{
	dpy = XOpenDisplay(displayName);
	if (dpy == NULL)
		throw std::runtime_error(
			"Unable to open a connection to the X server");
	if (!mustBeLocal)
		return;

	int s = ConnectionNumber(dpy);
	struct sockaddr_in address;
	int len = sizeof address;
	if (getpeername(s, &address, &len) == -1)
		throw std::runtime_error("unable to get remote network address");

	if (address.sin_family != AF_INET)
		// UNIX domain socket to X, must be local
		return;

	// walk through interfaces and check that the X server is on
	// the local machine.

	char if_buf[BUFSIZ];
	struct ifconf ifc;
	ifc.ifc_len = sizeof if_buf;
	ifc.ifc_buf = if_buf;
	if (ioctl(s, SIOCGIFCONF, (char *) &ifc) < 0)
		throw std::runtime_error("unable to get list of network interfaces");

	unsigned int n = ifc.ifc_len / sizeof (struct ifreq);
	for (ifreq *ifr = ifc.ifc_req; n != 0; --n, ++ifr) {
		if (ioctl(s, SIOCGIFADDR, ifr) < 0)
			continue;
		if (((sockaddr_in *) &ifr->ifr_addr)->sin_addr.s_addr
				== ((sockaddr_in *) &address)->sin_addr.s_addr)
			return;
	}
	throw std::runtime_error("must run locally to change video to stereo format");
}

X11Gui::~X11Gui()
{
	if (dpy == NULL)
		return;

	if (property != None) {
		// unannounce
		Window root = RootWindow(dpy, DefaultScreen(dpy));
		XDeleteProperty(dpy, root, property);
	}
	XCloseDisplay(dpy);
}

void
X11Gui::closed()
{
	dpy = NULL;
}

bool
X11Gui::announce(const char *propertyName, int port)
{
	property = XInternAtom(dpy, propertyName, False);
	Window root = RootWindow(dpy, DefaultScreen(dpy));
	Atom type = XA_INTEGER;
	int format = 32;
	unsigned long nitems, bytes_after;
	long *prop;
	int status = XGetWindowProperty(dpy, root, property, 0, 1, False,
			AnyPropertyType, &type, &format, &nitems, &bytes_after,
			(unsigned char **) &prop);
	if (status == Success && type != None) {
		// another daemon is already running
		XFree(prop);
		return false;
	}
	// Place property on root with with port
	// number for library to rendevous with.
	long p = port;
	XChangeProperty(dpy, root, property, XA_INTEGER, 32,
				PropModeReplace, (unsigned char *) &p, 1);
	// For some reason, we need to get the property now
	// for it to appear on the root window.
	status = XGetWindowProperty(dpy, root, property, 0, 1, False,
			AnyPropertyType, &type, &format, &nitems, &bytes_after,
			(unsigned char **) &prop);
	if (status != Success || type != XA_INTEGER)
		throw std::runtime_error("unable to register network control port");
	return true;
}

Display *
X11Gui::display() const
{
	return dpy;
}

int
X11Gui::fd() const
{
	if (dpy)
		return ConnectionNumber(dpy);
	throw std::runtime_error("X11 connection closed");
}

void
X11Gui::exitMessage(const char *message) const
{
#if 0
	// debug version
	if (appName != NULL)
		std::cerr << appName << ": ";
	std::cerr << message << '\n';
#else
	// gui version
	// TODO: quote single and double quotes in message
	std::ostringstream cmd;
	cmd << "/usr/bin/X11/xconfirm -c -header '" << appName << ": error' -icon error -B OK -t '" << message << "' > /dev/null &";
	::system(cmd.str().c_str());
#endif
}
