// Copyright (c) 2002 The Regents of the University of California.
// All rights reserved.
//
// Redistribution and use in source and binary forms, with or without
// modification, are permitted provided that the following conditions
// are met:
//   1. Redistributions of source code must retain the above copyright
//      notice, this list of conditions, and the following disclaimer.
//   2. Redistributions in binary form must reproduce the above
//      copyright notice, this list of conditions, and the following
//      disclaimer in the documentation and/or other materials provided
//      with the distribution.
//   3. Redistributions must acknowledge that this software was
//      originally developed by the UCSF Computer Graphics Laboratory
//      under support by the NIH National Center for Research Resources,
//      grant P41-RR01081.
//
// THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDER "AS IS" AND ANY
// EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
// IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR
// PURPOSE ARE DISCLAIMED.  IN NO EVENT SHALL THE REGENTS BE LIABLE
// FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
// CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT
// OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR
// BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY,
// WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE
// OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE,
// EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
//
// $Id: autostereod.cpp,v 1.7 2002/02/07 22:49:15 gregc Exp $

#define FD_SETSIZE 8196		/* pick something larger than getdtablesize() */
#include <iostream>
#include <exception>
#include <vector>
#include "VideoControl.h"
#include "Daemon.h"
#include "GetOpt.h"
#include <unistd.h>
#include <sys/types.h>
#include <sys/time.h>
#include <sys/select.h>
#include <sys/resource.h>
#include <sys/types.h>
#include <sys/socket.h>
#include <netinet/in.h>
#include <signal.h>

int unlimitNoFiles();

X11Gui *gui = NULL;		// global so signal handler can see it

class fdIsSet {
	// predicate for C++ algorithms that tests if a bit is set in a fd_set
	const fd_set	*fdSet;
public:
	fdIsSet(const fd_set *f): fdSet(f) {}
	bool operator()(int i) {
		return FD_ISSET(i, fdSet);
	}
};

// need to detect if we get a signal
volatile int signalled = 0;

void
handleSignal(int sig)
{
	signalled = sig;
}

int
main(int argc, char **argv)
{
	bool background = true;		// true if background after startup
	bool verbose = false;

	int opt;
	otf::GetOpt getopt(argc, argv, "fv");
	while ((opt = getopt()) != -1) switch (opt) {
	  default:
		goto usage;
	  case 'f':
		background = false;
		break;
	  case 'v':
		verbose = true;
		break;
	}
	if (getopt.index() != argc) {
usage:
		std::cerr << "usage: " << argv[0] << " [-f] [-v]\n";
		return 2;
	}

	// catch signals so we can cleanup
	signal(SIGHUP, handleSignal);
	signal(SIGINT, handleSignal);
	signal(SIGTERM, handleSignal);
	signal(SIGUSR1, handleSignal);
	signal(SIGUSR2, handleSignal);

	// allow for the maximum number of connections
	int MAX_FD = unlimitNoFiles();
	if (MAX_FD >= FD_SETSIZE)
		MAX_FD = FD_SETSIZE - 1;

	// try to connect to the display
	try {
		gui = new X11Gui(NULL, "autostereod", true);
	} catch (std::exception &e) {
		std::cerr << "autostereod: " << e.what() << '\n';
		return 1;
	}

	// we now have a X connection, so errors messages can be gui'd.
	typedef std::vector<int> IntVec;
	IntVec connections;
	try {
		// find stereo video mode
		VideoControl vc(gui);
		if (verbose) {
			std::cerr << "original video format: " << vc.originalVideoFormat() << '\n';
			std::cerr << "stereo video format: " << vc.stereoVideoFormat() << '\n';
		}

		// setup daemon
		Daemon daemon(0);
		if (!gui->announce("SGI_AUTOSTEREO", daemon.port())) {
			// another autostereod is already running
			delete gui;
			return 0;
		}

		if (background) {
			switch (fork()) {
			  case -1:	// failed, so run in foreground
				break;
			  case 0:	// in child
				break;
			  default:	// in parent, so disappear
				_exit(0);
			}
		}

		// initially select on X display and daemon
		fd_set active;
		FD_ZERO(&active);
		FD_SET(gui->fd(), &active);
		FD_SET(daemon.fd(), &active);
		for (;;) {
			fd_set readFds = active;
			if (::select(getdtablehi(), &readFds, NULL, NULL, NULL)
									== -1) {
				// we were probably interrupted, check signals
				switch (signalled) {
				  case 0:
					continue;
				  case SIGHUP:
					if (connections.size() > 0)
						vc.setStereo(false);
					for (IntVec::iterator i = connections.begin(); i != connections.end(); ++i) {
						FD_CLR(*i, &active);
						::close(*i);
					}
					connections.clear();
					signalled = 0;
					continue;
				  default:
					if (connections.size() > 0)
						vc.setStereo(false);
					break;
				}
				break;
			}
			if (FD_ISSET(gui->fd(), &readFds)) {
				FD_CLR(gui->fd(), &readFds);
				// if display goes away, exit
				// (need to find out if video gets changed
				//	back, if not we're screwed)
				gui->closed();
				break;
			}
			if (FD_ISSET(daemon.fd(), &readFds)) {
				// process connection to daemon
				FD_CLR(daemon.fd(), &readFds);
				struct sockaddr_in address;
				int len = sizeof address;
				int a = ::accept(daemon.fd(), &address, &len);
				if (a < 0)
					continue;
				FD_SET(a, &active);
				connections.push_back(a);
				if (connections.size() == 1)
					vc.setStereo(true);
			}
			IntVec::iterator closed = std::remove_if(
				connections.begin(), connections.end(),
				fdIsSet(&readFds));
			for (IntVec::iterator i = closed; i != connections.end();
									++i) {
				// TODO: only clear if read zero bytes
				FD_CLR(*i, &active);
				::close(*i);
			}
			connections.erase(closed, connections.end());
			if (connections.size() == 0)
				vc.setStereo(false);
		}
	} catch (std::exception &e) {
		gui->exitMessage(e.what());
		delete gui;
		return 1;
	}
	delete gui;	// also deregisters
	return 0;
}

int
unlimitNoFiles()
{
	struct rlimit limit;
	getrlimit(RLIMIT_NOFILE, &limit);
	limit.rlim_cur = limit.rlim_max;
	setrlimit(RLIMIT_NOFILE, &limit);
	return limit.rlim_cur;
}
