// Copyright (c) 1998-2000 The Regents of the University of California.
// All rights reserved.
//
// Redistribution and use in source and binary forms are permitted
// provided that the above copyright notice and this paragraph are
// duplicated in all such forms and that any documentation,
// distribution and/or use acknowledge that the software was developed
// by the Computer Graphics Laboratory, University of California,
// San Francisco.  The name of the University may not be used to
// endorse or promote products derived from this software without
// specific prior written permission.
// 
// THIS SOFTWARE IS PROVIDED ``AS IS'' AND WITHOUT ANY EXPRESS OR
// IMPLIED WARRANTIES, INCLUDING, WITHOUT LIMITATION, THE IMPLIED
// WARRANTIES OF MERCHANTIBILITY AND FITNESS FOR A PARTICULAR PURPOSE.
// IN NO EVENT SHALL THE REGENTS OF THE UNIVERSITY OF CALIFORNIA BE
// LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY,
// OR CONSEQUENTIAL DAMAGES ARISING OUT OF THE USE OF THIS SOFTWARE.

// $Id: WrapPy.h,v 1.52 2003/05/09 22:30:35 gregc Exp $

#ifndef otf_WrapPy_h
# define otf_WrapPy_h

# ifndef OTF_WRAPPY_DLL
#  define OTF_WRAPPY_IMEX
# elif defined(OTF_WRAPPY_EXPORT)
#  define OTF_WRAPPY_IMEX __declspec(dllexport)
# else
#  define OTF_WRAPPY_IMEX __declspec(dllimport)
# endif

// see wrappy's documentation for more details
//
// _WC is the wrapped C++ class
//
// A wrapped C++ class, _WC, has an additional base class, WrapPy<_WC>.
// The wpyGetObject() member function is used to return an owned Python object.
//
// Use WrapPyType<_WC>::check(PyObject*) to check if the PyObject is
// of the wrapped type (typically WCObjectType).
//

#include <Python.h>
#include <string>
#include <vector>
#include <list>
#include <set>
#include <map>
#include <iostream>

// TODO: need to register this somehow
#define Py_TPFLAGS_WRAPPY_TYPE (1L<<31)

namespace otf {

// forward declarations for templates
template <class _c> PyObject* pyObject(_c _x);
template <class _iterator> PyObject* cvtSequenceToPyList(_iterator _b, _iterator _e);
template <class _iterator> PyObject* cvtSequenceToPyTuple(_iterator _b, _iterator _e);
template <class _iterator> PyObject* cvtMapToPyDict(_iterator _b, _iterator _e);
template <class _MultiMap> PyObject* cvtMultiMapToPyDict(_MultiMap const& _mm);
template <class _iterator> PyObject* cvtMultiMapToPyDict(_iterator _b, _iterator _e);

class OTF_WRAPPY_IMEX WrapPyArgTuple {
	PyObject* arg;
public:
	WrapPyArgTuple(): arg(0) {}
	~WrapPyArgTuple() {
		Py_XDECREF(arg);
	}
	PyObject* operator()() {
		Py_XDECREF(arg);
		arg = PyTuple_New(0);
		return arg;
	}
	PyObject* operator()(PyObject* o) {
		Py_XDECREF(arg);
		arg = PyTuple_New(1);
		Py_XINCREF(o);
		PyTuple_SetItem(arg, 0, o);
		return arg;
	}
	PyObject* operator()(PyObject* o, PyObject* p) {
		Py_XDECREF(arg);
		arg = PyTuple_New(2);
		Py_XINCREF(o);
		PyTuple_SetItem(arg, 0, o);
		Py_XINCREF(p);
		PyTuple_SetItem(arg, 1, p);
		return arg;
	}
};

template <class _WC> class WrapPy;

template <class _WC> class WrapPyType {
#if 0
#if defined(__sgi) && _COMPILER_VERSION <= 721
public:
#endif
	static void clear();
	static PyObject* create();
	friend class WrapPy<_WC>;
#endif
public:
	static bool check(PyObject*, bool noneOk = true);
};

enum WrapPyCreate { PWC_DONT_CREATE, PWC_CREATE, PWC_CREATE_AND_OWN };

class OTF_WRAPPY_IMEX WrapPyObj {
	// use as a non-template base class, so we can dynamic_cast to/from it
protected:
	mutable PyObject* pyObj;
	mutable bool owner;	// true if Python dealloc deletes C++ object
public:
	WrapPyObj(): pyObj(0), owner(false) {}
	virtual PyObject* wpyGetObject(WrapPyCreate pwc = PWC_CREATE) const = 0;
	virtual void wpyDisassociate();	// for Python Object's dealloc method
	virtual void wpyAssociate(PyObject* o) const;
	bool pyOwned() const { return owner; }
	void setPyOwned() { owner = true; }
	virtual ~WrapPyObj() {}
};

template <class _WC>
class WrapPy: virtual public WrapPyObj {
public:
	WrapPy() {}
	// returns owned object
	virtual PyObject* wpyGetObject(WrapPyCreate pwc = PWC_CREATE) const;
	virtual ~WrapPy();
	PyObject* wpyNew() const;
	void wpyDelete();
};

template <class _WC> PyObject*
WrapPy<_WC>::wpyGetObject(WrapPyCreate pwc) const
{
	if (pyObj) {
		if (pwc == PWC_CREATE_AND_OWN)
			owner = true;
		else
			Py_INCREF(pyObj);
		return pyObj;
	}
	if (pwc == PWC_DONT_CREATE)
		return NULL;
	pyObj = wpyNew();
	if (pwc == PWC_CREATE_AND_OWN)
		owner = true;
	else
		Py_XINCREF(pyObj);	// can be NULL, iff badly initialized
					// Python class
	return pyObj;
}

template <class _WC>
WrapPy<_WC>::~WrapPy()
{
	if (pyObj)
		wpyDelete();
}

#if 0
// Can't specialize function templates?
template <class _c> PyObject*
pyObject<_c*>(_c* xp)
{
	if (xp == NULL) {
		Py_INCREF(Py_None);
		return Py_None;
	}
	return xp->wpyGetObject();
}
#endif

template <> inline PyObject*
pyObject(char _x) { return PyString_FromStringAndSize(&_x, int(_x != '\0')); }

template <> inline PyObject*
pyObject(bool _x)
{	
	PyObject *_o = (_x) ? Py_True : Py_False;
	Py_INCREF(_o);
	return _o;
}

// TODO: figure out how to give optional allocator to std::vector<bool>
template <> inline PyObject*
pyObject(std::vector<bool>::reference _x)
{
	return PyInt_FromLong(static_cast<long>(_x));
}

template <> inline PyObject*
pyObject(short _x) { return PyInt_FromLong(static_cast<long>(_x)); }

template <> inline PyObject*
pyObject(int _x) { return PyInt_FromLong(static_cast<long>(_x)); }

template <> inline PyObject*
pyObject(long _x) { return PyInt_FromLong(_x); }

template <> inline PyObject*
pyObject(float _x) { return PyFloat_FromDouble(static_cast<double>(_x)); }

template <> inline PyObject*
pyObject(double _x) { return PyFloat_FromDouble(_x); }

template <> inline PyObject*
pyObject(char const* _x)
{
	if (_x == NULL) {
		Py_INCREF(Py_None);
		return Py_None;
	}
	return PyString_FromString(_x);
}

template <> inline PyObject*
pyObject(std::string _x) { return PyString_FromStringAndSize(_x.c_str(), _x.size()); }

#ifdef otf_Symbol_h
template <> inline PyObject*
pyObject(otf::Symbol _x) { return PyString_FromStringAndSize(_x.c_str(), _x.size()); }
#endif

template <> inline PyObject*
pyObject(PyObject* _x) { Py_XINCREF(_x); return _x; }

template <class _c> PyObject*
pyObject(std::vector<_c> const& container)
{
	return cvtSequenceToPyList(container.begin(), container.end());
}

template <class _c> PyObject*
pyObject(std::list<_c> const& container)
{
	return cvtSequenceToPyList(container.begin(), container.end());
}

template <class _c> PyObject*
pyObject(std::set<_c> const& container)
{
	return cvtSequenceToPyList(container.begin(), container.end());
}

template <class _c> PyObject*
pyObject(std::multiset<_c> const& container)
{
	return cvtSequenceToPyList(container.begin(), container.end());
}

template <class _c, class _d> PyObject*
pyObject(std::map<_c, _d> const& container)
{
	return cvtMapToPyDict(container.begin(), container.end());
}

template <class _c, class _d> PyObject*
pyObject(std::multimap<_c, _d> const& container)
{
	return cvtMultiMapToPyDict(container);
}

template <class _c, class _d> PyObject*
pyObject(std::pair<_c, _d> const& _p)
{
	PyObject* _r = PyTuple_New(2);
	PyObject* _fo = pyObject(_p.first);
	if (_fo == NULL) {
		Py_DECREF(_r);
		return NULL;
	}
	PyTuple_SET_ITEM(_r, 0, _fo);
	PyObject* _so = pyObject(_p.second);
	if (_so == NULL) {
		Py_DECREF(_r);
		return NULL;
	}
	PyTuple_SET_ITEM(_r, 1, _so);
	return _r;
}

// nomenclature:
//
//	_r	result
//	_b	beginning of range
//	_e	end of range
//	_i	integer index
//	_n	next item in range
//	_k	key
//	_ko	key object
//	_vo	value object

template <class _iterator> PyObject*
cvtSequenceToPyList(_iterator _b, _iterator _e)
{
	PyObject* _r = PyList_New(std::distance(_b, _e));
	if (_r == NULL)
		return NULL;
	for (int _i = 0; _b != _e; ++_i, ++_b) {
#ifndef OTF_NEED_EXPLICIT_TEMPLATES
		PyObject* _vo = pyObject(*_b);
#else
		PyObject* _vo = pyObject<std::iterator_traits<_iterator>::value_type>(*_b);
#endif
		if (_vo == NULL) {
			Py_DECREF(_r);
			return NULL;
		}
#ifdef PyList_SET_ITEM
		PyList_SET_ITEM(_r, _i, _vo);
#else
		PyList_SetItem(_r, _i, _vo);
#endif
	}
	return _r;
}

template <class _iterator> PyObject*
cvtSequenceToPyTuple(_iterator _b, _iterator _e)
{
	PyObject* _r = PyTuple_New(std::distance(_b, _e));
	if (_r == NULL)
		return NULL;
	for (int _i = 0; _b != _e; ++_i, ++_b) {
#ifndef OTF_NEED_EXPLICIT_TEMPLATES
		PyObject* _vo = pyObject(*_b);
#else
		PyObject* _vo = pyObject<std::iterator_traits<_iterator>::value_type>(*_b);
#endif
		if (_vo == NULL) {
			Py_DECREF(_r);
			return NULL;
		}
#ifdef PyTuple_SET_ITEM
		PyTuple_SET_ITEM(_r, _i, _vo);
#else
		PyTuple_SetItem(_r, _i, _vo);
#endif
	}
	return _r;
}

#ifdef OTF_NEED_EXPLICIT_TEMPLATES
template <class _t>
struct StripConst
{
	typedef _t type;
	typedef _t const const_type;
};

#if defined(__BORLANDC__) && __BORLANDC__ < 0x0550
#ifdef otf_Symbol_h
template <>
struct StripConst<Symbol const>
{
	typedef Symbol type;
	typedef Symbol const const_type;
};
#endif
template <>
struct StripConst<int const>
{
	typedef int type;
	typedef int const const_type;
};
#endif
#endif

inline Py_complex
makePy_complex(double r, double i)
{
	Py_complex c;
	c.real = r;
	c.imag = i;
	return c;
}

template <class _iterator> PyObject*
cvtMapToPyDict(_iterator _b, _iterator _e)
{
	PyObject* _r = PyDict_New();
	if (_r == NULL)
		return NULL;
	for (; _b != _e; ++_b) {
#ifndef OTF_NEED_EXPLICIT_TEMPLATES
		PyObject* _ko = pyObject(_b->first);
#else
		PyObject* _ko = pyObject<StripConst<std::iterator_traits<_iterator>::value_type::first_type>::type>(_b->first);
#endif
		if (_ko == NULL) {
			Py_DECREF(_r);
			return NULL;
		}
#ifndef OTF_NEED_EXPLICIT_TEMPLATES
		PyObject* _vo = pyObject(_b->second);
#else
		PyObject* _vo = pyObject<std::iterator_traits<_iterator>::value_type::second_type>(_b->second);
#endif
		if (_vo == NULL) {
			Py_DECREF(_ko);
			Py_DECREF(_r);
			return NULL;
		}
		PyDict_SetItem(_r, _ko, _vo);
		Py_DECREF(_ko);
		Py_DECREF(_vo);
	}
	return _r;
}

template <class _MultiMap> PyObject*
cvtMultiMapToPyDict(_MultiMap const& _mm)
{
	PyObject* _r = PyDict_New();
	if (_r == NULL)
		return NULL;
	for (typename _MultiMap::const_iterator _b = _mm.begin(); _b != _mm.end();) {
		typename _MultiMap::key_type const _k = _b->first;
		typename _MultiMap::const_iterator _n = _mm.upper_bound(_k);
		PyObject* _vo = PyList_New(std::distance(_b, _n));
		if (_vo == NULL) {
			Py_DECREF(_r);
			return NULL;
		}
		for (int _i = 0; _b != _n; ++_i, ++_b) {
#ifndef OTF_NEED_EXPLICIT_TEMPLATES
			PyObject* _o = pyObject(_b->second);
#else
			PyObject* _o = pyObject<_MultiMap::mapped_type>(_b->second);
#endif
			if (_o == NULL) {
				Py_DECREF(_vo);
				Py_DECREF(_r);
				return NULL;
			}
#ifdef PyList_SET_ITEM
			PyList_SET_ITEM(_vo, _i, _o);
#else
			PyList_SetItem(_vo, _i, _o);
#endif
		}
#ifndef OTF_NEED_EXPLICIT_TEMPLATES
		PyObject* _ko = pyObject(_k);
#else
		PyObject* _ko = pyObject<_MultiMap::key_type>(_k);
#endif
		if (_ko == NULL) {
			Py_DECREF(_vo);
			Py_DECREF(_r);
			return NULL;
		}
		PyDict_SetItem(_r, _ko, _vo);
		Py_DECREF(_ko);
		Py_DECREF(_vo);
	}
	return _r;
}

template <class _iterator> PyObject*
cvtMultiMapToPyDict(_iterator _b, _iterator _e)
{
	PyObject* _r = PyDict_New();
	if (_r == NULL)
		return NULL;
	_iterator _n;
	for (; _b != _e; _b = _n) {
		PyObject* _vo = PyList_New(0);
		if (_vo == NULL) {
			Py_DECREF(_r);
			return NULL;
		}
		for (_n = _b; _n != _e && _n->first == _b->first; ++_n) {
#ifndef OTF_NEED_EXPLICIT_TEMPLATES
			PyObject* _o = pyObject(_n->second);
#else
			PyObject* _o = pyObject<std::iterator_traits<_iterator>::value_type::second_type>(_n->second);
#endif
			if (_o == NULL) {
				Py_DECREF(_vo);
				Py_DECREF(_r);
				return NULL;
			}
			PyList_Append(_vo, _o);
			Py_DECREF(_o);
		}
#ifndef OTF_NEED_EXPLICIT_TEMPLATES
		PyObject* _ko = pyObject(_b->first);
#else
		PyObject* _ko = pyObject<StripConst<std::iterator_traits<_iterator>::value_type::first_type>::type>(_b->first);
#endif
		if (_ko == NULL) {
			Py_DECREF(_r);
			return NULL;
		}
		PyDict_SetItem(_r, _ko, _vo);
		Py_DECREF(_ko);
		Py_DECREF(_vo);
	}
	return _r;
}

OTF_WRAPPY_IMEX extern PyObject* WrapPyString;	// "__wrappy__"
OTF_WRAPPY_IMEX extern WrapPyObj const* wrappyObject(PyObject* o);
OTF_WRAPPY_IMEX extern void* wrappyVoidObject(PyObject* o);

} // namespace otf

#endif
