// Copyright (c) 2000 The Regents of the University of California.
// All rights reserved.
//
// Redistribution and use in source and binary forms are permitted
// provided that the above copyright notice and this paragraph are
// duplicated in all such forms and that any documentation,
// distribution and/or use acknowledge that the software was developed
// by the Computer Graphics Laboratory, University of California,
// San Francisco.  The name of the University may not be used to
// endorse or promote products derived from this software without
// specific prior written permission.
// 
// THIS SOFTWARE IS PROVIDED ``AS IS'' AND WITHOUT ANY EXPRESS OR
// IMPLIED WARRANTIES, INCLUDING, WITHOUT LIMITATION, THE IMPLIED
// WARRANTIES OF MERCHANTIBILITY AND FITNESS FOR A PARTICULAR PURPOSE.
// IN NO EVENT SHALL THE REGENTS OF THE UNIVERSITY OF CALIFORNIA BE
// LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY,
// OR CONSEQUENTIAL DAMAGES ARISING OUT OF THE USE OF THIS SOFTWARE.

// $Id: WrapPyProxy.cpp,v 1.6 2001/06/27 00:06:28 gregc Exp $

#define OTF_NO_SYMBOL
#define OTF_WRAPPY_EXPORT
#include "WrapPyProxy.h"
#include <sstream>

namespace otf {

struct WrapPyProxy: public PyObject {
	WrapPyObj* wpo;
	const char* description;
	int* debug;
	bool destroyed;
};

PyObject *
WrapPyProxy_FromWrapPyObj(const WrapPyObj* wpo, const char* description, int* debug)
{
	if (wpo == NULL) {
		PyErr_SetString(PyExc_RuntimeError,
				"can not create WrapPyProxy from null pointer");
		return NULL;
	}
	WrapPyProxy* self = PyObject_NEW(WrapPyProxy, &WrapPyProxy_Type);
	if (self == NULL)
		return NULL;
	self->wpo = const_cast<WrapPyObj *>(wpo);
	self->description = description;
	self->debug = debug;
	self->destroyed = false;
	return (PyObject *)self;
}

const WrapPyObj*
WrapPyProxy_AsWrapPyObj(PyObject* self)
{
	if (self == NULL) {
		if (!PyErr_Occurred())
			PyErr_SetString(PyExc_TypeError,
			  "WrapPyProxy_AsWrapPyObj called with null pointer");
		return NULL;
	}
	if (!WrapPyProxy_Check(self)) {
		PyErr_SetString(PyExc_TypeError,
			"WrapPyProxy_AsWrapPyObj with non-WrapPyProxy-object");
		return NULL;
	}
	WrapPyProxy* wpp = static_cast<WrapPyProxy*>(self);
	if (wpp->destroyed) {
		std::ostringstream ost;
		ost << "C++ ";
		if (wpp->description)
			ost << wpp->description;
		else
			ost << "class";
		ost << " instance gone";
#ifdef OTF_SGI_STRSTREAM
		ost << ends;
#endif
		PyErr_SetString(PyExc_RuntimeError, ost.str().c_str());
		return NULL;
	}
	return wpp->wpo;
}

void
WrapPyProxy_SetDestroyed(PyObject* self)
{
	if (self == NULL) {
		if (!PyErr_Occurred())
			PyErr_SetString(PyExc_TypeError,
			  "WrapPyProxy_AsWrapPyObj called with null pointer");
		return;
	}
	if (!WrapPyProxy_Check(self)) {
		PyErr_SetString(PyExc_TypeError,
			"WrapPyProxy_AsWrapPyObj with non-WrapPyProxy-object");
		return;
	}
	WrapPyProxy* wpp = static_cast<WrapPyProxy*>(self);
	wpp->destroyed = true;
}

extern "C" {

static void
WrapPyProxy_dealloc(PyObject* self)
{
	WrapPyProxy* wpp = static_cast<WrapPyProxy*>(self);
	if (wpp->debug && *wpp->debug >= 6)
		std::cerr << "Deallocating " << wpp->description << ": "
							<< wpp->wpo << '\n';
	if (!wpp->destroyed) {
		wpp->wpo->wpyDisassociate();
		if (wpp->wpo->pyOwned())
			delete wpp->wpo;
	}
	PyMem_DEL(self);
}

static PyObject*
WrapPyProxy_GetAttrO(PyObject* self, PyObject* attr)
{
	WrapPyProxy* wpp = static_cast<WrapPyProxy*>(self);
	char const* aname = PyString_AsString(attr);
	if (aname == NULL) {
		PyErr_SetObject(PyExc_AttributeError, attr);
		return NULL;
	}
	if (aname[0] == '_' && aname[1] == '_') {
		if (strcmp(&aname[2], "members__") == 0) {
			PyObject* result = PyList_New(1);
			PyList_SetItem(result, 0, PyString_FromString("destroyed"));
			return result;
		}
		if (strcmp(&aname[2], "methods__") == 0) {
			PyObject* result = PyList_New(0);
			return result;
		}
	}
	if (strcmp(aname, "destroyed") == 0)
		return PyInt_FromLong(wpp->destroyed);
	PyErr_SetObject(PyExc_AttributeError, attr);
	return NULL;
}

} // extern "C"

static char WrapPyProxy_Type__doc__[] = 
"Proxy class to keep track of C++ WrapPy<> objects";

PyTypeObject WrapPyProxy_Type = {
	PyObject_HEAD_INIT(&PyType_Type)
	0,			// ob_size
	"WrapPyProxy",		// tp_name
	sizeof(WrapPyProxy),	// tp_basicsize
	0,			// tp_itemsize
	WrapPyProxy_dealloc,	// tp_dealloc
	0,			// tp_print
	0,			// tp_getattr
	0,			// tp_setattr
	0,			// tp_compare
	0,			// tp_repr
	0,			// tp_as_number
	0,			// tp_as_sequence
	0,			// tp_as_mapping
	0,			// tp_hash
	0,			// tp_call
	0,			// tp_str
	WrapPyProxy_GetAttrO,	// tp_getattro
	0,			// tp_setattro
	0,			// tp_as_buffer
	0,			// tp_flags
	WrapPyProxy_Type__doc__	// tp_doc
};

} // namespace otf
