// Copyright (c) 1998-1999 The Regents of the University of California.
// All rights reserved.
//
// Redistribution and use in source and binary forms are permitted
// provided that the above copyright notice and this paragraph are
// duplicated in all such forms and that any documentation,
// distribution and/or use acknowledge that the software was developed
// by the Computer Graphics Laboratory, University of California,
// San Francisco.  The name of the University may not be used to
// endorse or promote products derived from this software without
// specific prior written permission.
// 
// THIS SOFTWARE IS PROVIDED ``AS IS'' AND WITHOUT ANY EXPRESS OR
// IMPLIED WARRANTIES, INCLUDING, WITHOUT LIMITATION, THE IMPLIED
// WARRANTIES OF MERCHANTIBILITY AND FITNESS FOR A PARTICULAR PURPOSE.
// IN NO EVENT SHALL THE REGENTS OF THE UNIVERSITY OF CALIFORNIA BE
// LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY,
// OR CONSEQUENTIAL DAMAGES ARISING OUT OF THE USE OF THIS SOFTWARE.

// $Id: CvtType.cpp,v 1.67 2003/05/15 04:50:48 gregc Exp $

#include "common.h"
#include "compute.h"
#include "CvtType.h"
#include <otf/Reg.h>
#include <stdlib.h>
#include <assert.h>

using std::string;
using otf::Symbol;

static string
getSubType(const string &ttype, /*INOUT*/ string::size_type *start)
{
	// ttype is a template type,
	// start is the offset to start looking for a subtype
	string::size_type s = *start;
	if (s == 0) {
		s = ttype.find("<");
		if (s == string::npos) {
			*start = ttype.size();
			return ttype;
		}
	}
	switch (ttype[s]) {
	  case '<':
	  case ',':
		++s;
		break;
	}
	std::vector<char> stack;
	string::size_type i = s;
	for (; i < ttype.size(); ++i) {
		switch (ttype[i]) {
		  case '<':
			stack.push_back(ttype[i]);
			break;
		  case ')':
			if (stack.empty() || stack.back() != '(')
				goto done;
			stack.pop_back();
			break;
		  case ']':
			if (stack.empty() || stack.back() != '[')
				goto done;
			stack.pop_back();
			break;
		  case '>':
			if (stack.empty() || stack.back() != '<')
				goto done;
			stack.pop_back();
			break;
		  case ',':
			if (stack.empty())
				goto done;
			break;
		}
	}
done:
	*start = i;
	return removeBlanks(ttype.substr(s, i - s));
}

static otf::Reg identifier("[[:alpha:]_][[:alpha:]_:[:digit:]]*");
static otf::Reg cvtPrefix(
	"^(inline|static|extern|explicit)[[:>:]][[:space:]]*",
	otf::Reg::EXTENDED
);

CvtType::CvtType(const Decl *scope, const string &cpp_type):
	cacheMe(false), aptNeedCheck(false)
	// cacheMe should be true iff aptFormat_ is 'O'
	// pyToCppPattern is needed iff aptFormat_ is not 'O'
{
	// cpp_type must have extra blanks removed already
	cppType_ = cpp_type;

	bool isReference = (*cpp_type.rbegin() == '&');

	string fullType(qualify(scope, cppType_));
	// strip off leading explicit/inline/static
	for (;;) {
		string::size_type len;
		if (cvtPrefix.find(fullType, &len) != 0)
			break;
		fullType.replace(0, len, "");
	}
	if (*fullType.rbegin() == '&') {
		fullType.erase(fullType.size() - 1);
		otf::Reg constPrefix("^const[[:space:]]*",
			otf::Reg::EXTENDED
		);
		for (;;) {
			string::size_type len;
			if (constPrefix.find(fullType, &len) != 0)
				break;
			fullType.replace(0, len, "");
		}
	}

	// get base type (follow typedef's, const's, remove reference)
	type_ = baseType(scope, cpp_type);

	if (*type_.rbegin() == ']') {
		std::cerr << "unable to wrap arrays: \"" << type_ << "\"\n";
		return;
	}

	// first, special case pointer types
	if (type_ == "char*") {
		pyToCppPattern = "PyString_AsString(@)";
		aptType_ = "char*";
		aptFormat_ = 'z';
		aptToCppPattern = '@';
		// cppToAptPattern = bvPattern;
		typeCheckPattern = "PyString_Check(@)";
		bvFormat_ = 'z';
		bvPattern = '@';
		return;
	}
	if (type_ == "PyObject*" || type_ == "struct _object*") {
		// bypass wrappy argument conversions
		cacheMe = true;
		pyToCppPattern = '@';
		aptType_ = "PyObject*";
		aptFormat_ = 'O';
		// cppToAptPattern = bvPattern;
		typeCheckPattern = "true";
		bvFormat_ = 'N';
		bvPattern = '@';
		return;
	}

	if (type_ == "void*") {
		pyToCppPattern = "PyCObject_AsVoidPtr(@)";
		aptType_ = "PyObject*";
		aptFormat_ = 'O';
		aptNeedCheck = true;
		// cppToAptPattern = bvPattern;
		typeCheckPattern = "PyCObject_Check(@)";
		bvFormat_ = 'N';
		bvPattern = "PyCObject_FromVoidPtr(@, NULL)";
		return;
	}
	// TODO: if still a pointer or an array
	//	strip pointer-ness, recurse, and reassemble

	if (type_ == "signed char" || type_ == "unsigned char") {
		pyToCppPattern = "static_cast<" + type_ + ">(PyInt_AsLong(@))";
		aptType_ = "char";
		aptFormat_ = 'b';
		aptToCppPattern = '@';
		cppToAptPattern = "static_cast<" + aptType_ + ">(@)";
		typeCheckPattern = "PyInt_Check(@)";
		bvFormat_ = 'b';
		bvPattern = '@';
		return;
	}
	if (type_ == "char") {
		// To get C/C++ char semantics of a null character being
		// false, we use an empty string for the null character.
		pyToCppPattern = "PyString_AsString(@)[0]"; // has trailing null
		aptType_ = "PyObject*";
		aptFormat_ = 'O';
		// aptToCppPattern = pyToCppPattern;
		// cppToAptPattern = bvPattern;
		aptNeedCheck = true;
		typeCheckPattern = "(PyString_Check(@) && PyString_Size(@) <= 1)";
		bvFormat_ = 'N';
		bvPattern = "otf::pyObject<char>(@)";
		return;
	}
	if (type_ == "int" || type_ == "unsigned int" || type_ == "unsigned") {
		pyToCppPattern = "static_cast<" + type_ + ">(PyInt_AsLong(@))";
		aptType_ = "int";
		aptFormat_ = 'i';
		aptToCppPattern = '@';
		// cppToAptPattern = bvPattern;
		typeCheckPattern = "PyInt_Check(@)";
		bvFormat_ = 'i';
		bvPattern = '@';
		return;
	}
	if (type_ == "short" || type_ == "unsigned short"
	|| type_ == "short int" || type_ == "unsigned short int") {
		pyToCppPattern = "static_cast<" + type_ + ">(PyInt_AsLong(@))";
		aptType_ = "short int";
		aptFormat_ = 'h';
		aptToCppPattern = '@';
		// aptToCppPattern = pyToCppPattern;
		typeCheckPattern = "PyInt_Check(@)";
		bvFormat_ = 'h';
		bvPattern = '@';
		return;
	}
	if (type_ == "long" || type_ == "unsigned long"
	|| type_ == "long int" || type_ == "unsigned long int") {
		if (type_ == "long" || type_ == "long int")
			pyToCppPattern = "PyInt_AsLong(@)";
		else
			pyToCppPattern = "static_cast<" + type_
							+ ">(PyInt_AsLong(@))";
		aptType_ = "long";
		aptFormat_ = 'l';
		aptToCppPattern = '@';
		// cppToAptPattern = bvPattern;
		typeCheckPattern = "PyInt_Check(@)";
		bvFormat_ = 'l';
		bvPattern = '@';
		return;
	}
	if (type_ == "long long" || type_ == "unsigned long long") {
		if (type_ == "long long")
			pyToCppPattern = "PyLong_AsLongLong(@)";
		else
			pyToCppPattern = "static_cast<" + type_
						+ ">(PyLong_AsLongLong(@))";
		aptType_ = "long long";
		aptFormat_ = 'L';
		aptToCppPattern = '@';
		// cppToAptPattern = bvPattern;
		typeCheckPattern = "(PyInt_Check(@) || PyLong_Check(@))";
		bvFormat_ = 'L';
		bvPattern = '@';
		return;
	}
	if (type_ == "bool") {
		pyToCppPattern = "bool(PyInt_AsLong(@))";
		aptType_ = "int";
		aptFormat_ = 'i';
		aptToCppPattern = "bool(@)";
		cppToAptPattern = "int(@)";
		typeCheckPattern = "PyInt_Check(@)";
		bvFormat_ = 'O';
		bvPattern = "otf::pyObject(bool(@))";
		return;
	}
	if (type_ == "float") {
		pyToCppPattern = "float(PyFloat_AsDouble(@))";
		aptType_ = "float";
		aptFormat_ = 'f';
		aptToCppPattern = '@';
		// cppToAptPattern = bvPattern;
		typeCheckPattern = "(PyInt_Check(@) || PyFloat_Check(@))";
		bvFormat_ = 'f';
		bvPattern = '@';
		return;
	}
	if (type_ == "double") {
		pyToCppPattern = "PyFloat_AsDouble(@)";
		aptType_ = "double";
		aptFormat_ = 'd';
		aptToCppPattern = '@';
		// cppToAptPattern = bvPattern;
		typeCheckPattern = "(PyInt_Check(@) || PyFloat_Check(@))";
		bvFormat_ = 'd';
		bvPattern = '@';
		return;
	}
	if (type_.compare(0, 5, "enum ") == 0) {
		cppType_ = type_.substr(5);
		pyToCppPattern = cppType_ + "(PyInt_AsLong(@))";
		aptType_ = "int";
		aptFormat_ = 'i';
		aptToCppPattern = cppType_ + "(@)";
		cppToAptPattern = '@';
		typeCheckPattern = "PyInt_Check(@)";
		bvFormat_ = 'i';
		bvPattern = '@';
		return;
	}
	if (type_ == "std::string" || type_ == "string") {
		// C++ strings preserve length in the presense of null bytes
		pyToCppPattern = "std::string(PyString_AS_STRING(@), PyString_GET_SIZE(@))";
		aptType_ = "PyObject*";
		aptFormat_ = 'S';
		// aptToCppPattern = pyToCppPattern;
		// cppToAptPattern = bvPattern;
		typeCheckPattern = "PyString_Check(@)";
		bvFormat_ = 'N';
		bvPattern = "PyString_FromStringAndSize(@.data(), @.size())";
		return;
	}
	if (type_ == "otf::Symbol" || type_ == "Symbol") {
		pyToCppPattern = "otf::Symbol(std::string(PyString_AS_STRING(@), PyString_GET_SIZE(@)))";
		aptType_ = "PyObject*";
		aptFormat_ = 'S';
		// aptToCppPattern = pyToCppPattern;
		// cppToAptPattern = bvPattern;
		typeCheckPattern = "PyString_Check(@)";
		bvFormat_ = 'N';
		bvPattern = "PyString_FromStringAndSize(@.str().data(), @.size())";
		return;
	}

	if (type_ == "std::complex<float>" || type_ == "complex<float>"
	|| type_ == "std::complex<double>" || type_ == "std::complex<double>") {
		pyToCppPattern = type_ + "(PyComplex_RealAsDouble(@), PyComplex_ImagAsDouble(@))";
		aptType_ = "Py_complex";
		aptFormat_ = 'D';
		aptToCppPattern = type_ + "(@.real, @.imag)";
		cppToAptPattern = "otf::makePy_complex(@.real(), @.imag())";
		typeCheckPattern = "PyComplex_Check(@)";
		bvFormat_ = 'N';
		bvPattern = "PyComplex_FromDoubles(@.real(), @.imag())";
		return;
	}
	if (type_.compare(0, 10, "std::pair<") == 0
	|| type_.compare(0, 5, "pair<") == 0) {
		pyToCppPattern = "?TODO?";
		aptType_ = "PyObject*";
		aptFormat_ = 'O';
		// aptToCppPattern = pyToCppPattern;
		// cppToAptPattern = bvPattern;
		aptNeedCheck = true;
		typeCheckPattern = "?TODO?";
		// bvFormat_ and bvPatteren are figured out below
		string::size_type s = 0;
		string first = getSubType(type_, &s);
		string second = getSubType(type_, &s);
		string::size_type len = first.rfind("::");
		if (first != second || first.size() < 11 || len == string::npos
		|| !(first.substr(first.size() - 8) == "iterator")) {
			CvtType firstArg(scope, first);
			CvtType secondArg(scope, second);
			cacheMe = firstArg.cache() || secondArg.cache();
			typeCheckPattern
				= "PyTuple_Check(@) && PyTuple_Size(@) == 2"
				" && "
				+ firstArg.typeCheck("PyTuple_GetItem(@, 0)");
				+ " && "
				+ secondArg.typeCheck("PyTuple_GetItem(@, 1)");
			// TODO: pyToCppPattern
			bvFormat_ = "(" + firstArg.bvFormat()
						+ secondArg.bvFormat() + ")";
			bvPattern = firstArg.bvArg("@.first") + ", "
						+ secondArg.bvArg("@.second");
			return;
		}
		// ::.*iterator
		bvFormat_ = 'N';
		CvtType subType(scope, first.substr(0, len));
		cacheMe = subType.cache();
		const string &tmp = subType.baseCppType();
		if (tmp.compare(0, 9, "std::map<") == 0
		|| tmp.compare(0, 4, "map<") == 0
		|| tmp.compare(0, 14, "std::hash_map<") == 0
		|| tmp.compare(0, 9, "hash_map<") == 0) {
			bvPattern = "otf::cvtMapToPyDict(@.first, @.second)";
			return;
		}
		if (tmp.compare(0, 14, "std::multimap<") == 0
		|| tmp.compare(0, 9, "multimap<") == 0
		|| tmp.compare(0, 19, "std::hash_multimap<") == 0
		|| tmp.compare(0, 14, "hash_multimap<") == 0) {
			bvPattern = "otf::cvtMultiMapToPyDict(@.first, @.second)";
			return;
		}
		bvPattern = "otf::cvtSequenceToPyList(@.first, @.second)";
		return;
	}

	if (type_.compare(0, 9, "std::map<") == 0
	|| type_.compare(0, 4, "map<") == 0
	|| type_.compare(0, 14, "std::hash_map<") == 0
	|| type_.compare(0, 9, "hash_map<") == 0) {
		string::size_type s = 0;
		string key = getSubType(type_, &s);
		string value = getSubType(type_, &s);
		cacheMe = CvtType(scope, key).cache()
					|| CvtType(scope, value).cache();
		pyToCppPattern = "?TODO?";
		aptType_ = "PyObject*";
		aptFormat_ = 'O';
		// aptToCppPattern = pyToCppPattern;
		// cppToAptPattern = bvPattern;
		aptNeedCheck = true;
		typeCheckPattern = "?TODO?";
		bvFormat_ = 'N';
		bvPattern = "otf::cvtMapToPyDict(@.begin(), @.end())";
		return;
	}

	if (type_.compare(0, 14, "std::multimap<") == 0
	|| type_.compare(0, 9, "multimap<") == 0
	|| type_.compare(0, 19, "std::hash_multimap<") == 0
	|| type_.compare(0, 14, "hash_multimap<") == 0) {
		string::size_type s = 0;
		string key = getSubType(type_, &s);
		string value = getSubType(type_, &s);
		cacheMe = CvtType(scope, key).cache()
					|| CvtType(scope, value).cache();
		pyToCppPattern = "?TODO?";
		aptType_ = "PyObject*";
		aptFormat_ = 'O';
		// aptToCppPattern = pyToCppPattern;
		aptNeedCheck = true;
		typeCheckPattern = "?TODO?";
		bvFormat_ = 'N';
		bvPattern = "otf::cvtMultiMapToPyDict(@)";
		return;
	}

	if (type_.compare(0, 9, "std::set<") == 0
	|| type_.compare(0, 4, "set<") == 0
	|| type_.compare(0, 19, "std::hash_set<") == 0
	|| type_.compare(0, 14, "hash_set<") == 0) {
		// TODO? convert to dictionary keys?
		string::size_type s = 0;
		string value = getSubType(type_, &s);
		cacheMe = CvtType(scope, value).cache();
		pyToCppPattern = "?TODO?";
		aptType_ = "PyObject*";
		aptFormat_ = 'O';
		// aptToCppPattern = pyToCppPattern;
		// cppToAptPattern = bvPattern;
		aptNeedCheck = true;
		typeCheckPattern = "?TODO?";
		bvFormat_ = 'N';
		// TODO: use a special type instead of a Python list
		bvPattern = "otf::cvtSequenceToPyList(@.begin(), @.end())";
		return;
	}

	if (type_.compare(0, 9, "std::multiset<") == 0
	|| type_.compare(0, 4, "multiset<") == 0
	|| type_.compare(0, 19, "std::hash_multiset<") == 0
	|| type_.compare(0, 14, "hash_multiset<") == 0) {
		// TODO? convert to dictionary keys?
		string::size_type s = 0;
		string value = getSubType(type_, &s);
		cacheMe = CvtType(scope, value).cache();
		pyToCppPattern = "?TODO?";
		aptType_ = "PyObject*";
		aptFormat_ = 'O';
		// aptToCppPattern = pyToCppPattern;
		// cppToAptPattern = bvPattern;
		aptNeedCheck = true;
		typeCheckPattern = "?TODO?";
		bvFormat_ = 'N';
		// TODO: use a special type instead of a Python list
		bvPattern = "otf::cvtSequenceToPyList(@.begin(), @.end())";
		return;
	}

	if (type_.compare(0, 10, "std::list<") == 0
	|| type_.compare(0, 5, "list<") == 0
	|| type_.compare(0, 11, "std::slist<") == 0
	|| type_.compare(0, 6, "slist<") == 0
	|| type_.compare(0, 11, "std::deque<") == 0
	|| type_.compare(0, 6, "deque<") == 0
	|| type_.compare(0, 12, "std::vector<") == 0
	|| type_.compare(0, 7, "vector<") == 0) {
		string::size_type s = 0;
		string value = getSubType(type_, &s);
		cacheMe = CvtType(scope, value).cache();
		pyToCppPattern = "?TODO?";
		aptType_ = "PyObject*";
		aptFormat_ = 'O';
		// aptToCppPattern = pyToCppPattern;
		// cppToAptPattern = bvPattern;
		aptNeedCheck = true;
		typeCheckPattern = "?TODO?";
		bvFormat_ = 'N';
		bvPattern = "otf::cvtSequenceToPyList(@.begin(), @.end())";
		return;
	}

	if (type_.compare(0, 11, "otf::AArray<") == 0
	|| type_.compare(0, 6, "AArray<") == 0
	|| type_.compare(0, 11, "otf::Array<") == 0
	|| type_.compare(0, 6, "Array<") == 0) {
		// TODO: convert to/from numeric arrays
		string::size_type s = 0;
		string value = getSubType(type_, &s);
		CvtType valueArg(scope, value);
		string arrayLen = getSubType(type_, &s);
		int len = atoi(arrayLen.c_str());
		cacheMe = valueArg.cache();
		pyToCppPattern = "?TODO?";
		aptType_ = "PyObject*";
		aptFormat_ ='O';
		// aptToCppPattern = pyToCppPattern;
		// cppToAptPattern = bvPattern;
		aptNeedCheck = true;
		typeCheckPattern = "(PySequence_Check(@) && PySequence_Size(@) == " + arrayLen + ")";
		bvFormat_ = 'N';
		bvPattern = "otf::cvtSequenceToPyTuple(@.begin(), @.end())";
		return;
	}

	// assumed wrapped class
	pyToCppPattern = "?TODO?";
	aptType_ = "PyObject*";
	aptFormat_ = 'O';
	// aptToCppPattern = pyToCppPattern;
	// cppToAptPattern = bvPattern;
	aptNeedCheck = true;
	typeCheckPattern = "?TODO?";
	bvFormat_ = 'N';
	bvPattern = "?TODO?";
	string::size_type len;
	string::size_type start = identifier.find(type_, &len);
	if (start == string::npos)
		return;
	Symbol id(type_.substr(start, len));
	bvPattern = "otf::pyObject<" + fullType + ">(@)";
	string objectName;
	string::size_type i = id.str().rfind("::");
	if (i == string::npos)
		objectName = id.str();
	else
		objectName = id.str().substr(i + 2);
	objectName += "Object";
	typeCheckPattern = "otf::WrapPyType<" + id.str() + ">::check(@)";
	CDCIMap::iterator x = classInfoMap.find(classDeclsMap[id]);
	const ClassInfo *ci = x == classInfoMap.end() ? NULL : (*x).second;
	if (ci != NULL && ci->isPythonClass) {
		if (!ci->dontCache)
			cacheMe = true;
		if (isReference || !ci->isWrappySubclass)
			pyToCppPattern = "*";
		else
			pyToCppPattern = "@ == Py_None ? NULL : ";
		if (ci->isWrappySubclass)
			pyToCppPattern += "dynamic_cast<" + id.str() + "*>(const_cast<otf::WrapPyObj*>(otf::wrappyObject(@)))";
		else
			pyToCppPattern += "reinterpret_cast<" + id.str()
					+ "*>(otf::wrappyVoidObject(@))";
		return;
	}
	if (ci != NULL && ci->isWrappySubclass) {
		if (!ci->dontCache)
			cacheMe = true;
		if (isReference)
			pyToCppPattern = "*";
		else
			pyToCppPattern = "@ == Py_None ? NULL : ";
		pyToCppPattern += "static_cast<" + objectName + "*>(@)->inst";
		return;
	}
	if (ci != NULL && !ci->isWrappySubclass && !ci->isAbstractType) {
		pyToCppPattern = "*static_cast<" + objectName + "*>(@)->inst";
		return;
	}
	if (derivedClasses.find(id) != derivedClasses.end()) {
		// id is a base class and we're generating a Python type
		if (ci && !ci->dontCache)
			cacheMe = true;
		std::pair<DerivedClasses::iterator, DerivedClasses::iterator>
					bounds = derivedClasses.equal_range(id);
		aptNeedCheck = true;
		typeCheckPattern = "(";
		if (isReference)
			pyToCppPattern = "*(";
		else
			pyToCppPattern = "@ == Py_None ? NULL : ";
		for (DerivedClasses::iterator i = bounds.first;
						i != bounds.second; ++i) {
			if (i != bounds.first) {
				typeCheckPattern += " || ";
				pyToCppPattern += " : ";
			}
			typeCheckPattern += "otf::WrapPyType<" + (*i).second.str()
								+ ">::check(@)";
			pyToCppPattern += "otf::WrapPyType<" + (*i).second.str()
				+ ">::check(@) ? static_cast<" + type_
				+ ">(static_cast<" + (*i).second.str()
				+ "Object*>(@)->inst)";
		}
		typeCheckPattern += ")";
		pyToCppPattern += " : NULL";
		if (isReference)
			pyToCppPattern += ")";
		return;
	}
	std::cerr << programName << ": unable to wrap: \"" << type_ << "\"\n";
	// TODO: Otherwise treat class as a POD
}

string
CvtType::aptToCpp(const string &arg) const
{
	string tmp(aptToCppPattern);
	if (tmp.empty()) {
		assert(aptType_ == "PyObject*");
		tmp = pyToCppPattern;
	}
	string::size_type x = 0;
	for (;;) {
		x = tmp.find('@', x);
		if (x == string::npos)
			break;
		tmp.replace(x, 1, arg);
		x += arg.size() - 1;
	}
	return tmp;
}

string
CvtType::cppToApt(const string &arg) const
{
	string tmp(cppToAptPattern);
	if (tmp.empty())
		tmp = bvPattern;
	string::size_type x = 0;
	for (;;) {
		x = tmp.find('@', x);
		if (x == string::npos)
			break;
		tmp.replace(x, 1, arg);
		x += arg.size() - 1;
	}
	return tmp;
}

string
CvtType::pyToCpp(const string &arg) const
{
	string tmp(pyToCppPattern);
	string::size_type x = 0;
	for (;;) {
		x = tmp.find('@', x);
		if (x == string::npos)
			break;
		tmp.replace(x, 1, arg);
		x += arg.size() - 1;
	}
	return tmp;
}

string
CvtType::bvArg(const string &arg) const
{
	string tmp(bvPattern);
	string::size_type x = 0;
	for (;;) {
		x = tmp.find('@', x);
		if (x == string::npos)
			break;
		tmp.replace(x, 1, arg);
		x += arg.size() - 1;
	}
	return tmp;
}

string
CvtType::typeCheck(const string &arg) const
{
	string tmp(typeCheckPattern);
	string::size_type x = 0;
	for (;;) {
		x = tmp.find('@', x);
		if (x == string::npos)
			break;
		tmp.replace(x, 1, arg);
		x += arg.size() - 1;
	}
	return tmp;
}

// generic type support

static Decl *
checkBaseClassesForType(const ClassInfo *ci, Symbol id)
{
	for (CIList::const_iterator i = ci->baseClasses.begin();
					i != ci->baseClasses.end(); ++i) {
		Decl *idDecl = (*i)->cd->members.unique(id);
		if (idDecl != NULL
		&& idDecl->dtype != Decl::FUNCTION)
			// constructors are not types
			return idDecl;
		idDecl = checkBaseClassesForType(*i, id);
		if (idDecl != NULL)
			return idDecl;
	}
	return NULL;
}

const Decl *
typeDecl(const Decl *scope, Symbol id)
{
	for (;;) {
		const ClassDecl *cd = NULL;
		const ClassInfo *ci = NULL;
		const NamespaceDecl *nd = NULL;
		const DeclList *decls;
		if (scope == NULL)
			decls = &globalDecls;
		else if (nd = dynamic_cast<const NamespaceDecl *>(scope))
			decls = &nd->decls;
		else if (cd = dynamic_cast<const ClassDecl *>(scope)) {
			decls = &cd->members;
			CDCIMap::iterator x = classInfoMap.find(cd);
			if (x != classInfoMap.end())
				ci = (*x).second;
		} else
			return NULL;
		const Decl *idDecl = decls->unique(id);
		if (idDecl != NULL && idDecl->dtype != Decl::FUNCTION)
			// constructors are not types
			return idDecl;
		if (ci != NULL) {
			Decl *idDecl = checkBaseClassesForType(ci, id);
			if (idDecl != NULL)
				return idDecl;
		}
		if (nd) {
			// This doesn't find types, but it does make
			// qualify() of namespace functions work.
			DeclList::RAconst_iterator i = decls->rAfind(id);
			if (i != decls->rAend())
				return i->second;
		}
		if (scope == NULL)
			return NULL;
		scope = scope->scope;
	}
}

static DeclList::RAConstRange
checkBaseClassesForFuncVarType(const ClassInfo *ci, Symbol id)
{
	for (CIList::const_iterator i = ci->baseClasses.begin();
					i != ci->baseClasses.end(); ++i) {
		DeclList::RAConstRange idDecls
					= (*i)->cd->members.rAequal_range(id);
		if (idDecls.first != idDecls.second)
			return idDecls;
		idDecls = checkBaseClassesForFuncVarType(*i, id);
		if (idDecls.first != idDecls.second)
			return idDecls;
	}
	DeclList::RAconst_iterator dummy;
	return DeclList::RAConstRange(dummy, dummy);
}

DeclList::RAConstRange
funcVarTypeDecls(const Decl *scope, string name)
{
	for (;;) {
		string::size_type delim = name.find("::");
		if (delim == string::npos)
			break;
		const Decl *symDecl = typeDecl(scope,
						Symbol(name.substr(0, delim)));
		if (symDecl == NULL) {
			DeclList::RAconst_iterator dummy;
			return DeclList::RAConstRange(dummy, dummy);
		}
		scope = symDecl;
		name = name.substr(delim + 2);
	}
	Symbol id(name);

	for (;;) {
		const ClassDecl *cd = NULL;
		const ClassInfo *ci = NULL;
		const NamespaceDecl *nd = NULL;
		const DeclList *decls;
		if (scope == NULL)
			decls = &globalDecls;
		else if (nd = dynamic_cast<const NamespaceDecl *>(scope))
			decls = &nd->decls;
		else if (cd = dynamic_cast<const ClassDecl *>(scope)) {
			decls = &cd->members;
			CDCIMap::iterator x = classInfoMap.find(cd);
			if (x != classInfoMap.end())
				ci = (*x).second;
		} else {
			DeclList::RAconst_iterator dummy;
			return DeclList::RAConstRange(dummy, dummy);
		}
		const DeclList::RAConstRange idDecls = decls->rAequal_range(id);
		if (idDecls.first != idDecls.second) {
			// constructors are not types
			return idDecls;
		}
		if (ci != NULL) {
			DeclList::RAConstRange idDecls
				= checkBaseClassesForFuncVarType(ci, id);
			if (idDecls.first != idDecls.second)
				return idDecls;
		}
		if (scope == NULL) {
			DeclList::RAconst_iterator dummy;
			return DeclList::RAConstRange(dummy, dummy);
		}
		scope = scope->scope;
	}
}

string
qualify(const Decl *scope, const string &str, bool keepNamespace)
{
	// look through string for identifiers that need to be qualified
	// with class and/or namespace names when used outside of the
	// class/namespace they were defined in.
	string tmp(str);
	string result;
	string::size_type idLen;
	for (;;) {
		string::size_type start = identifier.find(tmp, &idLen);
		if (start == string::npos) {
			result += tmp;
			break;
		}
		string id = tmp.substr(start, idLen);
		result += tmp.substr(0, start);
		tmp.replace(0, start + idLen, "");
		string::size_type delim = id.find("::");
		const Decl *symDecl = typeDecl(scope,
						Symbol(id.substr(0, delim)));
		if (symDecl != NULL)
			for (; symDecl->scope != NULL; symDecl = symDecl->scope)
				id = symDecl->scope->tag.str() + "::" + id;
		if (!keepNamespace && !nameSpace.empty()
		&& id.size() > nameSpace.size()
		&& id.compare(0, nameSpace.size(), nameSpace) == 0
		&& id.compare(nameSpace.size(), 2, "::") == 0)
			id = id.substr(nameSpace.size() + 2);
		result += id;
	}
	return result;
}

static otf::Reg btPrefix(
	"^(inline|static|extern|explicit|const|unsigned)[[:>:]][[:space:]]*",
	otf::Reg::EXTENDED
);
static otf::Reg btConst(
	"[[:space:]]*[[:<:]]const[[:>:]]",
	otf::Reg::EXTENDED
);

string
baseType(const Decl *scope, const string &name)
{
	string tmp(name);
startOver:
	// strip off leading const/unsigned/explicit/inline/static
	for (;;) {
		string::size_type len;
		if (btPrefix.find(tmp, &len) != 0)
			break;
		tmp.replace(0, len, "");
	}
	// strip off embedded const
	for (;;) {
		string::size_type len, where;
		where = btConst.find(tmp, &len);
		if (where == string::npos)
			break;
		tmp.replace(where, len, "");
	}
	// reference doesn't change base type
	if (*tmp.rbegin() == '&')
		tmp.erase(tmp.size() - 1);
	// check if name is a typedef, if so, expand it and start over
	const Decl *curScope = scope;
	string::size_type idLen;
	string::size_type start = identifier.find(tmp, &idLen);
	if (start == string::npos)
		return "??";
	string type(tmp.substr(start, idLen));
	for (;;) {
		if (type.empty())
			return "???";
		string::size_type delim = type.find("::");
		// lookup name in current scope, then enclosing scope
		Symbol sym(type.substr(0, delim));
		const Decl *symDecl = typeDecl(curScope, sym);
		if (delim == string::npos) {
			if (symDecl == NULL)
				break;
			if (symDecl->dtype == Decl::ENUM) {
				tmp.replace(start, 0, "enum ");
				break;
			}
			if (symDecl->dtype != Decl::TYPEDEF)
				break;
			// it's a typedef, so recurse
			tmp.replace(start, idLen, symDecl->text);
			scope = symDecl->scope;
			goto startOver;
		}
		// we have a :: delimitor, so the symbol we just found must
		// be a namespace or class and thus the scope we want to
		// look up the next part in.
		curScope = symDecl;
		type = type.substr(delim + 2);
	}
	return qualify(scope, tmp);
}
