// Copyright (c) 1998-2000 The Regents of the University of California.
// All rights reserved.
//
// Redistribution and use in source and binary forms are permitted
// provided that the above copyright notice and this paragraph are
// duplicated in all such forms and that any documentation,
// distribution and/or use acknowledge that the software was developed
// by the Computer Graphics Laboratory, University of California,
// San Francisco.  The name of the University may not be used to
// endorse or promote products derived from this software without
// specific prior written permission.
// 
// THIS SOFTWARE IS PROVIDED ``AS IS'' AND WITHOUT ANY EXPRESS OR
// IMPLIED WARRANTIES, INCLUDING, WITHOUT LIMITATION, THE IMPLIED
// WARRANTIES OF MERCHANTIBILITY AND FITNESS FOR A PARTICULAR PURPOSE.
// IN NO EVENT SHALL THE REGENTS OF THE UNIVERSITY OF CALIFORNIA BE
// LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY,
// OR CONSEQUENTIAL DAMAGES ARISING OUT OF THE USE OF THIS SOFTWARE.

// $Id: CvtType.h,v 1.26 2003/05/09 22:49:36 gregc Exp $

#ifndef CvtType_h
# define CvtType_h

# include <set>
# include <map>
# include <otf/Symbol.h>
# include "ParseHeader.h"

class CvtType {
	// C++ parameter support (cpp)
	//
	// Given a C++ type (cppType in a given scope), describe how to
	// convert it to and from PyObject* and its PyArg_ParseTuple type.
public:
	CvtType(const Decl *scope, const std::string &cppType);

	// cache() returns true if attributes of type should be cached
	// -- usually true if C++ type is a C++ class.
	bool		cache() const;

	const std::string &cppType() const;	// C++ argument type
	const std::string &baseCppType() const;	// base type (followed
						// typedef's, remove reference)

	// PyArg_ParseTuple support (apt)
	// 	PyArg_ParseTuple is used to convert PyObject*'s to C++ types
	// 	
	// 	aptType is the C type corresponding to the aptFormat.  If
	// 	aptType is PyObject* (i.e, no conversion), then we
	// 	typically need to type check (needTypeCheck, typeCheck) the
	// 	parsed PyObject* and then convert it with pyToCpp.
	//
	// 	aptToCpp returns code that converts the aptType variable
	// 	to the C++ type.
	// 	
	// 	cppToApt returns code that converts the C++ type to the
	// 	aptType (used for converting default argument values).
	const std::string &aptType() const;
	const std::string &aptFormat() const;
	std::string	aptToCpp(const std::string &arg) const;
	std::string	cppToApt(const std::string &arg) const;
	bool		needTypeCheck() const;

	// Knowledge about PyObject*'s:
	//
	//	typeCheck returns code that checks if a PyObject* can be
	//	mapped to a given C++ type.
	//
	//	pyToCpp return code that converts a PyObject* to the C++ type
	//	(you need to call PyErr_Occured afterwards).
	std::string	typeCheck(const std::string &arg) const;
	std::string	pyToCpp(const std::string &arg) const;

	// Py_BuildValue support (bv)
	// 	Py_BuildValue is used to convert C++ types to PyObject*'s
	//
	// 	bvFormat returns what's needed for Py_BuildValue format
	// 	and bvArg
	const std::string &bvFormat() const;
	std::string	bvArg(const std::string &arg) const;
private:
	std::string	cppType_;
	std::string	type_;		// baseCppType
	bool		cacheMe;
	std::string	pyToCppPattern;
	std::string	aptType_;
	std::string	aptFormat_;
	std::string	aptToCppPattern;
	std::string	cppToAptPattern;
	bool		aptNeedCheck;
	std::string	typeCheckPattern;
	std::string	bvFormat_;
	std::string	bvPattern;
};

inline bool
CvtType::cache() const
{
	return cacheMe;
}

inline const std::string &
CvtType::cppType() const
{
	return cppType_;
}

inline const std::string &
CvtType::baseCppType() const
{
	return type_;
}

inline const std::string &
CvtType::aptFormat() const
{
	return aptFormat_;
}

inline const std::string &
CvtType::aptType() const
{
	return aptType_;
}

inline bool
CvtType::needTypeCheck() const
{
	return aptNeedCheck;
}

inline const std::string &
CvtType::bvFormat() const
{
	return bvFormat_;
}

extern DeclList::RAConstRange
		funcVarTypeDecls(const Decl *scope, std::string name);
extern const Decl *
		typeDecl(const Decl *scope, otf::Symbol id);
extern std::string
		qualify(const Decl *scope, const std::string &str,
							bool keepNS = false);
extern std::string
		baseType(const Decl *scope, const std::string &name);

#endif
