// Copyright (c) 1998-2000 The Regents of the University of California.
// All rights reserved.
//
// Redistribution and use in source and binary forms are permitted
// provided that the above copyright notice and this paragraph are
// duplicated in all such forms and that any documentation,
// distribution and/or use acknowledge that the software was developed
// by the Computer Graphics Laboratory, University of California,
// San Francisco.  The name of the University may not be used to
// endorse or promote products derived from this software without
// specific prior written permission.
// 
// THIS SOFTWARE IS PROVIDED ``AS IS'' AND WITHOUT ANY EXPRESS OR
// IMPLIED WARRANTIES, INCLUDING, WITHOUT LIMITATION, THE IMPLIED
// WARRANTIES OF MERCHANTIBILITY AND FITNESS FOR A PARTICULAR PURPOSE.
// IN NO EVENT SHALL THE REGENTS OF THE UNIVERSITY OF CALIFORNIA BE
// LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY,
// OR CONSEQUENTIAL DAMAGES ARISING OUT OF THE USE OF THIS SOFTWARE.

// $Id: class.cpp,v 1.51 2003/06/17 22:26:46 gregc Exp $

// dump output for Python types

#include <fstream>
#include "common.h"
#include "method.h"
#include "type.h"
#include "class.h"
#include "CvtType.h"

using std::string;
using otf::Symbol;

typedef std::map<Symbol, string> FuncMap;

static void
dumpWrapClassInline(std::ostream &output, const ClassInfo *ci)
{
	string doInline;
	if (!skipInline)
		doInline = "inline ";
	string prefix(qualify(ci->cd->scope, ci->name.str(), true));
	string objectName;
	if (!nameSpace.empty())
		objectName = nameSpace + "::";
	objectName += ci->name.str() + "Object";
	output <<
		"\n"
		"namespace otf {\n"
		// type check function
		"\n"
		"template <> " << doInline << "bool\n"
		"WrapPyType<" << prefix
			<< ">::check(PyObject* o, bool";
	if (ci->isWrappySubclass)
		output << " noneOk";
	output <<
		")\n"
		"{\n";
	if (ci->isWrappySubclass)
		output <<
			"\tif (noneOk && o == Py_None)\n"
			"\t\treturn true;\n";
	output <<
		"\tif (!PyInstance_Check(o))\n"
		"\t\treturn false;\n"
		"\tPyInstanceObject* obj = reinterpret_cast<PyInstanceObject*>(o);\n"
		"\tPyObject* klass = reinterpret_cast<PyObject*>(obj->in_class);\n"
		"\treturn PyClass_IsSubclass(klass, " << objectName << "Class);\n"
		"}\n";
	output <<
		"\n"
		"template <> " << exportTag << "PyObject* pyObject(" << prefix;
	if (ci->isWrappySubclass || ci->isAbstractType)
		output << " const*";
	output << " o);\n";
	if (ci->isWrappySubclass || ci->isAbstractType)
		// don't understand yet why this is necessary 12/2000
		output <<
			"template <> inline PyObject* pyObject(" << prefix
				<< "* o) { return pyObject<" << prefix
				<< " const*>(o); }\n";
	if (ci->isWrappySubclass)
		output <<
			"\n"
			"template <> " << exportTag << "PyObject* WrapPy<"
					<< prefix << ">::wpyNew() const;\n"
			"template <> " << exportTag << "void WrapPy<"
					<< prefix << ">::wpyDelete();\n";
	output <<
		"\n"
		"} // namespace otf\n";
}

static void
dumpWrapClassOutline(std::ostream &output, const ClassInfo *ci)
{
	string prefix(qualify(ci->cd->scope, ci->name.str(), true));
	string ns;
	if (!nameSpace.empty())
		ns = nameSpace + "::";
	string objectName = ns + ci->name.str() + "Object";

	output <<
		"\n"
		"namespace otf {\n"
		"\n"
		"template <> " << exportTag << "PyObject*\n"
		"pyObject(" << prefix;
	if (ci->isWrappySubclass || ci->isAbstractType)
		output << " const*";
	output << " o)\n"
		"{\n";
	if (ci->isWrappySubclass)
		output <<
			"\tif (o == NULL) {\n"
			"\t\tPy_INCREF(Py_None);\n"
			"\t\treturn Py_None;\n"
			"\t}\n"
			"\treturn o->wpyGetObject();\n";
	else if (!ci->isAbstractType) {
		output <<
			"\t// use a PyInstance_New that doesn't call __init__\n"
			"\tPyObject* self = PyInstance_New(" << ns << module
						<< "DummyClass, NULL, NULL);\n"
			"\tif (self == NULL)\n"
			"\t\treturn NULL;\n"
			"\tPyInstanceObject* wco = reinterpret_cast<PyInstanceObject*>(self);\n"
			"\tPy_DECREF(wco->in_class);\n"
			"\tPy_INCREF(" << objectName << "Class);\n"
			"\twco->in_class = reinterpret_cast<PyClassObject*>(" << objectName << "Class);\n"
			"\t// now fill in new instance\n"
			"\t" << prefix << "* inst = new " << prefix << "(o);\n"
			"\tif (" << ns << module << "Debug >= 6)\n"
			"\t\tstd::cerr << \"Allocate " << ci->name
						<< ": \" << inst << '\\n';\n"
			"\tPyObject* tmp = PyCObject_FromVoidPtr(inst, " << ns
						<< ci->name << "Destroy);\n"
			"\tPyDict_SetItem(wco->in_dict, WrapPyString, tmp);\n"
			"\tPy_DECREF(tmp);\n";
		dumpClassAttrInit(output, 1, ci, true);
		output <<
			"\treturn self;\n";
	} else {
		std::cerr << ci->name << " should derive from otf::WrapPy<"
			<< ci->name << "> for better code.\n";
		output <<
			"\tif (o == NULL) {\n"
			"\t\tPy_INCREF(Py_None);\n"
			"\t\treturn Py_None;\n"
			"\t}\n";
		// TODO: really need to descend through derived classes to
		// the first ones that derive from otf::WrapPy<>.
		std::pair<DerivedClasses::iterator, DerivedClasses::iterator>
				bounds = derivedClasses.equal_range(ci->name);
		// TODO: ci->name above should be qualified
		std::cerr << "unimplemented cast from base class that doesn't derived from otf::WrapPy\n";
		for (DerivedClasses::iterator i = bounds.first;
						i != bounds.second; ++i) {
		}
		output <<
			"\tthrow std::runtime_error(\"unknown derived class from " << ci->name << "\");\n";
	}
	output <<
		"}\n";

	if (!ci->isWrappySubclass) {
		output <<
			"\n" /*{*/
			"} // namespace otf\n";
		return;
	}

	output <<
		// create function
		"\n"
		"template <> " << exportTag << "PyObject*\n"
		"WrapPy<" << prefix << ">::wpyNew() const\n"
		"{\n";
	if (ci->isAbstractType)
		output <<
			"\tthrow std::logic_error(\"should be in subclass' wpyNew\");\n";
	else {
		output <<
			"\t// Do a subset of PyInstance_New that doesn't call __init__\n"
			"\tPyObject* self = PyInstance_New(" << ns << module
						<< "DummyClass, NULL, NULL);\n"
			"\tif (self == NULL)\n"
			"\t\treturn NULL;\n"
			"\tPyInstanceObject* wco = reinterpret_cast<PyInstanceObject*>(self);\n"
			"\tPy_DECREF(wco->in_class);\n"
			"\tPy_INCREF(" << objectName << "Class);\n"
			"\twco->in_class = reinterpret_cast<PyClassObject*>("
						<< objectName << "Class);\n"
			"\t// now fill in new instance\n"
			"\twpyAssociate(self);\n"
			"\tif (" << ns << module << "Debug >= 6)\n"
			"\t\tstd::cerr << \"Allocate " << ci->name
				<< ": \" << static_cast<" << prefix
						<< " const*>(this) << '\\n';\n"
			"\tPyObject* tmp = WrapPyProxy_FromWrapPyObj(this, \""
				<< ci->name << "\", &" << ns << module
				<< "Debug);\n"
			"\tPyDict_SetItem(wco->in_dict, WrapPyString, tmp);\n"
			"\tPy_DECREF(tmp);\n";
		dumpClassAttrInit(output, 1, ci, true);
		output <<
			"\treturn self;\n";
	}
	output <<
		"}\n"
		// clear instance function
		"\n"
		"template <> " << exportTag << "void\n"
		"WrapPy<" << prefix << ">::wpyDelete()\n"
		"{\n";
	if (ci->isAbstractType)
		output <<
			"\tthrow std::logic_error(\"should be in subclass' wpyDelete\");\n";
	else
		output <<
			"\t// called by wrapped class destructor\n"
			"\t//assert(pyObj && PyInstance_Check(pyObj));\n"
			"\tPyObject* saveObj = pyObj;\n"
			"\twpyDisassociate();\n"
			"\tPyInstanceObject* inst = reinterpret_cast<PyInstanceObject*>(saveObj);\n"
			"\tPyObject *wpp = PyDict_GetItem(inst->in_dict, WrapPyString);\n"
			"\tif (wpp != NULL)\n"
			"\t\tWrapPyProxy_SetDestroyed(wpp);\n"
#if 0
			// Bad idea because instance dictionary may hold stuff
			// (e.g., a Tk widget) that needs to be cleaned up.
			"\tPyDict_Clear(inst->in_dict);\n"
#else
			"\t// clear cached attributes\n"
			"\tint pos = 0;\n"
			"\tPyObject* key = 0;\n"
			"\twhile (PyDict_Next(inst->in_dict, &pos, &key, NULL)) {\n"
			"\t\tif (!PyString_Check(key))\n"
			"\t\t\tcontinue;\n"
			"\t\tchar const* s = PyString_AS_STRING(key);\n"
			"\t\tif (s[0] == '_' && strncmp(s, \"__cached_\", 9) == 0)\n"
			"\t\t\tPyDict_DelItem(inst->in_dict, key);\n"
			"\t}\n"
#endif
			"\tif (!owner)\n"
			"\t\tPy_DECREF(saveObj);\n";
	output <<
		"}\n";
	output <<
		"\n"
		"} // namespace otf\n";
}

bool
dumpClassHeader(const ClassInfo *ci)
{
	string filename(ci->name.str());
	filename += "Object.h";
	std::auto_ptr<std::ostream> outs(outputStream(filename).release());
	std::ostream &output = *outs.get();

	string define(ci->name.str() + "Object_h");
	if (!nameSpace.empty())
		define = nameSpace + '_' + define;
	output <<
		"#ifndef " << define << "\n"
		"# define " << define << "\n"
		"\n"
		"# include <Python.h>\n"
		"# include <otf/WrapPy.h>\n";
	output << ci->includeFile << "\n";
	if (!nameSpace.empty())
		output <<
			"\n"
			"namespace " << nameSpace << " {\n";
	output <<
		"\n"
		<< exportTag << "extern int " << module << "Debug;\n"
	// object definition
		<< exportTag << "extern PyObject* " << ci->name
							<< "ObjectClass;\n"
		<< exportTag << "extern void " << ci->name
					<< "ClassInit(PyObject *moduleDict);\n"
		"typedef PyInstanceObject " << ci->name << "Object;\n";
	if (!ci->isWrappySubclass && !ci->isAbstractType)
		output << "extern \"C\" " << exportTag << "void " << ci->name
						<< "Destroy(void *cobj);\n";

	if (ci->isPythonClass) {
		// Need function prototypes so attributes may be initialized
		// in subclasses.
		for (AttrVec::const_iterator i = ci->attrs.begin();
						i != ci->attrs.end(); ++i) {
			if (!i->cache)
				continue;	// skip primitive types
			string func(i->get.substr(0, i->get.find('(')));
			output << "extern \"C\" " << exportTag << "PyObject* "
				<< func
				<< "(PyObject*, PyObject*, PyObject*);\n";
		}
	}

	if (!nameSpace.empty())
		output <<
			"\n"
			"} // namespace " << nameSpace << "\n";

	dumpWrapClassInline(output, ci);

	output <<
		"\n"
		"#endif\n";
	// flush output and check return status
	output.flush();
	if (output.good())
		return true;
	std::cerr << programName << ": error writing " << filename << '\n';
	return false;
}

static void
dumpDestroy(std::ostream &output, const ClassInfo *ci)
{
	if (ci->isWrappySubclass)
		return;

	string prefix(qualify(ci->cd->scope, ci->name.str()));
	output <<
		"\n"
		"extern \"C\"\n"
		<< exportTag << "void\n" <<
		ci->name << "Destroy(void *cobj)\n"
		"{\n"
		"\tif (" << module << "Debug >= 6)\n"
		"\t\tstd::cerr << \"Deallocate " << ci->name
					<< ": \" << cobj << '\\n';\n";
	bool publicDestructor = ci->cd->members.has
					(Symbol('~' + ci->name.str()), PUBLIC);
	if (publicDestructor)
		output <<
			"\t" << prefix << "* inst = reinterpret_cast<"
						<< prefix << "*>(cobj);\n"
			"\tdelete inst;\n";
	// TODO: (else) need annotation to tell how to delete C++ object
	output <<
		"}\n";
}

static void
dumpStr(std::ostream &output, const ClassInfo *ci, const string &fname)
{
	string prefix(qualify(ci->cd->scope, ci->name.str(), true));
	output <<
		"\n"
		"extern \"C\"\n"
		<< exportTag << "PyObject*\n" <<
		fname << "(PyObject*, PyObject *args, PyObject *keywds)\n"
		"{\n"
		"\tPyObject* self;\n"
		"\tstatic char* kwlist[] = { \"self\", NULL };\n"
		"\tif (!PyArg_ParseTupleAndKeywords(args, keywds, \"O:str\", kwlist, &self))\n"
		"\t\treturn NULL;\n"
		"\t" << prefix << " const* inst = ";
	if (ci->isWrappySubclass)
		output << "dynamic_cast<" << prefix
				<< " const*>(otf::wrappyObject(self));\n";
	else
		output << "reinterpret_cast<" << prefix
					<< "*>(otf::wrappyVoidObject(self));\n";
	output <<
		"\tif (inst == NULL)\n"
		"\t\treturn NULL;\n";
	output <<
		"\ttry {\n";
	if (!ci->str.empty()
	&& ci->str.find(Symbol("str")) != ci->str.end())
		output <<
			"\t\tstd::string const& str = inst->str();\n"
			"\t\treturn PyString_FromStringAndSize(str.data(), str.size());\n";
	else if (!ci->str.empty()
	&& ci->str.find(Symbol("c_str")) != ci->str.end())
		output <<
			"\t\treturn PyString_FromString(inst->c_str());\n";
	else if (ci->print)
		output <<
			"\t\tstd::ostringstream buf;\n"
			"\t\tbuf << " <<  "*inst;\n"
			"\t\treturn PyString_FromString(buf.str().c_str());\n";
	output <<
		"\t} catch (...) {\n"
		"\t\t" << module << "Error();\n"
		"\t}\n"
		"\treturn NULL;\n"
		"}\n"
		"\n"
		"/*const*/ char " << fname << "_doc[] = \n\"" << prefix
							<< " -> string\";\n";
}

#if 0
static void
dumpDelAttrO(std::ostream &output, const ClassInfo *ci, const string &fname)
{
	output <<
		"\n"
		"extern \"C\"\n"
		<< exportTag << "PyObject*\n" <<
		fname << "(PyObject*, PyObject* args)\n"
		"{\n"
		"\tPyObject* self;\n"
		"\tPyObject* attr;\n"
		"\tif (!PyArg_ParseTuple(args, \"OO\", &self, &attr))\n"
		"\t\treturn NULL;\n"
		"\tPyInstanceObject *wco = reinterpret_cast<PyInstanceObject*>(self);\n"
		"\tPyObject *dict = PyDict_GetItemString(wco->in_class->cl_dict, \"__wrappyReadonly__\");\n"
		"\tif (dict != NULL && PyDict_GetItem(dict, attr)) {\n"
		"\t\tPyErr_SetString(PyExc_AttributeError, \"can not delete read-only attribute\");\n"
		"\t\treturn NULL;\n"
		"\t}\n";
		"\tif (PyDict_SetItem(wco->in_dict, attr, NULL) == -1)\n"
		"\t\treturn NULL;\n"
		"\tPy_INCREF(Py_None);\n"
		"\treturn Py_None;\n"
		"}\n";
	output <<
		"\n"
		"PyMethodDef " << ci->name << "MD_" << fname << " = {\n"
		"\t\"" << fname << "\",\n"
		"\t(PyCFunction) " << ci->name << fname << ",\n"
		"\tMETH_VARARGS, \"delete attribute\"\n"
		"};\n";
}
#endif


static void
dumpSetAttrO(std::ostream &output, const ClassInfo *ci, const string &fname)
{
	output <<
		"\n"
		"extern \"C\"\n"
		<< exportTag << "PyObject*\n"
		<< ci->name << fname
			<< "(PyObject*, PyObject* args)\n"
		"{\n"
		"\tPyObject* self;\n"
		"\tPyObject* attr;\n"
		"\tPyObject* value;\n"
		"\tif (!PyArg_ParseTuple(args, \"OOO\", &self, &attr, &value))\n"
		"\t\treturn NULL;\n"
		"\tPyInstanceObject* wco = reinterpret_cast<PyInstanceObject*>(self);\n";
	output <<
		"\tif (PyDict_GetItem(roDict, attr)) {\n"
		"\t\tPyErr_SetString(PyExc_AttributeError, \"can not set read-only attribute\");\n"
		"\t\treturn NULL;\n"
		"\t}\n";
	output <<
		"\t// hardcoded attributes\n"
		"\tPyObject* tmp = PyDict_GetItem(setDict, attr);\n"
		"\tif (tmp) {\n"
		"\t\totf::WrapPyArgTuple arg;\n"
		"\t\tPyCFunctionWithKeywords func = otf::WrapPyPCFWK_AsPCFWK(tmp);\n"
		"\t\treturn (*func)(self, arg(self, value), NULL);\n"
		"\t}\n";
	output <<
		"\tPyClassObject* klass = reinterpret_cast<PyClassObject*>("
						<< ci->name << "ObjectClass);\n"
		"\tint numBases = PyTuple_Size(klass->cl_bases);\n"
		"\tfor (int i = 0; i < numBases; ++i) {\n"
		"\t\tPyObject* baseObj = PyTuple_GetItem(klass->cl_bases, i);\n"
		"\t\tif (!PyClass_Check(baseObj))\n"
		"\t\t\tcontinue;\n"
		"\t\tPyClassObject* base = reinterpret_cast<PyClassObject*>(baseObj);\n"
		"\t\tif (base->cl_setattr == NULL)\n"
		"\t\t\tcontinue;\n"
		"\t\tPyObject* tmp = PyEval_CallObject(base->cl_setattr, args);\n"
		"\t\tif (tmp != NULL\n"
		"\t\t|| !PyErr_ExceptionMatches(PyExc_AttributeError))\n"
		"\t\t\treturn tmp;\n"
		"\t\tPyErr_Clear();\n"
		"\t}\n";
	output <<
		"\tif (" << module << "Debug >= 7)\n"
		"\t\tstd::cerr << \"Setting " << ci->name
			<< ".\" << PyString_AsString(attr) << \"\\n\";\n"
		"\tif (PyDict_SetItem(wco->in_dict, attr, value) == -1)\n"
		"\t\treturn NULL;\n"
		"\tPy_INCREF(Py_None);\n"
		"\treturn Py_None;\n"
		"}\n";
	output <<
		"\n"
		"PyMethodDef " << ci->name << "MD_" << fname << " = {\n"
		"\t\"" << fname << "\",\n"
		"\t(PyCFunction) " << ci->name << fname << ",\n"
		"\tMETH_VARARGS, \"set attribute\"\n"
		"};\n";
}

static void
dumpGetAttrO(std::ostream &output, const ClassInfo *ci, const string &fname)
{
	output <<
		"\n"
		"extern \"C\"\n"
		<< exportTag << "PyObject*\n"
		<< ci->name << fname << "(PyObject*, PyObject* args)\n"
		"{\n"
		"\tPyObject* self;\n"
		"\tPyObject* attr;\n"
		"\tif (!PyArg_ParseTuple(args, \"OO\", &self, &attr))\n"
		"\t\treturn NULL;\n"
		"\tchar const* aname = PyString_AsString(attr);\n"
		"\tif (aname == NULL)\n"
		"\t\treturn NULL;\n";
	AttrVec::size_type size = ci->attrs.size();
	output <<
		"\tif (aname[0] == '_' && aname[1] == '_') {\n";
	output <<
		"\t\tif (strcmp(&aname[2], \"members__\") == 0) {\n"
		"\t\t\tPyObject* result = PyList_New(" << size
							<< ");\n";
	int k = 0;
	for (AttrVec::const_iterator i = ci->attrs.begin();
					i != ci->attrs.end(); ++i, ++k)
		output << "\t\t\tPyList_SetItem(result, " << k
			<< ", PyString_FromString(\"" << i->name
			<< "\"));\n";
	output <<
		"\t\t\treturn result;\n"
		"\t\t}\n"
		"\t}\n";
	output <<
		//"\tPyInstanceObject* wco = reinterpret_cast<PyInstanceObject*>(self);\n"
		"\tif (PyDict_GetItem(woDict, attr)) {\n"
		"\t\tPyErr_SetString(PyExc_AttributeError, \"can not get write-only attribute\");\n"
		"\t\treturn NULL;\n"
		"\t}\n";
	output <<
		"\t// hardcoded attributes\n"
		"\tPyObject* tmp = PyDict_GetItem(getDict, attr);\n"
		"\tif (tmp) {\n"
		"\t\totf::WrapPyArgTuple arg;\n"
		"\t\tPyCFunctionWithKeywords func = otf::WrapPyPCFWK_AsPCFWK(tmp);\n"
		"\t\treturn (*func)(NULL, arg(self), NULL);\n"
		"\t}\n";
	output <<
		"\tPyClassObject* klass = reinterpret_cast<PyClassObject*>("
						<< ci->name << "ObjectClass);\n"
		"\tint numBases = PyTuple_Size(klass->cl_bases);\n"
		"\tfor (int i = 0; i < numBases; ++i) {\n"
		"\t\tPyObject* baseObj = PyTuple_GetItem(klass->cl_bases, i);\n"
		"\t\tif (!PyClass_Check(baseObj))\n"
		"\t\t\tcontinue;\n"
		"\t\tPyClassObject* base = reinterpret_cast<PyClassObject*>(baseObj);\n"
		"\t\tif (base->cl_getattr == NULL)\n"
		"\t\t\tcontinue;\n"
		"\t\tPyObject* tmp = PyEval_CallObject(base->cl_getattr, args);\n"
		"\t\tif (tmp != NULL\n"
		"\t\t|| !PyErr_ExceptionMatches(PyExc_AttributeError))\n"
		"\t\t\treturn tmp;\n"
		"\t\tPyErr_Clear();\n"
		"\t}\n";
	output <<
		"\tPyErr_SetObject(PyExc_AttributeError, attr);\n"
		"\treturn NULL;\n";
	output <<
		"}\n";
	output <<
		"\n"
		"PyMethodDef " << ci->name << "MD_" << fname << " = {\n"
		"\t\"" << fname << "\",\n"
		"\t(PyCFunction) " << ci->name << fname << ",\n"
		"\tMETH_VARARGS, \"get attribute\"\n"
		"};\n";
}

static void
dumpVariables(std::ostream &output, const ClassInfo *ci)
{
	string prefix(qualify(ci->cd->scope, ci->name.str()));
	// These are functions for public class variables.
	for (AttrVec::const_iterator i = ci->variables.begin();
						i != ci->variables.end(); ++i) {
		if (!i->get.empty()) {
			output <<
				"\n"
				"extern \"C\"\n"
				<< exportTag << "PyObject*\n" <<
				ci->name << "Get" << i->name << "(PyObject*, PyObject* args, PyObject* keywds)\n"
				"{\n"
				"\tPyObject* self;\n"
				"\tstatic char* kwlist[] = { \"self\", NULL };\n"
				"\tif (!PyArg_ParseTupleAndKeywords(args, keywds, \"O:get\", kwlist, &self))\n"
				"\t\treturn NULL;\n"
				"\t" << prefix << " const* inst = ";
			if (ci->isWrappySubclass)
				output << "dynamic_cast<" << prefix
					<< " const*>(otf::wrappyObject(self));\n";
			else
				output << "reinterpret_cast<" << prefix
					<< "*>(otf::wrappyVoidObject(self));\n";
			output <<
				"\tif (inst == NULL)\n"
				"\t\treturn NULL;\n"
				"\tPyObject *o = " << i->get << ";\n";
			if (i->cache)
				output <<
					"\t// cache attribute\n"
					"\tPyInstanceObject* wco = reinterpret_cast<PyInstanceObject*>(self);\n"
					"\tstatic PyObject *key = NULL;\n"
					"\tif (key == NULL)\n"
					"\t\tkey = PyString_FromString(\"__cached_" << i->name << "__\");\n"
					"\tPyDict_SetItem(wco->in_dict, key, o);\n";
			output <<
				"\treturn o;\n"
				"}\n";
		}
		if (i->set.empty())
			continue;
		output <<
			"\n"
			"extern \"C\"\n"
			<< exportTag << "PyObject*\n" <<
			ci->name << "Set" << i->name << "(PyObject*, PyObject* args, PyObject* keywds)\n"
			"{\n"
			"\tPyObject* self;\n"
			"\tstatic char* kwlist[] = { \"self\", \"arg\", NULL };\n"
			<< i->set << ";\n";
		if (i->cache)
			output <<
				"\t// cache attribute\n"
				"\tPyInstanceObject* wco = reinterpret_cast<PyInstanceObject*>(self);\n"
				"\tstatic PyObject *key = NULL;\n"
				"\tif (key == NULL)\n"
				"\t\tkey = PyString_FromString(\"__cached_" << i->name << "__\");\n"
				"\tif (PyDict_SetItem(wco->in_dict, key, ptArg) == -1)\n"
				"\t\treturn NULL;\n";
		output <<
			"\tPy_INCREF(Py_None);\n"
			"\treturn Py_None;\n"
			"}\n";
	}
}

static void
dumpMethods(std::ostream &output, const ClassInfo *ci, FuncMap *cf)
{
	dumpDestroy(output, ci);

	// first, create all of the method definitions
	MethodMap::const_iterator next;
	for (MethodMap::const_iterator i = ci->attrMethods.begin();
					i != ci->attrMethods.end(); i = next) {
		next = ci->attrMethods.upper_bound(i->first);
		(void) dumpMethod(output, ci, i, next, true, string(),
								string(), true);
		// These methods are accessed by [GS]etAttrO directly
	}

	for (MethodMap::const_iterator i = ci->methods.begin();
					i != ci->methods.end(); i = next) {
		next = ci->methods.upper_bound(i->first);
		string proto = dumpMethod(output, ci, i, next, true);
		proto += "\n" + i->first.str() + " documentation.";
		output <<
			"\n"
			"const char " << ci->name << '_' << i->first
				<< "_doc[] = \n\"" << stringize(proto)
				<< "\";\n";
		cf->insert(FuncMap::value_type(i->first,
					ci->name.str() + "_" + i->first.str()));
	}

	if (!ci->isAbstractType && !ci->constructors.empty()) {
		string fname = ci->name.str() + "_" + ci->name.str();
		string proto = dumpMethod(output, ci, ci->constructors.begin(),
						ci->constructors.end(), true);
		proto += "\n" + ci->name.str() + " constructor documentation.";
		output <<
			"\n"
			"const char " << fname << "_doc[] = \n\""
						<< stringize(proto) << "\";\n";
		cf->insert(FuncMap::value_type(Symbol("__init__"), fname));
	}

	output <<
		"\n"
		"PyMethodDef " << ci->name << "Methods[] = {\n";
	for (FuncMap::iterator i = cf->begin(); i != cf->end(); ++i) {
		Symbol pyname((*i).first);
		const string &fname((*i).second);
		output <<
			"\t{\n"
			"\t\t\"" << pyname << "\", (PyCFunction) " << fname
								<< ",\n"
			"\t\tMETH_KEYWORDS, const_cast<char*>(" << fname
								<< "_doc)\n"
			"\t},\n";
	}
	output <<
		"\t{ NULL, NULL }\n"
		"};\n";
}

static void
dumpCompare(std::ostream &output, const ClassInfo *ci, const MethodMap &mm,
							const string &fname)
{
	// TODO: support comparisons with multiple types
	string prefix(qualify(ci->cd->scope, ci->name.str(), true));
	output <<
		"\n"
		"extern \"C\"\n"
		<< exportTag << "PyObject*\n" <<
		fname << "(PyObject*, PyObject* args, PyObject* keywds)\n"
		"{\n"
		"\tPyObject* self;\n"
		"\tPyObject* right;\n"
		"\tstatic char* kwlist[] = { \"self\", \"right\", NULL };\n"
		"\tif (!PyArg_ParseTupleAndKeywords(args, keywds, \"OO:__cmp__\", kwlist, &self, &right))\n"
		"\t\treturn NULL;\n"
		"\t" << prefix << " const* lInst = ";
	if (ci->isWrappySubclass)
		output << "dynamic_cast<" << prefix
				<< " const*>(otf::wrappyObject(self));\n";
	else
		output << "reinterpret_cast<" << prefix
					<< "*>(otf::wrappyVoidObject(self));\n";
	output <<
		"\tif (lInst == NULL)\n"
		"\t\treturn NULL; // should never happen\n"
		"\ttry {\n";
	int count = 0;
	for (MethodMap::const_iterator i = mm.begin(); i != mm.end();
								++i, ++count) {
		const FuncDecl *fd = i->second;
		int memberFunc = dynamic_cast<const ClassDecl *>(fd->scope)
									? 0 : 1;
		CvtType arg(fd->scope, fd->args[memberFunc].type);
		string getValue(arg.pyToCpp("right"));
		string star;
		if (getValue[0] != '*')
			// TODO: fix this hack 
			star = "*";
		output << "\t\t" << qualify(fd->scope, arg.cppType())
				<< " right" << count << " = " << getValue
				<< ";\n"
			"\t\tif (!PyErr_Occurred()) {\n"
			"\t\t\tif (*lInst < " << star << "right" << count << ")\n"
			"\t\t\t\treturn PyInt_FromLong(-1);\n"
			"\t\t\telse if (" << star << "right" << count << " < *lInst)\n"
			"\t\t\t\treturn PyInt_FromLong(1);\n"
			"\t\t\treturn PyInt_FromLong(0);\n"
			"\t\t}\n"
			"\t\tPyErr_Clear();\n";
	}
	output <<
		"\t\tPyErr_SetString(PyExc_TypeError, \"uncomparable types\");\n"
		"\t} catch (...) {\n"
		"\t\t" << module << "Error();\n"
		"\t}\n"
		"\treturn NULL;\n"
		"}\n";
	string doc = ci->name.str() + " comparison function";
	output <<
		"\n"
		"/*const*/ char " << fname << "_doc[] = \n\"" << stringize(doc)
								<< "\";\n";
}

static void
dumpSpecial(std::ostream &output, const ClassInfo *ci,
	const MethodMap &mm, const char *mname, const char *pyname,
	FuncMap *cf)
{
	string fname = ci->name.str() + mname;
	string doc = dumpMethod(output, ci, mm.begin(), mm.end(), true, fname,
									pyname);
	output <<
		"\n"
		"/*const*/ char " << fname << "_doc[] = \n\"" << stringize(doc)
								<< "\";\n";
	cf->insert(FuncMap::value_type(Symbol(pyname), fname));
}

static bool
checkBaseClassesForFunc(const ClassInfo *ci, Symbol id)
{
	for (CIList::const_iterator i = ci->baseClasses.begin();
					i != ci->baseClasses.end(); ++i) {
		Decl *idDecl = (*i)->cd->members.unique(id);
		if (idDecl != NULL
		&& idDecl->dtype == Decl::FUNCTION)
			return true;
		if (checkBaseClassesForFunc(*i, id))
			return true;
	}
	return false;
}

static void
dumpHash(std::ostream &output, const ClassInfo *ci, FuncMap *cf)
{
	if (!ci->hash.empty()) {
		string fname = ci->name.str() + "_hash";
		cf->insert(FuncMap::value_type(Symbol("__hash__"), fname));
		return;
	}
	if (checkBaseClassesForFunc(ci, Symbol("hash")))
		return;
	string fname = ci->name.str() + "Hash";
	output <<
		"\n"
		"extern \"C\"\n"
		<< exportTag << "PyObject*\n" <<
		fname << "(PyObject*, PyObject* args, PyObject* keywds)\n"
		"{\n"
		"\tstatic char* kwlist[] = { \"self\", NULL };\n"
		"\tPyObject* self;\n"
		"\tif (!PyArg_ParseTupleAndKeywords(args, keywds, \"O:__hash__\", kwlist, &self))\n"
		"\t\treturn NULL;\n"
		"\treturn PyInt_FromLong(_Py_HashPointer(self));\n"
		"}\n"
		"\n"
		"const char " << fname << "_doc[] = \"hash instance\";\n";
	cf->insert(FuncMap::value_type(Symbol("__hash__"), fname));
}

static void
dumpSpecialMethods(std::ostream &output, const ClassInfo *ci, FuncMap *cf)
{
	//
	// find member functions that act like special Python type functions
	//

	if (!ci->compare.empty()) {
		string fname = ci->name.str() + "Compare";
		dumpCompare(output, ci, ci->compare, fname);
		cf->insert(FuncMap::value_type(Symbol("__cmp__"), fname));
	}
	if (!ci->call.empty())
		dumpSpecial(output, ci, ci->call, "Call", "__call__", cf);

	dumpHash(output, ci, cf);

	// string
	if (!ci->str.empty() || ci->print) {
		string fname = ci->name.str() + "Str";
		dumpStr(output, ci, fname);
		cf->insert(FuncMap::value_type(Symbol("__str__"), fname));
	}

	// number methods
	if (!ci->nb_add.empty())
		dumpSpecial(output, ci, ci->nb_add, "Add", "__add__", cf);
	if (!ci->nb_radd.empty())
		dumpSpecial(output, ci, ci->nb_radd, "RAdd", "__radd__", cf);
	if (!ci->nb_subtract.empty())
		dumpSpecial(output, ci, ci->nb_subtract, "Subtract", "__sub__", cf);
	if (!ci->nb_rsubtract.empty())
		dumpSpecial(output, ci, ci->nb_rsubtract, "RSubtract", "__rsub__", cf);
	if (!ci->nb_multiply.empty())
		dumpSpecial(output, ci, ci->nb_multiply, "Multiply", "__mul__", cf);
	if (!ci->nb_rmultiply.empty())
		dumpSpecial(output, ci, ci->nb_rmultiply, "RMultiply", "__rmul__", cf);
	if (!ci->nb_divide.empty())
		dumpSpecial(output, ci, ci->nb_divide, "Divide", "__div__", cf);
	if (!ci->nb_rdivide.empty())
		dumpSpecial(output, ci, ci->nb_rdivide, "RDivide", "__rdiv__", cf);
	if (!ci->nb_remainder.empty())
		dumpSpecial(output, ci, ci->nb_remainder, "Remainder", "__mod__", cf);
	if (!ci->nb_rremainder.empty())
		dumpSpecial(output, ci, ci->nb_rremainder, "RRemainder", "__rmod__", cf);
	if (!ci->nb_divmod.empty())
		dumpSpecial(output, ci, ci->nb_divmod, "Divmod", "__div__", cf);
	if (!ci->nb_rdivmod.empty())
		dumpSpecial(output, ci, ci->nb_rdivmod, "RDivmod", "__rdiv__", cf);
	if (!ci->nb_power.empty())
		dumpSpecial(output, ci, ci->nb_power, "Power", "__pow__", cf);
	if (!ci->nb_rpower.empty())
		dumpSpecial(output, ci, ci->nb_rpower, "RPower", "__rpow__", cf);
	if (!ci->nb_negative.empty())
		dumpSpecial(output, ci, ci->nb_negative, "Negative", "__neg__", cf);
	if (!ci->nb_positive.empty())
		dumpSpecial(output, ci, ci->nb_positive, "Positive", "__pos__", cf);
	if (!ci->nb_absolute.empty())
		dumpSpecial(output, ci, ci->nb_absolute, "Absolute", "__abs__", cf);
	if (!ci->nb_nonzero.empty())
		dumpSpecial(output, ci, ci->nb_nonzero, "Nonzero", "__nonzero__", cf);
	if (!ci->nb_invert.empty())
		dumpSpecial(output, ci, ci->nb_invert, "Invert", "__invert__", cf);
	if (!ci->nb_lshift.empty())
		dumpSpecial(output, ci, ci->nb_lshift, "Lshift", "__lshift__", cf);
	if (!ci->nb_rlshift.empty())
		dumpSpecial(output, ci, ci->nb_rlshift, "RLshift", "__rlshift__", cf);
	if (!ci->nb_rshift.empty())
		dumpSpecial(output, ci, ci->nb_rshift, "Rshift", "__rshift__", cf);
	if (!ci->nb_rrshift.empty())
		dumpSpecial(output, ci, ci->nb_rrshift, "RRshift", "__rrshift__", cf);
	if (!ci->nb_and.empty())
		dumpSpecial(output, ci, ci->nb_and, "And", "__and__", cf);
	if (!ci->nb_rand.empty())
		dumpSpecial(output, ci, ci->nb_rand, "RAnd", "__rand__", cf);
	if (!ci->nb_xor.empty())
		dumpSpecial(output, ci, ci->nb_xor, "Xor", "__xor__", cf);
	if (!ci->nb_rxor.empty())
		dumpSpecial(output, ci, ci->nb_rxor, "RXor", "__rxor__", cf);
	if (!ci->nb_or.empty())
		dumpSpecial(output, ci, ci->nb_or, "Oor", "__or__", cf);
	if (!ci->nb_ror.empty())
		dumpSpecial(output, ci, ci->nb_ror, "ROr", "__ror__", cf);
#if 0
	if (ci->nb_coerce) {
		string fname = ci->name.str() + "Coerce";
		dumpCoercionFunc(output, fname);
		cf->insert(FuncMap::value_type(Symbol("__coerce__"), fname));
	}
#endif
	if (!ci->nb_int.empty())
		dumpSpecial(output, ci, ci->nb_int, "Int", "__int__", cf);
	if (!ci->nb_long.empty())
		dumpSpecial(output, ci, ci->nb_long, "Long", "__long__", cf);
	if (!ci->nb_float.empty())
		dumpSpecial(output, ci, ci->nb_float, "Float", "__float__", cf);
	if (!ci->nb_oct.empty())
		dumpSpecial(output, ci, ci->nb_oct, "Oct", "__oct__", cf);
	if (!ci->nb_hex.empty())
		dumpSpecial(output, ci, ci->nb_hex, "Hex", "__hex__", cf);

	// sequence methods
	if (!ci->sq_length.empty())
		dumpSpecial(output, ci, ci->sq_length, "Length", "__len__", cf);
	// no support instances for these methods
	if (!ci->sq_concat.empty())
		dumpSpecial(output, ci, ci->sq_concat, "Concat", "__add__", cf);
	if (!ci->sq_repeat.empty())
		dumpSpecial(output, ci, ci->sq_repeat, "Repeat", "__mul__", cf);
	if (!ci->sq_item.empty())
		dumpSpecial(output, ci, ci->sq_item, "GetItem", "__getitem__", cf);
	if (!ci->sq_slice.empty())
		dumpSpecial(output, ci, ci->sq_slice, "GetSlice", "__getslice__", cf);
	if (!ci->sq_ass_item.empty())
		dumpSpecial(output, ci, ci->sq_ass_item, "SetItem", "__setitem__", cf);
	if (!ci->sq_ass_slice.empty())
		dumpSpecial(output, ci, ci->sq_ass_slice, "SetSlice", "__setslice__", cf);

	// map methods
	if (!ci->mp_length.empty())
		dumpSpecial(output, ci, ci->mp_length, "Length", "__len__", cf);
	if (!ci->mp_subscript.empty())
		dumpSpecial(output, ci, ci->mp_subscript, "GetItem", "__getitem__", cf);
	if (!ci->mp_ass_subscript.empty())
		dumpSpecial(output, ci, ci->mp_ass_subscript, "SetItem", "__setitem__", cf);

#if 0
	// buffer methods
	if (!ci->bf_getreadbuffer.empty())
		dumpSpecial(output, ci, ci->bf_getreadbuffer, "?", "____", cf);
	if (!ci->bf_getwritebuffer.empty())
		dumpSpecial(output, ci, ci->bf_getwritebuffer, "?", "____", cf);
	if (!ci->bf_getsegcount.empty())
		dumpSpecial(output, ci, ci->bf_getsegcount, "?", "____", cf);
#endif
}

static void
dumpClassAttributes(std::ostream &output, const ClassInfo *ci)
{
	for (AttrVec::const_iterator i = ci->attrs.begin();
						i != ci->attrs.end(); ++i) {
		if (i->set.empty()) {
			output <<
				"\tsetItem(roDict, \"" << i->name
						<< "\", PyTuple_New(0));\n";
		} else {
			string::size_type pos = i->set.find('(');
			output <<
				"\tsetItem(setDict, \"" << i->name
					<< "\", otf::WrapPyPCFWK_FromPCFWK("
					<< i->set.substr(0, pos)
					<< "));\n";
		}
		if (i->get.empty()) {
			output <<
				"\tsetItem(woDict, \"" << i->name
						<< "\", PyTuple_New(0));\n";
		} else {
			string::size_type pos = i->get.find('(');
			output <<
				"\tsetItem(getDict, \"" << i->name
					<< "\", otf::WrapPyPCFWK_FromPCFWK("
					<< i->get.substr(0, pos)
					<< "));\n";
		}
	}
	output <<
		"\n"
		"\tsetItem(classDict, \"__setattr__\", PyCFunction_New(&" << ci->name << "MD_SetAttrO, NULL));\n"
		"\tsetItem(classDict, \"__getattr__\", PyCFunction_New(&" << ci->name << "MD_GetAttrO, NULL));\n";
#if 0
		"\tsetItem(classDict, \"__delattr__\", PyCFunction_New(&" << ci->name << "MD_DelAttrO, NULL));\n";
#endif
}

bool
dumpClassCode(const ClassInfo *ci)
{
	FuncMap classFuncs;

	string filename(ci->name.str());
	filename += "Object.cpp";
	std::auto_ptr<std::ostream> outs(outputStream(filename).release());
	std::ostream &output = *outs.get();

	output << "#include \"" << module << ".h\"\n";
	if (ci->str.empty() && ci->print)
		output << "#include <sstream>\n";
	if (ci->isWrappySubclass)
		output << "#include <otf/WrapPyProxy.h>\n";
	output << "#include <otf/WrapPyPCFWK.h>\n";

	if (!nameSpace.empty())
		output <<
			"\n"
			"namespace " << nameSpace << " {\n";
	string prefix(qualify(ci->cd->scope, ci->name.str()));
	output <<
		"\n"
		"PyObject* " << ci->name << "ObjectClass;\n"
		"\n"
		"static void\n"
		"mustBeInstance(PyObject *o)\n"
		"{\n"
		"\tif (otf::WrapPyType<" << prefix << ">::check(o, false))\n"
		"\t\treturn;\n"
		"\tthrow std::runtime_error(\"not an instance of class "
			<< prefix << "\");\n"
		"}\n";

	dumpSpecialMethods(output, ci, &classFuncs);

	dumpVariables(output, ci);

	dumpMethods(output, ci, &classFuncs);

	if (!ci->attrs.empty()) {
		output <<
			"\n"
			"static PyObject* setDict = PyDict_New();\n"
			"static PyObject* roDict = PyDict_New();\n"
			"static PyObject* getDict = PyDict_New();\n"
			"static PyObject* woDict = PyDict_New();\n";
		string setattro("SetAttrO");
		dumpSetAttrO(output, ci, setattro);
		string getattro("GetAttrO");
		dumpGetAttrO(output, ci, getattro);
#if 0
		string delattro("DelAttrO");
		dumpDelAttrO(output, ci, delattro);
#endif
	}

	output <<
		"\n"
		"/*const*/ char " << ci->name << "_doc[] = \"" << ci->name
						<< " documentation\";\n";
	output <<
		"\n"
		"static void\n"
		"setItem(PyObject *dict, const char *key, PyObject *item)\n"
		"{\n"
		"\t// decrements reference count of item iff successful\n"
		"\tif (item == NULL\n"
		"\t|| PyDict_SetItemString(dict, const_cast<char*>(key), item) != 0)\n"
		"\t\tthrow std::runtime_error(\"PyDict_SetItemString failed\");\n"
		"\tPy_DECREF(item);\n"
		"}\n";
	output <<
		"\n"
		"void\n"
		<< ci->name << "ClassInit(PyObject *moduleDict)\n"
		"{\n"
		"\tPyObject* name = PyString_FromString(\"" << ci->name
								<< "\");\n"
		"\tPyObject* classDict = PyDict_New();\n"
		"\tsetItem(classDict, \"__doc__\", PyString_FromString("
						<< ci->name << "_doc));\n";
	if (!ci->attrs.empty())
		dumpClassAttributes(output, ci);

	if (ci->baseClasses.empty())
		output <<
			"\n"
			"\tPyObject *bases = NULL;\n";
	else {
		output <<
			"\n"
			"\tPyObject *bases = PyTuple_New("
					<< ci->baseClasses.size() << ");\n";
		int j = 0;
		for (CIList::const_iterator i = ci->baseClasses.begin();
					i != ci->baseClasses.end(); ++i, ++j) {
			Symbol id = (*i)->name;
			output <<
				"\tPy_INCREF(" << id << "ObjectClass);\n"
				"\tPyTuple_SetItem(bases, " << j << ", " << id
							<< "ObjectClass);\n";
		}
	}
	output <<
		"\t" << ci->name
		<< "ObjectClass = PyClass_New(bases, classDict, name);\n";

	// TODO: could make constants immutable by placing them in "attributes"
	for (AttrVec::const_iterator i = ci->constants.begin();
						i != ci->constants.end(); ++i)
		output <<
			"\tsetItem(classDict, \"" << i->name << "\", "
							<< i->get << ");\n";
	//
	output <<
		"\tfor (PyMethodDef *ml = " << ci->name
				<< "Methods; ml->ml_name != NULL; ++ml) {\n"
		"\t\tPyObject* func = PyCFunction_New(ml, NULL);\n"
		"\t\tsetItem(classDict, ml->ml_name, PyMethod_New(func, NULL, "
					<< ci->name << "ObjectClass));\n"
		"\t\tPy_DECREF(func);\n"
		"\t}\n"
		"\tPyDict_SetItem(moduleDict, name, " << ci->name
							<< "ObjectClass);\n";
	output <<
		"\t//Py_DECREF(" << ci->name << "ObjectClass); keep around\n"
		"\t//Py_DECREF(name);\n"
		"\tPy_DECREF(classDict);\n"
		"}\n";

	if (!nameSpace.empty())
		output <<
			"\n"
			"} // namespace " << nameSpace << "\n";

	dumpWrapClassOutline(output, ci);

	// flush output and check return status
	output.flush();
	if (output.good())
		return true;
	std::cerr << programName << ": error writing " << filename << '\n';
	return false;
}

void
fillAttrs(AttrVec *attrs, const ClassInfo *ci)
{
	for (CIList::const_iterator i = ci->baseClasses.begin();
					i != ci->baseClasses.end(); ++i)
		fillAttrs(attrs, *i);
	for (AttrVec::const_iterator i = ci->attrs.begin();
						i != ci->attrs.end(); ++i)
		attrs->push_back(*i);
}

void
dumpClassAttrInit(std::ostream &output, int indent, const ClassInfo *ci,
							bool needNamespace)
{
	// TODO: do all attributes, esp. subclass attributes
	string bi(tab(indent));		// base indent
	string ns;
	if (needNamespace && !nameSpace.empty())
		ns = nameSpace + "::";
	bool first = true;
	AttrVec attrs;
	fillAttrs(&attrs, ci);
	for (AttrVec::const_iterator i = attrs.begin(); i != attrs.end(); ++i) {
		if (!i->cache)
			continue;	// skip primitive types
		if (first) {
			first = false;
			output << bi << "// initialize cached attributes\n"
				<< bi << "otf::WrapPyArgTuple arg;\n"
				<< bi << "PyObject* attrTmp;\n";
		}
		string func(i->get.substr(0, i->get.find('(')));
		output << bi << "attrTmp = " << ns << func
					<< "(NULL, arg(self), NULL);\n";
		output << bi << "Py_XDECREF(attrTmp);\n";
	}
	if (!first)
		output << bi << "PyErr_Clear();\n";
}
