// Copyright (c) 1998-2000 The Regents of the University of California.
// All rights reserved.
//
// Redistribution and use in source and binary forms are permitted
// provided that the above copyright notice and this paragraph are
// duplicated in all such forms and that any documentation,
// distribution and/or use acknowledge that the software was developed
// by the Computer Graphics Laboratory, University of California,
// San Francisco.  The name of the University may not be used to
// endorse or promote products derived from this software without
// specific prior written permission.
// 
// THIS SOFTWARE IS PROVIDED ``AS IS'' AND WITHOUT ANY EXPRESS OR
// IMPLIED WARRANTIES, INCLUDING, WITHOUT LIMITATION, THE IMPLIED
// WARRANTIES OF MERCHANTIBILITY AND FITNESS FOR A PARTICULAR PURPOSE.
// IN NO EVENT SHALL THE REGENTS OF THE UNIVERSITY OF CALIFORNIA BE
// LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY,
// OR CONSEQUENTIAL DAMAGES ARISING OUT OF THE USE OF THIS SOFTWARE.

// $Id: compute.cpp,v 1.33 2003/06/18 01:03:06 gregc Exp $

#include <set>
#include <locale>
#include <algorithm>
#include <stdarg.h>
#include <otf/Reg.h>
#include "common.h"
#include "compute.h"
#include "CvtType.h"
#ifdef OTF_NO_LOCALE
# include <ctype.h>
#endif

// We use Python's copy semantics (increment the reference count),
// instead of C++'s -- so remove copy constructors from class declarations
// because we can't deal with them yet.
#define REMOVE_COPY_CONSTRUCTORS

using std::string;
using otf::Symbol;

DeclList	globalDecls;
NamespaceDecl	*moduleScope = NULL;

DerivedClasses	derivedClasses;
SymCDMap	classDeclsMap;
CDCIMap		classInfoMap;

static otf::Reg	btStatic("static[[:>:]][[:space:]]*", otf::Reg::EXTENDED);

static bool
isWrapPySubclass(const ClassDecl *cd)
{
	// A class is wrapped if it is subclassed from WrapPy<name>
	string pattern("[[:<:]]WrapPy[[:space:]]*<[[:space:]]*");
	pattern += cd->name.str() + "[[:space:]]*>";
	otf::Reg containsClass(pattern.c_str());
	for (DeclList::const_iterator i = cd->baseClasses.begin();
					i != cd->baseClasses.end(); ++i) {
		if (string::npos != containsClass.find((*i)->text))
			return true;
	}
	return false;
}

static void
computeDerivedClasses(const ClassDecl *cd)
{
	Symbol derived(qualify(cd->scope, cd->name.str()));
	for (DeclList::const_iterator i = cd->baseClasses.begin();
					i != cd->baseClasses.end(); ++i) {
		Symbol base(qualify(cd->scope, (*i)->tag.str()));
		derivedClasses.insert(std::make_pair(base, derived));
	}
}

// A "get" attribute function is a function that has a non-void return
// value, possibily through output arguments, and has no input arguments.

inline const FuncDecl *
findGetFunc(const ClassDecl *cd, Symbol fname)
{
	if (cd == NULL)
		return NULL;
	const Decl *d = cd->members.unique(fname);
	if (d == NULL)
		return NULL;
	const FuncDecl *fd = dynamic_cast<const FuncDecl *>(d);
	if (fd == NULL
	|| (fd->args.size() == 0 && fd->returnType == "void"))
		return NULL;
	for (ArgList::const_iterator i = fd->args.begin(); i != fd->args.end();
									++i)
		if (i->in)
			return NULL;
	return fd;
}

static const FuncDecl *
findGetFunc(const ClassInfo *ci, Symbol fname)
{
	if (ci == NULL)
		return NULL;
	const FuncDecl *result = findGetFunc(ci->cd, fname);
	if (result != NULL)
		return result;
	// recurse on base classes
	for (CIList::const_iterator i = ci->baseClasses.begin();
					i != ci->baseClasses.end(); ++i) {
		const ClassInfo *bci = *i;
		if (ci->isPythonClass && bci->isPythonClass)
			continue;
		result = findGetFunc(bci, fname);
		if (result != NULL)
			return result;
	}
	return NULL;
}

// A "set" attribute function is a function that does not return a value,
// not even through output arguments, and has at least one input argument.

inline const FuncDecl *
findSetFunc(const DeclList::RAConstRange &range)
{
	for (DeclList::RAconst_iterator i = range.first; i != range.second;
									++i) {
		const FuncDecl *fd
				= dynamic_cast<const FuncDecl *>(i->second);
		if (fd == NULL || !(fd->returnType == "void")
		|| fd->args.size() == 0)
			continue;
		for (ArgList::const_iterator i = fd->args.begin();
						i != fd->args.end(); ++i)
			if (i->out)
				goto tryNext;
		return fd;
	tryNext:
		;
	}
	return NULL;
}

static const FuncDecl *
findSetFunc(const ClassInfo *ci, Symbol fname)
{
	if (ci == NULL)
		return NULL;
	DeclList::RAConstRange range = ci->cd->members.rAequal_range(fname);
	const FuncDecl *result = findSetFunc(range);
	if (result != NULL)
		return result;
	// recurse on base classes
	for (CIList::const_iterator i = ci->baseClasses.begin();
					i != ci->baseClasses.end(); ++i) {
		const ClassInfo *bci = *i;
		if (ci->isPythonClass && bci->isPythonClass)
			continue;
		result = findSetFunc(bci, fname);
		if (result != NULL)
			return result;
	}
	return NULL;
}

typedef std::set<const Decl *> DeclSet;

inline void
markVisited(const ClassDecl *cd, Symbol fname, DeclSet *visited, MethodMap *mm,
							Symbol attrName)
{
	DeclList::RAConstRange range = cd->members.rAequal_range(fname);
	for (DeclList::RAconst_iterator i = range.first; i != range.second;
									++i) {
		(*visited).insert(i->second);
		if (mm == NULL)
			continue;
		const FuncDecl *fd = dynamic_cast<const FuncDecl *>(i->second);
		if (fd == NULL)
			continue;
		// assert(fd->tag == fname);
		fd->attrName = attrName;
		mm->insert(MethodMap::value_type(fd->tag, fd));
	}
}

static void
markVisited(const ClassInfo *ci, Symbol fname, DeclSet *visited, MethodMap *mm,
							Symbol attrName)
{
	markVisited(ci->cd, fname, visited, mm, attrName);
	// recurse on base classes
	for (CIList::const_iterator i = ci->baseClasses.begin();
						i != ci->baseClasses.end(); ++i)
		markVisited(*i, fname, visited, mm, attrName);
}

// all "count" extra arguments must be of type "const char*"

static bool
fmatch(const FuncDecl *fd, const char *rtype, string::size_type count, ...)
{
	if (fd == NULL)
		return false;
	if (rtype != NULL)
		if (!(fd->returnType == rtype)
		&& !(fd->returnType == string("inline ") + rtype))
			return false;
	if (fd->args.size() != count)
		return false;

	va_list argp;
	va_start(argp, count);
	for (ArgList::const_iterator i = fd->args.begin(); i != fd->args.end();
									++i) {
		const char *type = va_arg(argp, const char *);
		if (type == NULL)
			continue;	// don't care arg
		if (!(i->type == type)) {
			va_end(argp);
			return false;
		}
	}
	va_end(argp);
	return true;
}

static const FuncDecl *
ostreamOperator(const ClassDecl *cd)
{
	Symbol op("operator<<");
	string pattern("[[:<:]]");
	pattern += cd->name.str() + "[[:>:]]";
	otf::Reg containsClass(pattern.c_str());
	const DeclList *scopeDecls;
	if (cd->scope == NULL)
		scopeDecls = &globalDecls;
	else if (cd->scope->dtype == Decl::NAMESPACE) {
		const NamespaceDecl *nd
				= static_cast<const NamespaceDecl *>(cd->scope);
		scopeDecls = &nd->decls;
	} else { // dtype == CLASS or STRUCT
		const ClassDecl *nd = static_cast<const ClassDecl *>(cd->scope);
		scopeDecls = &nd->members;
	}
	DeclList::RAConstRange range = scopeDecls->rAequal_range(op);
	for (DeclList::RAconst_iterator i = range.first; i != range.second;
									++i) {
		const FuncDecl *fd
				= dynamic_cast<const FuncDecl *>(i->second);
		if (fd == NULL)
			continue;
		if (!fmatch(fd, "std::ostream&", 2, NULL, NULL)
		&& !fmatch(fd, "ostream&", 2, NULL, NULL))
			continue;
		// see if second argument contains class name
		if (containsClass.find(fd->args[1].type) != string::npos)
			return fd;
	}
	return NULL;
}

static bool
unaryOp(const ClassDecl *cd, const char *unaryop, const char *returnType, MethodMap *mm)
{
	// a class unary operator can either be a member function that
	// takes zero arguments or a function that takes one.
	MethodMap tmpMM;
	if (mm == NULL)
		mm = &tmpMM;
	Symbol op(unaryop);
	string pattern("[[:<:]]");
	pattern += cd->name.str() + "[[:>:]]";
	otf::Reg containsClass(pattern.c_str());
	// first look in class
	DeclList::RAConstRange range = cd->members.rAequal_range(op);
	for (DeclList::RAconst_iterator i = range.first; i != range.second;
									++i) {
		const FuncDecl *fd
				= dynamic_cast<const FuncDecl *>(i->second);
		if (fd == NULL || !fmatch(fd, returnType, 0))
			continue;
		mm->insert(MethodMap::value_type(op, fd));
		return true;
	}
	// next look at class' scope
	const DeclList *scopeDecls;
	if (cd->scope == NULL)
		scopeDecls = &globalDecls;
	else if (cd->scope->dtype == Decl::NAMESPACE) {
		const NamespaceDecl *nd
				= static_cast<const NamespaceDecl *>(cd->scope);
		scopeDecls = &nd->decls;
	} else { // dtype == CLASS or STRUCT
		const ClassDecl *nd = static_cast<const ClassDecl *>(cd->scope);
		scopeDecls = &nd->members;
	}
	DeclList::RAConstRange erange = scopeDecls->rAequal_range(op);
	for (DeclList::RAconst_iterator i = erange.first; i != erange.second;
									++i) {
		const FuncDecl *fd
				= dynamic_cast<const FuncDecl *>(i->second);
		if (fd == NULL || !fmatch(fd, returnType, 1, NULL))
			continue;
		// see if first arguments contains class name
		if (!containsClass.find(fd->args[0].type) != string::npos)
			continue;
		mm->insert(MethodMap::value_type(op, fd));
		return true;
	}
	return false;
}

static bool
binOp(const ClassDecl *cd, const char *binop, const char *returnType, MethodMap *mm)
{
	// a class binary operator can either be a member function that
	// takes one argument or a function that takes two.
	MethodMap tmpMM;
	if (mm == NULL)
		mm = &tmpMM;
	Symbol op(binop);
	string pattern("[[:<:]]");
	pattern += cd->name.str() + "[[:>:]]";
	otf::Reg containsClass(pattern.c_str());
	DeclList::RAConstRange range = cd->members.rAequal_range(op);
	for (DeclList::RAconst_iterator i = range.first; i != range.second;
									++i) {
		const FuncDecl *fd
				= dynamic_cast<const FuncDecl *>(i->second);
		if (fd && fmatch(fd, returnType, 1, NULL))
			mm->insert(MethodMap::value_type(op, fd));
	}
	const DeclList *scopeDecls;
	if (cd->scope == NULL)
		scopeDecls = &globalDecls;
	else if (cd->scope->dtype == Decl::NAMESPACE) {
		const NamespaceDecl *nd
				= static_cast<const NamespaceDecl *>(cd->scope);
		scopeDecls = &nd->decls;
	} else { // dtype == CLASS or STRUCT
		const ClassDecl *nd = static_cast<const ClassDecl *>(cd->scope);
		scopeDecls = &nd->members;
	}
	DeclList::RAConstRange erange = scopeDecls->rAequal_range(op);
	for (DeclList::RAconst_iterator i = erange.first; i != erange.second;
									++i) {
		const FuncDecl *fd
				= dynamic_cast<const FuncDecl *>(i->second);
		if (fd == NULL || !fmatch(fd, returnType, 2, NULL, NULL))
			continue;
		// see if first argument contains class name
		if (containsClass.find(fd->args[0].type) != string::npos)
			mm->insert(MethodMap::value_type(op, fd));
	}
	return !mm->empty();
}

static bool
revBinOp(const ClassDecl *cd, const char *binop, const char *returnType, MethodMap *mm)
{
	// a reverse binary operator is a function that takes two
	// arguments with the class being the second and not the first.
	MethodMap tmpMM;
	if (mm == NULL)
		mm = &tmpMM;
	Symbol op(binop);
	string pattern("[[:<:]]");
	pattern += cd->name.str() + "[[:>:]]";
	otf::Reg containsClass(pattern.c_str());
	const DeclList *scopeDecls;
	if (cd->scope == NULL)
		scopeDecls = &globalDecls;
	else if (cd->scope->dtype == Decl::NAMESPACE) {
		const NamespaceDecl *nd
				= static_cast<const NamespaceDecl *>(cd->scope);
		scopeDecls = &nd->decls;
	} else { // dtype == CLASS or STRUCT
		const ClassDecl *nd = static_cast<const ClassDecl *>(cd->scope);
		scopeDecls = &nd->members;
	}
	DeclList::RAConstRange erange = scopeDecls->rAequal_range(op);
	for (DeclList::RAconst_iterator i = erange.first; i != erange.second;
									++i) {
		const FuncDecl *fd
				= dynamic_cast<const FuncDecl *>(i->second);
		if (fd == NULL || !fmatch(fd, returnType, 2, NULL, NULL))
			return false;
		// see if first argument contains class name
		if (containsClass.find(fd->args[0].type) != string::npos)
			continue;
		// see if second argument contains class name
		if (containsClass.find(fd->args[1].type) != string::npos)
			mm->insert(MethodMap::value_type(op, fd));
	}
	return !mm->empty();
}

static void
computeBuiltins(ClassInfo *ci)
{
	const FuncDecl *fd;

	// look for type methods
	// print method
	ci->print = NULL != ostreamOperator(ci->cd);
	// compare method
	binOp(ci->cd, "operator<", "bool", &ci->compare);
	// TODO: repr method
	// hash method
	fd = findGetFunc(ci, Symbol("hash"));
	if (fd)
		ci->hash.insert(MethodMap::value_type(Symbol("hash"), fd));
	// call method
	Symbol op("operator()");
	DeclList::RAConstRange range = ci->cd->members.rAequal_range(op);
	for (DeclList::RAconst_iterator i = range.first; i != range.second; ++i)
		if (i->second->dtype == Decl::FUNCTION) {
			fd = dynamic_cast<FuncDecl *>(i->second);
			ci->call.insert(MethodMap::value_type(op, fd));
		}

	// str method TODO? prevent from being a method
	fd = dynamic_cast<FuncDecl *>(ci->cd->members.unique(Symbol("str")));
	if (fd
	&& (fmatch(fd, "std::string", 0) || fmatch(fd, "const std::string&", 0)
	|| fmatch(fd, "string", 0) || fmatch(fd, "const string&", 0)))
		ci->str.insert(MethodMap::value_type(Symbol("str"), fd));
	else {
		fd = dynamic_cast<FuncDecl *>
				(ci->cd->members.unique(Symbol("c_str")));
		if (fd
		&& (fmatch(fd, "char*", 0) || fmatch(fd, "const char*", 0)))
			ci->str.insert(MethodMap::value_type(Symbol("c_str"),
									fd));
	}

	// look for number methods
	if (ci->hasNumberMethods) {
		if (binOp(ci->cd, "operator+", 0, &ci->nb_add)) {
			if (!ci->isPythonClass)
				ci->nb_coerce = true;
		}
		revBinOp(ci->cd, "operator+", 0, &ci->nb_radd);
		if (binOp(ci->cd, "operator-", 0, &ci->nb_subtract)) {
			if (!ci->isPythonClass)
				ci->nb_coerce = true;
		}
		revBinOp(ci->cd, "operator-", 0, &ci->nb_rsubtract);
		if (binOp(ci->cd, "operator*", 0, &ci->nb_multiply)) {
			if (!ci->isPythonClass)
				ci->nb_coerce = true;
		}
		revBinOp(ci->cd, "operator*", 0, &ci->nb_rmultiply);
		if (binOp(ci->cd, "operator/", 0, &ci->nb_divide)) {
			if (!ci->isPythonClass)
				ci->nb_coerce = true;
		}
		revBinOp(ci->cd, "operator/", 0, &ci->nb_rdivide);
		if (binOp(ci->cd, "operator%", 0, &ci->nb_remainder)) {
			if (!ci->isPythonClass)
				ci->nb_coerce = true;
		}
		revBinOp(ci->cd, "operator%", 0, &ci->nb_rremainder);
		// TODO: divmod, rdivmod
		// TODO: power, rpower
		unaryOp(ci->cd, "operator-", 0, &ci->nb_negative);
		unaryOp(ci->cd, "operator+", 0, &ci->nb_positive);
		// TODO: absolute
		if (!unaryOp(ci->cd, "operator!", "bool", &ci->nb_nonzero))
			unaryOp(ci->cd, "operator bool", "", &ci->nb_nonzero);
		unaryOp(ci->cd, "operator~", 0, &ci->nb_invert);
		if (binOp(ci->cd, "operator<<", 0, &ci->nb_lshift)) {
			if (!ci->isPythonClass)
				ci->nb_coerce = true;
		}
#ifdef TODO
		revBinOp(ci->cd, "operator<<", 0, &ci->nb_rlshift);
#endif
		if (binOp(ci->cd, "operator>>", 0, &ci->nb_rshift)) {
			if (!ci->isPythonClass)
				ci->nb_coerce = true;
		}
#ifdef TODO
		revBinOp(ci->cd, "operator>>", 0, &ci->nb_rrshift);
#endif
		if (binOp(ci->cd, "operator&", 0, &ci->nb_and)) {
			if (!ci->isPythonClass)
				ci->nb_coerce = true;
		}
		revBinOp(ci->cd, "operator&", 0, &ci->nb_rand);
		if (binOp(ci->cd, "operator^", 0, &ci->nb_xor)) {
			if (!ci->isPythonClass)
				ci->nb_coerce = true;
		}
		revBinOp(ci->cd, "operator^", 0, &ci->nb_rxor);
		if (binOp(ci->cd, "operator|", 0, &ci->nb_or)) {
			if (!ci->isPythonClass)
				ci->nb_coerce = true;
		}
		revBinOp(ci->cd, "operator|", 0, &ci->nb_ror);
		unaryOp(ci->cd, "operator int", "", &ci->nb_int);
		unaryOp(ci->cd, "operator long", "", &ci->nb_long);
		unaryOp(ci->cd, "operator float", "", &ci->nb_float);
		// TODO: oct
		// TODO: hex

		// TODO: Python 2.0 iadd, etc.
	}

	// look for sequence methods
	if (ci->hasSequenceMethods) {
		// length (TODO: return type should be int/size_t/::size_type)
		if (!unaryOp(ci->cd, "size", 0, &ci->sq_length))
			unaryOp(ci->cd, "length", 0, &ci->sq_length);
		// concat
		if (ci->nb_add.empty())
			binOp(ci->cd, "operator+", 0, &ci->sq_concat);
		// repeat
		if (ci->nb_multiply.empty())
			binOp(ci->cd, "operator*", 0, &ci->sq_repeat);
		// item (subscript should be int-compatible)
		binOp(ci->cd, "operator[]", 0, &ci->sq_item);
		// TODO: slice
		// TODO: ass_item	(assign item)
#if 0
		// TODO: want one that returns a non-const reference
		binOp(ci->cd, "operator[]", 0, &ci->sq_ass_item)) != 0) {
#endif
		// TODO: ass_slice	(assign slice)

		// TODO: Python 1.6 contains
		// TODO: Python 2.0 icontains
	}

	// look for mapping methods
	if (ci->hasMappingMethods) {
		// length (TODO: return type should be int/size_t/::size_type)
		if (!unaryOp(ci->cd, "size", 0, &ci->mp_length))
			unaryOp(ci->cd, "length", 0, &ci->mp_length);
		// subscript
		binOp(ci->cd, "operator[]", 0, &ci->mp_subscript);
		// TODO: ass_subscript
#if 0
		// TODO: want one that returns a non-const reference
		binOp(ci->cd, "operator[]", 0), &ci->mp_ass_subscript);
#endif
	}

	// look for buffer methods
	// TODO: getreadbuffer
	// TODO: getwritebuffer
	// TODO: getsegcount
}

static std::pair<bool, string>
theGetType(const Decl *scope, const FuncDecl *fd)
{
	// Return the type for a accessor get function.
	// This code should agree with dumpParseArgs(....).

	bool cache = false;
	string gt;	// get type
	if (!fd->returnType.empty() && fd->returnType != "void"
	&& fd->returnType != "static void") {
		CvtType cvt(scope, fd->returnType);
		cache = cvt.cache();
		gt += cvt.baseCppType();
	}
	bool tuple = false;
	for (ArgList::const_iterator i = fd->args.begin(); i != fd->args.end();
									++i) {

		if (!i->out)
			continue;
		string type = i->type;
		if (*type.rbegin() == '*')
			type.erase(type.size() - 1);	// remove trailing *
		CvtType arg(scope, type);
		if (!cache)
			cache = arg.cache();
		if (!gt.empty()) {
			gt += ',';
			tuple = true;
		}
		gt += arg.baseCppType();
	}
	if (!tuple)
		return std::make_pair(cache, gt);
	gt += ')';
	return std::make_pair(cache, '(' + gt);
}

static std::pair<bool, string>
theSetType(const Decl *scope, const FuncDecl *fd)
{
	// Return the type for a accessor set function.
	// This code should agree with dumpParseArgs(....).

	bool cache = false;
	string st;	// set type
	bool tuple = false;
	for (ArgList::const_iterator i = fd->args.begin(); i != fd->args.end();
									++i) {
		if (!i->in)
			continue;
		string type = i->type;
		if (i->out && *type.rbegin() == '*')
			type.erase(type.size() - 1);	// remove trailing *
		CvtType arg(scope, type);
		if (!cache)
			cache = arg.cache();
		if (!st.empty()) {
			st += ',';
			tuple = true;
		}
		st += arg.baseCppType();
	}
	if (!tuple)
		return std::make_pair(cache, st);
	st += ')';
	return std::make_pair(cache, '(' + st);
}

static void
addAttr(ClassInfo *ci, const ClassDecl *cd, DeclSet *visited, Symbol name,
				const FuncDecl *getF, const FuncDecl *setF)
{
	ClassAttrInfo ai;
	if (getF == NULL && setF == NULL)
		return;
	ai.name = name;
	if (getF != NULL) {
		ai.get = ci->name.str() + '_' + getF->tag.str()
						+ "(self, arg(), NULL)";
		markVisited(cd, getF->tag, visited, &ci->attrMethods,
								ai.name);
	}
	if (setF != NULL) {
		ai.set = ci->name.str() + '_' + setF->tag.str()
					+ "(self, arg(value), NULL)";
		markVisited(cd, setF->tag, visited, &ci->attrMethods,
								ai.name);
	}
	typedef std::pair<bool, string> info;
	if (getF != NULL && setF != NULL) {
		// determine attribute's type from its functions
		info gInfo = theGetType(getF->scope, getF);
		info sInfo = theSetType(getF->scope, setF);
		if (warnings && gInfo != sInfo)
			std::cerr << programName
				<< ": warning: attribute " << ai.name
				<< " type mismatch: '" << gInfo.second
				<< "' versus '" << sInfo.second << "'\n";
		ai.cache = gInfo.first || sInfo.first;
	} else if (getF != NULL) {
		info gInfo = theGetType(getF->scope, getF);
		ai.cache = gInfo.first;
	} else {
		info sInfo = theSetType(getF->scope, setF);
		ai.cache = sInfo.first;
	}
	ci->attrs.push_back(ai);
}

static void
processComment(ClassInfo *ci, const ClassDecl *cd, DeclSet *visited, const Decl *d)
{
	static const char ro[] = "READONLY:";
	static const string::size_type roLen = sizeof ro - 1;
	if (d->text.size() > roLen && d->text.compare(0, roLen, ro) == 0) {
		// change attribute to be read-only
		string::size_type start = d->text.find_first_not_of(SPACES,
								roLen);
		Symbol name(d->text.substr(start));
		AttrVec::iterator i = std::find_if(ci->attrs.begin(),
					ci->attrs.end(), AttrHasName(name));
		if (i != ci->attrs.end()) {
			i->set = string();
			i = std::find_if(ci->variables.begin(),
				ci->variables.end(), AttrHasName(name));
			if (i != ci->variables.end())
				i->set = string();
		}
		return;
	}

	static const char wo[] = "WRITEONLY:";
	static const string::size_type woLen = sizeof wo - 1;
	if (d->text.size() > woLen && d->text.compare(0, woLen, wo) == 0) {
		// change attribute to be write-only
		string::size_type start = d->text.find_first_not_of(SPACES,
								woLen);
		Symbol name(d->text.substr(start));
		AttrVec::iterator i = std::find_if(ci->attrs.begin(),
					ci->attrs.end(), AttrHasName(name));
		if (i != ci->attrs.end()) {
			i->get = string();
			i = std::find_if(ci->variables.begin(),
				ci->variables.end(), AttrHasName(name));
			if (i != ci->variables.end())
				i->get = string();
		}
		return;
	}

	static const char wr[] = "WEAKREF:";
	static const string::size_type wrLen = sizeof wr - 1;
	if (d->text.size() > wrLen && d->text.compare(0, wrLen, wr) == 0) {
		// change attribute to be not be cached
		string::size_type start = d->text.find_first_not_of(SPACES,
								wrLen);
		Symbol name(d->text.substr(start));
		AttrVec::iterator i = std::find_if(ci->attrs.begin(),
					ci->attrs.end(), AttrHasName(name));
		if (i != ci->attrs.end()) {
			i->cache = false;
			i = std::find_if(ci->variables.begin(),
				ci->variables.end(), AttrHasName(name));
			if (i != ci->variables.end())
				i->cache = false;
		}
		return;
	}

	static const char Attr[] = "ATTRIBUTE:";
	static const string::size_type AttrLen = sizeof Attr - 1;
	if (d->text.size() > AttrLen
	&& d->text.compare(0, AttrLen, Attr) == 0) {
		// look for user defined attributes,
		// could be missing a get or set function.
		string::size_type start
				= d->text.find_first_not_of(SPACES, AttrLen);
		string name = d->text.substr(start);
		string Name = name;
#ifdef OTF_NO_LOCALE
		if (islower(Name[0]))
			Name[0] = toupper(Name[0]);
#else
		if (ct.is(ct.lower, Name[0]))
			Name[0] = ct.toupper(Name[0]);
#endif
		Symbol nameSym(name);

		// look for get functions
		const FuncDecl *getF = findGetFunc(ci, nameSym);
		if (getF == NULL) {
			getF = findGetFunc(ci, Symbol("get" + Name));
			if (getF == NULL)
				getF = findGetFunc(ci, Symbol("Get" + Name));
		}
		// now look for set functions
		const FuncDecl *setF = findSetFunc(ci, Symbol("set" + Name));
		if (setF == NULL)
			setF = findSetFunc(ci, Symbol("Set" + Name));
		addAttr(ci, cd, visited, nameSym, getF, setF);
		return;
	}
}

static void
computeClassInfoAttributes(ClassInfo *ci, const ClassDecl *cd, DeclSet *visited)
{
	if (cd == ci->cd)
		computeBuiltins(ci);

	// When generating a Python class, derived class names
	// hide base class names.  The exception is when the
	// base class will be a Python type.

	// TODO: When generating Python types, C++ derived class
	// names should hide base class names.

	// recurse into base classes if appropriate
	const ClassInfo *curci = classInfoMap[cd];
	if (curci != NULL)
		for (CIList::const_iterator i = curci->baseClasses.begin();
					i != curci->baseClasses.end(); ++i) {
			const ClassInfo *bci = *i;
			if (curci->isPythonClass && bci->isPythonClass)
				continue;
			computeClassInfoAttributes(ci, bci->cd, visited);
		}

	// now do this class
	for (DeclList::const_iterator i = cd->members.begin();
						i != cd->members.end(); ++i) {
		const Decl *d = *i;
		if (visited->find(d) != visited->end())
			continue;
		visited->insert(d);
		if (d->access != PUBLIC
		|| (!d->tag.empty() && d->tag.str()[0] == '~'))
			continue;

		switch (d->dtype) {
		  default:
			// anything not covered, we ignore
			break;
		  case Decl::COMMENT:
			processComment(ci, cd, visited, d);
			break;
		  case Decl::CONSTANT: {
			const VarDecl *vd = static_cast<const VarDecl *>(d);
			CvtType cvt(cd, vd->type);
			ClassAttrInfo ai;
			ai.name = d->tag;
			ai.get = make_buildvalue(cvt.bvFormat(), 
				qualify(cd, cvt.bvArg(vd->tag.str())));
			ai.set = "";
			ci->constants.push_back(ai);
			break;
		  }
		  case Decl::VARIABLE: {
			const VarDecl *vd = static_cast<const VarDecl *>(d);
			CvtType cvt(cd, vd->type);
			ClassAttrInfo ai;
			ai.name = d->tag;
			ai.cache = cvt.cache();
			string attr;
			if (!ci->isPythonClass)
				attr = "wco->";
			attr += "inst->" + vd->tag.str();
			ai.get = make_buildvalue(cvt.bvFormat(), cvt.bvArg(attr));
			// TODO? ideally we'd move this code generation to
			// class.cpp and type.cpp.
			if (ci->isPythonClass) {
				string prefix(qualify(ci->cd->scope, ci->name.str()));
				ai.set = "\t" + qualify(cd, cvt.aptType())
					+ " ptArg;\n"
					"\tif (!PyArg_ParseTupleAndKeywords(args, keywds, \"O" + cvt.aptFormat() + ":set\", kwlist, &self, &ptArg))\n"
					"\t\treturn NULL;\n"
					"\t" + prefix + "* inst = ";
				if (ci->isWrappySubclass)
					ai.set += "dynamic_cast<" + prefix
					+ "*>(const_cast<otf::WrapPyObj*>(otf::wrappyObject(self)));\n";
				else
					ai.set += "reinterpret_cast<" + prefix
					+ "*>(otf::wrappyVoidObject(self));\n";
				ai.set +=
					"\tif (inst == NULL)\n"
					"\t\treturn NULL;\n";
			} else
				ai.set = "\t" + qualify(cd, cvt.aptType())
					+ " ptArg;\n"
					+ "\tif (!PyArg_ParseTuple(arg(value), \""
					+ cvt.aptFormat() + ":set\", &ptArg))\n"
					+ "\t\treturn;\n";
			if (cvt.needTypeCheck()) {
				ai.set += "\tif (!"
					+ cvt.typeCheck("ptArg") + ") {\n"
					"\t\tPyErr_SetString(PyExc_TypeError, \"attribute "
					+ vd->tag.str() + " should be a "
					// TODO: strip * from cppType()
					+ cvt.cppType() + "\");\n";
				if (ci->isPythonClass)
					ai.set += "\t\treturn NULL;\n";
				else
					ai.set += "\t\treturn;\n";
				ai.set += "\t}\n";
			}
			ai.set += "\t" + attr + " = "
					+ qualify(cd, cvt.aptToCpp("ptArg"));
			ci->variables.push_back(ai);
			// simple attribute function (inherit cache from above)
			ai.get = ci->name.str() + "Get" + vd->tag.str()
								+ "(wco)";
			ai.set = ci->name.str() + "Set" + vd->tag.str()
							+ "(wco, value)";
			ci->attrs.push_back(ai);
			break;
		  }
		  case Decl::FUNCTION: {
			const FuncDecl *fd = static_cast<const FuncDecl *>(d);
			if (fd->tag.str().compare(0, 8, "operator") == 0)
				break;
			if (fd->tag == ci->name) {
				// save away constructors
				ci->constructors.insert(
					MethodMap::value_type(fd->tag, fd));
				break;
			}
			if (fd->tag == cd->name)
				// skip subclass constructors
				break;
			if (btStatic.find(fd->returnType) == 0) {
				// save away class methods
				ci->classMethods.insert(
					MethodMap::value_type(fd->tag, fd));
				break;
			}

			string name = fd->tag.str();
			const FuncDecl *getF = NULL;
			const FuncDecl *setF = NULL;
			Symbol nameSym;
			if (name.compare(0, 3, "set") == 0
			|| name.compare(0, 3, "Set") == 0) {
				// if name is set* then look for * or get*
				setF = findSetFunc(ci, fd->tag);
				if (setF == NULL) {
					ci->methods.insert(
						MethodMap::value_type(fd->tag,
									fd));
					break;
				}
				string attr = name.substr(3);
#ifdef OTF_NO_LOCALE
				if (isupper(attr[0]))
					attr[0] = tolower(attr[0]);
#else
				if (ct.is(ct.upper, attr[0]))
					attr[0] = ct.tolower(attr[0]);
#endif
				nameSym = Symbol(attr);
				getF = findGetFunc(ci, nameSym);
				if (getF == NULL) {
					Symbol get = Symbol(
						(name[0] == 's' ? 'g' : 'G')
						+ name.substr(1));
					getF = findGetFunc(ci, get);
				}
			}
			else if (name.compare(0, 3, "get") == 0
			|| name.compare(0, 3, "Get") == 0) {
				// else if name is get* then look for set*
				getF = findGetFunc(ci, fd->tag);
				if (getF == NULL) {
					ci->methods.insert(
						MethodMap::value_type(fd->tag,
									fd));
					break;
				}
				string attr = name.substr(3);
#ifdef OTF_NO_LOCALE
				if (isupper(attr[0]))
					attr[0] = tolower(attr[0]);
#else
				if (ct.is(ct.upper, attr[0]))
					attr[0] = ct.tolower(attr[0]);
#endif
				nameSym = Symbol(attr);
				Symbol set = Symbol((name[0] == 'g' ? 's' : 'S')
								+ name.substr(1));
				setF = findSetFunc(ci, set);
			}
			else if ((getF = findGetFunc(ci, fd->tag)) != NULL) {
				// if name is * then look for set*
				nameSym = getF->tag;
#ifdef OTF_NO_LOCALE
				if (islower(name[0]))
					name[0] = toupper(name[0]);
#else
				if (ct.is(ct.lower, name[0]))
					name[0] = ct.toupper(name[0]);
#endif
				setF = findSetFunc(ci, Symbol("set" + name));
				if (setF == NULL)
					setF = findSetFunc(ci,
							Symbol("Set" + name));
			}
			if (getF == NULL || setF == NULL) {
				// attributes have both set() and get()
				ci->methods.insert(
					MethodMap::value_type(fd->tag, fd));
				break;
			}
			addAttr(ci, cd, visited, nameSym, getF, setF);
			break;
		  }
		}
	}
	// TODO: remove following bandaid
	if (cd == ci->cd) {
		// remove duplicates
		std::stable_sort(ci->attrs.begin(), ci->attrs.end(),
							AttrNameLt());
		AttrVec::iterator dups = std::unique(ci->attrs.begin(),
					ci->attrs.end(), AttrNameEq());
		ci->attrs.erase(dups, ci->attrs.end());
	}
}

static ClassInfo *
computeClassInfo(ClassDecl *cd)
{
	ClassInfo *ci = new ClassInfo(cd);
	classInfoMap[cd] = ci;

	// change default if a C++ class instead of a struct
	if (ci->cd->dtype == Decl::CLASS)
		ci->isPythonClass = true;

	if (isWrapPySubclass(cd))
		ci->isWrappySubclass = true;

	// scan comments for interesting stuff
	for (DeclList::iterator i = cd->members.begin();
						i != cd->members.end(); ++i) {
		Decl *d = *i;
		if (d->dtype != Decl::COMMENT)
			continue;
		if (d->text == "WRAP CLASS")
			ci->skipClass = false;
		else if (d->text == "PYTHON CLASS")
			ci->isPythonClass = true;
		else if (d->text == "PYTHON TYPE")
			ci->isPythonClass = false;
		else if (d->text == "ABSTRACT")
			ci->isAbstractType = true;
		else if (d->text == "NUMBER METHODS")
			ci->hasNumberMethods = true;
		else if (d->text == "SEQUENCE METHODS")
			ci->hasSequenceMethods = true;
		else if (d->text == "MAPPING METHODS")
			ci->hasMappingMethods = true;
		else if (d->text == "BUFFER PROCS")
			ci->hasBufferProcs = true;
		else if (d->text == "DON'T CACHE")
			ci->dontCache = true;
	}

	// if includefile isn't set, then set it
	if (ci->includeFile.empty()) {
		ci->includeFile = "# include ";
		if (cd->filename == "standard input")
			ci->includeFile += '"' + cd->name.str() + ".h\"";
		else if (cd->filename.c_str()[0] == '<'
		|| cd->filename.c_str()[0] == '"')
			ci->includeFile += cd->filename.str();
		else
			ci->includeFile += '"' + cd->filename.str() + '"';
	}

	// remove all declarations we're not going to look at
	typedef std::vector<Decl *> DeclVec;
	DeclVec removeDecls;
	for (DeclList::iterator i = cd->members.begin();
						i != cd->members.end(); ++i) {
		Decl *d = *i;
		if (d->dtype == Decl::COMMENT)
			// save comments
			continue;
		if (!d->tag.empty()
		&& (d->tag == cd->name || d->tag.str()[0] == '~'))
			// keep private constructors and destructors
			continue;
		if (d->access != PUBLIC)
			// eliminate private and protected regions of class
			removeDecls.push_back(d);
		else if (d->tag.empty())
			removeDecls.push_back(d);
	}
	for (DeclVec::iterator i = removeDecls.begin(); i != removeDecls.end();
									++i)
		cd->members.remove(*i);
	return ci;
}

static void
fillInClass(ClassDecl *cd)
{
	// Try to mimic C++ behavior of creating a default constructor,
	// a default destructor, a copy constructor, and an assignment
	// operator if they don't already exist.

	// look for constructors
	bool hasConstructor = false;
#ifndef REMOVE_COPY_CONSTRUCTORS
	bool hasCopyConstructor = false;
#endif
	DeclList::RARange range = cd->members.rAequal_range(cd->name);
	for (DeclList::RAiterator i = range.first; i != range.second;) {
		DeclList::RAiterator next = i;
		++next;
		FuncDecl *fd = dynamic_cast<FuncDecl *>(i->second);
		if (fd == NULL)
			continue;
		if (fd->args.size() != 1)
			hasConstructor = true;
		else {
			// check for copy constructor
			const string &type = fd->args[0].type;
			if (type == cd->name.str() + '&'
			|| type == "const " + cd->name.str() + '&') {
#ifdef REMOVE_COPY_CONSTRUCTORS
				cd->members.remove(fd);
#else
				hasCopyConstructor = true;
#endif
			} else
				hasConstructor = true;
		}
		i = next;
	}
	if (!hasConstructor) {
		// no constructor found, fake one
		FuncDecl *fd = new FuncDecl(PUBLIC, cd, cd->filename);
		fd->tag = cd->name;
		fd->returnType = "";
		cd->members.append(fd);
	}
#ifndef REMOVE_COPY_CONSTRUCTORS
	if (!hasCopyConstructor) {
		// no copy constructor found, fake one
		FuncDecl *fd = new FuncDecl(PUBLIC, cd, cd->filename);
		fd->tag = cd->name;
		fd->returnType = "";
		Argument arg;
		arg.name = Symbol("_x");
		arg.type = "const " + cd->name.str() + '&';
		fd->args.push_back(arg);
		cd->members.append(fd);
	}
#endif
	// check for destructor
	Symbol dtag = Symbol('~' + cd->name.str());
	if (!cd->members.has(dtag, ANY)) {
		// no destructor found, fake one
		FuncDecl *fd = new FuncDecl(PUBLIC, cd, cd->filename);
		fd->tag = dtag;
		fd->returnType = "";
		cd->members.append(fd);
	}
	if (!binOp(cd, "operator=", NULL, NULL)) {
		// no operator=, fake one
		FuncDecl *fd = new FuncDecl(PUBLIC, cd, cd->filename);
		fd->tag = Symbol("operator=");
		fd->returnType = "void";
		Argument arg;
		arg.name = Symbol("_x");
		arg.type = "const " + cd->name.str() + '&';
		fd->args.push_back(arg);
		cd->members.append(fd);
	}
}

static void
computeClassDeclMap(DeclList *dl)
{
	for (DeclList::iterator i = dl->begin(); i != dl->end(); ++i) {
		switch ((*i)->dtype) {
		  default:
			break;
		  case Decl::CLASS:
		  case Decl::STRUCT: {
			ClassDecl *cd = static_cast<ClassDecl *>(*i);
			fillInClass(cd);
			string qualName = qualify(cd->scope, cd->name.str());
			classDeclsMap[Symbol(qualName)] = cd;
			computeClassDeclMap(&cd->members);
			break;
		  }
		  case Decl::NAMESPACE: {
			NamespaceDecl *nd = static_cast<NamespaceDecl *>(*i);
			computeClassDeclMap(&nd->decls);
			break;
		  }
		}
	}
}

static void
computeClassInfoMap(DeclList *dl, bool genScope)
{
	for (DeclList::iterator i = dl->begin(); i != dl->end(); ++i) {
		switch ((*i)->dtype) {
		  default:
			break;
		  case Decl::CLASS:
		  case Decl::STRUCT: {
			ClassDecl *cd = static_cast<ClassDecl *>(*i);
			computeDerivedClasses(cd);
			computeClassInfoMap(&cd->members, genScope);
			ClassInfo *ci = computeClassInfo(cd);
			if (genScope
			&& (cd->access == PUBLIC || cd->access == GLOBAL))
				ci->skipClass = false;
			break;
		  }
		  case Decl::NAMESPACE: {
			NamespaceDecl *nd = static_cast<NamespaceDecl *>(*i);
			if (!genScope)
				if (nameSpace == nd->tag) {
					genScope = true;
					moduleScope = nd;
				}
			computeClassInfoMap(&nd->decls, genScope);
			break;
		  }
		}
	}
}

void
computeBaseClasses()
{
	for (CDCIMap::iterator i = classInfoMap.begin();
						i != classInfoMap.end(); ++i) {
		ClassInfo *ci = const_cast<ClassInfo*>(i->second);
		ClassDecl *cd = const_cast<ClassDecl*>(ci->cd);	// or i->first
		for (DeclList::const_iterator i = cd->baseClasses.begin();
					i != cd->baseClasses.end(); ++i) {
			string qualName = qualify(cd->scope, (*i)->tag.str());
			const ClassDecl *bcd = classDeclsMap[Symbol(qualName)];
			if (bcd == NULL)
				continue;
			CDCIMap::iterator j = classInfoMap.find(bcd);
			if (j == classInfoMap.end())
				continue;
			ClassInfo *base = const_cast<ClassInfo*>(j->second);
			ci->baseClasses.push_back(base);
		}
	}
}

void
computeUsing(DeclList *dl)
{
	typedef std::vector<Decl *> DeclVec;
	DeclVec removeDecls;
	DeclVec addDecls;
	for (DeclList::iterator i = dl->begin(); i != dl->end(); ++i) {
		Decl *d = *i;
		switch (d->dtype) {
		  default:
			break;
		  case Decl::USING: {
			static const int uSz = sizeof "using";
			string name = d->text.substr(uSz, d->text.size() - uSz);
			if (name.compare(0, 9, "namespace") == 0) {
				std::cerr << programName
				    << ": 'using namespace' is not supported\n";
				removeDecls.push_back(d);
				continue;
			}
			removeDecls.push_back(d);
			if (*name.rbegin() == ';')
				// remove trailing ;
				name.erase(name.size() - 1);
			DeclList::RAConstRange ads
					= funcVarTypeDecls(d->scope, name);
			for (DeclList::RAconst_iterator j = ads.first;
							j != ads.second; ++j)
				addDecls.push_back(j->second);
			break;
		  }
		  case Decl::CLASS:
		  case Decl::STRUCT: {
			ClassDecl *cd = static_cast<ClassDecl *>(*i);
			computeUsing(&cd->members);
			break;
		  }
		  case Decl::NAMESPACE: {
			NamespaceDecl *nd = static_cast<NamespaceDecl *>(*i);
			computeUsing(&nd->decls);
			break;
		  }
		}
	}
	for (DeclVec::iterator i = removeDecls.begin(); i != removeDecls.end();
									++i)
		dl->remove(*i);
	for (DeclVec::iterator i = addDecls.begin(); i != addDecls.end(); ++i) {
		Decl *d = *i;

		// need to avoid duplicates (esp. of types)
		switch (d->dtype) {
		  default:
			break;
		  case Decl::TYPEDEF:
		  case Decl::STRUCT:
		  case Decl::CLASS: {
			DeclList::RAConstRange range
						= dl->rAequal_range(d->tag);
			if (range.first != range.second)
				continue;
			break;
		  }
		  case Decl::FUNCTION: {
			FuncDecl *fd = dynamic_cast<FuncDecl *>(d);
			DeclList::RAConstRange range
						= dl->rAequal_range(d->tag);
			for (DeclList::RAconst_iterator i = range.first;
						i != range.second; ++i) {
				FuncDecl *cfd
					= dynamic_cast<FuncDecl *>(i->second);
				if (cfd == NULL)
					continue;
				if (*fd == *cfd)
					goto skip_it;
			}
			break;
		skip_it:
			continue;
		  }
		}
		// TODO? Change scope of declaration
		dl->append(d->copy());
	}
}

static void
fixupAbstractBaseClasses(ClassInfo *ci)
{
	for (CIList::iterator i = ci->baseClasses.begin();
					i != ci->baseClasses.end(); ++i) {
		ClassInfo *base = *i;
		if (base->skipClass)
			continue;
		if (base->isAbstractType)
			base->isPythonClass = true;
		if (!base->baseClasses.empty())
			fixupAbstractBaseClasses(base);
	}
}

static void
confirmSubclassConsistency()
{
	// C++ class hierarchies should be converted to either all
	// Python types or all Python classes.
	// If a C++ class is subclassed from otf::WrapPy<class>
	// and will be a Python class, then convert abstract base
	// classes to be Python classes if not already set.

	if (derivedClasses.empty())
		return;

	// First do our silent fixup of abstract base classes
	// so that abstract base classes of classes that will
	// be Python classes are Python classes as well.
	for (CDCIMap::iterator i = classInfoMap.begin();
						i != classInfoMap.end(); ++i) {
		ClassInfo *ci = const_cast<ClassInfo*>((*i).second);
		if (ci->skipClass || !ci->isPythonClass
		|| ci->baseClasses.empty())
			continue;
		fixupAbstractBaseClasses(ci);
	}

	// now look for consistency
	for (CDCIMap::iterator i = classInfoMap.begin();
						i != classInfoMap.end(); ++i) {
		const ClassInfo *ci = (*i).second;
		if (ci->skipClass)
			continue;
		for (CIList::const_iterator i = ci->baseClasses.begin();
					i != ci->baseClasses.end(); ++i) {
			ClassInfo *base = *i;
			if ((ci->isPythonClass && !base->isPythonClass)
			|| (!ci->isPythonClass && base->isPythonClass)) {
				std::cerr << "class " << ci->name
					<< " and class " << base->name
					<< " need to be both Python classes or types\n";
			}
		}
	}
}

static void
computeAttributes()
{
	// find attributes
	for (CDCIMap::iterator i = classInfoMap.begin();
						i != classInfoMap.end(); ++i) {
		ClassInfo *ci = const_cast<ClassInfo*>((*i).second);
		if (ci->skipClass)
			continue;
		DeclSet visited;
		computeClassInfoAttributes(ci, ci->cd, &visited);
	}
}

void
computeWrapInformation()
{
	computeClassDeclMap(&globalDecls);
	computeClassInfoMap(&globalDecls, nameSpace.empty());
	computeBaseClasses();
	computeUsing(&globalDecls);
	confirmSubclassConsistency();
	computeAttributes();
}
