// Copyright (c) 1998-2000 The Regents of the University of California.
// All rights reserved.
//
// Redistribution and use in source and binary forms are permitted
// provided that the above copyright notice and this paragraph are
// duplicated in all such forms and that any documentation,
// distribution and/or use acknowledge that the software was developed
// by the Computer Graphics Laboratory, University of California,
// San Francisco.  The name of the University may not be used to
// endorse or promote products derived from this software without
// specific prior written permission.
// 
// THIS SOFTWARE IS PROVIDED ``AS IS'' AND WITHOUT ANY EXPRESS OR
// IMPLIED WARRANTIES, INCLUDING, WITHOUT LIMITATION, THE IMPLIED
// WARRANTIES OF MERCHANTIBILITY AND FITNESS FOR A PARTICULAR PURPOSE.
// IN NO EVENT SHALL THE REGENTS OF THE UNIVERSITY OF CALIFORNIA BE
// LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY,
// OR CONSEQUENTIAL DAMAGES ARISING OUT OF THE USE OF THIS SOFTWARE.

// $Id: module.cpp,v 1.19 2003/06/24 19:02:14 gregc Exp $

#include <iostream>
#include <algorithm>
#include <list>
#include <otf/Reg.h>
#include "ParseHeader.h"
#include "CvtType.h"
#include "common.h"
#include "compute.h"
#include "method.h"

using std::string;

bool dumpModuleHeader();

void
dump_class_info(std::ostream &output, const ClassInfo *ci)
{
	output << "\tinfo = PyDict_New();\n";
	for (AttrVec::const_iterator i = ci->attrs.begin();
						i != ci->attrs.end(); ++i)
		output << "\tPyDict_SetItemString(info, \"" << i->name
							<< "\", attribute);\n";
	if (ci->isPythonClass)
		return;

	for (AttrVec::const_iterator i = ci->constants.begin();
						i != ci->constants.end(); ++i)
		output << "\tPyDict_SetItemString(info, \"" << i->name
							<< "\", constant);\n";
	MethodMap::const_iterator next;
	for (MethodMap::const_iterator i = ci->methods.begin();
					i != ci->methods.end(); i = next) {
		next = ci->methods.upper_bound(i->first);
		output <<
			"\textern PyMethodDef " << ci->name << "MD_"
							<< i->first << ";\n"
			"\to = PyCFunction_New(&" << ci->name << "MD_"
						<< i->first << ", NULL);\n"
			"\tPyDict_SetItemString(info, \"" << i->first
								<< "\", o);\n"
			"\tPy_DECREF(o);\n";
	}
}

void
dump_wrappy_doc(std::ostream &output, const DeclList *modDecls)
{
	output <<
		"\n"
		"void\n"
		"make_wrappy_doc(PyObject* modDict)\n"
		"{\n"
		"\tPyObject* d = PyDict_New();\n"
		"\tPyDict_SetItemString(modDict, \"wrappy_doc\", d);\n"
		"\tPyObject* constant = PyString_FromString(\"constant\");\n";
	// modules constants
	for (DeclList::const_iterator i = modDecls->begin();
						i != modDecls->end(); ++i) {
		if ((*i)->dtype != Decl::CONSTANT)
			continue;
		output <<
			"\tPyDict_SetItemString(d, \"" << (*i)->tag
							<< "\", constant);\n";
	}
	// class constants for Python types
	for (CDCIMap::iterator i = classInfoMap.begin();
						i != classInfoMap.end(); ++i) {
		const ClassInfo *ci = (*i).second;
		if (ci->skipClass || ci->isPythonClass)
			continue;
		for (DeclList::const_iterator j = ci->cd->members.begin();
					j != ci->cd->members.end(); ++j) {
			Decl *d = *j;
			if (d->dtype != Decl::CONSTANT)
				continue;
			output <<
				"\tPyDict_SetItemString(d, \"" << ci->name
					<< '_' << d->tag << "\", constant);\n";
		}
	}

	// dump Python class and type information
	output <<
		"\tPyObject* attribute = PyString_FromString(\"attribute\");\n"
		"\tPyObject* method = PyString_FromString(\"method\");\n"
		"\tPyObject* info;\n"
		"\tPyObject* o;\n";
	for (CDCIMap::iterator i = classInfoMap.begin();
						i != classInfoMap.end(); ++i) {
		const ClassInfo *ci = (*i).second;
		if (ci->skipClass || (ci->isAbstractType && !ci->isPythonClass))
			continue;
		dump_class_info(output, ci);
		output <<
			"\tPyDict_SetItemString(d, \"" << ci->name;
		if (!ci->isPythonClass)
			output << "Type";
		output << "\", info);\n"
			"\tPy_DECREF(info);\n";
	}
	output <<
		"\tPy_DECREF(method);\n"
		"\tPy_DECREF(attribute);\n"
		"\tPy_DECREF(constant);\n"
		"\tPy_DECREF(d);\n"
		"}\n";
}

void
dumpInitClasses(std::ostream &output)
{
	string nsPrefix;
	if (!nameSpace.empty())
		nsPrefix = nameSpace + "::";

	typedef std::list<const ClassInfo*> CIDList;
	CIDList cidl;
	for (CDCIMap::iterator i = classInfoMap.begin();
						i != classInfoMap.end(); ++i) {
		const ClassInfo *ci = (*i).second;
		if (ci->skipClass || !ci->isPythonClass)
			continue;
		if (ci->baseClasses.empty())
			cidl.push_front(ci);
		else
			cidl.push_back(ci);
	}
	if (cidl.empty())
		return;

	output <<
		"\n"
		"\ttry {\n";

	typedef std::set<const ClassInfo*> CISet;
	CISet visited;
	while (!cidl.empty()) {
		const ClassInfo *ci = cidl.front();
		cidl.pop_front();
		for (CIList::const_iterator i = ci->baseClasses.begin();
					i != ci->baseClasses.end(); ++i) {
			if ((*i)->skipClass || !(*i)->isPythonClass)
				continue;
			if (visited.find(*i) == visited.end()) {
				// One of the base classes hasn't been
				// visited yet, save for later.
				cidl.push_back(ci);
				goto try_again;
			}
		}
		visited.insert(ci);
		output <<
			"\t\t" << nsPrefix << ci->name
						<< "ClassInit(modDict);\n";
	try_again:;
	}
	output <<
		"\t} catch (std::exception &) {\n"
		"\t\tPyErr_SetString(" << nsPrefix << module
			<< "ErrorObj, \"class initialization failed\");\n"
		"\t}\n";
}

bool
dumpModuleCode()
{
	string filename(module);
	filename += ".cpp";
	std::auto_ptr<std::ostream> outs(outputStream(filename).release());
	std::ostream &output = *outs.get();
	string nsPrefix;
	if (!nameSpace.empty())
		nsPrefix = nameSpace + "::";

	output << "#define WrapPy_Functions\n"
		"#include \"" << module << ".h\"\n";

	const DeclList *modDecls;
	if (moduleScope != NULL)
		modDecls = &moduleScope->decls;
	else
		modDecls = &globalDecls;

	// find all include files for C++ global (namespace) functions
	SymbolSet includes;
	for (DeclList::RAconst_iterator i = modDecls->rAbegin();
					i != modDecls->rAend(); ++i) {
		const Decl *d = i->second;
		if (d->tag.empty() || d->dtype != Decl::FUNCTION
		|| d->tag.str().find(':') != string::npos
		|| d->tag.str().compare(0, 8, "operator") == 0)
			// skip non-functions, class function declarations,
			// and operators
			continue;
		if (d->filename != "standard input")
			includes.insert(d->filename);
	}
	for (SymbolSet::iterator i = includes.begin(); i != includes.end();
									++i) {
		string tmp = i->str();
		if (tmp[0] == '<' || tmp[0] == '"')
			output << "#include " << tmp << '\n';
		else
			output << "#include \"" << tmp << "\"\n";
	}

	output <<
		"#include <stdlib.h>	// for getenv() and atoi()\n";

	if (!derivedClasses.empty()) {
		// generate pyObject function for base classes
		output <<
			"\n"
			"namespace otf {\n";
		for (DerivedClasses::iterator i = derivedClasses.begin();
						i != derivedClasses.end(); ) {
			DerivedClasses::iterator end
					= derivedClasses.upper_bound(i->first);
			if (i->first == "otf::WrapPy") {
				i = end;
				continue;
			}
			const ClassDecl *bcd = classDeclsMap[i->first];
			if (bcd == NULL) {
				i = end;
				continue;
			}
			CDCIMap::iterator j = classInfoMap.find(bcd);
			if (j == classInfoMap.end()) {
				i = end;
				continue;
			}
			const ClassInfo *base = j->second;
			if (base->isPythonClass) {
				i = end;
				continue;
			}
			string prefix(qualify(moduleScope,
						i->first.str(), true));
			output <<
				"\n"
				"template <> PyObject*\n"
				"pyObject(" << prefix << "* o)\n"
				"{\n"
				"\totf::WrapPyObj* wo = dynamic_cast<otf::WrapPyObj*>(o);\n"
				"\tif (wo == NULL) {\n"
				"\t\tPy_INCREF(Py_None);\n"
				"\t\treturn Py_None;\n"
				"\t}\n"
				"\treturn wo->wpyGetObject();\n"
				"}\n";
			i = end;
		}
		output <<
			"\n"
			"} // namespace otf\n";
	}

	if (!nameSpace.empty())
		output <<
			"\n"
			"namespace " << nameSpace << " {\n";
	output <<
		"\n"
		"PyObject* " << module << "ErrorObj;\n"
		"PyObject* " << module << "DummyClass;\n"
		"int " << module << "Debug;\n";

	// TODO: use exception specification knowledge
	output <<
		"\n"
		"void\n"
		<< module << "Error()\n"
		"{\n"
		"\t// generic exception handler\n"
		"\ttry {\n"
		"\t\t// rethrow exception to look at it\n"
		"\t\tthrow;\n"
#if 0
		"\t} catch (otf::PyException &) {\n"
		"\t\t// nothing to do, already set\n"
#endif
		"\t} catch (otf::WrapPyObj &wo) {\n"
		"\t\tPyObject *obj = wo.wpyGetObject();\n"
		"\t\tPyErr_SetObject(" << module << "ErrorObj, obj);\n"
		"\t\tPy_DECREF(obj);\n"
		"\t} catch (std::exception &e) {\n"
		"\t\tPyErr_SetString(" << module << "ErrorObj, e.what());\n"
		"\t} catch (...) {\n"
		"\t\tif (" << module << "Debug)\n"
		"\t\t\tthrow; // fatal exception\n"
		"\t\tPyErr_SetString(" << module << "ErrorObj, \"unknown C++ exception\");\n"
		"\t}\n"
		"}\n";

	typedef std::vector<string> StrVec;
	StrVec modMethods;

	// C++ class constructors and functions are Python module methods
	for (CDCIMap::iterator i = classInfoMap.begin();
						i != classInfoMap.end(); ++i) {
		const ClassInfo *ci = (*i).second;
		if (ci->skipClass)
			continue;
		MethodMap::const_iterator next;
		for (MethodMap::const_iterator j = ci->classMethods.begin();
					j != ci->classMethods.end(); j = next) {
			next = ci->classMethods.upper_bound(j->first);
			string proto = dumpMethod(output, ci, j, next, false);
			proto = prefixLines(ci->name.str() + '_', proto);
			proto += "\n" + ci->name.str() + "::" + j->first.str()
							+ " documentation.";
			output <<
				"\n"
				"const char " << module << '_' << ci->name
					<< '_' << j->first << "_doc[] = \n\""
					<< stringize(proto) << "\";\n";
			modMethods.push_back(ci->name.str() + '_'
							+ j->first.str());
		}
		if (ci->isAbstractType || ci->isPythonClass)
			continue;
		if (!ci->constructors.empty()) {
			string proto = dumpMethod(output, ci,
				ci->constructors.begin(),
				ci->constructors.end(), false);
			proto += "\n" + ci->name.str()
						+ " constructor documentation.";
			output <<
				"\n"
				"const char " << module << '_' << ci->name
					<< "_doc[] = \n\"" << stringize(proto)
					<< "\";\n";
			modMethods.push_back(ci->name.str());
		}
	}
	// C++ global (namespace) functions are Python module methods
	DeclList::RAconst_iterator next;
	for (DeclList::RAconst_iterator i = modDecls->rAbegin();
					i != modDecls->rAend(); i = next) {
		next = modDecls->rAupper_bound(i->first);
		const Decl *d = i->second;
		if (d->dtype == Decl::VARIABLE) {
			// just an easy place to test for this
			static bool warned = false;
			if (!warned)
				std::cerr << programName
				<< ": global variables are not supported\n";
			warned = true;
			continue;
		}
		if (d->tag.empty() || d->dtype != Decl::FUNCTION
		|| d->tag.str().find(':') != string::npos
		|| d->tag.str().compare(0, 8, "operator") == 0)
			// skip non-functions, class function declarations,
			// and operators
			continue;
		MethodMap funcs;
		for (DeclList::RAconst_iterator j = i; j != next; ++j) {
			const FuncDecl *fd
				= dynamic_cast<const FuncDecl *>(j->second);
			if (fd == NULL)
				continue;
			funcs.insert(MethodMap::value_type(fd->tag, fd));
		}
		string proto = dumpMethod(output, NULL, funcs.begin(),
							funcs.end(), false);
		proto += "\n" + d->tag.str() + " documentation.";
		output <<
			"\n"
			"const char " << module << '_' << d->tag
				<< "_doc[] = \"" << stringize(proto) << "\";\n";
		modMethods.push_back(d->tag.str());
	}
	// fill in method table
	output <<
		"\n"
		"PyMethodDef " << module << "Methods[] = {\n";
	for (StrVec::iterator i = modMethods.begin(); i != modMethods.end();
									++i)
		output <<
			"\t{\n"
			"\t\t\"" << *i << "\", (PyCFunction) " << module << '_'
				<< *i << ",\n"
			<< "\t\tMETH_KEYWORDS, const_cast<char*>(" << module
						<< '_' << *i << "_doc)\n"
			<< "\t},\n";
	output <<
		"\t{ NULL, NULL }\n"
		"};\n";

	if (wrappyDoc)
		dump_wrappy_doc(output, modDecls);

	// generate module documentation string
	output <<
		"\n"
		"const char* " << module << "_doc = \"TODO: module doc\";\n";
	if (!nameSpace.empty())
		output <<
			"\n"
			"} // namespace " << nameSpace << "\n";

	output <<
		"\n"
		"extern \"C\"\n"
		<< exportTag << "void\n" <<
		"init" << module << "()\n"
		"{\n";
	output <<
		"\tif (getenv(\"" << module << "Debug\") != NULL)\n"
		"\t\t" << nsPrefix << module << "Debug = atoi(getenv(\""
						<< module << "Debug\"));\n"
		"\tPyObject* module = Py_InitModule4(\"" << module << "\", "
					<< nsPrefix << module << "Methods,\n"
		"\t\t\tconst_cast<char*>(" << nsPrefix << module
							<< "_doc), NULL,\n"
		"\t\t\tPYTHON_API_VERSION);\n"
		"\tPyObject* modDict = PyModule_GetDict(module);\n"
		"\t" << nsPrefix << module
				<< "ErrorObj = PyErr_NewException(\"" 
				<< module << ".error\", NULL, NULL);\n"
		"\tif (" << nsPrefix << module << "ErrorObj)\n"
		"\t\tPyDict_SetItemString(modDict, \"error\", " << nsPrefix
						<< module << "ErrorObj);\n"
		"\tPy_XDECREF(" << nsPrefix << module << "ErrorObj);\n";

	output <<
		"\n"
		"\tPyObject* tmpDict = PyDict_New();\n"
		"\tPyObject* tmpName = PyString_FromString(\"_DummyClass\");\n"
		"\t" << nsPrefix << module
			<< "DummyClass = PyClass_New(NULL, tmpDict, tmpName);\n"
		"\tPy_DECREF(tmpDict);\n"
		"\tPy_DECREF(tmpName);\n";

	// module constants
	output <<
		"\n"
		"\tPyObject* o;\n";
	for (DeclList::const_iterator i = modDecls->begin();
						i != modDecls->end(); ++i) {
		if ((*i)->dtype != Decl::CONSTANT)
			continue;
		static bool warned = false;
		if (!warned)
			std::cerr << programName
			<< ": warning: module constants can be overriden\n";
		warned = true;
		const VarDecl *vd = static_cast<VarDecl *>(*i);
		CvtType cvt(moduleScope, vd->type);
		output <<
			"\to = " << make_buildvalue(cvt.bvFormat(),
				qualify(moduleScope, cvt.bvArg(vd->tag.str()),
									true))
				<< ";\n"
			"\tPyDict_SetItemString(modDict, \"" << (*i)->tag
				<< "\", o);\n"
			"\tPy_DECREF(o);\n";
	}
	for (CDCIMap::iterator i = classInfoMap.begin();
						i != classInfoMap.end(); ++i) {
		const ClassInfo *ci = (*i).second;
		if (ci->skipClass || ci->isPythonClass)
			continue;
		const ClassDecl *cd = ci->cd;
		for (DeclList::const_iterator j = cd->members.begin();
						j != cd->members.end(); ++j) {
			if ((*j)->dtype != Decl::CONSTANT)
				continue;
			const VarDecl *vd = static_cast<VarDecl *>(*j);
			CvtType cvt(cd, vd->type);
			output <<
				"\to = " << make_buildvalue(cvt.bvFormat(),
				    qualify(cd, cvt.bvArg(vd->tag.str()), true))
				    << ";\n"
				"\tPyDict_SetItemString(modDict, \"" << cd->tag
					<< '_' << vd->tag << "\", o);\n"
				"\tPy_DECREF(o);\n";
		}
	}
	
	// module Python types
	for (CDCIMap::iterator i = classInfoMap.begin();
						i != classInfoMap.end(); ++i) {
		const ClassInfo *ci = (*i).second;
		if (ci->skipClass || ci->isPythonClass || ci->isAbstractType)
			continue;
		output <<
			'\t' << nsPrefix << ci->name
				<< "ObjectType.ob_type = &PyType_Type;\n"
			"\tPyDict_SetItemString(modDict, \"" << ci->name
								<< "Type\",\n"
			"\t\t\treinterpret_cast<PyObject*>(&" << nsPrefix
					<< ci->name << "ObjectType));\n";
	}

	// module Python classes
	// make sure we initialize base classes before derived ones
	dumpInitClasses(output);

	if (wrappyDoc)
		output <<
			"\n"
			"\t" << "if (Py_OptimizeFlag == 0)\n"
			"\t\t" << nsPrefix << "make_wrappy_doc(modDict);\n";
	output <<
		"}\n";

	// flush output and check return status
	output.flush();
	if (output.good())
		return dumpModuleHeader();
	std::cerr << programName << ": error writing " << filename << '\n';
	return false;
}

bool
dumpModuleHeader()
{
	// output common includes and shared conversion functions

	string filename(module);
	filename += ".h";
	std::auto_ptr<std::ostream> outs(outputStream(filename).release());
	std::ostream &output = *outs.get();

	output <<
		"#ifndef " << module << "_h\n"
		"# define " << module << "_h\n"
		"\n"
		"# include <stdexcept>\n"
		"# include <new>\n";
	if (!extraHeader.empty())
		output << "# include \"" << extraHeader << "\"\n";
	if (classInfoMap.empty())
		// TODO: or if all classes are skipped
		output << "# include <Python.h>\n";
	for (CDCIMap::iterator i = classInfoMap.begin();
						i != classInfoMap.end(); ++i) {
		const ClassInfo *ci = (*i).second;
		if (ci->skipClass || (ci->isAbstractType && !ci->isPythonClass))
			continue;
		output << "# include \"" << ci->name << "Object.h\"\n";
	}

	if (!nameSpace.empty())
		output <<
			"\n"
			"namespace " << nameSpace << " {\n";
	output <<
		"\n"
		<< exportTag << "extern void " << module << "Error();\n"
		"extern PyObject* " << module << "DummyClass;\n";
	if (!nameSpace.empty())
		output <<
			"\n"
			"} // namespace " << nameSpace << "\n";

	output <<
		"\n"
		"#endif\n";
	// flush output and check return status
	output.flush();
	if (output.good())
		return true;
	std::cerr << programName << ": error writing " << filename << '\n';
	return false;
}
