// Copyright (c) 1998-2000 The Regents of the University of California.
// All rights reserved.
//
// Redistribution and use in source and binary forms are permitted
// provided that the above copyright notice and this paragraph are
// duplicated in all such forms and that any documentation,
// distribution and/or use acknowledge that the software was developed
// by the Computer Graphics Laboratory, University of California,
// San Francisco.  The name of the University may not be used to
// endorse or promote products derived from this software without
// specific prior written permission.
// 
// THIS SOFTWARE IS PROVIDED ``AS IS'' AND WITHOUT ANY EXPRESS OR
// IMPLIED WARRANTIES, INCLUDING, WITHOUT LIMITATION, THE IMPLIED
// WARRANTIES OF MERCHANTIBILITY AND FITNESS FOR A PARTICULAR PURPOSE.
// IN NO EVENT SHALL THE REGENTS OF THE UNIVERSITY OF CALIFORNIA BE
// LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY,
// OR CONSEQUENTIAL DAMAGES ARISING OUT OF THE USE OF THIS SOFTWARE.

// $Id: wrappy.cpp,v 1.124 2001/02/07 01:17:12 gregc Exp $

#include <otf/GetOpt.h>
#include <otf/Symbol.h>
#include <string>
#include <iostream>
#include <algorithm>
#include "common.h"
#include "compute.h"
#include "type.h"
#include "class.h"
#include "module.h"
#include <stdlib.h>
#include <stdio.h>	// for remove
#include <string.h>	// for strrchr

using std::string;
using otf::Symbol;

int processInput();

int
main(int argc, char **argv)
{
	std::ios::sync_with_stdio(false); // we don't need no stinking C
	const char *cp;
	if (argc == 0)
		programName = "unknown";
	else if ((cp = ::strrchr(argv[0], '/')) != NULL)
		programName = cp + 1;
	else
		programName = argv[0];

	int opt;
	otf::GetOpt getopt(argc, argv, "BDITde:g:h:i:m:N:ns:w");
	while ((opt = getopt()) != EOF) switch (opt) {
	  case 'd':
		wrappyDoc = true;
		break;
	  case 'e':
		exportTag = string(getopt.arg()) + ' ';
		ignoreTokens.push_back(getopt.arg());
		break;
	  case 'g':
		gperfMinimum = atoi(getopt.arg());
		break;
	  case 'h':
		extraHeader = getopt.arg();
		break;
	  case 'i':
		ignoreTokens.push_back(getopt.arg());
		break;
	  case 'm':
		module = string(getopt.arg());
		break;
	  case 'N':
		nameSpace = string(getopt.arg());
		break;
	  case 'n':
		noOutput = true;
		break;
	  case 's':
		singleClass = Symbol(getopt.arg());
		break;
	  case 'w':
		warnings = true;
		break;
	  case 'B':
		explicitDestructor = true;
		explicitTemplates = true;
		skipInline = true;
		break;
	  case 'D':
		explicitDestructor = true;
		break;
	  case 'I':
		skipInline = true;
		break;
	  case 'T':
		explicitTemplates = true;
		break;
	  case '?':
		goto usage;
	}

	if (getopt.index() == argc)
		parseHeader("", &globalDecls);
	else for (int i = getopt.index(); i < argc; i++)
		parseHeader(argv[i], &globalDecls);
	return processInput();
usage:
	std::cerr << "usage: " << argv[0] << " [ options ] [ input-file ]\n"
		"-d\t-- generate wrappy_doc dictionary\n"
		"-e tag\t-- import/export tag for Windows DLL's\n"
		"-g min\t-- minimum number of attributes/methods to use gperf\n"
		"-h file\t-- add extra header file to module.h\n"
		"-i token\t-- ignore token\n"
		"-m name\t-- module name\n"
		"-n\t-- no output (just error check input)\n"
		"-N name\t-- namespace name\n"
		"-s name\t-- single class\n"
		"-w\t-- turn on (possibily incorrect) warnings\n"
		"-B\t-- turn on fixes for Borland 5.[45] compilers (-DIT)\n"
		"-D\t-- turn on explicit naming of destructors\n"
		"-I\t-- turn off inline in one case\n"
		"-T\t-- turn on explict naming of template functions\n";
	return 1;
}

int
processInput()
{
	// figure out set of wrapped classes
	computeWrapInformation();

	// if a single class
	if (singleClass == "__module__") {
		if (!dumpModuleCode())
			return 1;
		return 0;
	}
	if (!singleClass.empty()) {
		const ClassDecl *cd = classDeclsMap[Symbol(singleClass)];
		if (cd == NULL && !nameSpace.empty()) {
			string tmp = nameSpace + "::" + singleClass.str();
			cd = classDeclsMap[Symbol(tmp)];
		}
		if (cd == NULL) {
			std::cerr << programName << ": unable to find class "
							<< singleClass << '\n';
			return 1;
		}
		const ClassInfo *ci = classInfoMap[cd];
		if (ci == NULL || ci->skipClass
		|| (ci->isAbstractType && !ci->isPythonClass))
			return 1;
		std::cout << "on class " << cd->name << std::endl;
		switch (ci->isPythonClass) {
		  case true:
			if (!dumpClassHeader(ci) || !dumpClassCode(ci))
				return 1;
			break;
		  case false:
			if (!dumpTypeHeader(ci) || !dumpTypeCode(ci))
				return 1;
			break;
		}
		std::cout << "on module " << module << std::endl;
		if (!dumpModuleCode())
			return 1;
		return 0;
	}
	// foreach wrapped class dump header and code
	for (CDCIMap::iterator i = classInfoMap.begin();
						i != classInfoMap.end(); ++i) {
		const ClassInfo *ci = (*i).second;
		if (ci->skipClass || (ci->isAbstractType && !ci->isPythonClass))
			continue;
		std::cout << "on class " << ci->name << std::endl;
		switch (ci->isPythonClass) {
		  case true:
			if (!dumpClassHeader(ci) || !dumpClassCode(ci))
				return 1;
			break;
		  case false:
			if (!dumpTypeHeader(ci) || !dumpTypeCode(ci))
				return 1;
			break;
		}
	}
	std::cout << "on module " << module << std::endl;
	if (!dumpModuleCode())
		return 1;
	return 0;
}
