// Copyright (c) 1998-2000 The Regents of the University of California.
// All rights reserved.
//
// Redistribution and use in source and binary forms are permitted
// provided that the above copyright notice and this paragraph are
// duplicated in all such forms and that any documentation,
// distribution and/or use acknowledge that the software was developed
// by the Computer Graphics Laboratory, University of California,
// San Francisco.  The name of the University may not be used to
// endorse or promote products derived from this software without
// specific prior written permission.
// 
// THIS SOFTWARE IS PROVIDED ``AS IS'' AND WITHOUT ANY EXPRESS OR
// IMPLIED WARRANTIES, INCLUDING, WITHOUT LIMITATION, THE IMPLIED
// WARRANTIES OF MERCHANTIBILITY AND FITNESS FOR A PARTICULAR PURPOSE.
// IN NO EVENT SHALL THE REGENTS OF THE UNIVERSITY OF CALIFORNIA BE
// LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY,
// OR CONSEQUENTIAL DAMAGES ARISING OUT OF THE USE OF THIS SOFTWARE.

// $Id: method.cpp,v 1.33 2003/07/14 21:32:34 gregc Exp $

#include <sstream>
#include "common.h"
#include "method.h"
#include "compute.h"
#include "CvtType.h"
#include "class.h"
#include "type.h"

using std::string;
using otf::Symbol;

struct ParseInfo {
	string	returnFormat;
	string	returnParams;
	string	funcParams;
	Symbol	cacheReturn;
};

string
dumpParseArgs(std::ostream &output, int indent, const FuncDecl *fd,
	string pyname, ParseInfo *pi, bool attribute, const ClassInfo *ci)
{
	bool isStatic = fd->returnType.compare(0, 7, "static ") == 0;
	string bi(tab(indent));		// base indent
	string proto(fd->tag.str() + '(');
	string returnProto;
	// declare local variables for each argument
	int argCount = 0;
	std::ostringstream parseFormat, parseParams, kwParams;
	std::ostringstream toCppArgs;
	bool hasDefault = false;
	bool needWco = ci ? (ci->name == pyname) : false;
	bool cacheAttribute = false;
	if (attribute) {
		AttrVec::const_iterator i = std::find_if(ci->attrs.begin(),
			ci->attrs.end(), AttrHasName(fd->attrName));
		if (i != ci->attrs.end())
			cacheAttribute = i->cache;
	}
	if (ci && ci->isPythonClass && !isStatic) {
		parseFormat << 'O';
		parseParams << ", &self";
		kwParams << " \"\",";
		toCppArgs << bi << "mustBeInstance(self);\n";
	}
	if (!fd->returnType.empty() && fd->returnType != "void"
	&& fd->returnType != "static void") {
		CvtType rt(fd->scope, fd->returnType);
		pi->returnFormat = rt.bvFormat();
		pi->returnParams = ", "
				+ qualify(fd->scope, rt.bvArg("result"));
		if (cacheAttribute && rt.cache()) {
			pi->cacheReturn = fd->attrName;
			needWco = true;
		}
		returnProto = rt.baseCppType();
	}
	int numInArgs = 0;
	for (ArgList::const_iterator i = fd->args.begin(); i != fd->args.end();
									++i) {
		argCount += 1;
		string type = i->type;
		if (i->out && *type.rbegin() == '*') {
			// TODO: handle output arrays too
			// strip trailing *
			type.erase(type.size() - 1);
		}
		CvtType arg(fd->scope, type);
		if (i->in) {
			++numInArgs;
			output << bi << qualify(fd->scope, arg.aptType());
			output << " ptArg" << argCount;
			if (!i->defValue.empty()) {
				output << "= ";
				if (arg.aptType() == "O")
					output << "NULL";
				else
					output << qualify(fd->scope,
						arg.cppToApt(i->defValue));
			}
			output << ";\n";
			if (!attribute) {
				if (*(proto.end() - 1) != '(')
					proto += ", ";
				proto += arg.baseCppType() + ' '
								+ i->name.str();
			}
		}
		if (!pi->funcParams.empty())
			pi->funcParams += ", ";
		if (i->out)
			pi->funcParams += '&';	// out params need leading &
		pi->funcParams += "cppArg" + itostr(argCount);
		if (i->out) {
			pi->returnFormat += arg.bvFormat();
			pi->returnParams += ", "
				+ qualify(fd->scope,
					arg.bvArg("cppArg" + itostr(argCount)));
			if (cacheAttribute && arg.cache()
			&& pi->returnFormat.size() == 1) {
				pi->cacheReturn = fd->attrName;
				needWco = true;
			}
			if (!attribute) {
				if (!returnProto.empty())
					returnProto += ", ";
				returnProto += arg.baseCppType();
			}
		}
		if (i->in && arg.needTypeCheck()) {
			toCppArgs
				<< bi << "if (!"
					<< arg.typeCheck("ptArg"
							+ itostr(argCount))
					<< ") {\n"
				<< bi << "\tPyErr_SetString(PyExc_TypeError, \"argument "
					<< argCount << " should be a "
					// TODO: strip * from cppType()
					<< arg.cppType() << "\");\n"
				<< bi << "\treturn NULL;\n"
				<< bi << "}\n";
		}
		if (cacheAttribute && !i->out && arg.cache()
		&& !fd->attrName.empty()) {
			// Note: all arguments that need to be cached
			// are PyObject*'s.
			// TODO: cache other arguments
			needWco = true;
			toCppArgs << bi << "// cache attribute\n"
				<< bi << "static PyObject* ca" << argCount
									<< ";\n"
				<< bi << "if (ca" << argCount << " == NULL)\n"
				<< bi << "\tca" << argCount
					<< " = PyString_InternFromString(\"__cached_"
					<< fd->attrName << "__\");\n"
				<< bi << "PyDict_SetItem(wco->in_dict, ca"
					<< argCount << ", ptArg" << argCount
					<< ");\n";
		}
		toCppArgs << bi << qualify(fd->scope, arg.cppType())
						<< " cppArg" << argCount;
		if (i->in)
			toCppArgs << " = " << qualify(fd->scope,
				arg.aptToCpp("ptArg" + itostr(argCount)));
		toCppArgs << ";\n";
		if (!i->in)
			continue;
		if (!i->defValue.empty() && !hasDefault) {
			hasDefault = true;
			parseFormat << '|';
		}
		parseFormat << arg.aptFormat();
		parseParams << ", &ptArg" << argCount;
		if (hasDefault)
			kwParams << " \"" << i->name << "\",";
		else
			kwParams << " \"\",";
	}
	parseFormat << ':' << pyname;
	output << bi << "static char* kwlist[] = {" << kwParams.str()
								<< " NULL };\n"
		<< bi << "if (!PyArg_ParseTupleAndKeywords(args, keywds, \""
			<< parseFormat.str() << "\", kwlist"
			<< parseParams.str() << "))\n"
		<< bi << "\treturn NULL;\n";
	if (ci && ci->isPythonClass) {
		if (needWco)
			output
				<< bi << "PyInstanceObject* wco = reinterpret_cast<PyInstanceObject*>(self);\n";
		if (!isStatic) {
			string qualName
				= qualify(ci->cd->scope, ci->name.str());
			output << bi << qualName << "* inst = ";
			if (ci->isWrappySubclass)
				output << "dynamic_cast<" << qualName <<
					"*>(const_cast<otf::WrapPyObj*>(otf::wrappyObject(self)));\n";
			else
				output << "reinterpret_cast<" << qualName <<
					"*>(otf::wrappyVoidObject(self));\n";
			output << bi << "if (inst == NULL)\n";
			bool isConstructor = ci->name == fd->tag;
			if (isConstructor)
				output << bi << "\tPyErr_Clear();\n";
			else
				output << bi << "\treturn NULL;\n";
		}
	}
	argCount = 0;
	for (ArgList::const_iterator i = fd->args.begin(); i != fd->args.end();
									++i) {
		// handle default arguments that are PyObject's
		argCount += 1;
		if (!i->in || i->defValue.empty())
			continue;
		string type = i->type;
		if (i->out && *type.rbegin() == '*') {
			// strip trailing *
			type.erase(type.size() - 1);
		}
		CvtType arg(fd->scope, type);
		if (!(arg.aptType() == "O"))
			continue;
		output << bi << "if (ptArg" << argCount << " == NULL)\n"
			<< bi << "\tptArg" << argCount << " =  "
				<< make_buildvalue(arg.bvFormat(), 
					qualify(fd->scope,
							arg.bvArg(i->defValue)))
				<< ";\n";
	}
	output << toCppArgs.str();
	if (attribute)
		return string();
	proto += ')';
	if (!returnProto.empty())
		proto = proto + " -> " + returnProto;
	return proto + '\n';
}

string
dumpConstructorBody(std::ostream &output, int indent, const ClassInfo *ci,
					const FuncDecl *fd, string pyname)
{
	string bi(tab(indent));		// base indent
	ParseInfo pi;
	string proto = dumpParseArgs(output, indent, fd, pyname, &pi, false,
									ci);
	if (ci->isPythonClass) {
		string qualName = qualify(ci->cd->scope, ci->name.str());
		output
			<< bi << "if (inst != NULL) {\n";
		
		if (ci->isWrappySubclass)
			output << bi << "\tinst->wpyDisassociate();\n";
		output
			<< bi << "\tdelete inst;\n"
			<< bi << "}\n"
			<< bi << "inst = new " << qualName << '('
						<< pi.funcParams << ");\n"
			<< bi << "if (" << module << "Debug >= 6)\n"
			<< bi << "\tstd::cerr << \"Allocate " << ci->name
						<< ": \" << inst << '\\n';\n";
		if (ci->isWrappySubclass)
			output
				<< bi << "inst->wpyAssociate(self);\n"
				<< bi << "inst->setPyOwned();\n"
				<< bi << "PyObject* tmp = otf::WrapPyProxy_FromWrapPyObj(inst, \"" << ci->name << "\", &" << module << "Debug);\n";
		else
			output
				<< bi << "PyObject* tmp = PyCObject_FromVoidPtr(inst, " << ci->name << "Destroy);\n";
		output
			<< bi << "PyDict_SetItem(wco->in_dict, otf::WrapPyString, tmp);\n"
			<< bi << "Py_DECREF(tmp);\n";
	} else {
		string qualName = qualify(ci->cd->scope, ci->name.str());
		if (ci->isWrappySubclass)
			output << bi << qualName << "* inst = new "
					<< qualName << '(' << pi.funcParams
					<< ");\n"
				<< bi << "pInst = inst->wpyGetObject(otf::PWC_CREATE_AND_OWN);\n";
		else {
			output << bi << qualName << " tmp";
			if (!pi.funcParams.empty())
				output << '(' << pi.funcParams << ')';
			output << ";\n" << bi;
			if (!explicitTemplates)
				output << "pInst = otf::pyObject(tmp);\n";
			else
				output << "pInst = otf::pyObject<" << qualName
								<< ">(tmp);\n";
		}
	}
	return proto;
}

string
dumpMethodBody(std::ostream &output, int indent, const ClassInfo *ci,
			const FuncDecl *fd, string pyname, bool attribute)
{
	string bi(tab(indent));		// base indent
	ParseInfo pi;
	string proto = dumpParseArgs(output, indent, fd, pyname, &pi,
								attribute, ci);
	string returnType;
	bool isStatic = fd->returnType.compare(0, 7, "static ") == 0;
	if (isStatic)
		returnType = fd->returnType.substr(7);
	else
		returnType = fd->returnType;
	output << bi;
	if (!(returnType == "void"))
		output << qualify(fd->scope, returnType) << " result = ";
	if (ci != NULL && !isStatic)
		// class function
		output << "inst->" << fd->tag;
	else if (ci != NULL)
		// class static function
		output << qualify(ci->cd->scope, ci->name.str()) << "::"
								<< fd->tag;
	else
		// module function
		output << qualify(fd->scope, fd->tag.str());
	output << '(' << pi.funcParams << ");\n";
	if (!pi.cacheReturn.empty())
		output << bi << "// cache attribute\n"
			<< bi << "static PyObject* ca;\n"
			<< bi << "if (ca == NULL)\n"
			<< bi << "\tca = PyString_InternFromString(\"__cached_"
						<< pi.cacheReturn << "__\");\n"
			<< bi << "PyObject* pyResult = ";
	else
		output << bi << "return ";
	output << make_buildvalue(pi.returnFormat, pi.returnParams) << ";\n";
	if (!pi.cacheReturn.empty()) {
		output << bi << "PyDict_SetItem(wco->in_dict, ca, pyResult);\n"
			<< bi << "return pyResult;\n";
	}
	return proto;
}

typedef std::multimap<int, const FuncDecl *> FMethods;

bool
sortMethods(FMethods *methods, MethodMap::const_iterator start,
						MethodMap::const_iterator stop)
{
	// return true if all methods have the same number of arguments.
	int count = -1;
	// sort methods by number of arguments
	for (MethodMap::const_iterator i = start; i != stop; ++i) {
		const FuncDecl *fd = i->second;
		int numInputArgs = 0;
		for (ArgList::const_iterator j = fd->args.begin();
						j != fd->args.end(); ++j) {
			if (!j->defValue.empty())
				break;
			if (j->in)
				++numInputArgs;
		}
		if (count == -1)
			count = numInputArgs;
		else if (count > -1 && count != numInputArgs)
			count = -2;
		methods->insert(FMethods::value_type(numInputArgs, fd));
	}
	return count != -2;
}

typedef std::map<const FuncDecl *, ArgList::size_type> UniArgs;

void
uniqueArgs(UniArgs *uniargs, FMethods::iterator first, FMethods::iterator last)
{
	for (FMethods::iterator i = first; i != last; ++i) {
		const FuncDecl *fd = i->second;
		ArgList::size_type diffArgPos = ~0;
		for (ArgList::size_type j = 0; j != fd->args.size(); ++j) {
			// TODO: only consider input arguments
			// (this means arguments in different functions
			// could match different positions!)
			const Argument &a = fd->args[j];
			string type(a.type);
			if (a.out && *type.rbegin() == '*') {
				// TODO: handle output arrays
				// strip trailing *
				type.erase(type.size() - 1);
			}
			CvtType arg(fd->scope, type);
			// is this arg different from all the other args?
			for (FMethods::iterator k = first; k != last; ++k) {
				if (k == i)
					continue;
				const FuncDecl *fd2 = k->second;
				if (j >= fd2->args.size())
					goto try_again;
				const Argument &a2 = fd2->args[j];
				string type2(a2.type);
				if (a2.out && *type2.rbegin() == '*') {
					// TODO: handle output arrays
					// strip trailing *
					type2.erase(type2.size() - 1);
				}
				CvtType arg2(fd2->scope, type2);
				if (arg.typeCheck("?") == arg2.typeCheck("?"))
					goto try_again;
			}
			diffArgPos = j;
			goto found;
		try_again:;
		}
		std::cerr << "unable to discern different methods: "
				<< qualify(fd->scope, fd->tag.str()) << '\n';
	found:
		(*uniargs)[fd] = diffArgPos;
	}
}

string
dumpMethod(std::ostream &output, const ClassInfo *ci,
	MethodMap::const_iterator start, MethodMap::const_iterator stop,
	bool needSelf, string fname, string pyname, bool attribute)
{
	bool isConstructor = ci != NULL && ci->name == start->first;
	bool isStatic = start->second->returnType.compare(0, 7, "static ") == 0;
	string prefix;
	if (ci == NULL || (isConstructor && !ci->isPythonClass))
		prefix = module;
	else if (isStatic)
		prefix = module + '_' + ci->name.str();
	else
		prefix = ci->name.str();
	if (fname.empty())
		fname = prefix + '_' + start->first.str();
	if (pyname.empty())
		pyname = start->first.str();
	string self;
	if (needSelf && (ci == NULL || !ci->isPythonClass))
		self = " self";
	output <<
		"\n"
		"extern \"C\"\n"
		<< exportTag << "PyObject*\n" <<
		fname << "(PyObject*" << self
				<< ", PyObject* args, PyObject* keywds)\n"
		<< "{\n";
	if (ci != NULL && !isStatic && !self.empty()) {
		if (!ci->isPythonClass) {
			string qualName
				= qualify(ci->cd->scope, ci->name.str());
			output <<
				"\t" << prefix
					<< "Object* wco = reinterpret_cast<"
					<< prefix << "Object*>(self);\n"
				"\t" << qualName << "* inst = wco->inst;\n";
			if (!isConstructor && ci->isWrappySubclass)
				output <<
					"\tif (inst == NULL) {\n"
					"\t\tthrow std::runtime_error(\"C++ "
						<< prefix
						<< " instance gone\");\n"
					"\t}\n";
		}
	}
	output <<
		"\ttry {\n";
	int maxInputArgs = 0;
	for (MethodMap::const_iterator i = start; i != stop; ++i) {
		const FuncDecl &fd = *i->second;
		int numInputArgs = 0;
		for (ArgList::const_iterator j = fd.args.begin();
							j != fd.args.end(); ++j)
			if (j->in)
				++numInputArgs;
		if (numInputArgs > maxInputArgs)
			maxInputArgs = numInputArgs;
	}
	if (attribute && maxInputArgs > 1) {
		// attribute hack
		output <<
			"\t\t// make 'i.a = x, y, z' work\n";
		if (!ci->isPythonClass)
			output <<
				"\t\tif (PyTuple_Size(args) == 1\n"
				"\t\t&& PyTuple_Check(PyTuple_GetItem(args, 0)))\n"
				"\t\t\targs = PyTuple_GetItem(args, 0);\n";
		else {
			output <<
				"\t\tPyObject* newargs = NULL;\n"
				"\t\tif (PyTuple_Size(args) == 2\n"
				"\t\t&& PyTuple_Check(PyTuple_GetItem(args, 1))) {\n"
				"\t\t\tPyObject* tmp = PyTuple_GetItem(args, 1);\n"
				"\t\t\tint len = PyTuple_Size(tmp);\n"
				"\t\t\tnewargs = PyTuple_New(1 + len);\n"
				"\t\t\tPyObject* self = PyTuple_GetItem(args, 0);\n"
				"\t\t\tPy_INCREF(self);\n"
				"\t\t\tPyTuple_SetItem(newargs, 0, self);\n"
				"\t\t\tfor (int i = 0; i < len; ++i) {\n"
				"\t\t\t\tPyObject* arg = PyTuple_GetItem(tmp, i);\n"
				"\t\t\t\tPy_INCREF(arg);\n"
				"\t\t\t\tPyTuple_SetItem(newargs, i + 1, arg);\n"
				"\t\t\t}\n"
				"\t\t\targs = newargs;\n"
				"\t\t}\n";
		}
	}
	if (ci && ci->isPythonClass && !isStatic)
		output <<
			"\t\tPyObject* self;\n";
	else if (isConstructor && !ci->isPythonClass)
		output <<
			"\t\tPyObject *pInst = 0;\n";
	string proto;
	if (std::distance(start, stop) == 1)
		if (isConstructor)
			proto = dumpConstructorBody(output, 2, ci,
							start->second, pyname);
		else
			proto = dumpMethodBody(output, 2, ci, start->second,
							pyname, attribute);
	else {
		// sort by number of arguments and handle multiple
		// versions with same number of arguments
		FMethods methods;
		bool needSwitch = !sortMethods(&methods, start, stop);
		if (needSwitch)
			output <<
				"\t\tswitch (PyTuple_Size(args)) {\n"
				"\t\t  default:\n"
				"\t\t\tthrow std::runtime_error(\"wrong number of arguments\");\n";
		string bi(tab(2));		// base indent
		if (needSwitch)
			bi += '\t';
		for (FMethods::iterator i = methods.begin();
							i != methods.end(); ) {
			int numArgs = i->first;
			FMethods::iterator next = methods.upper_bound(numArgs);
			if (ci && ci->isPythonClass && !isStatic)
				numArgs += 1;
			if (needSwitch)
				output << "\t\t  case " << numArgs << ": {\n";
			FMethods::size_type count = std::distance(i, next);
			UniArgs uniargs;
			if (count > 1)
				uniqueArgs(&uniargs, i, next);
			for (FMethods::iterator j = i; j != next; ++j) {
				const FuncDecl *fd = j->second;
				int indent = needSwitch ? 3 : 2;
				if (count > 1) {
					// discern this case
					ArgList::size_type argNum = uniargs[fd];
					if (argNum == ~0)
						continue;
					const Argument &a = fd->args[argNum];
					if (ci && ci->isPythonClass && !isStatic)
						++argNum;
					string type(a.type);
					if (a.out && *type.rbegin() == '*') {
						// TODO: handle output arrays
						// strip trailing *
						type.erase(type.size() - 1);
					}
					CvtType arg(fd->scope, type);
					output << bi;
					if (j != i)
						output << /*{*/ "} else ";
					output << "if (PyTuple_Size(args) > "
						<< argNum << "\n"
						<< bi << "&& ("
						<< arg.typeCheck(
							"PyTuple_GetItem(args, "
							+ itostr(argNum) + ")")
						<< ")) {\n"; /*}*/
					indent = needSwitch ? 4 : 3;
				}
				if (isConstructor) {
					proto += dumpConstructorBody(output,
						indent, ci, fd, pyname);
					if (needSwitch)
						output << bi << "break;\n";
				} else
					proto += dumpMethodBody(output, indent,
						ci, fd, pyname, attribute);
				if (needSwitch && count > 1)
					output << bi << /*}*/ "}\n";
			}
			if (count > 1) {
				output << bi << /*{*/ "} else {\n"
					<< bi << "\tthrow std::runtime_error(\"argument list mismatch\");\n"
					<< bi << "}\n";
				if (needSwitch)
					output << bi << "break;\n";
			}
			if (needSwitch)
				output << "\t\t  }\n";
			i = next;
		}
		if (needSwitch)
			output <<
				"\t\t}\n";
	}
	if (isConstructor) {
		if (ci->isPythonClass) {
			dumpClassAttrInit(output, 2, ci, false);
			output <<
				"\t\tPy_INCREF(Py_None);\n"
				"\t\treturn Py_None;\n";
		} else {
			dumpTypeAttrInit(output, 2, ci);
			output <<
				"\t\treturn pInst;\n";
		}
	}
	output <<
		"\t} catch (...) {\n"
		"\t\t" << module << "Error();\n"
		"\t}\n"
		"\treturn NULL;\n"
		"}\n";
	return proto;
}
