// Copyright (c) 1998-2000 The Regents of the University of California.
// All rights reserved.
//
// Redistribution and use in source and binary forms are permitted
// provided that the above copyright notice and this paragraph are
// duplicated in all such forms and that any documentation,
// distribution and/or use acknowledge that the software was developed
// by the Computer Graphics Laboratory, University of California,
// San Francisco.  The name of the University may not be used to
// endorse or promote products derived from this software without
// specific prior written permission.
// 
// THIS SOFTWARE IS PROVIDED ``AS IS'' AND WITHOUT ANY EXPRESS OR
// IMPLIED WARRANTIES, INCLUDING, WITHOUT LIMITATION, THE IMPLIED
// WARRANTIES OF MERCHANTIBILITY AND FITNESS FOR A PARTICULAR PURPOSE.
// IN NO EVENT SHALL THE REGENTS OF THE UNIVERSITY OF CALIFORNIA BE
// LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY,
// OR CONSEQUENTIAL DAMAGES ARISING OUT OF THE USE OF THIS SOFTWARE.

// $Id: type.cpp,v 1.23 2003/06/17 22:26:46 gregc Exp $

// dump output for Python types

#include <fstream>
#include "common.h"
#include "method.h"
#include "type.h"
#include "CvtType.h"
#include <stdio.h>
#include <stdlib.h>

using std::string;
using otf::Symbol;

static void
dumpWrapTypeInline(std::ostream &output, const ClassInfo *ci)
{
	string prefix(qualify(ci->cd->scope, ci->name.str(), true));
	string objectName;
	if (!nameSpace.empty())
		objectName = nameSpace + "::";
	objectName += ci->name.str() + "Object";
	output <<
		"\n"
		"namespace otf {\n"
		// type check function
		"\n"
		"template <> inline bool\n"
		"WrapPyType<" << prefix << ">::check(PyObject* o, bool";
	if (ci->isWrappySubclass)
		output << " noneOk";
	output <<
		")\n"
		"{\n";
	if (ci->isWrappySubclass)
		output <<
			"\tif (noneOk && o == Py_None)\n"
			"\t\treturn true;\n";
	output <<
		"\treturn o->ob_type == &" << objectName << "Type;\n"
		"}\n";

	if (ci->isWrappySubclass)
		output <<
			"\n"
			"template <> " << exportTag << "PyObject* pyObject("
						<< prefix << " const* o);\n"
			// don't understand yet why this is necessary 12/2000
			"template <> inline PyObject* pyObject(" << prefix
				<< "* o) { return pyObject<" << prefix
				<< " const*>(o); }\n"
			"template <> " << exportTag << "PyObject* WrapPy<"
					<< prefix << ">::wpyNew() const;\n"
			"template <> " << exportTag << "void WrapPy<"
					<< prefix << ">::wpyDelete();\n";
	else
		output <<
			"\n"
			"template <> " << exportTag << "PyObject* pyObject("
							<< prefix << " o);\n";
	output <<
		"\n"
		"} // namespace otf\n";
}

void
dumpWrapTypeOutline(std::ostream &output, const ClassInfo *ci)
{
	string prefix(qualify(ci->cd->scope, ci->name.str(), true));
	string objectName;
	if (!nameSpace.empty())
		objectName = nameSpace + "::";
	objectName += ci->name.str() + "Object";
	output <<
		"\n"
		"namespace otf {\n";
	string debugPrefix;
	if (!nameSpace.empty())
		debugPrefix = nameSpace + "::";
	if (!ci->isWrappySubclass) {
		output <<
			"\n"
			"template <> " << exportTag << "PyObject*\n"
			"pyObject(" << prefix << " o)\n"
			"{\n"
			"\t" << objectName << "* wco = PyObject_NEW("
				<< objectName << ", &" << objectName
				<< "Type);\n"
			"\tif (" << debugPrefix << module << "Debug >= 6)\n"
			"\t\tstd::cerr << \"Allocate " << ci->name
						<< ": \" << wco << '\\n';\n"
			"\tif (wco == NULL)\n"
			"\t\treturn NULL;\n"
			"\twco->in_dict = PyDict_New();\n"
			"\tnew (&wco->instData) " << prefix << "(o);\n"
			"\twco->inst = reinterpret_cast<" << prefix
						<< "*>(&wco->instData);\n"
			"\treturn static_cast<PyObject*>(wco);\n"
			"}\n";
	}

	if (!ci->isWrappySubclass) {
		output <<
			"\n" /*{*/
			"} // namespace otf\n";
		return;
	}

	output <<
		"\n"
		"template <> " << exportTag << "PyObject*\n"
		"pyObject(" << prefix << " const* o)\n"
		"{\n"
		"\tif (o == NULL) {\n"
		"\t\tPy_INCREF(Py_None);\n"
		"\t\treturn Py_None;\n"
		"\t}\n"
		"\treturn o->wpyGetObject();\n"
		"}\n";

	output <<
		// create function
		"\n"
		"template <> " << exportTag << "PyObject*\n"
		"WrapPy<" << prefix << ">::wpyNew() const\n"
		"{\n"
		"\t" << objectName << "* wco = PyObject_NEW("
			<< objectName << ", &" << objectName
			<< "Type);\n"
		"\tif (" << debugPrefix << module << "Debug >= 6)\n"
		"\t\tstd::cerr << \"Allocate " << ci->name
					<< ": \" << wco << '\\n';\n"
		"\tif (wco == NULL)\n"
		"\t\treturn NULL;\n"
		"\twco->in_dict = PyDict_New();\n"
		"\twco->inst = const_cast<" << prefix << "*>(static_cast<"
					<< prefix << " const*>(this));\n"
		"\towner = false;\n"
		"\treturn static_cast<PyObject*>(wco);\n"
		"}\n"
		// clear instance function
		"\n"
		"template <> " << exportTag << "void\n"
		"WrapPy<" << prefix << ">::wpyDelete()\n"
		"{\n"
		"\t// called by wrapped class destructor\n"
		"\t" << objectName << "* wco = static_cast<" << objectName
							<< "*>(pyObj);\n"
		"\twco->inst = NULL;\n"
#if 0
		// Bad idea because instance dictionary may hold stuff
		// (e.g., a Tk widget) that needs to be cleaned up.
		"\tPyDict_Clear(inst->in_dict);\n"
#else
		"\t// clear cached attributes\n"
		"\tint pos = 0;\n"
		"\tPyObject* key = 0;\n"
		"\twhile (PyDict_Next(wco->in_dict, &pos, &key, NULL)) {\n"
		"\t\tif (!PyString_Check(key))\n"
		"\t\t\tcontinue;\n"
		"\t\tchar const* s = PyString_AS_STRING(key);\n"
		"\t\tif (s[0] == '_' && strncmp(s, \"__cached_\", 9) == 0)\n"
		"\t\t\tPyDict_DelItem(wco->in_dict, key);\n"
		"\t}\n"
#endif
		"\tif (!owner)\n"
		"\t\tPy_DECREF(pyObj);\n"
		"}\n"
		"\n"
		"} // namespace otf\n";
}

bool
dumpTypeHeader(const ClassInfo *ci)
{
	string filename(ci->name.str());
	filename += "Object.h";
	std::auto_ptr<std::ostream> outs(outputStream(filename).release());
	std::ostream &output = *outs.get();

	string define(ci->name.str() + "Object_h");
	if (!nameSpace.empty())
		define = nameSpace + '_' + define;
	output <<
		"#ifndef " << define << "\n"
		"# define " << define << "\n"
		"\n"
		"# include <Python.h>\n"
		"# include <otf/WrapPy.h>\n";
	output << ci->includeFile << "\n";
	if (!nameSpace.empty())
		output <<
			"\n"
			"namespace " << nameSpace << " {\n";
	output <<
		"\n"
		<< exportTag << "extern int " << module << "Debug;\n"
	// object definition
		"\n"
		"extern \"C\" {\n"
		"struct " << exportTag << ci->name
					<< "Object: public PyObject {\n"
		"\tPyObject* in_dict;\n"
		"\t" << qualify(ci->cd->scope, ci->name.str()) << "* inst;\n";
	if (!ci->isWrappySubclass)
		output <<
			"\tchar instData[sizeof ("
				<< qualify(ci->cd->scope, ci->name.str())
				<< ")];\n";
	output <<
		"};\n"
		"\n"
		<< exportTag << "extern PyTypeObject " << ci->name
							<< "ObjectType;\n";
	if (!ci->attrs.empty()) {
		bool first = true;
		for (AttrVec::const_iterator i = ci->attrs.begin();
						i != ci->attrs.end(); ++i) {
			if (!i->cache)
				continue;	// skip primitive types
			if (first) {
				first = false;
				output <<
					"// externs for cached attributes\n";
			}
			string func(i->get.substr(0, i->get.find('(')));
			if (ci->variables.end()
			!= std::find_if(ci->variables.begin(),
					ci->variables.end(),
					AttrHasName(i->name)))
				// a class variable
				output <<
					"extern PyObject* " << func << "("
						<< ci->name << "Object*);\n";
			else // a class function
				output <<
					"extern PyObject* " << func
					<< "(PyObject*, PyObject*, PyObject*);\n";
		}
	}
	output <<
		"\n"
		"} // extern \"C\"\n";

	if (!nameSpace.empty())
		output <<
			"\n"
			"} // namespace " << nameSpace << "\n";

	dumpWrapTypeInline(output, ci);

	output <<
		"\n"
		"#endif\n";
	// flush output and check return status
	output.flush();
	if (output.good())
		return true;
	std::cerr << programName << ": error writing " << filename << '\n';
	return false;
}

static void
dumpDeallocator(std::ostream &output, const ClassInfo *ci)
{
	output <<
		"\n"
		"extern \"C\"\n"
		<< exportTag << "void\n" <<
		ci->name << "_dealloc(PyObject* self)\n"
		"{\n"
		"\t" << ci->name << "Object* wco = static_cast<" << ci->name
							<< "Object*>(self);\n"
		"\tif (" << module << "Debug >= 6)\n"
		"\t\tstd::cerr << \"Deallocate " << ci->name
						<< ": \" << wco << '\\n';\n"
		"\tif (wco->inst) {\n";
	if (!ci->isWrappySubclass) {
		output << "\t\twco->inst->";
		if (explicitDestructor)
			output << qualify(ci->cd->scope, ci->name.str())
								<< "::";
		output <<
			"~" << ci->name
			<< "(); // freed in Python's C layer\n";
	} else if (ci->cd->members.has(Symbol('~' + ci->name.str()), PUBLIC))
		output <<
			"\t\tif (wco->inst->pyOwned())\n"
			"\t\t\tdelete wco->inst;\n"
			"\t\telse\n"
			"\t\t\twco->inst->wpyDisassociate();\n";
	output <<
		"\t\twco->inst = 0;\n"
		"\t}\n"
		"\tif (wco->ob_refcnt > 0) {\n"
		"\t\tif (" << module << "Debug >= 6)\n"
		"\t\t\tstd::cerr << \"Resurrected " << ci->name
					<< ": \" << wco << '\\n';\n"
		"\t\treturn; // somehow the object was resurrected\n"
		"\t}\n"
		"\tPy_XDECREF(wco->in_dict);\n"
		"\tPyMem_DEL(self);\n"
		"}\n";
}

static void
dumpPrint(std::ostream &output, const ClassInfo *ci, const string &fname)
{
	// TODO: if str() method, then use it
	output <<
		"\n"
		"extern \"C\"\n"
		<< exportTag << "int\n" <<
		fname << "(PyObject* self, FILE* fp, int /*flags*/)\n"
		"{\n"
		"\t" << ci->name << "Object* wco = static_cast<" << ci->name
							<< "Object*>(self);\n";
	if (ci->isWrappySubclass)
		output <<
			"\tif (wco->inst == NULL)\n"
			"\t\treturn 0;\n";
	output <<
		"\ttry {\n";
	if (ci->print)
		output <<
			"\t\tstd::ostringstream buf;\n"
			"\t\tbuf << " << (ci->isWrappySubclass ? "*" : "")
							<< "wco->inst;\n"
			"\t\tfputs(buf.str().c_str(), fp);\n";
	if (!ci->str.empty()
	&& ci->str.find(Symbol("str")) != ci->str.end())
		output <<
			"\t\tfputs(wco->inst->str().c_str(), fp);\n";
	else if (!ci->str.empty()
	&& ci->str.find(Symbol("c_str")) != ci->str.end())
		output <<
			"\t\tfputs(wco->inst->c_str(), fp);\n";
	output <<
		"\t} catch (...) {\n"
		"\t\t" << module << "Error();\n"
		"\t}\n"
		"\treturn 0;\n"
		"}\n";
}

static void
dumpStr(std::ostream &output, const ClassInfo *ci, const string &fname)
{
	output <<
		"\n"
		"extern \"C\"\n"
		<< exportTag << "PyObject*\n" <<
		fname << "(PyObject* self)\n"
		"{\n"
		"\t" << ci->name << "Object* wco = static_cast<" << ci->name
							<< "Object*>(self);\n";
	if (ci->isWrappySubclass)
		output <<
			"\tif (wco->inst == NULL) {\n"
			"\t\tPy_INCREF(Py_None);\n"
			"\t\treturn Py_None;\n"
			"\t}\n";
	output <<
		"\ttry {\n";
	if (!ci->str.empty()
	&& ci->str.find(Symbol("str")) != ci->str.end())
		output <<
			"\t\treturn PyString_FromString(wco->inst->str().c_str());\n";
	else if (!ci->str.empty()
	&& ci->str.find(Symbol("c_str")) != ci->str.end())
		output <<
			"\t\treturn PyString_FromString(wco->inst->c_str());\n";
	else if (ci->print)
		output <<
			"\t\tstd::ostringstream buf;\n"
			"\t\tbuf << " <<  "*wco->inst;\n"
			"\t\treturn PyString_FromString(buf.str().c_str());\n";
	output <<
		"\t} catch (...) {\n"
		"\t\t" << module << "Error();\n"
		"\t}\n"
		"\treturn NULL;\n"
		"}\n";
}

static void
dumpCompare(std::ostream &output, const ClassInfo *ci, const string &fname)
{
	output <<
		"\n"
		"extern \"C\"\n"
		<< exportTag << "int\n" <<
		fname << "(PyObject* o0, PyObject* o1)\n"
		"{\n"
		"\t" << ci->name << "Object* wco0 = dynamic_cast<" << ci->name
							<< "Object*>(o0);\n"
		"\t" << ci->name << "Object* wco1 = dynamic_cast<" << ci->name
							<< "Object*>(o1);\n"
		"\tif (wco0 != NULL && wco1 != NULL) {\n"
		"\t\tif (*wco0->inst < *wco1->inst)\n"
		"\t\t\treturn -1;\n"
		"\t\telse if (*wco1->inst < *wco0->inst)\n"
		"\t\t\treturn 1;\n"
		"\t\treturn 0;\n"
		"\t}\n"
		"\t// TODO: compare with other types\n"
		"\treturn strcmp(o0->ob_type->tp_name, o1->ob_type->tp_name);\n"
		"}\n";
}

static void
dumpCall(std::ostream &output, const ClassInfo *ci, const string &fname)
{
	output <<
		"\n"
		"extern \"C\"\n"
		<< exportTag << "long\n" <<
		fname << "(PyObject* o)\n"
		"{\n"
		"\t" << ci->name << "Object* wco = static_cast<" << ci->name
							<< "Object*>(o);\n";
	// TODO
	output <<
		"}\n";
}

static void
dumpHash(std::ostream &output, const ClassInfo *ci, const string &fname)
{
	output <<
		"\n"
		"extern \"C\"\n"
		<< exportTag << "long\n" <<
		fname << "(PyObject* o)\n"
		"{\n"
		"\t" << ci->name << "Object* wco = static_cast<" << ci->name
							<< "Object*>(o);\n"
		"\treturn wco->inst->hash();\n"
		"}\n";
}

static void
dumpPtrHash(std::ostream &output, const ClassInfo */*ci*/, const string &fname)
{
	// assert(ci->isWrappySubclass);
	output <<
		"\n"
		"extern \"C\"\n"
		<< exportTag << "long\n" <<
		fname << "(PyObject* o)\n"
		"{\n"
		"\treturn _Py_HashPointer(o);"
		"}\n";
}

static bool
createGperf(const ClassInfo *ci)
{
	if (noOutput)
		return true;
	// randomize names
	typedef std::vector<Symbol> SymVec;
	SymVec names;
	AttrVec::size_type size = ci->attrs.size() + ci->methods.size() + ci->constants.size();
	names.reserve(size);
	for (AttrVec::const_iterator i = ci->attrs.begin();
						i != ci->attrs.end(); ++i)
		names.push_back(i->name);
	MethodMap::const_iterator next;
	for (MethodMap::const_iterator i = ci->methods.begin();
					i != ci->methods.end(); i = next) {
		next = ci->methods.upper_bound(i->first);
		names.push_back(i->first);
	}
	for (AttrVec::const_iterator i = ci->constants.begin();
						i != ci->constants.end(); ++i)
		names.push_back(i->name);
#if 0
	std::random_shuffle(names.begin(), names.end());
#endif

	// TODO: replace /tmp/ with TMPDIR for NT compat?
	static char tmpFilename[L_tmpnam] = "";
	if (tmpFilename[0] == '\0')
		(void) ::tmpnam(tmpFilename);
	if (tmpFilename[0] == '\0')
		return false;
	std::ofstream os(tmpFilename);
	os << "%{\n"
		"enum " << ci->name << "AttrEnum {\n";
	const char *sep = "";
	for (SymVec::iterator i = names.begin(); i != names.end(); ++i) {
		os << sep << "\t" << ci->name << "Attr_" << *i;
		sep = ",\n";
	}
	os << "\n"
		"};\n"
		"%}\n"
		"struct " << ci->name <<
				"Attr { const char* name; " << ci->name <<
				"AttrEnum attr; }\n"
		"%%\n";
	for (SymVec::iterator i = names.begin(); i != names.end(); ++i)
		os << *i << ", " << ci->name << "Attr_" << *i << "\n";
	os.close();

	string cmd("gperf -tCE -L C++ -Z " + ci->name.str()
		+ "AttrLookup -N find " + tmpFilename + " > " + ci->name.str()
		+ "Attr.cpp");
	bool success = ::system(cmd.c_str()) == 0;
	if (!success) {
		// try again with the -D option to handle collisions
		cmd.insert(9, "D");
		success = ::system(cmd.c_str()) == 0;
	}
	::remove(tmpFilename);
	return success;
}

static void
dumpGperfSetAttrO(std::ostream &output, const ClassInfo *ci, const string &fname)
{
	output <<
		"\n"
		"extern \"C\"\n"
		<< exportTag << "int\n" <<
		fname << "(PyObject* self, PyObject* attr, PyObject* value)\n"
		"{\n"
		"\t" << ci->name << "Object* wco = static_cast<" << ci->name
							<< "Object*>(self);\n";
	if (ci->isWrappySubclass)
		output <<
			"\tif (wco->inst == NULL) {\n"
			"\t\tPyErr_SetString(PyExc_RuntimeError, \"C++ "
					<< ci->name << " instance gone\");\n"
			"\t\treturn -1;\n"
			"\t}\n";
	output <<
		"\tchar const* aname = PyString_AsString(attr);\n"
		"\tif (aname == NULL)\n"
		"\t\treturn -1;\n"
		"\t" << ci->name << "Attr const* ca = " << ci->name
				<< "AttrLookup::find(aname, PyString_Size(attr));\n"
		"\tif (ca == NULL) {\n"
		"\t\tif (value == NULL) {\n"
		"\t\t\tif (" << module << "Debug >= 7)\n"
		"\t\t\t\tstd::cerr << \"Deleting " << ci->name
						<< ".\" << aname << \"\\n\";\n"
		"\t\t\treturn PyDict_DelItem(wco->in_dict, attr);\n"
		"\t\t}\n"
		"\t\tif (" << module << "Debug >= 7)\n"
		"\t\t\tstd::cerr << \"Setting " << ci->name
						<< ".\" << aname << \"\\n\";\n"
		"\t\treturn PyDict_SetItem(wco->in_dict, attr, value);\n"
		"\t}\n";
	if (!ci->attrs.empty())
		output <<
			"\totf::WrapPyArgTuple arg;\n";
	bool has_del_builtin = false;
	bool has_del_readonly = false;
	bool has_set_readonly = false;
	bool has_replace_method = false;
	output <<
		"\tswitch (ca->attr) {\n";
	for (AttrVec::const_iterator i = ci->attrs.begin();
						i != ci->attrs.end(); ++i) {
		output <<
			"\t  case " << ci->name << "Attr_" << i->name
								<< ": {\n"
			"\t\tif (value == NULL)\n";
		if (i->set.empty()) {
			output <<
				"\t\t\tgoto del_readonly;\n"
				"\t\tgoto set_readonly;\n";
			has_del_readonly = true;
			has_set_readonly = true;
		} else {
			output <<
				"\t\t\tgoto del_builtin;\n"
				"\t\t" << i->set << ";\n"
				"\t\treturn -(PyErr_Occurred() != NULL);\n";
			has_del_builtin = true;
		}
		output <<
			"\t  }\n";
	}
	MethodMap::const_iterator next;
	for (MethodMap::const_iterator i = ci->methods.begin();
					i != ci->methods.end(); i = next) {
		next = ci->methods.upper_bound(i->first);
		output <<
			"\t  case " << ci->name << "Attr_" << i->first
								<< ":\n";
	}
	if (!ci->methods.empty()) {
		output <<
			"\t\tif (value == NULL)\n"
			"\t\t\tgoto del_builtin;\n"
			"\t\tgoto replace_method;\n";
		has_del_builtin = true;
		has_replace_method = true;
	}
	if (!ci->constants.empty()) {
		for (AttrVec::const_iterator i = ci->constants.begin();
						i != ci->constants.end(); ++i)
			output <<
				"\t  case " << ci->name << "Attr_" << i->name
								<< ":\n";
		output <<
			"\t\tif (value == NULL)\n"
			"\t\t\tgoto del_readonly;\n"
			"\t\tgoto set_readonly;\n";
		has_del_readonly = true;
		has_set_readonly = true;
	}
	output <<
		"\t}\n"
		"\t// NOTREACHED\n"
		"\treturn NULL; // shutup compiler, all cases covered above\n";
	if (has_del_readonly)
		output <<
			"del_readonly:\n"
			"\tPyErr_SetString(PyExc_AttributeError, \"can not del read-only attribute\");\n"
			"\treturn -1;\n";
	if (has_set_readonly)
		output <<
			"set_readonly:\n"
			"\tPyErr_SetString(PyExc_AttributeError, \"can not set read-only attribute\");\n"
			"\treturn -1;\n";
	if (has_del_builtin)
		output <<
			"del_builtin:\n"
			"\tPyErr_SetString(PyExc_AttributeError, \"can not del built-in attribute\");\n"
			"\treturn -1;\n";
	if (has_replace_method)
		output <<
			"replace_method:\n"
			"\tPyErr_SetString(PyExc_AttributeError, \"can not replace built-in method\");\n"
			"\treturn -1;\n";
	output <<
		"}\n";
}

static void
dumpSetAttrO(std::ostream &output, const ClassInfo *ci, const string &fname)
{
	output <<
		"\n"
		"extern \"C\"\n"
		<< exportTag << "int\n" <<
		fname << "(PyObject* self, PyObject* attr, PyObject* value)\n"
		"{\n";
	output <<
		"\t" << ci->name << "Object* wco = static_cast<" << ci->name
							<< "Object*>(self);\n";
	if (ci->isWrappySubclass)
		output <<
			"\tif (wco->inst == NULL) {\n"
			"\t\tPyErr_SetString(PyExc_RuntimeError, \"C++ "
					<< ci->name << " instance gone\");\n"
			"\t\treturn -1;\n"
			"\t}\n";
	output <<
		"\tchar const* aname = PyString_AsString(attr);\n"
		"\tif (aname == NULL)\n"
		"\t\treturn -1;\n";
	bool has_del_builtin = false;
	bool has_del_readonly = false;
	bool has_set_readonly = false;
	bool has_replace_method = false;
	if (!ci->attrs.empty())
		output <<
			"\totf::WrapPyArgTuple arg;\n";
	string sep = "\t";
	for (AttrVec::const_iterator i = ci->attrs.begin();
						i != ci->attrs.end(); ++i) {
		output <<
			sep << "if (strcmp(aname, \"" << i->name
							<< "\") == 0) {\n"
			"\t\tif (value == NULL)\n";
		if (i->set.empty()) {
			output << "\t\t\tgoto del_readonly;\n"
				"\t\tgoto set_readonly;\n";
			has_del_readonly = true;
			has_set_readonly = true;
		} else {
			output <<
				"\t\t\tgoto del_builtin;\n"
				"\t\t" << i->set << ";\n"
				"\t\treturn -(PyErr_Occurred() != NULL);\n";
			has_del_builtin = true;
		}
		output <<
			"\t}\n";
		sep = "\telse ";
	}
	// error to override a member function
	if (!ci->methods.empty()) {
		sep += "if (";
		MethodMap::const_iterator next;
		for (MethodMap::const_iterator i = ci->methods.begin();
					i != ci->methods.end(); i = next) {
			next = ci->methods.upper_bound(i->first);
			output << sep << "strcmp(aname, \"" << i->first
								<< "\") == 0";
			sep = "\n\t|| ";
		}
		output <<
			") {\n"
			"\t\tif (value == NULL)\n"
			"\t\t\tgoto del_builtin;\n"
			"\t\tgoto replace_method;\n"
			"\t}\n";
		has_del_builtin = true;
		has_replace_method = true;
		sep = "\telse ";
	}
	if (!ci->constants.empty()) {
		sep += "if (";
		for (AttrVec::const_iterator i = ci->constants.begin();
						i != ci->constants.end(); ++i) {
			output << sep << "strcmp(aname, \"" << i->name
								<< "\") == 0";
			sep = "\n\t|| ";
		}
		output <<
			") {\n"
			"\t\tif (value == NULL)\n"
			"\t\t\tgoto del_readonly;\n"
			"\t\tgoto set_readonly;\n"
			"\t}\n";
		has_del_readonly = true;
		has_set_readonly = true;
		sep = "\telse ";
	}
	output <<
		"\tif (value == NULL) {\n"
		"\t\tif (" << module << "Debug >= 7)\n"
		"\t\t\tstd::cerr << \"Deleting " << ci->name
						<< ".\" << aname << \"\\n\";\n"
		"\t\treturn PyDict_DelItem(wco->in_dict, attr);\n"
		"\t}\n"
		"\tif (" << module << "Debug >= 7)\n"
		"\t\tstd::cerr << \"Setting " << ci->name
						<< ".\" << aname << \"\\n\";\n"
		"\treturn PyDict_SetItem(wco->in_dict, attr, value);\n";
	if (has_del_readonly)
		output <<
			"del_readonly:\n"
			"\tPyErr_SetString(PyExc_AttributeError, \"can not del read-only attribute\");\n"
			"\treturn -1;\n";
	if (has_set_readonly)
		output <<
			"set_readonly:\n"
			"\tPyErr_SetString(PyExc_AttributeError, \"can not set read-only attribute\");\n"
			"\treturn -1;\n";
	if (has_del_builtin)
		output <<
			"del_builtin:\n"
			"\tPyErr_SetString(PyExc_AttributeError, \"can not del built-in attribute\");\n"
			"\treturn -1;\n";
	if (has_replace_method)
		output <<
			"replace_method:\n"
			"\tPyErr_SetString(PyExc_AttributeError, \"can not replace built-in method\");\n"
			"\treturn -1;\n";
	output <<
		"}\n";
}

static void
dumpGperfGetAttrO(std::ostream &output, const ClassInfo *ci, const string &fname)
{
	output <<
		"\n"
		"extern \"C\"\n"
		<< exportTag << "PyObject*\n" <<
		fname << "(PyObject* self, PyObject* attr)\n"
		"{\n"
		"\t" << ci->name << "Object* wco = static_cast<" << ci->name
							<< "Object*>(self);\n"
		"\tchar const* aname = PyString_AsString(attr);\n"
		"\tif (aname == NULL) {\n"
		"\t\tPyErr_SetObject(PyExc_AttributeError, attr);\n"
		"\t\treturn NULL;\n"
		"\t}\n"
		"\tif (aname[0] == '_' && aname[1] == '_') {\n"
		"\t\tif (strcmp(&aname[2], \"members__\") == 0) {\n"
		"\t\t\tPyObject* result = PyList_New("
			<< ci->attrs.size() + ci->constants.size() << ");\n";
	int k = 0;
	for (AttrVec::const_iterator i = ci->attrs.begin();
						i != ci->attrs.end(); ++i, ++k)
		output << "\t\t\tPyList_SetItem(result, " << k
			<< ", PyString_FromString(\"" << i->name << "\"));\n";
	for (AttrVec::const_iterator i = ci->constants.begin();
					i != ci->constants.end(); ++i, ++k)
		output << "\t\t\tPyList_SetItem(result, " << k
			<< ", PyString_FromString(\"" << i->name << "\"));\n";
	output <<
		"\t\t\treturn result;\n"
		"\t\t}\n"
		"\t\tif (strcmp(&aname[2], \"methods__\") == 0) {\n";
	typedef std::vector<Symbol> SymVec;
	SymVec methods;
	methods.reserve(ci->methods.size());
	MethodMap::const_iterator next;
	for (MethodMap::const_iterator i = ci->methods.begin();
					i != ci->methods.end(); i = next) {
		next = ci->methods.upper_bound(i->first);
		methods.push_back(i->first);
	}
	output <<
		"\t\t\tPyObject* result = PyList_New(" << methods.size()
								<< ");\n";
	k = 0;
	for (SymVec::iterator i = methods.begin(); i != methods.end(); ++i, ++k)
		output << "\t\t\tPyList_SetItem(result, " << k
			<< ", PyString_FromString(\"" << *i << "\"));\n";
	output <<
		"\t\t\treturn result;\n"
		"\t\t}\n"
		"\t}\n"
		"\t" << ci->name << "Attr const* ca = " << ci->name
				<< "AttrLookup::find(aname, PyString_Size(attr));\n"
		"\tif (ca == NULL) {\n"
		"\t\tPyObject* value = PyDict_GetItem(wco->in_dict, attr);\n"
		"\t\tif (value) {\n"
		"\t\t\tPy_INCREF(value);\n"
		"\t\t\treturn value;\n"
		"\t\t}\n"
		"\t\tPyErr_SetObject(PyExc_AttributeError, attr);\n"
		"\t\treturn NULL;\n"
		"\t}\n";
	if (!ci->attrs.empty())
		output <<
			"\t// hardcoded attributes\n"
			"\totf::WrapPyArgTuple arg;\n";
	bool has_get_writeonly = false;
	output <<
		"\tswitch (ca->attr) {\n";
	for (AttrVec::const_iterator i = ci->attrs.begin();
						i != ci->attrs.end(); ++i) {
		output <<
			"\t  case " << ci->name << "Attr_" << i->name
								<< ":\n";
		if (i->get.empty()) {
			output << "\t\tgoto get_writeonly;\n";
			has_get_writeonly = true;
		} else
			output << "\t\treturn " << i->get << ";\n";
	}
	for (MethodMap::const_iterator i = ci->methods.begin();
					i != ci->methods.end(); i = next) {
		next = ci->methods.upper_bound(i->first);
		output <<
			"\t  case " << ci->name << "Attr_" << i->first << ":\n"
			"\t\treturn PyCFunction_New(&" << ci->name
					<< "MD_" << i->first << ", self);\n";
	}
	for (AttrVec::const_iterator i = ci->constants.begin();
						i != ci->constants.end(); ++i)
		output <<
			"\t  case " << ci->name << "Attr_" << i->name << ":\n"
			"\t\treturn " << i->get << ";\n";
	output <<
		"\t}\n"
		"\t// NOTREACHED\n"
		"\treturn NULL; // shutup compiler, all cases covered above\n";
	if (has_get_writeonly)
		output <<
			"get_writeonly:\n"
			"\tPyErr_SetString(PyExc_AttributeError, \"can not get write-only attribute\");\n"
			"\treturn NULL;\n";
	output <<
		"}\n";
}

static void
dumpGetAttrO(std::ostream &output, const ClassInfo *ci, const string &fname)
{
	output <<
		"\n"
		"extern \"C\"\n"
		<< exportTag << "PyObject*\n" <<
		fname << "(PyObject* self, PyObject* attr)\n"
		"{\n"
		"\t" << ci->name << "Object* wco = static_cast<" << ci->name
							<< "Object*>(self);\n"
		"\tchar const* aname = PyString_AsString(attr);\n"
		"\tif (aname == NULL) {\n"
		"\t\tPyErr_SetObject(PyExc_AttributeError, attr);\n"
		"\t\treturn NULL;\n"
		"\t}\n";
	if (ci->isWrappySubclass)
		output <<
			"\tif (wco->inst == NULL) {\n"
			"\t\tPyErr_SetString(PyExc_RuntimeError, \"C++ "
					<< ci->name << " instance gone\");\n"
			"\t\treturn NULL;\n"
			"\t}\n";
	output <<
		"\tif (aname[0] == '_' && aname[1] == '_') {\n";
	AttrVec::size_type size = ci->attrs.size() + ci->constants.size();
	if (size != 0) {
		output <<
			"\t\tif (strcmp(&aname[2], \"members__\") == 0) {\n"
			"\t\t\tPyObject* result = PyList_New(" << size
								<< ");\n";
		int k = 0;
		for (AttrVec::const_iterator i = ci->attrs.begin();
						i != ci->attrs.end(); ++i, ++k)
			output << "\t\t\tPyList_SetItem(result, " << k
				<< ", PyString_FromString(\"" << i->name
				<< "\"));\n";
		for (AttrVec::const_iterator i = ci->constants.begin();
					i != ci->constants.end(); ++i, ++k)
			output << "\t\t\tPyList_SetItem(result, " << k
				<< ", PyString_FromString(\"" << i->name
				<< "\"));\n";
		output <<
			"\t\t\treturn result;\n"
			"\t\t}\n";
	}
	output <<
		"\t\tif (strcmp(&aname[2], \"methods__\") == 0) {\n";
	typedef std::vector<Symbol> SymVec;
	SymVec methods;
	methods.reserve(ci->methods.size());
	MethodMap::const_iterator next;
	for (MethodMap::const_iterator i = ci->methods.begin();
					i != ci->methods.end(); i = next) {
		next = ci->methods.upper_bound(i->first);
		methods.push_back(i->first);
	}
	output <<
		"\t\t\tPyObject* result = PyList_New(" << methods.size()
								<< ");\n";
	int k = 0;
	for (SymVec::iterator i = methods.begin(); i != methods.end(); ++i, ++k)
		output << "\t\t\tPyList_SetItem(result, " << k
			<< ", PyString_FromString(\"" << *i
			<< "\"));\n";
	output <<
		"\t\t\treturn result;\n"
		"\t\t}\n"
		"\t}\n";
	if (!ci->attrs.empty())
		output <<
			"\t// hardcoded attributes\n"
			"\totf::WrapPyArgTuple arg;\n";
	bool has_get_writeonly = false;
	const char *sep = "\t";
	for (AttrVec::const_iterator i = ci->attrs.begin();
						i != ci->attrs.end(); ++i) {
		output <<
			sep << "if (strcmp(aname, \"" << i->name
							<< "\") == 0)\n";
		if (i->get.empty()) {
			output << "\t\tgoto get_writeonly;\n";
			has_get_writeonly = true;
		} else
			output << "\t\treturn " << i->get << ";\n";
		sep = "\telse ";
	}
	for (MethodMap::const_iterator i = ci->methods.begin();
					i != ci->methods.end(); i = next) {
		next = ci->methods.upper_bound(i->first);
		output <<
			sep << "if (strcmp(aname, \"" << i->first
							<< "\") == 0)\n"
			"\t\treturn PyCFunction_New(&" << ci->name
					<< "MD_" << i->first << ", self);\n";
		sep = "\telse ";
	}
	if (!ci->constants.empty()) {
		for (AttrVec::const_iterator i = ci->constants.begin();
						i != ci->constants.end(); ++i) {
			output <<
				sep << "if (strcmp(aname, \"" << i->name
								<< "\") == 0)\n"
				"\t\treturn " << i->get << ";\n";
			sep = "\telse ";
		}
	}
	output <<
		"\tPyObject* value;\n";

	string indent;
	output << "\tvalue = PyDict_GetItem(wco->in_dict, attr);\n"
		"\tif (value) {\n"
		"\t\tPy_INCREF(value);\n"
		"\t\treturn value;\n"
		"\t}\n"
		"\tPyErr_SetObject(PyExc_AttributeError, attr);\n"
		"\treturn NULL;\n";
	if (has_get_writeonly)
		output <<
			"get_writeonly:\n"
			"\tPyErr_SetString(PyExc_AttributeError, \"can not get write-only attribute\");\n"
			"\treturn NULL;\n";
	output <<
		"}\n";
}

static void
dumpVariables(std::ostream &output, const ClassInfo *ci)
{
	// These are functions for public class variables.
	for (AttrVec::const_iterator i = ci->variables.begin();
						i != ci->variables.end(); ++i) {
		if (!i->get.empty()) {
			output <<
				"\n"
				"PyObject*\n" <<
				ci->name << "Get" << i->name << "("
					<< ci->name << "Object* wco)\n"
				"{\n"
				"\tPyObject *o = " << i->get << ";\n";
			if (i->cache)
				output <<
					"\t// cache attribute\n"
					"\tstatic PyObject *key = NULL;\n"
					"\tif (key == NULL)\n"
					"\t\tkey = PyString_FromString(\"__cached_" << i->name << "__\");\n"
					"\tPyDict_SetItem(wco->in_dict, key, o);\n";
			output <<
				"\treturn o;\n"
				"}\n";
		}
		if (i->set.empty())
			continue;
		output <<
			"\n"
			"void\n" <<
			ci->name << "Set" << i->name << "(" << ci->name
					<< "Object* wco, PyObject* value)\n"
			"{\n"
			"\totf::WrapPyArgTuple arg;\n"
			<< i->set << ";\n";
		if (i->cache)
			output <<
				"\t// cache attribute\n"
				"\tstatic PyObject *key = NULL;\n"
				"\tif (key == NULL)\n"
				"\t\tkey = PyString_FromString(\"__cached_" << i->name << "__\");\n"
				"\tPyDict_SetItem(wco->in_dict, key, value);\n"
				"\tPy_DECREF(value);\n";
		output <<
			"}\n";
	}
}

static void
dumpMethods(std::ostream &output, const ClassInfo *ci)
{
	// first, create all of the method definitions
	MethodMap::const_iterator next;
	for (MethodMap::const_iterator i = ci->attrMethods.begin();
					i != ci->attrMethods.end(); i = next) {
		next = ci->attrMethods.upper_bound(i->first);
		(void) dumpMethod(output, ci, i, next, true, string(),
								string(), true);
	}

	for (MethodMap::const_iterator i = ci->methods.begin();
					i != ci->methods.end(); i = next) {
		next = ci->methods.upper_bound(i->first);
		string proto = dumpMethod(output, ci, i, next, true);
		proto += "\n" + i->first.str() + " documentation.";
		output <<
			"\n"
			"const char " << ci->name << '_' << i->first
				<< "_doc[] = \n\"" << stringize(proto)
				<< "\";\n";
		output <<
			"\n"
			"PyMethodDef " << ci->name << "MD_" << i->first
								<< " = {\n"
			"\t\"" << i->first << "\", (PyCFunction) " << ci->name
						<< '_' << i->first << ",\n"
			"\tMETH_KEYWORDS, const_cast<char*>(" << ci->name
						<< '_' << i->first << "_doc)\n"
			"};\n";
	}
}

static void
dumpCoercionFunc(std::ostream &output, const string &name)
{
	output <<
		"\n"
		"extern \"C\"\n"
		<< exportTag << "int\n" <<
		name << "(PyObject** left, PyObject** right)\n"
		"{\n"
		"\tif (!PyType_HasFeature((*left)->ob_type, Py_TPFLAGS_WRAPPY_TYPE)\n"
		"\t|| !PyType_HasFeature((*right)->ob_type, Py_TPFLAGS_WRAPPY_TYPE))\n"
		"\t\treturn 1;\n"
		"\t// we do further type checking in the binary operators\n"
		"\tPy_INCREF(*left);\n"
		"\tPy_INCREF(*right);\n"
		"\treturn 0;\n"
		"}\n";
}

static void
dumpUnaryFunc(std::ostream &output, const ClassInfo *ci, const MethodMap &mm, const string &name)
{
	// PyObject *func(PyObject *)
	output <<
		"\n"
		"extern \"C\"\n"
		<< exportTag << "PyObject*\n" <<
		name << "(PyObject* self)\n"
		"{\n"
		"\ttry {\n"
		"\t\t" << ci->name << "Object* wco = static_cast<" << ci->name
							<< "Object*>(self);\n"
#if 0
		"\t\treturn "
		"wco->inst->" << fd->tag;
#else
		;
		output << "\t\tPyErr_SetString(PyExc_NotImplementedError, \""
					<< name << " unimplemented\");\n";
#endif
	output <<
#if 0
		"();\n"
#endif
		"\t} catch (...) {\n"
		"\t\t" << module << "Error();\n"
		"\t}\n"
		"\treturn NULL;\n"
		"}\n";
}

static void
dumpBinaryFunc(std::ostream &output, const ClassInfo *ci, const MethodMap &mm, const string &name)
{
	// PyObject *func(PyObject *, PyObject *)
	output <<
		"\n"
		"extern \"C\"\n"
		<< exportTag << "PyObject*\n" <<
		name << "(PyObject* self, PyObject* right)\n"
		"{\n"
		"\ttry {\n"
		"\t\t" << ci->name << "Object* wco = static_cast<" << ci->name
							<< "Object*>(self);\n";
#if 0
	// TODO: could be several versions with different types on the right
#else
	output << "\t\tPyErr_SetString(PyExc_NotImplementedError, \""
					<< name << " unimplemented\");\n";
#endif
	output <<
		"\t} catch (...) {\n"
		"\t\t(" << module << "Error();\n"
		"\t}\n"
		"\treturn NULL;\n"
		"}\n";
}

static void
dumpInquiryFunc(std::ostream &output, const ClassInfo *ci, const MethodMap &mm, const string &name)
{
	// int func(PyObject *)
	output <<
		"\n"
		"extern \"C\"\n"
		<< exportTag << "int\n" <<
		name << "(PyObject* self)\n"
		"{\n"
		"\ttry {\n"
		"\t\t" << ci->name << "Object* wco = static_cast<" << ci->name
							<< "Object*>(self);\n"
#if 0
		"\t\treturn "
		"wco->inst->" << fd->tag;
#else
		;
		output << "\t\tPyErr_SetString(PyExc_NotImplementedError, \""
					<< name << " unimplemented\");\n";
#endif
	output <<
#if 0
		"();\n"
#endif
		"\t} catch (...) {\n"
		"\t\t" << module << "Error();\n"
		"\t}\n"
		"\treturn -1;\n"
		"}\n";
}

static void
dumpNumberMethods(std::ostream &output, const ClassInfo *ci, const string &name)
{
	string nb_add("0");
	string nb_subtract("0");
	string nb_multiply("0");
	string nb_divide("0");
	string nb_remainder("0");
	string nb_divmod("0");
	string nb_power("0");
	string nb_negative("0");
	string nb_positive("0");
	string nb_absolute("0");
	string nb_nonzero("0");
	string nb_invert("0");
	string nb_lshift("0");
	string nb_rshift("0");
	string nb_and("0");
	string nb_xor("0");
	string nb_or("0");
	string nb_coerce("0");
	string nb_int("0");
	string nb_long("0");
	string nb_float("0");
	string nb_oct("0");
	string nb_hex("0");
	if (!ci->nb_add.empty()) {
		nb_add = ci->name.str() + "Add";
		dumpBinaryFunc(output, ci, ci->nb_add, nb_add);
	}
	if (!ci->nb_subtract.empty()) {
		nb_subtract = ci->name.str() + "Subtract";
		dumpBinaryFunc(output, ci, ci->nb_subtract, nb_subtract);
	}
	if (!ci->nb_multiply.empty()) {
		nb_multiply = ci->name.str() + "Multiply";
		dumpBinaryFunc(output, ci, ci->nb_multiply, nb_multiply);
	}
	if (!ci->nb_divide.empty()) {
		nb_divide = ci->name.str() + "Divide";
		dumpBinaryFunc(output, ci, ci->nb_divide, nb_divide);
	}
	if (!ci->nb_remainder.empty()) {
		nb_remainder = ci->name.str() + "Remainder";
		dumpBinaryFunc(output, ci, ci->nb_remainder, nb_remainder);
	}
	if (!ci->nb_divmod.empty()) {
		nb_divmod = ci->name.str() + "Divmod";
		dumpBinaryFunc(output, ci, ci->nb_divmod, nb_divmod);
	}
#if 0
	if (!ci->nb_power.empty()) {
		nb_power = ci->name.str() + "Power";
		dumpTernaryFunc(output, ci, ci->nb_power, nb_power);
	}
#endif
	if (!ci->nb_negative.empty()) {
		nb_negative = ci->name.str() + "Negative";
		dumpUnaryFunc(output, ci, ci->nb_negative, nb_negative);
	}
	if (!ci->nb_positive.empty()) {
		nb_positive = ci->name.str() + "Positive";
		dumpUnaryFunc(output, ci, ci->nb_positive, nb_positive);
	}
	if (!ci->nb_absolute.empty()) {
		nb_absolute = ci->name.str() + "Absolute";
		dumpUnaryFunc(output, ci, ci->nb_absolute, nb_absolute);
	}
	if (!ci->nb_nonzero.empty()) {
		nb_nonzero = ci->name.str() + "Nonzero";
		dumpInquiryFunc(output, ci, ci->nb_nonzero, nb_nonzero);
	}
	if (!ci->nb_invert.empty()) {
		nb_invert = ci->name.str() + "Invert";
		dumpUnaryFunc(output, ci, ci->nb_invert, nb_invert);
	}
	if (!ci->nb_lshift.empty()) {
		nb_lshift = ci->name.str() + "Lshift";
		dumpBinaryFunc(output, ci, ci->nb_lshift, nb_lshift);
	}
	if (!ci->nb_rshift.empty()) {
		nb_rshift = ci->name.str() + "Rshift";
		dumpBinaryFunc(output, ci, ci->nb_rshift, nb_rshift);
	}
	if (!ci->nb_and.empty()) {
		nb_and = ci->name.str() + "And";
		dumpBinaryFunc(output, ci, ci->nb_and, nb_and);
	}
	if (!ci->nb_xor.empty()) {
		nb_xor = ci->name.str() + "Xor";
		dumpBinaryFunc(output, ci, ci->nb_xor, nb_xor);
	}
	if (!ci->nb_or.empty()) {
		nb_or = ci->name.str() + "Or";
		dumpBinaryFunc(output, ci, ci->nb_or, nb_or);
	}
	if (ci->nb_coerce) {
		nb_coerce = ci->name.str() + "Coerce";
		dumpCoercionFunc(output, nb_coerce);
	}
	if (!ci->nb_int.empty()) {
		nb_int = ci->name.str() + "Int";
		dumpUnaryFunc(output, ci, ci->nb_int, nb_int);
	}
	if (!ci->nb_long.empty()) {
		nb_long = ci->name.str() + "Long";
		dumpUnaryFunc(output, ci, ci->nb_long, nb_long);
	}
	if (!ci->nb_float.empty()) {
		nb_float = ci->name.str() + "Float";
		dumpUnaryFunc(output, ci, ci->nb_float, nb_float);
	}
	if (!ci->nb_oct.empty()) {
		nb_oct = ci->name.str() + "Oct";
		dumpUnaryFunc(output, ci, ci->nb_oct, nb_oct);
	}
	if (!ci->nb_hex.empty()) {
		nb_hex = ci->name.str() + "Hex";
		dumpUnaryFunc(output, ci, ci->nb_hex, nb_hex);
	}
	output <<
		"\n"
		"PyNumberMethods " << name << " = {\n"
		"\t" << nb_add << ", // nb_add\n"
		"\t" << nb_subtract << ", // nb_subtract\n"
		"\t" << nb_multiply << ", // nb_multiply\n"
		"\t" << nb_divide << ", // nb_divide\n"
		"\t" << nb_remainder << ", // nb_remainder\n"
		"\t" << nb_divmod << ", // nb_divmod\n"
		"\t" << nb_power << ", // nb_power\n"
		"\t" << nb_negative << ", // nb_negative\n"
		"\t" << nb_positive << ", // nb_positive\n"
		"\t" << nb_absolute << ", // nb_absolute\n"
		"\t" << nb_nonzero << ", // nb_nonzero\n"
		"\t" << nb_invert << ", // nb_invert\n"
		"\t" << nb_lshift << ", // nb_lshift\n"
		"\t" << nb_rshift << ", // nb_rshift\n"
		"\t" << nb_and << ", // nb_and\n"
		"\t" << nb_xor << ", // nb_xor\n"
		"\t" << nb_or << ", // nb_or\n"
		"\t" << nb_coerce << ", // nb_coerce\n"
		"\t" << nb_int << ", // nb_int\n"
		"\t" << nb_long << ", // nb_long\n"
		"\t" << nb_float << ", // nb_float\n"
		"\t" << nb_oct << ", // nb_oct\n"
		"\t" << nb_hex << ", // nb_hex\n"
		"};\n";
}

static void
dumpSequenceMethods(std::ostream &output, const ClassInfo *ci, const string &name)
{
	string sq_length("0");
	string sq_concat("0");
	string sq_repeat("0");
	string sq_item("0");
	string sq_slice("0");
	string sq_ass_item("0");
	string sq_ass_slice("0");

	if (!ci->sq_length.empty()) {
		sq_length = ci->name.str() + "Size";
		dumpInquiryFunc(output, ci, ci->sq_length, sq_length);
	}
	if (!ci->sq_concat.empty()) {
		sq_concat = ci->name.str() + "Concat";
		dumpBinaryFunc(output, ci, ci->sq_concat, sq_concat);
	}
#if 0
	intargfunc sq_repeat;
	intargfunc sq_item;
	intintargfunc sq_slice;
	intobjargproc sq_ass_item;
	intintobjargproc sq_ass_slice;
#endif
	output <<
		"\n"
		"PySequenceMethods " << name << " = {\n"
		"\t" << sq_length << ", // sq_length\n"
		"\t" << sq_concat << ", // sq_concat\n"
		"\t" << sq_repeat << ", // sq_repeat\n"
		"\t" << sq_item << ", // sq_item\n"
		"\t" << sq_slice << ", // sq_slice\n"
		"\t" << sq_ass_item << ", // sq_ass_item\n"
		"\t" << sq_ass_slice << ", // sq_ass_slice\n"
		"};\n";
}

static void
dumpMappingMethods(std::ostream &output, const ClassInfo *ci, const string &name)
{
	string mp_length("0");
	string mp_subscript("0");
	string mp_ass_subscript("0");

	if (!ci->mp_length.empty()) {
		mp_length = ci->name.str() + "Size";
		dumpInquiryFunc(output, ci, ci->mp_length, mp_length);
	}
	if (!ci->mp_subscript.empty()) {
		mp_subscript = ci->name.str() + "Size";
		dumpBinaryFunc(output, ci, ci->mp_subscript, mp_subscript);
	}
#if 0
	objobjargproc mp_ass_subscript;
#endif
	output <<
		"\n"
		"PyMappingMethods " << name << " = {\n"
		"\t" << mp_length << ", // mp_length\n"
		"\t" << mp_subscript << ", // mp_subscript\n"
		"\t" << mp_ass_subscript << ", // mp_ass_subscript\n"
		"};\n";
}

bool
dumpTypeCode(const ClassInfo *ci)
{
	string tp_print("0");
	string tp_compare("0");
	string tp_repr("0");
	string tp_as_number("0");
	string tp_as_sequence("0");
	string tp_as_mapping("0");
	string tp_hash("0");
	string tp_call("0");
	string tp_str("0");
	string tp_getattro("0");
	string tp_setattro("0");
	string tp_as_buffer("0");
	string tp_doc("0");

	string filename(ci->name.str());
	filename += "Object.cpp";
	std::auto_ptr<std::ostream> outs(outputStream(filename).release());
	std::ostream &output = *outs.get();

	output << "#include \"" << module << ".h\"\n";
	if (ci->print)
		output << "#include <sstream>\n";

	dumpWrapTypeOutline(output, ci);

	if (!nameSpace.empty())
		output <<
			"\n"
			"namespace " << nameSpace << " {\n";

	dumpDeallocator(output, ci);
	
	//
	// find member functions that act like special Python type functions
	//

	// print to file
	if (ci->print || !ci->str.empty()) {
		tp_print = ci->name.str() + "Print";
		dumpPrint(output, ci, tp_print);
	}

	if (!ci->compare.empty()) {
		tp_compare = ci->name.str() + "Compare";
		dumpCompare(output, ci, tp_compare);
	}

	if (!ci->call.empty()) {
		tp_call = ci->name.str() + "Call";
		dumpCall(output, ci, tp_call);
	}

	if (ci->isWrappySubclass || !ci->hash.empty()) {
		tp_hash = ci->name.str() + "Hash";
		if (ci->hash.empty())
			dumpPtrHash(output, ci, tp_hash);
		else
			dumpHash(output, ci, tp_hash);
	}

	// string
	if (!ci->str.empty() || ci->print) {
		tp_str = ci->name.str() + "Str";
		dumpStr(output, ci, tp_str);
	}

	if (ci->hasNumberMethods) {
		tp_as_number = "&" + ci->name.str() + "AsNumber";
		dumpNumberMethods(output, ci, tp_as_number.substr(1));
	}
	if (ci->hasSequenceMethods) {
		tp_as_sequence = "&" + ci->name.str() + "AsSequence";
		dumpSequenceMethods(output, ci, tp_as_sequence.substr(1));
	}
	if (ci->hasMappingMethods) {
		tp_as_mapping = "&" + ci->name.str() + "AsMap";
		dumpMappingMethods(output, ci, tp_as_mapping.substr(1));
	}

	dumpVariables(output, ci);

	// methods table
	dumpMethods(output, ci);

	bool useGperf = false;
	AttrVec::size_type size = ci->attrs.size() + ci->constants.size() + ci->methods.size();
	if (gperfMinimum <= size && createGperf(ci))
		useGperf = true;
	if (useGperf) {
		output <<
			"\n"
			"#include \"" << ci->name << "Attr.cpp\"\n";
	}
	tp_setattro = ci->name.str() + "SetAttrO";
	if (useGperf)
		dumpGperfSetAttrO(output, ci, tp_setattro);
	else
		dumpSetAttrO(output, ci, tp_setattro);
	tp_getattro = ci->name.str() + "GetAttrO";
	if (useGperf)
		dumpGperfGetAttrO(output, ci, tp_getattro);
	else
		dumpGetAttrO(output, ci, tp_getattro);

	output << "\n"
		"const char " << ci->name << "_doc[] = \"" << ci->name
						<< " documentation\";\n";
	tp_doc = "const_cast<char*>(" + ci->name.str() + "_doc)";

	// Python 1.5 type definition -- must be last
	output <<
		exportTag << "PyTypeObject " << ci->name << "ObjectType = {\n"
		"\tPyObject_HEAD_INIT(0)\n"
		"\t0, // ob_size\n"
		"\t\"" << ci->name << "\", // tp_name\n"
		"\tsizeof(" << ci->name << "Object), // tp_basicsize\n"
		"\t0, // tp_itemsize\n"
		"\t" << ci->name << "_dealloc, // tp_dealloc\n"
		"\t" << tp_print << ", // tp_print\n"
		"\t0, // tp_getattr\n"
		"\t0, // tp_setattr\n"
		"\t" << tp_compare << ", // tp_compare\n"
		"\t" << tp_repr << ", // tp_repr\n"
		"\t" << tp_as_number << ", // tp_as_number\n"
		"\t" << tp_as_sequence << ", // tp_as_sequence\n"
		"\t" << tp_as_mapping << ", // tp_as_mapping\n"
		"\t" << tp_hash << ", // tp_hash\n"
		"\t" << tp_call << ", // tp_call\n"
		"\t" << tp_str << ", // tp_str\n"
		"\t" << tp_getattro << ", // tp_getattro\n"
		"\t" << tp_setattro << ", // tp_setattro\n"
		"\t" << tp_as_buffer << ", // tp_as_buffer\n"
		"\tPy_TPFLAGS_WRAPPY_TYPE, // tp_flags\n"
		"\t" << tp_doc << ", // tp_doc\n"
		"};\n";

	if (!nameSpace.empty())
		output <<
			"\n"
			"} // namespace " << nameSpace << "\n";
	// flush output and check return status
	output.flush();
	if (output.good())
		return true;
	std::cerr << programName << ": error writing " << filename << '\n';
	return false;
}

void
dumpTypeAttrInit(std::ostream &output, int indent, const ClassInfo *ci)
{
	string bi(tab(indent));		// base indent
	bool first = true;
	for (AttrVec::const_iterator i = ci->attrs.begin();
						i != ci->attrs.end(); ++i) {
		if (!i->cache)
			continue;	// skip primitive types
		if (first) {
			first = false;
			output << bi << "// initialize cached attributes\n"
				<< bi << "otf::WrapPyArgTuple arg;\n"
				<< bi << ci->name
					<< "Object* wco = static_cast<"
					<< ci->name << "Object*>(pInst);\n"
				<< bi << "PyObject* attrTmp;\n";
		}
		string func(i->get.substr(0, i->get.find('(')));
		if (ci->variables.end()
		!= std::find_if(ci->variables.begin(), ci->variables.end(),
							AttrHasName(i->name))) {
			// a class variable
			output << bi << "attrTmp = " << func << "(wco);\n";
		} else {
			// a class function
			output << bi << "attrTmp = " << func
						<< "(pInst, arg(), NULL);\n";
		}
		output << bi << "Py_XDECREF(attrTmp);\n";
	}
	if (!first)
		output << bi << "PyErr_Clear();\n";
}
