/* ----------------------------------------------------------------------------
*  Interface to all binary file IO used by the nmr library.
*/

#include <stdio.h>
#include <stddef.h>	/* Defines size_t. */

#include "binaryIO.h"
#include "utility.h"		// Use fatal_error()

/* ----------------------------------------------------------------------------
*  Routines to determine and change byte order of numeric types.
*/
static bool little_endian();			/* LSB lowest in memory. */
static void byte_reverse_x32_values(byte *b, size_t count);

/* ----------------------------------------------------------------------------
*  Routines for handling machine integer sizes bigger than integer sizes
*  saved in file.  The pad out and sign extend on reading and truncate on
*  writing.
*/
static bool read_integer(byte *b, size_t ts, size_t rs, bool sign_extend,
			 FILE *fp);
static bool read_little_endian_integer(byte *b, size_t ts, size_t rs,
				       bool sign_extend, FILE *fp);
static bool write_integer(long x, int bc, FILE *fp);

/* ----------------------------------------------------------------------------
*  Little endian means the machine stores the least significant byte of
*  integral types lowest in memory, most significant byte highest in memory.
*  The reverse order is called big endian.  The binary file format uses
*  big endian so byte order needs to be reversed on little endian machines.
*/
static bool little_endian(void)
{
  int i = 1;

  return *(byte *)&i;
}

/* ----------------------------------------------------------------------------
*  Reverse the order of an array of bytes.
*/
void byte_reverse(byte *b, size_t bc)
{
  byte *e, temp;

  for (e = b + bc - 1 ; b < e ; ++b, --e)
    {
      temp = *e;
      *e = *b;
      *b = temp;
    }
}

/* ----------------------------------------------------------------------------
*  Reverse the order of the 4 bytes in an IEEE float.
*/
static void byte_reverse_x32_values(byte *b, size_t count)
{
  if (sizeof(unsigned int) == 4)
    {
      //
      // Reversing the floats can take about 1/2 of the time in
      // simple array processing.  Using the shift operator and
      // bitwise or is about a factor of 4 faster than permuting
      // an array of bytes on a DEC Alpha compiled with g++ 2.95.
      // This will also be called on Intel x86 but I have not
      // profiled for that platform.
      //
      unsigned int *i = (unsigned int *) b;
      for (size_t k = 0 ; k < count ; ++k, ++i)
	*i = ((*i << 24) |
	      ((*i & 0x0000ff00) << 8) |
	      ((*i & 0x00ff0000) >> 8) |
	      (*i >> 24));
    }
  else
    {
      //
      // The generic float reversal routine.
      //
      for (size_t k = 0 ; k < count ; ++k, b += 4)
	{
	  byte temp;

	  temp = b[0];
	  b[0] = b[3];
	  b[3] = temp;

	  temp = b[1];
	  b[1] = b[2];
	  b[2] = temp;
	}
    }
}

/* ----------------------------------------------------------------------------
*  Initialize a bit field to zero.
*/
void zero_bit_field(bit_field bf)
{
  for (int k = 0 ; k < 4 ; ++k)
    bf[k] = 0;
}

/* ----------------------------------------------------------------------------
*  Extract a single bit from a bit field.  The return value is 0 or 1.
*/
unsigned long bit_field_bit(bit_field bf, int b)
{
  int byte_num = b / 8;
  byte bit = (0x80 >> (b % 8));

  return ((bf[byte_num] & bit) ? 1 : 0);
}

/* ----------------------------------------------------------------------------
*  Set a single bit in a bit field to zero or one.
*/
void set_bit_field_bit(bit_field bf, int b, bool state)
{
  int byte_num = b / 8;
  byte bit = (0x80 >> (b % 8));

  if (state)
    bf[byte_num] |= bit;
  else
    bf[byte_num] &= ~bit;
}

/* ----------------------------------------------------------------------------
*  Extract a string of bits from a bit field.  The desired bits are returned
*  as the least significant b2-b1 bits of the returned unsigned long.
*/
unsigned long bit_field_bits(bit_field bf, int b1, int b2)
{
  unsigned long bits = 0;

  for (int b = b1 ; b <= b2 ; ++b)
    {
      bits <<= 1;
      bits |= bit_field_bit(bf, b);
    }

  return bits;
}

/* ----------------------------------------------------------------------------
*  Set a string of bits in a bit field.  The least significant b2-b1 bits in
*  the bits argument contains the pattern to put in the bit field.
*/
void set_bit_field_bits(bit_field bf, int b1, int b2, unsigned long bits)
{
  for (int b = b2 ; b >= b1 ; --b)
    {
      set_bit_field_bit(bf, b, bits & 0x1);
      bits >>= 1;
    }
}

/* ----------------------------------------------------------------------------
*  Seek to beginning of file.
*/
bool beginning_of_file(FILE *fp)
{
  return seek_position(fp, 0, 0);
}

/* ----------------------------------------------------------------------------
*  Just a wrapper for fseek.
*/
bool seek_position(FILE *fp, long offset, int origin)
{
  return fseek(fp, offset, origin) == 0;
}

/* ----------------------------------------------------------------------------
*/
bool read_padding(size_t num, FILE *fp)
{
  char c;
  bool success = true;

  for (size_t k = 0 ; k < num && success ; ++k)
    success = read_u8(&c, fp);

  return success;
}

/* ----------------------------------------------------------------------------
*/
bool read_bit_field(bit_field bf, FILE *fp)
{
  return fread(bf, 1, 4, fp) == 4;
}

/* ----------------------------------------------------------------------------
*/
bool read_u8(char *c, FILE *fp)
{
  return fread(c, 1, 1, fp) == 1;
}

/* ----------------------------------------------------------------------------
*/
bool read_u8_array(char *c, size_t num, FILE *fp)
{
  return fread(c, 1, num, fp) == num;
}
  
/* ----------------------------------------------------------------------------
*/
bool read_s16(short *s, FILE *fp)
{
  return read_integer((byte *)s, sizeof(short), 2, true, fp);
}

/* ----------------------------------------------------------------------------
*/
bool read_u16(unsigned short *us, FILE *fp)
{
  return read_integer((byte *)us, sizeof(unsigned short), 2, false, fp);
}

/* ----------------------------------------------------------------------------
*/
bool read_s32(int *i, FILE *fp)
{
  return read_integer((byte *)i, sizeof(int), 4, true, fp);
}

/* ----------------------------------------------------------------------------
*/
bool read_s32_array(int *i, size_t num, bool big_endian, FILE *fp)
{
  bool success = true;

  if (sizeof(int) == 4)
    {
      //
      // Machine int is 4 bytes so no padding is needed.
      // So I can read the array more efficiently.
      //
      success = read_u8_array((char *) i, 4 * num, fp);
      if (success && little_endian() == big_endian)
	byte_reverse_x32_values((byte *)i, num);
    }
  else
    {
      if (big_endian)
	for (size_t k = 0 ; k < num && success ; ++k)
	  success = read_integer((byte *)&i[k], sizeof(int), 4, true, fp);
      else
	for (size_t k = 0 ; k < num && success ; ++k)
	  success = read_little_endian_integer((byte *)&i[k], sizeof(int), 4,
					       true, fp);
    }

  return success;
}

/* ----------------------------------------------------------------------------
*/
bool read_u32(unsigned *u, FILE *fp)
{
  return read_integer((byte *)u, sizeof(unsigned), 4, false, fp);
}

/* ----------------------------------------------------------------------------
*/
bool read_s32(long *l, FILE *fp)
{
  return read_integer((byte *)l, sizeof(long), 4, true, fp);
}

// ----------------------------------------------------------------------------
// Assume file uses specified byte order
//
bool read_f32(float *f, FILE *fp, bool big_endian)
{
  if (fread(f, 4, 1, fp) != 1)
    return false;

  bool machine_order = little_endian();
  bool file_order = !big_endian;
  if (file_order != machine_order)
    byte_reverse_x32_values((byte *)f, 1);

  return true;
}

/* ----------------------------------------------------------------------------
*/
bool read_f32(float *f, FILE *fp)
{
  if (fread(f, 4, 1, fp) != 1)
    return false;

  if (little_endian())
    byte_reverse_x32_values((byte *)f, 1);

  return true;
}

// ----------------------------------------------------------------------------
// Assume file uses specified byte order.
//
bool read_f32_array(float *floats, size_t num, FILE *fp, bool big_endian)
{
  if (fread(floats, 4, num, fp) != num)
    return false;

  bool machine_order = little_endian();
  bool file_order = !big_endian;
  if (file_order != machine_order)
    byte_reverse_x32_values((byte *)floats, num);

  return true;
}

/* ----------------------------------------------------------------------------
*/
bool read_f32_array(float *floats, size_t num, FILE *fp)
{
  if (fread(floats, 4, num, fp) != num)
    return false;

  if (little_endian())
    byte_reverse_x32_values((byte *)floats, num);

  return true;
}
  
/* ----------------------------------------------------------------------------
*  Read a big endian integer that is rs bytes long into a memory
*  representation that is ts bytes long sign extending if necessary.
*  If the machine is little endian make the memory representation
*  little endian.
*/
#define SIGN_BIT 0x80

static bool read_integer(byte *b, size_t ts, size_t rs, bool sign_extend,
			 FILE *fp)
{
  if (ts < rs)
    fatal_error("read_integer(): Machine integer size too small.\n");

  size_t pad = ts - rs;
  if (fread(b + pad, rs, 1, fp) != 1)
    return false;

  byte extend = ((sign_extend && (SIGN_BIT & b[pad])) ? 0xff : 0x00);
  for (size_t k = 0 ; k < pad ; ++k)
    b[k] = extend;

  if (little_endian())
    byte_reverse(b, ts);

  return true;
}
  
/* ----------------------------------------------------------------------------
*  Read a little endian integer that is rs bytes long into a memory
*  representation that is ts bytes long sign extending if necessary.
*  If the machine is big endian make the memory representation
*  big endian.
*/
static bool read_little_endian_integer(byte *b, size_t ts, size_t rs,
				       bool sign_extend, FILE *fp)
{
  if (ts < rs)
    fatal_error("read_little_endian_integer(): Machine integer size too small.\n");

  if (fread(b, rs, 1, fp) != 1)
    return false;

  byte extend = ((sign_extend && (SIGN_BIT & b[rs-1])) ? 0xff : 0x00);
  for (size_t k = rs ; k < ts ; ++k)
    b[k] = extend;

  if (!little_endian())
    byte_reverse(b, ts);

  return true;
}

/* ----------------------------------------------------------------------------
*/
#define PAD_CHARACTER 0

bool write_padding(size_t num, FILE *fp)
{
  bool success = true;

  for (size_t k = 0 ; k < num && success ; ++k)
    success = write_u8(PAD_CHARACTER, fp);

  return success;
}

/* ----------------------------------------------------------------------------
*/
bool write_bit_field(bit_field bf, FILE *fp)
{
  return fwrite(bf, 1, 4, fp) == 4;
}

/* ----------------------------------------------------------------------------
*/
bool write_u8(char c, FILE *fp)
{
  return fwrite(&c, 1, 1, fp) == 1;
}

/* ----------------------------------------------------------------------------
*/
bool write_u8_array(char *c, size_t num, FILE *fp)
{
  return fwrite(c, 1, num, fp) == num;
}
  
/* ----------------------------------------------------------------------------
*/
bool write_s16(short s, FILE *fp)
{
  return write_integer(s, 2, fp);
}

/* ----------------------------------------------------------------------------
*/
bool write_u16(unsigned short us, FILE *fp)
{
  return write_integer((long)us, 2, fp);
}

/* ----------------------------------------------------------------------------
*/
bool write_s32(int i, FILE *fp)
{
  return write_integer(i, 4, fp);
}

/* ----------------------------------------------------------------------------
*/
bool write_u32(unsigned u, FILE *fp)
{
  return write_integer((long)u, 4, fp);
}

/* ----------------------------------------------------------------------------
*/
bool write_s32(long l, FILE *fp)
{
  return write_integer(l, 4, fp);
}

/* ----------------------------------------------------------------------------
*/
bool write_f32(float f, FILE *fp)
{
  if (little_endian())
    byte_reverse_x32_values((byte *)&f, 1);

  return fwrite(&f, 4, 1, fp) == 1;
}

/* ----------------------------------------------------------------------------
*  Optimized.
*/
bool write_f32_array(float *floats, size_t num, FILE *fp)
{
  bool success;

  if (little_endian())
    {
      byte_reverse_x32_values((byte *)floats, num);
      success = (fwrite(floats, 4, num, fp) == num);
      byte_reverse_x32_values((byte *)floats, num);
    }
  else
    success = (fwrite(floats, 4, num, fp) == num);

  return success;
}

/* ----------------------------------------------------------------------------
*  Write a bc byte big endian representation of an integer x.
*  Must have bc <= 4.
*/
static bool write_integer(long x, int bc, FILE *fp)
{
  byte b[4];

  if (bc > 4)
    fatal_error("write_integer(): Requested file integer size > 32 bits.\n");

  for (int k = bc-1 ; k >= 0 ; --k)
    {
      b[k] = (byte) (x & 0xff);
      x >>= 8;
    }

  return fwrite(b, bc, 1, fp) == 1;
}
