// ----------------------------------------------------------------------------
// Read and write binary data saved in big endian byte order.
// If the machine's native byte order is little endian then byte
// order is swapped before writing to a file and when reading from
// a file.
//
// These routines also handle some variations in integer type sizes.
// Specifically if long has more than 32 bits or short has more than
// 16 bits then the read routines will work and the write routines
// will just write the low order bits.  Floats must be 32 bits and
// the float format (eg IEEE) in the file must match the machines
// native format except for byte order.
//

#ifndef BINARYIO_HEADER_INCLUDED
#define BINARYIO_HEADER_INCLUDED

#include <stdio.h>	// Use FILE
#include <stddef.h>	// Use size_t

typedef unsigned char byte;			// 8 bits
typedef byte bit_field[4];			// 32 bits

void byte_reverse(byte *b, size_t size);

//
//  The following routines return true on success, false on failure.
//

bool beginning_of_file(FILE *fp);
bool seek_position(FILE *fp, long offset, int origin);

bool read_u8_array(char *c, size_t num, FILE *fp);
bool read_u8(char *c, FILE *fp);
bool read_s16(short *s, FILE *fp);
bool read_u16(unsigned short *us, FILE *fp);
bool read_s32(int *i, FILE *fp);
bool read_s32(long *l, FILE *fp);
bool read_s32_array(int *i, size_t num, bool big_endian, FILE *fp);
bool read_u32(unsigned *u, FILE *fp);
bool read_f32(float *f, FILE *fp);
bool read_f32(float *f, FILE *fp, bool big_endian);
bool read_f32_array(float *floats, size_t num, FILE *fp);
bool read_f32_array(float *floats, size_t num, FILE *fp, bool big_endian);
bool read_padding(size_t num, FILE *fp);
bool read_bit_field(bit_field bf, FILE *fp);

bool write_u8(char c, FILE *fp);
bool write_u8_array(char *c, size_t num, FILE *fp);
bool write_s16(short s, FILE *fp);
bool write_u16(unsigned short us, FILE *fp);
bool write_s32(int i, FILE *fp);
bool write_s32(long l, FILE *fp);
bool write_u32(unsigned u, FILE *fp);
bool write_f32(float f, FILE *fp);
bool write_f32_array(float *floats, size_t num, FILE *fp);
bool write_padding(size_t num, FILE *fp);
bool write_bit_field(bit_field bf, FILE *fp);

/* ----------------------------------------------------------------------------
*  Bit field manipulation.
*/
void zero_bit_field(bit_field bf);
unsigned long bit_field_bit(bit_field bf, int b);
void set_bit_field_bit(bit_field bf, int b, bool state);
unsigned long bit_field_bits(bit_field bf, int b1, int b2);
void set_bit_field_bits(bit_field bf, int b1, int b2, unsigned long bits);

#endif
