// ----------------------------------------------------------------------------
// Reading and writing matrices of floats stored in files in block format.
//

#include "binaryIO.h"		// Use read_f32_array()
#include "blockfile.h"		// Use Block_File
#include "list.h"		// Use List
#include "memalloc.h"		// use new()
#include "memcache.h"		// use Memory_Cache
#include "spoint.h"		// Use IPoint, IRegion
#include "utility.h"		// Use fatal_error()

// ----------------------------------------------------------------------------
// Cached data block
//
class Data_Block
{
public:
  Data_Block(const IPoint &block, int volume);
  ~Data_Block();

  IPoint block;		// Position in block space.
  float *data;
  bool modified;
};

// ----------------------------------------------------------------------------
//
static bool ipoint_equal(TableKey k1, TableKey k2);
static unsigned long ipoint_hash(TableKey key);

// ----------------------------------------------------------------------------
//
Block_File::Block_File(const Stringy &path,
		       const IPoint &size, const IPoint &block_size,
		       Memory_Cache *mcache) :
  blocks(ipoint_equal, ipoint_hash)
{
  this->data_path = path;
  this->data_size = size;
  this->block_siz = block_size;
  this->mcache = mcache;
}

// ----------------------------------------------------------------------------
//
static bool ipoint_equal(TableKey k1, TableKey k2)
  { return (*(IPoint *) k1 == *(IPoint *) k2); }

// ----------------------------------------------------------------------------
//
static unsigned long ipoint_hash(TableKey key)
{
  IPoint *p = (IPoint *) key;
  unsigned long h = 0;
  int dim = p->dimension();
  for (int k = 0 ; k < dim ; ++k)
    h = hash_combine(h, (*p)[k]);
  return h;
}

// ----------------------------------------------------------------------------
//
Block_File::~Block_File()
{
  mcache->free_all();
}

// ----------------------------------------------------------------------------
//
Stringy Block_File::path() const
  { return data_path; }
IPoint Block_File::size() const
  { return data_size; }
IPoint Block_File::block_size() const
  { return block_siz; }

// ----------------------------------------------------------------------------
//
bool Block_File::read_value(const IPoint &position, float *value)
{
  if (!data_region().contains(position))
    {
      *value = 0;
      return true;
    }

  IPoint block;
  int block_index;
  block_location(position, &block, &block_index);
  Data_Block *b = data_block(block);
  if (b)
    {
      // mcache->lock(b);
      *value = b->data[block_index];
      // mcache->unlock(b);
    }


  return b != NULL;
}

// ----------------------------------------------------------------------------
//
void Block_File::block_location(const IPoint &position,
				IPoint *block, int *block_index) const
{
  int bindex = 0;
  int istep = 1;

  *block = IPoint(position.dimension());
  for (int a = data_size.dimension() - 1 ; a >= 0 ; --a)
    {
      int bsize = block_siz[a];
      (*block)[a] = position[a] / bsize;
      bindex += (position[a] % bsize) * istep;
      istep *= bsize;
    }

  *block_index = bindex;
}

// ----------------------------------------------------------------------------
//
void Block_File::flush_cache()
{
  List blist = blocks.values();
  for (int bi = 0 ; bi < blist.size() ; ++bi)
    write_block((Data_Block *) blist[bi]);
}
// ----------------------------------------------------------------------------
//
Memory_Cache *Block_File::memory_cache()
  { return mcache; }

// ----------------------------------------------------------------------------
//
Data_Block *Block_File::data_block(const IPoint &block)
{
  TableData dblock;
  if (blocks.find(&block, &dblock))
    return (Data_Block *) dblock;

  int volume = block_volume();
  Data_Block *b = new Data_Block(block, volume);
  if (!read_block(b))
    {
      delete b;
      return NULL;
    }

  int size = volume * sizeof(float) + sizeof(Data_Block);
  mcache->add_to_cache(b, size, free_data_block_cb, this);
  blocks.insert(&b->block, b);

  return b;
}

// ----------------------------------------------------------------------------
//
void Block_File::free_data_block_cb(void *dblock, void *bfile)
{
  Data_Block *b = (Data_Block *) dblock;
  Block_File *bf = (Block_File *) bfile;

  if (!bf->write_block(b))
    warn("Block_File::free_data_block_cb: Failed writing data block to %s\n",
	 bf->data_path.cstring());

  bf->blocks.remove(&b->block);
  delete b;
}

// ----------------------------------------------------------------------------
//
int Block_File::block_volume() const
  { return block_siz.product(); }

// ----------------------------------------------------------------------------
//
bool Block_File::read_block(Data_Block *b)
{
  bool success = read_block(b->block, b->data);

  if (success)
    b->modified = false;

  return success;
}

// ----------------------------------------------------------------------------
//
bool Block_File::write_block(Data_Block *b)
{
  if (!b->modified)
    return true;

  bool success = write_block(b->block, b->data);

  if (success)
    b->modified = false;

  return success;
}

// ----------------------------------------------------------------------------
//
Data_Block::Data_Block(const IPoint &block, int volume)
{
  this->block = block;
  this->data = new float [volume];
  this->modified = false;
}

// ----------------------------------------------------------------------------
//
Data_Block::~Data_Block()
{
  delete [] data;
  data = NULL;
}

// ----------------------------------------------------------------------------
//
bool Block_File::write_zeros()
{
  bool success = true;

  int size = block_volume();
  Data_Block b(IPoint(0), size);
  for (int k = 0 ; k < size ; ++k)
    b.data[k] = 0;

  Subarray_Iterator bi(block_space(), block_space(), true);
  for ( ; !bi.finished() ; bi.next())
    {
      b.block = bi.position();
      b.modified = true;
      success = write_block(bi.position(), b.data);
      if (!success)
	break;
    }

  return success;
}

// ----------------------------------------------------------------------------
//
bool Block_File::read_values(const IRegion &region, float *values)
{
  int vol = region.volume();
  for (int k = 0 ; k < vol ; ++k)
    values[k] = 0;

  IRegion r = region;
  r.clip(data_region());

  Subarray_Iterator b(block_subspace(r), block_space(), true);
  for ( ; !b.finished() ; b.next())
    if (!get_block_data(b.position(), region, values))
      return false;

  return true;
}

// ----------------------------------------------------------------------------
//
IRegion Block_File::block_space()
  { return ::block_space(data_size, block_siz); }

// ----------------------------------------------------------------------------
//
IRegion block_space(const IPoint &data_size, const IPoint &block_size)
{
  IRegion bs(IPoint(data_size.dimension()), IPoint(data_size.dimension()));

  for (int a = 0 ; a < data_size.dimension() ; ++a)
    {
      bs.min[a] = 0;
      bs.max[a] = (data_size[a] - 1) / block_size[a];
    }

  return bs;
}

// ----------------------------------------------------------------------------
//
IRegion Block_File::block_subspace(const IRegion &region)
{
  IRegion bregion = region;

  bregion.clip(data_region());
  for (int a = 0 ; a < region.dimension() ; ++a)
    {
      bregion.min[a] /= block_siz[a];
      bregion.max[a] /= block_siz[a];
    }

  return bregion;
}

// ----------------------------------------------------------------------------
//
IRegion Block_File::data_region()
{
  int dim = data_size.dimension();
  IPoint min(dim), max(dim);
  for (int a = 0 ; a < dim ; ++a)
    max[a] = data_size[a] - 1;

  return IRegion(min, max);
}

// ----------------------------------------------------------------------------
//
bool Block_File::get_block_data(const IPoint &b, const IRegion &region,
				float *data)
{
  Data_Block *blk = data_block(b);
  if (blk == NULL)
    return false;

  // mcache->lock(blk);

  float *block_data = blk->data;
  IRegion block = block_region(b);
  IRegion piece = block;
  piece.clip(region);
  piece.clip(data_region());

  //
  // Optimization: Copy data a row at a time along the longest axis.
  //

  IRegion base = piece;
  int long_axis = piece.longest_axis();
  base.max[long_axis] = base.min[long_axis];

  Subarray_Iterator bi(base, block, true);
  Subarray_Iterator di(base, region, true);

  int bstep = bi.step_size(long_axis);
  int dstep = di.step_size(long_axis);
  int rlength = piece.size(long_axis);

  for ( ; !bi.finished() ; bi.next(), di.next())
    {
      float *from = block_data + bi.index();
      float *to = data + di.index();
      for (int k = 0 ; k < rlength ; ++k)
	to[k * dstep] = from[k * bstep];
    }

  // mcache->unlock(blk);

  //
  //  Old point by point copy
  //
  //  for ( ; !bi.finished() ; bi.next(), di.next())
  //    data[di.index()] = block_data[bi.index()];

  return true;
}

// ----------------------------------------------------------------------------
//
IRegion block_region(const IPoint &block, const IPoint &block_size)
{
  int dim = block_size.dimension();
  IPoint min(dim), max(dim);

  for (int a = 0 ; a < dim ; ++a)
    {
      int size = block_size[a];
      min[a] = block[a] * size;
      max[a] = min[a] + size - 1;
    }

  return IRegion(min, max);
}

// ----------------------------------------------------------------------------
//
IRegion Block_File::block_region(const IPoint &block)
{
  return ::block_region(block, block_siz);
}

// ----------------------------------------------------------------------------
//
bool Block_File::write_value(const IPoint &position, float value)
{
  if (!data_region().contains(position))
    fatal_error("Block_File::write_value(): Position out of range.\n");

  IPoint block;
  int block_index;
  block_location(position, &block, &block_index);

  Data_Block *b = data_block(block);
  if (b == NULL)
    return false;

  // mcache->lock(b);

  b->data[block_index] = value;
  b->modified = true;

  // mcache->unlock(b);

  return true;
}

// ----------------------------------------------------------------------------
//
bool Block_File::write_values(const IRegion &region, float *values)
{
  Subarray_Iterator b(block_subspace(region), block_space(), true);

  for ( ; !b.finished() ; b.next())
    if (!put_block_data(b.position(), region, values))
      return false;

  return true;
}

// ----------------------------------------------------------------------------
//
bool Block_File::put_block_data(const IPoint &blk,
				const IRegion &region, float *data)
{
  Data_Block *cblock = data_block(blk);
  if (cblock == NULL)
    return false;

  // mcache->lock(cblock);

  float *block_data = cblock->data;

  IRegion block = block_region(blk);
  IRegion piece = block;
  piece.clip(region);

  //
  // Optimization: Copy data a row at a time along the longest axis.
  //

  IRegion base = piece;
  int long_axis = piece.longest_axis();
  base.max[long_axis] = base.min[long_axis];

  Subarray_Iterator bi(base, block, true);
  Subarray_Iterator di(base, region, true);

  int bstep = bi.step_size(long_axis);
  int dstep = di.step_size(long_axis);
  int rlength = piece.size(long_axis);

  for ( ; !bi.finished() ; bi.next(), di.next())
    {
      float *from = data + di.index();
      float *to = block_data + bi.index();
      for (int k = 0 ; k < rlength ; ++k)
	to[k * bstep] = from[k * dstep];
    }

  cblock->modified = true;

  // mcache->unlock(cblock);

  return true;
}

// ----------------------------------------------------------------------------
// Block file with header and no padding between blocks.
// Both the blocks and the data within blocks are indexed with the given
// indexing style (big_endian_indexing == true means highest axis
// varies fastest).
//
class Packed_Block_File : public Block_File
{
public:
  Packed_Block_File(FILE *fp, const Stringy &path,
		    const IPoint &size, const IPoint &block_size,
		    int data_start, bool big_endian_indexing,
		    Memory_Cache *);
  virtual ~Packed_Block_File();

  virtual bool read_block(const IPoint &block, float *values);
  virtual bool write_block(const IPoint &block, float *values);
private:
  FILE *fp;
  int data_start;
  bool big_endian_indexing;

  int block_location(const IPoint &block);
};


// ----------------------------------------------------------------------------
//
Block_File *packed_block_file(const Stringy &path, const Stringy &mode,
			      const IPoint &size, const IPoint &block_size,
			      int data_start, bool big_endian_indexing,
			      Memory_Cache *mcache)
{
  FILE *fp = fopen(path.cstring(), mode.cstring());
  if (fp == NULL)
    return NULL;
  return new Packed_Block_File(fp, path, size, block_size,
			       data_start, big_endian_indexing, mcache);
}

// ----------------------------------------------------------------------------
//
Packed_Block_File::Packed_Block_File(FILE *fp, const Stringy &path,
				     const IPoint &size,
				     const IPoint &block_size,
				     int data_start,
				     bool big_endian_indexing,
				     Memory_Cache *mcache) :
  Block_File(path, size, block_size, mcache)
{
  this->fp = fp;
  this->data_start = data_start;
  this->big_endian_indexing = big_endian_indexing;
}

// ----------------------------------------------------------------------------
//
Packed_Block_File::~Packed_Block_File()
{
  flush_cache();
  fclose(fp);
  fp = NULL;
}

// ----------------------------------------------------------------------------
//
bool Packed_Block_File::read_block(const IPoint &block, float *values)
{
  int position = block_location(block);
  int size = block_volume();

  bool read;
  if (big_endian_indexing)
    read = (seek_position(fp, position, 0) &&
	    read_f32_array(values, size, fp));
  else
    {
      float *temp = new float [size];
      read = (seek_position(fp, position, 0) &&
	      read_f32_array(temp, size, fp));
      if (read)
	little_to_big_endian_indexing(block_region(block), temp, values);
      delete [] temp;
    }

  return read;
}

// ----------------------------------------------------------------------------
//
bool Packed_Block_File::write_block(const IPoint &block, float *values)
{
  int position = block_location(block);
  int size = block_volume();

  bool wrote;
  if (big_endian_indexing)
    wrote = (seek_position(fp, position, 0) &&
	     write_f32_array(values, size, fp));
  else
    {
      float *temp = new float [size];
      big_to_little_endian_indexing(block_region(block), values, temp);
      wrote = (seek_position(fp, position, 0) &&
	       write_f32_array(temp, size, fp));
      delete [] temp;
    }

  return wrote;
}

// ----------------------------------------------------------------------------
//
int Packed_Block_File::block_location(const IPoint &block)
{
  int block_number = position_index(block, block_space(), big_endian_indexing);

  return data_start + block_number * block_volume() * (int) sizeof(float);
}

