/*
 * Contour.C:		Implementation of the Contour class
 *
 * The Contour class implements the "contour-broker".
 * The set of ContourPlanes is managed here.
 *
 * The Contour object contains a Set of ContourPlane objects each which
 * contain the methods for computing and drawing the contoured data.
 *
 */
#include <math.h>		// use pow()

#include "constant.h"		// Use DIM
#include "contour.h"
#include "contourplane.h"
#include "num.h"
#include "list.h"		// Use List
#include "memalloc.h"		// use new()
#include "memcache.h"		// use Memory_Cache
#include "notifier.h"		// use Notifier
#include "rectangle.h"		// Use Axis, Rectangle
#include "spectrum.h"
#include "table.h"		// Use Table
#include "utility.h"		// Use fatal_error()
#include "wait.h"		// use Wait

// ----------------------------------------------------------------------------
//
static IRegion planar_region(Spectrum *sp, const SRegion &r);
static int tile_count(Spectrum *sp, const IRegion &r);
static IRegion first_tile(Spectrum *sp, const IRegion &r);
static void tile_range(Spectrum *sp, int axis, const IRegion &r,
		       int *lo, int *hi, int *size);
static IRegion next_tile(Spectrum *sp, const IRegion &r,
			 const IRegion &previous_tile);
static int BlockSize(Spectrum *sp, int axis);

// ----------------------------------------------------------------------------
//
class Contour_Cache
{
public:
  Contour_Cache(Memory_Cache *, Notifier &);
  virtual ~Contour_Cache();

  ContourPlane *contour_plane(Spectrum *sp, const IRegion &tile,
			      const Contour_Levels &pos,
			      const Contour_Levels &neg);
  void release_plane(ContourPlane *plane);

private:
  Table contour_planes;
  Memory_Cache *memory_cache;
  Notifier &notifier;

  static void spectrum_deleted_cb(void *ccache, void *spectrum);
  static void free_plane_cb(void *cplane, void *ccache);
};

// ----------------------------------------------------------------------------
//
bool Contour_Levels::operator==(const Contour_Levels &levels) const
{
  return (this->lowest == levels.lowest &&
	  this->factor == levels.factor &&
	  this->levels == levels.levels);
}

// ----------------------------------------------------------------------------
//
double Contour_Levels::highest() const
  { return (levels > 0 ? lowest * pow(factor, levels - 1) : 0); }

// ----------------------------------------------------------------------------
//
void draw_contours(Spectrum *sp, const SRegion &r,
		   const Contour_Levels &pos, const Contour_Levels &neg,
		   bool subtract_fit_peaks,
		   Drawing_Routines &dr, Contour_Cache *cc,
		   Wait *wait)
{
  IRegion index_region = planar_region(sp, r);
  if (index_region.empty())
    return;

  int tiles = tile_count(sp, index_region);
  int tcount = 0;
  for (IRegion tile = first_tile(sp, index_region) ; 
       !tile.empty() ;
       tile = next_tile(sp, index_region, tile))
    {
      if (subtract_fit_peaks)
	{
	  ContourPlane cp(sp, tile, pos, neg, subtract_fit_peaks);
	  cp.display(dr);
	}
      else
	{
	  ContourPlane *cp = cc->contour_plane(sp, tile, pos, neg);
	  cp->display(dr);
	  cc->release_plane(cp);
	}

      tcount += 1;
      if (wait)
	if (wait->was_stop_requested())
	  break;
	else if (tcount % 100 == 0)
	  wait->progress_report("Drawing contours %.0f%% done.",
				(100.0 * tcount) / tiles);
    }
}

// ----------------------------------------------------------------------------
//
static IRegion planar_region(Spectrum *sp, const SRegion &r)
{
  int axis1, axis2;
  if (!r.plane_axes(&axis1, &axis2))
    return IRegion();

  int dim = sp->dimension();
  IRegion index_region(dim);
  for (int a = 0 ; a < dim ; ++a)
    if (a == axis1 || a == axis2)
      {
	// Should really use floor() and ceil() here but it creates
	// off by one rounding problems.

	index_region.min[a] = round(sp->map(r.max[a], a, PPM, INDEX));
	index_region.max[a] = round(sp->map(r.min[a], a, PPM, INDEX));
      }
    else
      {
	int center = round(sp->map(r.center(a), a, PPM, INDEX));
	index_region.min[a] = center;
	index_region.max[a] = center;
      }
  return index_region;
}

// ----------------------------------------------------------------------------
//
static int tile_count(Spectrum *sp, const IRegion &r)
{
  int axis1, axis2;
  if (!r.plane_axes(&axis1, &axis2))
    return 0;

  int t1_low, t1_high, size1, t2_low, t2_high, size2;
  tile_range(sp, axis1, r, &t1_low, &t1_high, &size1);
  tile_range(sp, axis2, r, &t2_low, &t2_high, &size2);

  return (t1_high - t1_low + 1) * (t2_high - t2_low + 1);
}

// ----------------------------------------------------------------------------
//
static IRegion first_tile(Spectrum *sp, const IRegion &r)
{
  int axis1, axis2;
  if (!r.plane_axes(&axis1, &axis2))
    return IRegion();

  int t1_low, t1_high, size1, t2_low, t2_high, size2;
  tile_range(sp, axis1, r, &t1_low, &t1_high, &size1);
  tile_range(sp, axis2, r, &t2_low, &t2_high, &size2);

  IRegion tile(axis1, t1_low * size1, (t1_low + 1) * size1,
	       axis2, t2_low * size2, (t2_low + 1) * size2, r.min);

  return tile;
}

// ----------------------------------------------------------------------------
// Note: Tiles overlap on their boundaries.
//
static void tile_range(Spectrum *sp, int axis, const IRegion &r,
		       int *lo, int *hi, int *size)
{
  int s = BlockSize(sp, axis);
  int x1 = r.min[axis];
  int x2 = r.max[axis];

  *lo = (x1 >= 0 ? x1/s : (x1 + 1)/s - 1);
  *hi = (x2 > x1 ? (x2 > 0 ? (x2 - 1)/s : x2/s - 1) :
	 (x2 == x1 ? *lo : *lo - 1));
  *size = s;
}

// ----------------------------------------------------------------------------
//
static int BlockSize(Spectrum *sp, int axis)
{
  int nPoints = max(1, sp->index_range().size(axis));

  return min(CONTOUR_TILE_SIZE, nPoints);
}

// ----------------------------------------------------------------------------
//
static IRegion next_tile(Spectrum *sp, const IRegion &r,
			 const IRegion &previous_tile)
{
  int axis1, axis2;
  if (!r.plane_axes(&axis1, &axis2))
    return IRegion();

  int t1_low, t1_high, size1, t2_low, t2_high, size2;
  tile_range(sp, axis1, r, &t1_low, &t1_high, &size1);
  tile_range(sp, axis2, r, &t2_low, &t2_high, &size2);

  IRegion tile = previous_tile;
  tile.translate(size1, axis1);

  if (tile.min[axis1] > t1_high * size1)
    {
      tile.min[axis1] = t1_low * size1;
      tile.max[axis1] = (t1_low + 1) * size1;

      tile.translate(size2, axis2);

      if (tile.min[axis2] > t2_high * size2)
	return IRegion();
    }

  return tile;
}

// ----------------------------------------------------------------------------
//
Contour_Cache *create_contour_cache(Memory_Cache *mc, Notifier &n)
  { return new Contour_Cache(mc, n); }
void delete_contour_cache(Contour_Cache *cc)
  { delete cc; }

// ----------------------------------------------------------------------------
//
Contour_Cache::Contour_Cache(Memory_Cache *mc, Notifier &n)
 : contour_planes(equal_contour_planes, contour_plane_hash),
   notifier(n)
{
  this->memory_cache = mc;
  notifier.notify_me(nt_will_delete_spectrum, spectrum_deleted_cb, this);
}

// ----------------------------------------------------------------------------
//
Contour_Cache::~Contour_Cache()
{
  notifier.dont_notify_me(nt_will_delete_spectrum, spectrum_deleted_cb, this);

  List cplist = contour_planes.keys();
  for (int cpi = 0 ; cpi < cplist.size() ; ++cpi)
    {
      ContourPlane *cp = (ContourPlane *) cplist[cpi];
      memory_cache->remove_from_cache((void *) cp);
      delete cp;
    }
  contour_planes.erase();
}

// ----------------------------------------------------------------------------
//
void Contour_Cache::spectrum_deleted_cb(void *ccache, void *spectrum)
{
  Contour_Cache *cc = (Contour_Cache *) ccache;
  Spectrum *sp = (Spectrum *) spectrum;

  List cplist = cc->contour_planes.keys();
  for (int cpi = 0 ; cpi < cplist.size() ; ++cpi)
    {
      ContourPlane *cp = (ContourPlane *) cplist[cpi];
      if (cp->spectrum() == sp)
	{
	  cc->contour_planes.remove(cp);
	  cc->memory_cache->remove_from_cache((void *) cp);
	  delete cp;
	}
    }
}

// ----------------------------------------------------------------------------
//
ContourPlane *Contour_Cache::contour_plane(Spectrum *sp, const IRegion &tile,
					   const Contour_Levels &pos,
					   const Contour_Levels &neg)
{
  ContourPlane *plane;

  TableData p;
  ContourPlane cp(sp, tile, pos, neg, false);
  if (contour_planes.find(&cp, &p))
    plane = (ContourPlane *) p;
  else
    {
      plane = new ContourPlane(sp, tile, pos, neg, false);
      contour_planes.insert(plane, plane);
      memory_cache->add_to_cache((void *) plane, plane->memory_use(),
				 free_plane_cb, this);
    }
  memory_cache->lock(plane);

  return plane;
}

// ----------------------------------------------------------------------------
//
void Contour_Cache::release_plane(ContourPlane *plane)
{
  memory_cache->unlock(plane);
}

// ----------------------------------------------------------------------------
// Add a ContourPlane to the set of planes.
//
void Contour_Cache::free_plane_cb(void *cplane, void *ccache)
{
  ContourPlane *cp = (ContourPlane *) cplane;
  Contour_Cache *cc = (Contour_Cache *) ccache;

  cc->contour_planes.remove(cp);
  delete cp;
}
