// ---------------------------------------------------------------------------
//
#include "dataregion.h"
#include "list.h"		// Use List
#include "memalloc.h"		// use new()
#include "spectrum.h"		// Use Spectrum
#include "spoint.h"		// Use IRegion
#include "wait.h"		// use Wait

static void delete_ipoint_list_entries(const List &ipoints);
static IRegion region_union(const IRegion &r1, const IRegion &r2);

// ---------------------------------------------------------------------------
//
Data_Region::Data_Region(Spectrum *sp, const IRegion &region)
{
  spect = sp;
  reg = region;
  values = NULL;
}

// ---------------------------------------------------------------------------
//
Data_Region::~Data_Region()
{
  delete [] values;

  values = NULL;
  spect = NULL;
}

// ---------------------------------------------------------------------------
//
Spectrum *Data_Region::spectrum() const
  { return spect; }

// ---------------------------------------------------------------------------
//
const IRegion &Data_Region::region() const
  { return reg; }

// ---------------------------------------------------------------------------
//
float *Data_Region::data() const
{
  if (values == NULL)
    {
      ((Data_Region *) this)->values = new float [region().volume()];
      spectrum()->heights_for_region(reg, values);
    }

  return values;
}

// ----------------------------------------------------------------------------
//
double Data_Region::height(const IPoint &p) const
{
  return data()[position_index(p, region(), true)];
}

// ----------------------------------------------------------------------------
//
void Data_Region::change_region(const IRegion &r)
{
  delete [] values;
  values = NULL;
  reg = r;
}

// ---------------------------------------------------------------------------
//
Marked_Region::Marked_Region(Spectrum *sp, const SRegion &limits)
  : Data_Region(sp, IRegion(sp->dimension()))
{
  this->marks = NULL;
  this->limits = sp->map(limits, PPM, INDEX).rounded();
}

// ---------------------------------------------------------------------------
//
Marked_Region::~Marked_Region()
{
  delete [] marks;
  marks = NULL;
}

// ---------------------------------------------------------------------------
//
bool *Marked_Region::marked()
{
  if (marks == NULL)
    {
      int size = region().volume();
      marks = new bool [size];
      for (int k = 0 ; k < size ; ++k)
	marks[k] = false;
    }
  return marks;
}

// ---------------------------------------------------------------------------
//
bool Marked_Region::mark_connected(const IPoint &start,
				 double lothresh, double hithresh,
				 int volume_limit, Wait *wait)
{
  List frontier, free_ipoints;
  frontier.append(new IPoint(start));
  return mark_connected(frontier, free_ipoints,
			lothresh, hithresh, volume_limit, wait);
}

// ---------------------------------------------------------------------------
//
bool Marked_Region::mark_connected(List &frontier, List &free_ipoints,
				 double lothresh, double hithresh,
				 int volume_limit, Wait *wait)
{
  List new_frontier;
  for (int pi = 0 ; pi < frontier.size() ; ++pi)
    {
      IPoint start = *(IPoint *) frontier[pi];

      if (!limits.contains(start))
	continue;

      if (!region().contains(start))
	if (!extend_region(start, volume_limit))
	  {
	    delete_ipoint_list_entries(frontier);
	    delete_ipoint_list_entries(free_ipoints);
	    return false;
	  }

      int p = position_index(start, region(), true);
      float h = data()[p];
      if (!marked()[p] && h >= lothresh && h <= hithresh)
	{
	  marked()[p] = true;
	  for (int d = 0 ; d < region().dimension() ; ++d)
	    {
	      if (free_ipoints.empty())
		free_ipoints.append(new IPoint());
	      IPoint *neighbor = (IPoint *) free_ipoints.pop();
	      *neighbor = start;
	      (*neighbor)[d] += 1;
	      new_frontier.append(neighbor);

	      if (free_ipoints.empty())
		free_ipoints.append(new IPoint());
	      neighbor = (IPoint *) free_ipoints.pop();
	      *neighbor = start;
	      (*neighbor)[d] -= 1;
	      new_frontier.append(neighbor);
	    }
	}
    }

  free_ipoints.append(frontier);
  frontier = new_frontier;
  if (frontier.empty())
    {
      delete_ipoint_list_entries(free_ipoints);
      return true;
    }

  if (wait && wait->was_stop_requested())
    {
      delete_ipoint_list_entries(frontier);
      delete_ipoint_list_entries(free_ipoints);
      return false;
    }

  return mark_connected(frontier, free_ipoints,
			lothresh, hithresh, volume_limit, wait);
}

// ----------------------------------------------------------------------------
//
static void delete_ipoint_list_entries(const List &ipoints)
{
  for (int pi = 0 ; pi < ipoints.size() ; ++pi)
    delete (IPoint *) ipoints[pi];
}

// ----------------------------------------------------------------------------
//
bool Marked_Region::extend_region(const IPoint &start, int volume_limit)
{
  int dim = start.dimension();
  int extension_radius = (dim == 2 ? 16 : (dim == 3 ? 8 : 4));
  IPoint lo = start, hi = start;
  for (int d = 0 ; d < dim ; ++d)
    {
      lo[d] -= extension_radius;
      hi[d] += extension_radius;
    }
  IRegion local = IRegion(lo, hi);

  IRegion extended = region_union(region(), local);
  if (extended.volume() > volume_limit)
    return false;

  resize_region(extended);

  return true;
}

// ----------------------------------------------------------------------------
//
void Marked_Region::mark_region(const SRegion &r)
{
  IRegion index_region = spectrum()->map(r, PPM, INDEX).rounded();
  if (!region().contains(index_region))
    resize_region(region_union(region(), index_region));

  bool *m = marked();
  Subarray_Iterator i(index_region, region(), true);
  for ( ; !i.finished() ; i.next())
    m[i.index()] = true;
}

// ----------------------------------------------------------------------------
//
static IRegion region_union(const IRegion &r1, const IRegion &r2)
{
  if (r1.empty())
    return r2;
  else if (r2.empty())
    return r1;

  IRegion sum = r1;
  sum.encompass(r2);
  return sum;
}

// ----------------------------------------------------------------------------
//
void Marked_Region::resize_region(const IRegion &r)
{
  if (marks)
    {
      int size = r.volume();
      bool *new_marks = new bool [size];
      for (int k = 0 ; k < size ; ++k)
	new_marks[k] = false;
      copy_array(marks, new_marks, sizeof(bool), region(), r);
      delete [] marks;
      marks = new_marks;
    }

  change_region(r);
}

// ---------------------------------------------------------------------------
//
bool Marked_Region::is_marked(const IPoint &p) const
{
  return (marks && region().contains(p) &&
	  marks[position_index(p, region(), true)]);
}

// ---------------------------------------------------------------------------
//
int Marked_Region::marked_volume() const
{
  if (marks == NULL)
    return 0;

  int volume = 0;
  int size = region().volume();
  for (int k = 0 ; k < size ; ++k)
    if (marks[k])
      volume += 1;

  return volume;
}

// ---------------------------------------------------------------------------
//
bool Marked_Region::enscribe_marked()
{
  if (marks == NULL)
    return false;

  IRegion bounding_region(region().max, region().min);
  Subarray_Iterator pi(region(), region(), true);
  for ( ; !pi.finished() ; pi.next())
    if (marks[pi.index()])
      bounding_region.encompass(pi.position());

  if (bounding_region.empty())
    return false;

  resize_region(bounding_region);

  return true;
}
