// ----------------------------------------------------------------------------
//
#include <iostream.h>		// Use ostream, istream
#include <stdio.h>		// Use sprintf(), fputc(), ...
#include <string.h>		// Use strchr(), strcmp()
#include <ctype.h>		// Use isspace(), isalpha()

#include "atom.h"		// use Atom
#include "brukerfile.h"		// use is_bruker_nmr_data()
#include "color.h"		// Use Color
#include "condition.h"		// Use Condition
#include "felixfile.h"		// Use is_felix_nmr_data()
#include "format.h"
#include "grid.h"		// Use Grid
#include "group.h"		// Use Group
#include "integrate.h"		// Use integrate_*()
#include "label.h"		// Use Label
#include "line.h"		// Use Line
#include "memalloc.h"		// use new()
#include "molecule.h"		// Use Molecule
#include "nmrdata.h"		// use standard_nucleus_name()
#include "num.h"		// Use min()
#include "objectid.h"		// use Object_Table
#include "ornament.h"		// Use Ornament
#include "peak.h"		// Use Peak
#include "peakgp.h"		// Use PeakGp
#include "print.h"		// Use Print_Options
#include "project.h"		// Use Project
#include "reporter.h"		// use Reporter
#include "resonance.h"		// Use Resonance
#include "savefile.h"		// Use SaveFile
#include "session.h"		// use Session
#include "spectrum.h"		// Use Spectrum
#include "spoint.h"		// Use SPoint
#include "system.h"		// Use file_exists(), tilde_expand()
#include "ucsffile.h"		// Use ucsf_nmr_data()
#include "uidialogs.h"		// Use request_new_data_location()
#include "uimain.h"		// Use query(), status_line()
#include "uimode.h"		// Use pointer_mode()
#include "notifier.h"		// use Notifier
#include "utility.h"		// Use fatal_error()
#include "uiview.h"		// Use View

#define DBFILENAME		"db"

#define COMMENT			'#'		// command input comment
#define ANAMESIZE		32	/* max size of group or atom name */
#define NO_ID			(-2)
#define SPECTRUM_NAME_SIZE	256
#define MAX_LINE_LENGTH		4096

/*
 * The following strings are found in spectrum annotation (save) files.
 */
#define SAVE_BEGINVIEW 		"<view>\n"
#define SAVE_ENDVIEW 		"<end view>\n"
#define SAVE_BEGINSPECTRUM	"<spectrum>\n"
#define SAVE_ENDSPECTRUM	"<end spectrum>\n"
#define SAVE_BEGINORNAMENT	"<ornament>\n"
#define SAVE_ENDORNAMENT	"<end ornament>\n"
#define SAVE_BEGINPARAMS	"<params>\n"
#define SAVE_ENDPARAMS		"<end params>\n"
#define SAVE_BEGINLINKS		"<links>\n"
#define SAVE_ENDLINKS		"<end links>\n"
#define SAVE_BEGINUSER		"<user>\n"
#define SAVE_ENDUSER		"<end user>\n"
#define SAVE_BEGINSYNC		"<syncs>\n"
#define SAVE_ENDSYNC		"<end syncs>\n"
#define SAVE_BEGINSYNCAXES	"<syncaxes>\n"
#define SAVE_ENDSYNCAXES	"<end syncaxes>\n"
#define SAVE_BEGINSET		"<sets>\n"
#define SAVE_ENDSET		"<end sets>\n"
#define SAVE_BEGINAXISMAP	"<axismaps>\n"
#define SAVE_ENDAXISMAP		"<end axismaps>\n"
#define SAVE_BEGINDBGROUPS	"<DBgroups>\n"
#define SAVE_ENDDBGROUPS	"<end DBgroups>\n"
#define SAVE_BEGINMOLECULE	"<molecule>\n"
#define SAVE_ENDMOLECULE	"<end molecule>\n"
#define SAVE_BEGINSEQUENCE	"<sequence>\n"
#define SAVE_ENDSEQUENCE	"<end sequence>\n"
#define SAVE_BEGINCONDITION	"<condition>\n"
#define SAVE_ENDCONDITION	"<end condition>\n"
#define SAVE_BEGINRESONANCES	"<resonances>\n"
#define SAVE_ENDRESONANCES	"<end resonances>\n"
#define SAVE_BEGINOVERLAYS	"<overlays>\n"
#define SAVE_ENDOVERLAYS	"<end overlays>\n"
#define SAVE_BEGINATTACHEDDATA	"<attached data>\n"
#define SAVE_ENDATTACHEDDATA	"<end attached data>\n"

//
// Flags for restoration from V1 save files.
//

#define RESTORE_ORNAMENT_FAILED		0
#define RESTORE_ORNAMENT_SUCCESS	1
#define RESTORE_ORNAMENT_CONTINUE	2

// ----------------------------------------------------------------------------
//
class View_Parameters
{
public:
  View_Parameters(Spectrum *sp, const View_Settings &settings);
  View *create_view(Session &);

  Spectrum *sp;
  View_Settings settings;
};

// ----------------------------------------------------------------------------
//
class Ornament_Properties
{
public:
  Ornament_Type type;
  bool select;
  bool lock;
  Stringy color;
  Stringy note;
};

// ----------------------------------------------------------------------------
//
static char *strncpy_terminate	(char *buf , const char *from , int bufsiz);
static char *strchrquote	(char *line , char ch);
static char *strtokquote	(char *line , char sep);
static bool isemptyline	(char *line);
static char *	sgetstring	(char **cpp);
static void	fputstring	(FILE *fp , const char *cp);
static bool	YorN		(char *cp);
static int	label2axis	(const char *str);

// ----------------------------------------------------------------------------
// String tokenizing routines.
//
static char *next_token(char **rest);
static char *next_line(char **rest);
static int next_int(char **rest);
static double next_double(char **rest);

// ----------------------------------------------------------------------------
// SPoint format
//
static SPoint read_point(char *text, char **next);
static SPoint read_point(istream &);

// ----------------------------------------------------------------------------
// Ornament format
//
static void ornament_save_v1(Ornament *op, FILE *fp);
static void ornament_save_identifier(Ornament *op, int id, ostream &o);
static Ornament *ornament_restore_for_undo(Session &s, istream &i,
					   Spectrum *sp,
					   Ornament_Type type, Ornament *op,
					   Object_Table &ids);
static Ornament *MakeOrnament(Session &, Spectrum *sp, Ornament_Type t, int id,
			      istream &strm, Object_Table &ids);
static Ornament *restore_embedded_ornament_for_undo(Session &, Spectrum *,
						    istream &i,
						    Object_Table &ids);
static void ornament_save_note_v1(Ornament *op, FILE *fp);
static void save_grid_v1(FILE *fp, Grid *g);
static Grid *restore_grid_v1(FILE *fp, Spectrum *sp);
static void save_grid_for_undo(ostream &o, Grid *grid);
static Grid *restore_grid_for_undo(istream &i, Spectrum *sp, Grid *grid);
static void save_line_v1(FILE *fp, Line *line);
static Line *restore_line_v1(FILE *fp, Spectrum *sp);
static void save_line_for_undo(ostream &o, Line *line);
static Line *restore_line_for_undo(istream &i, Spectrum *sp, Line *line);
static void save_label_v1(FILE *fp, Label *lbl);
static Label *restore_label_v1(FILE *fp, Spectrum *sp);
static LABELPOS label_positions(char *line, Spectrum *sp);
static void save_label_for_undo(ostream &o, Label *label, Object_Table &ids);
static Label *restore_label_for_undo(Project &, istream &i, Spectrum *sp,
				     Label *label, Object_Table &ids);
static void save_peak_v1(FILE *fp, Peak *pk);
static Peak *restore_peak_v1(FILE *fp, Spectrum *sp);
static SPoint decrement_point_dimension(const SPoint &p);
static void RestoreAssignment(CrossPeak *xp, const Stringy &str);
static void restore_peak_integration(char *line,
				     double *volume, Integration_Method *itype,
				     Stringy *lwtype);
static void restore_volume_error(char *line, double *verror, Stringy *method);
static Integration_Method integration_type(const char *name);
static void save_peak_for_undo(ostream &o, Peak *pk, Object_Table &ids);
static Peak *restore_peak_for_undo(Session &, istream &i, Spectrum *, Peak *,
				   Object_Table &ids);
static void read_volume(istream &strm, double *volume,
			Integration_Method *itype);
static void read_volume_error(istream &strm, double *verror, Stringy *method);
static void read_linewidth(istream &strm, Spectrum *sp,
			   SPoint *lw, Stringy *lwtype);
static void save_peakgroup_v1(FILE *fp, PeakGp *pg);
static PeakGp *restore_peakgroup_v1(FILE *fp, Spectrum *sp);
static void save_peakgroup_for_undo(ostream &o, PeakGp *pg, Object_Table &ids);
static PeakGp *restore_peakgroup_for_undo(Session &, istream &i, Spectrum *,
					  PeakGp *, Object_Table &ids);
static bool ornament_get_line_v1(Ornament_Properties &oprops,
				 char *buf, int bufsiz, FILE *fp,
				 char **tokp, char **rest, int *statusp,
				 Ornament_Type *next_type);
static void set_ornament_properties(Ornament *op, Ornament_Properties &oprops);
static Stringy ornament_get_note_v1(char **rest, FILE *fp);
static Ornament_Type RestoreV1Type();
static Stringy assignment_name(Resonance *r);
static void SaveAssignment(CrossPeak *xp, FILE *fp);
static void SaveAssignment(CrossPeak *xp, ostream &o);
static void SaveXRef(Ornament *op, Object_Table &ids, ostream &o);
static Ornament *FindXRef(Project &, const char *xRef, Object_Table &ids);
static Ornament *FindOrnament(Spectrum *sp, Ornament_Type t, int id,
			      Object_Table &ids);
static bool ParseIdentifier(Project &, Ornament_Type *typep, int *idp,
			    Spectrum **spp, const char *buf);
static Stringy colorread(const char *str, Ornament_Type type);
static Stringy colorformat(const Color &c, Ornament_Type t);
static void save_ornaments(Session &, Spectrum *sp, FILE *fp);
static bool has_owner(Ornament *orn);
static bool load_ornaments(Session &, Spectrum *sp, FILE *fp);
static Ornament *load_ornament(Ornament_Type type, Spectrum *sp, FILE *fp);
static void peak_list_option(char *line, Peak_List_Options *options);
static void save_peak_list_options(FILE *fp, Peak_List_Options *options);

static void save_view_params(View *view, const View_Settings &s, FILE *fp);
static void mode_save(Session &, FILE *fp);
static void save_view(View *view, FILE *fp);
static void save_contour_parameters(FILE *fp, const Contour_Parameters &pos,
				    const Contour_Parameters &neg);
static void read_contour_parameters(char *which, char *line,
				    Contour_Parameters *pos,
				    Contour_Parameters *neg);

static Spectrum *restore_savefile(Session &s, const Stringy &path,
				  List &views_to_show, Stringy *error_msg);
static bool skip_past_line(FILE *fp, const char *line);
static Spectrum *spectrum_restore_one(Session &, FILE *fp, SaveFile &sf,
				      int whichSpectrum, List &views_to_show,
				      bool *cancelled, Stringy *error_msg);
static void remove_spectrum_view_params(Spectrum *sp, List &views_to_show);
static bool restore_spectrum(Session &, Spectrum *, FILE *,
			     List &views_to_show);
static void LoadIntegrate(Spectrum *sp, Integration_Parameters &ip,
			  char *cp, char **rest);
static void SaveIntegrate(Spectrum *sp, Integration_Parameters &ip, FILE *fp);
static void LoadNoise(Spectrum *sp, char *cp, char **rest);
static void SaveNoise(Spectrum *sp, FILE *fp);
static void LoadOrnamentSizes(Spectrum *sp, double *ornament_sizes,
			      float *select_size, float *pointer_size,
			      float *line_end_size, char *which, char *value);
static void SaveOrnamentSizes(Spectrum *sp, double *ornament_sizes,
			      float select_size, float pointer_size,
			      float line_end_size, FILE *fp);
static void LoadPeakPick(Peak_Pick_Parameters &pp, char *cp, char **rest);
static void SavePeakPick(Peak_Pick_Parameters &pp, FILE *fp);
static void save_guessing_options(Spectrum *sp, FILE *fp);

static Stringy from_standard_path_separator(const Stringy &path);
static Stringy to_standard_path_separator(const Stringy &path);
static Stringy relative_path(const Stringy &path, const Stringy &directory);
static Stringy absolute_path(const Stringy &path, const Stringy &directory);
static Stringy spectrum_name_from_data_path(const Stringy &path);
static Spectrum *open_new_spectrum(Session &s, const Stringy &data_path,
				   Stringy *error_msg);
static NMR_Data *open_nmr_data(Session &, const Stringy &path);
static bool is_nmr_data(const Stringy &path);
static NMR_Data *read_nmr_data(const Stringy &path, Memory_Cache *);
static void read_view_geometry(FILE *fp, Spectrum *sp,
			       IPoint *axis_order, int *x, int *y,
			       int *xsize, int *ysize,
			       SPoint *center, SPoint *pixel_size,
			       SPoint *depth);
static View_Parameters *read_view_parameters(Spectrum *sp, FILE *fp);

static bool merge_db_data(const Stringy &db_path,
			  int mol_id, int cond_id, Condition *c);
static bool merge_molecule_db(const Stringy db_path, int mol_id, Molecule *m);
static bool merge_groups_atoms_sequence(const Stringy &db_file, Molecule *m);
static bool merge_condition_db(const Stringy &db_path, int mol_id, int cond_id,
			       Condition *c);

static bool merge_resonances(const Stringy &db_file, Condition *c);
static bool db_stuff_from_save_file(SaveFile &sf, Stringy *db_path,
				    int *molecule_id, Stringy *molecule_name,
				    int *condition_id, Stringy *condition_name,
				    int *spectrum_id, Stringy *spectrum_name,
				    Stringy *data_path, int *dim);
static void user_save(Session &, FILE *fp);
static void user_restore(Session &s, FILE *fp);
static bool user_saveauto_set(Session &, char *str);
static bool user_saveprompt_set(Session &, char *str);
static bool user_resize_views_set(Session &, char *str);
static bool user_keytimeout_set(Session &, char *str);
static bool user_cachesize_set(Session &, char *str);
static bool user_contour_graying_set(Session &, char *str);
static void set_restore(FILE *fp);
static void save_regions(const List &rlist, FILE *fp);
static bool read_region(Session &, char *cp);
static void print_save(const Print_Options &, FILE *fp);

static bool restore_view_axis_syncs(Session &, FILE *fp);
static void save_view_axis_syncs(Project &proj, FILE *fp);
static void save_view_overlays(Project &proj, FILE *fp);
static bool restore_view_overlays(Project &proj, FILE *fp);
static bool restore_project(Session &s, const Stringy &path,
			    Stringy *error_msg);
static void create_views(Session &, List &view_params);
static bool read_project_entries(SaveFile &sf, List *savefiles);
static bool restore_project_options(Project &proj, FILE *fp);
static bool restore_broadening_options(Project &proj, FILE *fp);
static bool is_project_file(const Stringy &path);
static bool save_project(Session &s, FILE *fp);
static void save_project_options(Project &proj, FILE *fp);
static void save_molecules(Project &, FILE *fp);
static bool some_spectrum_uses_molecule(Molecule *m, const List &slist);
static bool some_spectrum_uses_condition(Condition *c, const List &slist);
static bool restore_molecule(Project &proj, FILE *fp);
static bool get_line(FILE *fp, Stringy *line);
static Stringy line_tag(const Stringy &line, Stringy *rest);
static bool restore_sequence(Molecule *m, FILE *fp);
static void save_conditions(Project &, Molecule *m, FILE *fp);
static bool restore_condition(Molecule *m, FILE *fp);
static bool restore_resonances(Condition *c, FILE *fp);
static Stringy guess_nucleus(const Stringy &atom_name);
static bool restore_attached_data(AttachedData &ad, FILE *fp);
static void save_attached_data(AttachedData &ad, FILE *fp);
static bool save_spectra(Session &s, const List &spectra, bool backup);
static bool is_save_file(const Stringy &path);

static bool parse_group_atom(const Stringy &line,
		      Stringy *group, Stringy *atom, Stringy *rest);

static bool parse_line(Session &, const char *cmd);
static int old_file_color_index(const Color &color);	// Writing files
static Stringy old_file_color(int index);		// Reading files

static bool mode_set_mode(Session &, char *cp);
static Ornament_Type ornament_name2type(const char *name);
static bool ornament_restore_links(FILE *fp, Spectrum *sp);

// ----------------------------------------------------------------------------
// Used for old Sun Sparky compatibility
//
static int nextid;		// state variable

/*
 * strncpy_terminate:
 *
 * Like strncpy(), but terminates the buffer properly.
 */
static char *strncpy_terminate(char *buf, const char *from, int bufsiz)
{
	(void) strncpy(buf, from, bufsiz - 1);
	buf[bufsiz - 1] = '\0';
	return buf;
}


/*
 * Like strchr(), but the character searched for can be quoted with a '\'.
 */
static char *strchrquote(char *line, char ch)
{
	while (*line != '\0') {
		if (*line == ch)
			return line;
		if (*line == '\\') {
			if (line[1] == ch
			||  (line[1] == '\\' && line[2] == ch))
				(void) strcpy(line, line + 1);
		}
		line++;
	}
	return NULL;
}

/* 
 * strtokquote:
 *
 * Like strtok, but allows quoting with \
 * Parses lines like:
 *	cd xyzzy<sep> link make\<sep>  foobar
 * into:
 *	cd xyzzy\0
 *	link make<sep> foobar\0
 */
static char *strtokquote(char *line, char sep)
{
  static	char	*pos = NULL;		// state variable
	char		*retval;

	if (line != NULL)
		pos = line;
	if (pos == NULL || *pos == '\0')
		return NULL;
	retval = pos;
	pos = strchrquote(pos, sep);
	if (pos)
		*pos++ = '\0';
	return retval;
}


/*
 * isemptyline:
 *
 * Return true is this line is an empty line (only whitespace characters
 * before end-of-line or COMMENT)
 */
static bool isemptyline(char *line)
{
	while (*line) {
		if (*line == COMMENT)
			break;
		if (! isspace(*line))
			return false;
		line++;
	}
	return true;
}

/*
 * Get the double quoted string out of the character array, returning the
 * string and setting <*cpp> to the end of the string.
 */
static char *sgetstring(char **cpp)
{
	char	*start, *from, *to;

	while (**cpp && isspace(**cpp) && **cpp != '"')
		(*cpp)++;

	if (**cpp == '\0')
		return NULL;

	/*
	 * If we don't start with a double quote, this string stops at
	 * the first whitespace character.
	 */
	if (**cpp != '"') {
		start = from = *cpp;
		while (*from && !isspace(*from))
			from++;
		(*cpp) = from[0] ? from + 1 : from;
		*from = '\0';
		return start;
	}


	/*
	 * We start with double quote so we end with double quote.
	 */
	(*cpp)++;
	start = from = to = *cpp;

	while (*from) {
		if (from[0] == '"') {
			if (from[1] == '"')
				from++;		// "" is a literal "
			else 
				break;
		}
		*to++ = *from++;
	}
	*to = '\0';
	(*cpp) = from[1] ? from + 1 : from;
	return start;
}

/*
 * Put string <cp> to file <fp>.
 *
 * Any double quotes in the string are doubled to "".
 */
static void fputstring(FILE *fp, const char *cp)
{
	putc('"', fp);
	while (*cp) {
		if (*cp == '"')
			putc('"', fp);
		putc(*cp, fp);
		cp++;
	}
	putc('"', fp);
}

/*
 * Return true if:
 *	*<cp> is 'y', 'Y' or 't' (for "yes" "Yes" or "true")
 *	<cp> is a number greator than 0
 *	<cp> is the word "on" or "ON"
 *
 * Otherwise return false
 */
static bool YorN(char *cp)
{
	if (cp != NULL) {
		cp = skip_white(cp);
		if (*cp == 'y' || *cp == 'Y' || *cp == 't'
		||  (isdigit(*cp) && atoi(cp) > 0)
		||  ((cp[0]=='O' || cp[0]=='o') && (cp[1]=='N' || cp[1]=='n')))
			return true;
	}
	return false;
}

/*
 * Map the axis name <str> to a string.
 */
static int label2axis(const char *str)
{
  int axis;

  return (sscanf(str, "w%d", &axis) == 1 ? axis - 1 : 0);
}

// ----------------------------------------------------------------------------
//
static char *next_token(char **rest)
{
  if (*rest == NULL)
    return NULL;

  char *start = skip_white(*rest);
  if (*start == '\0')
    return NULL;

  char *end = skip_nonwhite(start);
  if (*end != '\0')
    {
      *end = '\0';
      *rest = end + 1;
    }
  else
    *rest = NULL;

  return start;
}

// ----------------------------------------------------------------------------
//
static char *next_line(char **rest)
{
  if (*rest == NULL)
    return NULL;

  char *start = skip_white(*rest);
  if (*start == '\0')
    return NULL;

  char *end = start;
  while (*end != '\0' && *end != '\n') ++end;

  if (*end != '\0')
    {
      *end = '\0';
      *rest = end + 1;
    }
  else
    *rest = NULL;

  return start;
}

// ----------------------------------------------------------------------------
//
static int next_int(char **rest)
{
  char *tok = next_token(rest);

  return (tok ? atoi(tok) : 0);
}

// ----------------------------------------------------------------------------
//
static double next_double(char **rest)
{
  char *tok = next_token(rest);

  return (tok ? atof(tok) : 0);
}

// ----------------------------------------------------------------------------
//
static SPoint read_point(char *text, char **next)
{
	int	dim;
	double	c[DIM];
	char	*cp;
	int	haveBracket;

	//
	// Set up reasonable defaults if the text is NULL or can't be read.
	//
	dim = 0;
	if (text != NULL) {

		//
		// Skip introductory '[' if any
		//
		cp = strchr(text, '[');
		if (cp) {
			cp++;
			haveBracket = true;
		}
		else {
			cp = text;
			haveBracket = false;
		}

		for (dim = 0;; dim++) {
			cp = skip_white(cp);

			//
			// Until closing ']' or end-of-string or dim >= DIM.
			//
			if (haveBracket && cp[0] == ']') {
				cp++;
				break;
			}
			else if (cp[0] == '\0' || dim >= DIM)
				break;

			//
			// Use strtod() to scan the number and advance the
			// <cp> pointer. If <cp> doesn't advance, we could
			// not read another number, and we are done.
			//
			char	*num = cp;
			c[dim] = strtod(num, &cp);
			if (num == cp)
				break;

			//
			// This handles parsing of lines like the following
			//   linewidth 1.23 3.45 1/2
			// where there is a final token which is not a number
			// but has an initial part that looks like a number.
			//
			if (!isspace(*cp) && *cp != '\0' && *cp != ']')
			  {
			    cp = num;
			    break;
			  }
		}

		if (next != NULL)
			*next = cp;
	}

	return SPoint(dim, c);
}

// ----------------------------------------------------------------------------
//
static SPoint read_point(istream &strm)
{
 	int dim;
	double c[DIM];
	
	char	*bp, ch;
	char	buf[MAX_LINE_LENGTH];

	//
	// Skip to the introductory character.
	//
	while (strm.get(ch) && isspace(ch))
		/* continue */;

	/*
	 * If starts with '[', ends with ']'
	 */
	if (ch == '[') {
		for (dim = 0; dim < DIM; ) {

			/*
			 * Skip leading whitespace
			 */
			while (strm.get(ch) && isspace(ch))
				/* continue */;

			/*
			 * Quit at end of stream or if ']' is seen.
			 */
			if (!strm || ch == ']')
				break;

			bp = buf;
			*bp++ = ch;
			while (strm.get(ch) && !isspace(ch) && ch != ']')
				*bp++ = ch;
			*bp++ = '\0';

			/*
			 * Handle optional units
			 */
			if (strcmp(buf, "ppm") == 0) ;
			else if (strcmp(buf, "hz") == 0) ;
			else if (strcmp(buf, "data") == 0) ;
			else
				c[dim++] = atof(buf);

			/*
			 * Quit at end of stream or if ']' is seen.
			 */
			if (!strm || ch == ']')
				break;
		}
	}
	else {
		/*
		 * No brackets, it is 2-D (old style)
		 */
		strm.putback(ch);
		for (dim = 0; dim < 2; dim++) {
			strm >> buf;
			if (strm)
				c[dim] = atof(buf);
		}
	}

	return SPoint(dim, c);
}

// ----------------------------------------------------------------------------
// Save the ornament's state to stream <o>. If <willDelete> is true, the
// ornament is going to be deleted and additional information must be saved.
//
void ornament_save_for_undo(Ornament *op, ostream &o, Object_Table &ids)
{
	ornament_save_identifier(op, ids.id(op, "ornament"), o);
	o	<< "co " << op->GetColor().name() << endl;	// Color
	o	<< "fl " << op->IsSelected()		// Flags
		<< ' '   << op->IsLocked()
		<< ' '   << (op->type() == label &&
			     ((Label *)op)->attached() != NULL) << endl;
	o	<< "--\n";			// mark end of common section

	switch (op->type())
	  {
	  case grid: save_grid_for_undo(o, (Grid *)op); break;
	  case line: save_line_for_undo(o, (Line *)op); break;
	  case label: save_label_for_undo(o, (Label *)op, ids); break;
	  case peak: save_peak_for_undo(o, (Peak *)op, ids); break;
	  case peakgroup: save_peakgroup_for_undo(o, (PeakGp *)op, ids); break;
	  default:
	    break;
	  }

	o	<< "<>\n";			// mark end of ornament
}

// ----------------------------------------------------------------------------
// Write the object identifier to stream <o>
//
static void ornament_save_identifier(Ornament *op, int id, ostream &o)
{
	o	<< "< "	<< op->type_name()
		<< ' '	<< id
		<< ' '	<< op->spectrum()->fullname()
		<< " >\n";
}

// ----------------------------------------------------------------------------
//
static Ornament *ornament_restore_for_undo(Session &s, istream &i,
					   Spectrum *sp,
					   Ornament_Type type, Ornament *op,
					   Object_Table &ids)
{
  Project &proj = s.project();
  char		keyword[MAX_LINE_LENGTH];
  bool		selected = false;
  bool		locked = false;
  bool		pointer = false;
  Stringy	colorname;
  Ornament *o = NULL;

  for ( ;; ) {
    i >> keyword;
    if (strcmp(keyword, "<>") == 0)
      break;

    if (strcmp(keyword, "co") == 0) {
      i >> colorname;
    } else if (strcmp(keyword, "fl") == 0) {
      i >> selected;
      i >> locked;
      i >> pointer;
    }
    else {
      switch (type)
	{
	case grid: o = restore_grid_for_undo(i, sp, (Grid *)op); break;
	case line: o = restore_line_for_undo(i, sp, (Line *)op); break;
	case label:
	  o = restore_label_for_undo(proj, i, sp, (Label *)op, ids); break;
	case peak:
	  o = restore_peak_for_undo(s, i, sp, (Peak *)op, ids); break;
	case peakgroup:
	  o = restore_peakgroup_for_undo(s, i, sp, (PeakGp *)op, ids); break;
	default:
	  break;
	}
      break;
    }
  }

  if (o) {
    o->select(selected);
    o->lock(locked);
    o->SetColor(colorname.cstring());
  }

  return o;
}

/*
 * Create and return the Ornament of type <t>, read from the <strm>
 */
static Ornament *MakeOrnament(Session &s, Spectrum *sp, Ornament_Type t,
			      int id, istream &strm, Object_Table &ids)
{
  Ornament *op = ornament_restore_for_undo(s, strm, sp, t, NULL, ids);
  if (op)
    ids.set_id(op, "ornament", id);

  return op;
}

/*
 * Restore an ornament of type <t>, DB ID <id> from the stream <strm>
 */
Ornament *ornament_restore_for_undo(Session &s, const Stringy &first_line,
				    istream &strm, Object_Table &ids)
{
  Ornament_Type		t;
  int		id;
  Spectrum	*sp;
  if (ParseIdentifier(s.project(), &t, &id, &sp, first_line.cstring())
      && valid_otype(t)) {
    Ornament *op = FindOrnament(sp, t, id, ids);
    if (op)	// Modify existing ornament
      return ornament_restore_for_undo(s, strm, sp, t, op, ids);
    else	// Create ornament
      return MakeOrnament(s, sp, t, id, strm, ids);
  }
  return NULL;
}

/*
 * Restore an embedded ornament (the kind that is defined inside of another
 * ornament's save structure), returning the ornament if restored properly.
 */
static Ornament *restore_embedded_ornament_for_undo(Session &s,
						    Spectrum *spect,
						    istream &i,
						    Object_Table &ids)
{
	char		buf[MAX_LINE_LENGTH];
	Spectrum	*sp;
	Ornament_Type		otype;
	int		oid;
	Project &proj = s.project();

	/*
	 * Get start of ornament
	 */
	buf[0] = '\0';
	while (i.eof() == false && buf[0] != '<' && buf[0] != ']')
		i.getline(buf, sizeof buf);


	/*
	 * An embedded ornament is restored, creating it.
	 */
	if (buf[0]=='<' && ParseIdentifier(proj, &otype, &oid, &sp, buf)) {
		Ornament *op = MakeOrnament(s, spect, otype, oid, i, ids);

		/*
		 * Skip until ']'
		 */
		do {
			i.getline(buf, sizeof buf);
		} while (i.eof() == false && buf[0] != ']');
		return op;
	}

	return NULL;
}

// ----------------------------------------------------------------------------
// Save the ornament non-specific information, then call the virtual _saveV1
// to save the specific information. Follow this by more non-specific
// information. The reason this is done is to preserve the general format
// of the version 1 save files.
//
static void ornament_save_v1(Ornament *op, FILE *fp)
{
	bool has_pointer = (op->type() == ::label &&
			    ((Label *)op)->attached());
	fprintf(fp, "type %s\n", op->type_name());
	fprintf(fp, "color %s\n",
		colorformat(op->GetColor(), op->type()).cstring());
	fprintf(fp, "flags %d %d 0 %d\n",
		op->IsSelected(), op->IsLocked(), has_pointer);
		

	if (!(op->GetNote() == ""))
		ornament_save_note_v1(op, fp);

	switch (op->type())
	  {
	  case grid: save_grid_v1(fp, (Grid *)op); break;
	  case line: save_line_v1(fp, (Line *)op); break;
	  case label: save_label_v1(fp, (Label *)op); break;
	  case peak: save_peak_v1(fp, (Peak *)op); break;
	  case peakgroup: save_peakgroup_v1(fp, (PeakGp *)op); break;
	  default:
	    break;
	  }
}

//
// Save a version 1 save file note to the file.
//
static void ornament_save_note_v1(Ornament *op, FILE *fp)
{
  Stringy note = op->GetNote();
  const char *cp = note.cstring();

	fprintf(fp, "note \"");
	while (*cp) {
		if (*cp == '"')
			fputc('"', fp);
		fputc(*cp, fp);
		cp++;
	}
	fputc('"', fp);
	fputc('\n', fp);
}

//
// Save the grid to a save file in old (version 1) format. The features
// important to grids are its orientation and location.
//
static void save_grid_v1(FILE *fp, Grid *g)
{
  fprintf(fp, "pos %d %s\n", g->axis,
	  point_format(g->location, "%.3f", false));
}

//
// Restore a version 1 save file format grid. This entails reading
// back the position and orientation.
//
static Grid *restore_grid_v1(FILE *fp, Spectrum *sp)
{
	char	buf[MAX_LINE_LENGTH], *cp, *rest;
	int	status;
	Ornament_Type	next_type;

	Ornament_Properties oprops;
	oprops.type = grid;

	int axis = 0;
	SPoint location(sp->dimension());

	while (ornament_get_line_v1(oprops, buf, sizeof buf, fp,
				    &cp, &rest, &status, &next_type)) {
		if (strcmp(cp, "pos") == 0) {
		  axis = atoi(next_token(&rest));
		  location = read_point(next_line(&rest), NULL);
		}
	}

	Grid *grid = NULL;
	if (status)
	  {
	    grid = new Grid(sp, location, axis);
	    set_ornament_properties(grid, oprops);
	  }

	return grid;
}

static void save_grid_for_undo(ostream &o, Grid *grid)
{
  o << "pos " << point_format(grid->location, "%.3f", true) << endl
    << "axis " << grid->axis << endl;
}

static Grid *restore_grid_for_undo(istream &i, Spectrum *sp, Grid *grid)
{
	char	buf[MAX_LINE_LENGTH];

	SPoint location(sp->dimension());
	int axis = 0;

	for ( ;; ) {
		i >> buf;
		if (strcmp(buf, "pos") == 0) {
			location = read_point(i);
		} else if (strcmp(buf, "axis") == 0)	// frequency
			i >> axis;
		else
			break;
	}

	if (grid == NULL)
	  grid = new Grid(sp, location, axis);

	grid->location = location;
	grid->axis = axis;
	sp->notifier().send_notice(nt_changed_ornament, (Ornament *)grid);

	return grid;
}

//
// Save the line to a save file in old (version 1) format.
//
static void save_line_v1(FILE *fp, Line *line)
{
  fprintf(fp, "bl %s\n", point_format(line->bl, "%.3f", false));
  fprintf(fp, "tr %s\n", point_format(line->tr, "%.3f", false));
  fprintf(fp, "ends %d\n", line->ends);
}

//
// Restore a version 1 save file format line.
//
static Line *restore_line_v1(FILE *fp, Spectrum *sp)
{
	char	buf[MAX_LINE_LENGTH], *cp, *rest;
	int	status;
	Ornament_Type	next_type;

	Ornament_Properties oprops;
	oprops.type = line;

	SPoint p1(sp->dimension()), p2(sp->dimension());

	//
	// Gather the line "bl" and "tr". Once we have the "tr",
	// compute the axis be seeing which of "bl" and "tr" are
	// equal.
	//
	while (ornament_get_line_v1(oprops, buf, sizeof buf, fp,
				    &cp, &rest, &status, &next_type)) {
		if (strcmp(cp, "bl") == 0) {
		  p1 = read_point(next_line(&rest), NULL);
		} else if (strcmp(cp, "tr") == 0) {
		  p2 = read_point(next_line(&rest), NULL);
		}

		//
		// The ends is just a number, which determines which
		// ends of the line have an arrow on the end.
		//
		else if (strcmp(cp, "ends") == 0)
		  atoi(next_token(&rest));		// not used

	}

	Line *line = NULL;
	if (status)
	  {
	    line = new Line(sp, p1, p2);
	    set_ornament_properties(line, oprops);
	  }

	return line;
}

static void save_line_for_undo(ostream &o, Line *line)
{
	o	<< "bl " << point_format(line->bl, "%.3f", true) << endl
		<< "tr " << point_format(line->tr, "%.3f", true) << endl
		<< "en " << line->ends << endl;
}

static Line *restore_line_for_undo(istream &i, Spectrum *sp, Line *line)
{
	char	buf[MAX_LINE_LENGTH];

	SPoint bl(sp->dimension()), tr(sp->dimension());
	int ends = 0;

	for ( ;; ) {
		i >> buf;
		if (strcmp(buf, "bl") == 0) {
			bl = read_point(i);
		} else if (strcmp(buf, "tr") == 0) {
			tr = read_point(i);
		} else if (strcmp(buf, "en") == 0)	// frequency
			i >> ends;
		else
			break;
	}

	if (line == NULL)
	  line = new Line(sp, bl, tr);

	line->bl = bl;
	line->tr = tr;
	line->ends = ends;
	sp->notifier().send_notice(nt_changed_ornament, (Ornament *)line);

	return line;
}

//
// Save member information to a version 1 save file
//
static void save_label_v1(FILE *fp, Label *lbl)
{
  fprintf(fp, "mode %d\n", (lbl->is_assignment() ? 2 : 0));
  fprintf(fp, "pos %s\n", point_format(lbl->location(), "%.3f", false));
  fprintf(fp, "label %s\n", lbl->GetString().cstring());
  fprintf(fp, "xy");
  int dim = lbl->spectrum()->dimension();
  for (int i = 0; i < dim; i++)
    for (int j = 0; j < dim; j++) {
      if (i != j)
	fprintf(fp, " %.3f,%.3f", lbl->xPosition(i, j),	lbl->yPosition(i, j));
    }
  (void) fprintf(fp, "\n");
}

//
// Restore label information from a version 1 save file.
//
static Label *restore_label_v1(FILE *fp, Spectrum *sp)
{
	char	buf[MAX_LINE_LENGTH], *cp, *rest;
	int	status;
	int	readOldCenter = true;
	Ornament_Type	next_type;

	Ornament_Properties oprops;
	oprops.type = label;

	Stringy text;
	SPoint location(sp->dimension());
	LABELPOS pos;
	bool assignment = false;

	while (ornament_get_line_v1(oprops, buf, sizeof buf, fp,
				    &cp, &rest, &status, &next_type)) {

		//
		// Get label text.
		//
		if (strcmp(cp, "label") == 0) {
		  text = next_line(&rest);
		}


		//
		// Gather the location of the label. This was done in
		// different fashions in the past. One used the center ...
		//
		else if (readOldCenter && strcmp(cp, "pos") == 0) {
			SPoint	r = read_point(next_line(&rest), NULL);
			location = r;
			pos = label_positions(r);
		}

		//
		// ... the new method uses xy pairs for each orientation
		//
		else if (strcmp(cp, "xy") == 0) {
		  pos = label_positions(next_line(&rest), sp);
		  readOldCenter = false;
		}

		//
		// display strings and other information
		//
		else if (strcmp(cp, "dp") == 0) {
			next_line(&rest);		// Obsolete mDisplayPos
		}
		else if (strcmp(cp, "dz") == 0)
			next_line(&rest);		// Obsolete mSize
		else if (strcmp(cp, "mode") == 0)
		  assignment = (atoi(next_line(&rest)) == 2);
	}

	Label *lbl = NULL;
	if (status)
	{
	  lbl = new Label(sp, text, location);
	  lbl->assignment(assignment);
	  lbl->set_positions(location, pos);
	  set_ornament_properties(lbl, oprops);
	}

	return lbl;
}

// ----------------------------------------------------------------------------
//
static LABELPOS label_positions(char *line, Spectrum *sp)
{
	LABELPOS pos;
	float	x, y;

	for (int i = 0; i < sp->dimension(); i++) {
	    for (int j = 0; j < sp->dimension(); j++) {
		if (i != j) {
			line = skip_white(line);
			if (sscanf(line, "%f,%f", &x, &y) == 2) {
				pos.x[i][j] = x;
				pos.y[i][j] = y;
				line = skip_nonwhite(line);
			}
		}
	    }
	}

	return pos;
}

/*
 * Save the Label state to stream <o>, for use when the label is undeleted (if
 * <willDelete> is true) or unmoved or when it is pasted from the clipboard.
 */
static void save_label_for_undo(ostream &o, Label *label, Object_Table &ids)
{
  o << "ce " << point_format(label->location(), "%.3f", true) << endl;

  o << "st " << label->GetString() << endl;
  o << "ty " << (label->is_assignment() ? 2 : 0) << endl;

  o << "xy";

  int dim = label->spectrum()->dimension();
  for (int i = 0; i < dim; i++) {
    for (int j = 0; j < dim; j++)
      if (i != j) {
	o << ' ' << label->xPosition(i, j) << ',' << label->yPosition(i, j);
      }
  }
  o << endl;

  CrossPeak *xp = label->attached();
  if (xp)
    SaveXRef(xp, ids, o);
}

/*
 * Restore the Label state from stream <i>, for use when the label is undeleted
 * or unmoved or when it is pasted from the clipboard.
 */
static Label *restore_label_for_undo(Project &proj, istream &i, Spectrum *sp,
				     Label *label, Object_Table &ids)
{
  char		buf[MAX_LINE_LENGTH];

  Stringy text;
  SPoint location(sp->dimension());
  LABELPOS pos;
  bool assignment = true;
  CrossPeak *xp = NULL;

  for ( ;; ) {
    i >> buf;


//
// For backward compatibility to v2.0x (x < 1), the label must have a simple
// center (location) as well as the orientation-dependent center (mDisplayPos)
//
    if (strcmp(buf, "ce") == 0) {		// frequency
      location = read_point(i);
    }
    else

      if (strcmp(buf, "st") == 0) {	// name
	i.getline(buf, sizeof buf);	//   get name
	text = trim_white(buf);
      }
      else if (strcmp(buf, "ty") == 0) {	// type
	int	type;
	i >> type;
	assignment = (type == 2);
      }
      else if (strcmp(buf, "xy") == 0) {	// type
	i.getline(buf, sizeof buf);
	pos = label_positions(buf, sp);
      }
      else if (strcmp(buf, "ds") == 0) {	// display string
	i.getline(buf, sizeof buf);	// Obsolete mDisplayStr
      }
      else if (strcmp(buf, "dp") == 0) {	// display pos
	double x, y;
	i >> x;				// Obsolete mDisplayPos
	i >> y;
      }
      else if (strcmp(buf, "dz") == 0) {	// display offset
	double x, y;
	i >> x;				// Obsolete mSize
	i >> y;
      }
      else if (strcmp(buf, "xr") == 0) {	// cross reference
	i.getline(buf, sizeof buf);	// read rest of line
	xp = (CrossPeak *) FindXRef(proj, buf, ids);
      }
      else if (strcmp(buf, "<>") == 0)
	break;
      else
	i.getline(buf, sizeof buf);	// skip rest of line
  }

  if (label == NULL)
    label = new Label(sp, text, location);

  label->SetString(text);
  label->assignment(assignment);
  label->set_positions(location, pos);
  if (xp && xp->label() == NULL)
    label->attach(xp);

  return label;
}

//
// Save the Peak to a save file in old (version 1) format.
//
static void save_peak_v1(FILE *fp, Peak *pk)
{
  Spectrum *sp = pk->spectrum();

  if (!pk->user_name().is_empty())
    fprintf(fp, "label %s\n", pk->user_name().cstring());

  fprintf(fp, "id %d\n", nextid++);  // For Sun compatibility.

  fprintf(fp, "pos %s\n", point_format(pk->position(), "%.3f", false));
  if (pk->IsAliased())
    fprintf(fp, "fq %s\n", point_format(pk->frequency(), "%.3f", false));

  fprintf(fp, "height %.3f %.3f\n", pk->FitHeight(), pk->DataHeight());

  //
  // Only if the LinewidthMethod is not "none", save the linewidth.
  //
  if (!pk->GetLinewidthMethod().is_empty()) {
    SPoint lw_hz = sp->scale(pk->linewidth(), PPM, HZ);
    fprintf(fp, "linewidth %s %s\n",
	    point_format(lw_hz, "%.3f", false),
	    pk->GetLinewidthMethod().cstring());
  }

  //
  // If integrated, save the volume, method, and limits.
  //
  double volume;
  if (pk->volume(&volume)) {
    fprintf(fp, "integral %.4e %s\n", volume,
	    index_name(Integration_Method_Short_Names,
		       pk->GetIntegrateMethod()).cstring());

    if (pk->IntegrateByFit() && pk->fit_residual() > 0)
      fprintf(fp, "fr %.3f\n", pk->fit_residual());

    double verror;
    Stringy method;
    if (pk->volume_error(&verror, &method))
      fprintf(fp, "ve %.3f %s\n", verror, method.cstring());
  }

  //
  // Save resonance information
  //
  SaveAssignment(pk, fp);

  //
  // If I have a label save the embedded label
  //
  if (pk->label()) {
    fputs("[\n", fp);
    ornament_save_v1(pk->label(), fp);
    fputs("]\n", fp);
  }
}

//
// Restore a version 1 save file format Peak.
//
static Peak *restore_peak_v1(FILE *fp, Spectrum *sp)
{
  char	buf[MAX_LINE_LENGTH], *cp, *cp1, *rest;
  int	status;
  Ornament_Type	next_type;

  Ornament_Properties oprops;
  oprops.type = peak;

  int dim = sp->dimension();
  SPoint freq(dim), alias(dim), lw(dim);
  double fheight = 0, dheight = 0, volume = 0, verror = 0, fit_residual = 0;
  Stringy verrormethod;
  bool havedh = false;
  Integration_Method itype = INTEGRATE_NONE;
  Stringy lwtype;
  Stringy assign;
  Label *label = NULL;

  while (ornament_get_line_v1(oprops, buf, sizeof buf, fp,
			      &cp, &rest, &status, &next_type)) {
    if (strcmp(cp, "pos") == 0) {
      freq = read_point(next_line(&rest), NULL);
    } else if (strcmp(cp, "id") == 0)
				  ;  // Obsolete id number.

    //
    // label becomes user label
    //
    else if (strcmp(cp, "label") == 0) {
      cp = next_line(&rest);	// Not used.
    }

    //
    // integral ###
    //
    else if (strcmp(cp, "integral") == 0)
      restore_peak_integration(next_line(&rest), &volume, &itype, &lwtype);

    //
    // fit error
    //
    else if (strcmp(cp, "fr") == 0)
      fit_residual = atof(next_token(&rest));

    //
    // volume error
    //
    else if (strcmp(cp, "ve") == 0)
      restore_volume_error(next_line(&rest), &verror, &verrormethod);

    //
    // height ###
    //
    else if (strcmp(cp, "height") == 0) {
      fheight = atof(next_token(&rest));
      cp = next_token(&rest);
      if (cp)
	{
	  dheight = atof(cp);
	  havedh = true;
	}
    }

    //
    // linewidth ### ...
    //
    else if (strcmp(cp, "linewidth") == 0) {
      cp1 = next_line(&rest);
      if (cp1) {
	SPoint lw_hz = read_point(cp1, &cp1);

	//
	// Fix a common save file error.
	//
	if (lw_hz.dimension() == dim + 1)
	  lw_hz = decrement_point_dimension(lw_hz);

	lw = sp->scale(lw_hz, HZ, PPM);
	lwtype = skip_white(cp1);
	if (lwtype == "none")
	  lwtype = "";
      }
    }

    //
    // goodness ###
    //
    else if (strcmp(cp, "goodness") == 0) ;		// obsolete
    else if (strcmp(cp, "gd") == 0) ;			// obsolete

    else if (strcmp(cp, "ri") == 0)
				  ;	// Obsolete assignment resonance indexes

    else if (strcmp(cp, "rs") == 0) {
      assign = next_line(&rest);
    }

    else if (strcmp(cp, "pl") == 0) 
				  ; // Per peak extra planes info no longer used.

    else if (strcmp(cp, "fq") == 0) {
      SPoint oldfq = freq;
      SPoint newfq = read_point(next_line(&rest), NULL);

      freq = newfq;
      alias = newfq - oldfq;
    }


    //
    // Watch for start of embedded ornament
    //
    else if (strcmp(cp, "[") == 0) {
      Ornament_Type next_type;
      Ornament_Properties oprops2;
      ornament_get_line_v1(oprops2, buf, sizeof buf, fp,
			   &cp, &rest, &status, &next_type);

      // If status == RESTORE_ORNAMENT_CONTINUE the type
      // of embedded ornament to restored is next_type.
      //
      if (status == RESTORE_ORNAMENT_CONTINUE) {
	if (next_type == ::label) {
	  label = (Label *) load_ornament(::label, sp, fp);
	}
      }
    }
  }

  Peak *pk = NULL;
  if (status)
    {
      SPoint pos = freq - alias;
      pk = new Peak(sp, pos);
      if (itype == INTEGRATE_BOX || itype == INTEGRATE_ELLIPSE)
	pk->box_volume(itype, volume);
      else if (itype == INTEGRATE_GAUSSIAN || itype == INTEGRATE_LORENTZIAN)
	pk->fit_volume(itype, volume, fit_residual);
      else if (itype == INTEGRATE_MANUAL)
	pk->manual_volume(volume);
      else
	pk->no_volume();
      if (verror > 0)
	pk->set_volume_error(verror, verrormethod);
      pk->FitHeight(fheight);
      if (havedh)
	pk->DataHeight(dheight);
      pk->linewidth(lwtype, lw);
      RestoreAssignment(pk, assign);
      pk->set_alias(alias);
      if (label)
	label->attach(pk);
      set_ornament_properties(pk, oprops);
    }

  return pk;
}

// ----------------------------------------------------------------------------
//
static SPoint decrement_point_dimension(const SPoint &p)
{
  SPoint dp(p.dimension()-1);

  for (int a = 0 ; a < p.dimension() - 1 ; ++a)
    dp[a] = p[a];

  return dp;
}

/*
 * Restore the assignment if it hasn't yet been restored. This will only
 * be called as long as the "rs" field exists.
 */
static void RestoreAssignment(CrossPeak *xp, const Stringy &str)
{
  Stringy rest = str;
  Stringy group, atom;
  for (int a = 0 ; a < xp->dimension() ; ++a)
    if (parse_group_atom(rest, &group, &atom, &rest))
      xp->ChangeAssignment(a, atom, group);
}

// ----------------------------------------------------------------------------
// 
static void restore_peak_integration(char *line,
				     double *volume, Integration_Method *itype,
				     Stringy *lwtype)
{
  char		*rest = line;

  *volume = atof(next_token(&rest));
  *itype = integration_type(next_token(&rest));

  if (*itype == INTEGRATE_GAUSSIAN || *itype == INTEGRATE_LORENTZIAN)
    *lwtype = "fit";
  else if (*itype == INTEGRATE_BOX || *itype == INTEGRATE_ELLIPSE)
    *lwtype = "";
}

// ----------------------------------------------------------------------------
// 
static void restore_volume_error(char *line, double *verror, Stringy *method)
{
  char *rest = line;

  *verror = atof(next_token(&rest));
  *method = next_line(&rest);
}

/*
 * Map the integration method name <name> to the integration method type.
 */
static Integration_Method integration_type(const char *name)
{
	if (name != NULL) {
		for (int i = 0; Integration_Method_Names[i]; i++) {
			if (strcmp(name, Integration_Method_Names[i]) == 0)
				return (Integration_Method) i;
		}
		for (int i = 0; Integration_Method_Short_Names[i]; i++) {
		  if (strcmp(name, Integration_Method_Short_Names[i]) == 0)
		    return (Integration_Method) i;
		}
	}
	return INTEGRATE_NONE;
}

/*
 * Save the Peak state to stream <o>, for use when the peak is undeleted (if
 * <willDelete> is true) or unmoved or when it is pasted from the clipboard.
 */
static void save_peak_for_undo(ostream &o, Peak *pk, Object_Table &ids)
{
  if (pk->peakgp() == NULL && !pk->user_name().is_empty())
    o << "nm " << pk->user_name() << endl;

  if (pk->IsAliased())
    o	<< "al " << point_format(pk->alias(), "%.3f", true) << endl;

  o	<< "fq " << point_format(pk->frequency(), "%.3f", true) << endl;

  double volume;
  if (pk->volume(&volume)) {
    o	<< "vo " << volume
	<< ' ' << index_name(Integration_Method_Short_Names,
			     pk->GetIntegrateMethod())
	<< endl;

    if (pk->IntegrateByFit())
      {
	o << "ht " << pk->FitHeight() << endl;
	if (pk->fit_residual() > 0)
	  o << "fr " << pk->fit_residual();
      }

    double verror;
    Stringy method;
    if (pk->volume_error(&verror, &method))
      o	<< "ve " << verror << ' ' << method << endl;
  }

  if (pk->HasLinewidth())
    o << "lw " <<  point_format(pk->linewidth(), "%.3f", true)
      << ' ' << pk->GetLinewidthMethod() << endl;

  SaveAssignment(pk, o);

  Ornament	*op = pk->label();
  if (op)
    SaveXRef(op, ids, o);

  op = pk->peakgp();
  if (op)
    SaveXRef(op, ids, o);
}

/*
 * Restore the Peak state from stream <i>, for use when the peak is undeleted
 * or unmoved or when it is pasted from the clipboard.
 */
static Peak *restore_peak_for_undo(Session &s, istream &i, Spectrum *sp,
				   Peak *pk, Object_Table &ids)
{
  char		buf[MAX_LINE_LENGTH];

  Project &proj = s.project();
  int dim = sp->dimension();
  SPoint freq(dim), alias(dim), lw(dim);
  double fheight = 0, dheight = 0, volume = 0, verror = 0, fit_residual = 0;
  Stringy verrormethod;
  bool havedh = false;
  Integration_Method itype = INTEGRATE_NONE;
  Stringy lwtype;
  Stringy assign;
  Label *label = NULL;
  PeakGp *pg = NULL;

  for ( ;; ) {
    i >> buf;
    if (strcmp(buf, "nm") == 0) {	// name
      i.getline(buf, sizeof buf);	// Not used.
    }
    else if (strcmp(buf, "al") == 0) {	// alias
      i.getline(buf, sizeof buf);
      alias = read_point(buf, NULL);
    }
    else if (strcmp(buf, "fq") == 0) {	// frequency
      freq = read_point(i);
    }
    else if (strcmp(buf, "ht") == 0)	// height
      i >> fheight;
    else if (strcmp(buf, "vo") == 0)	// volume
      read_volume(i, &volume, &itype);
    else if (strcmp(buf, "ve") == 0)	// volume error estimate
      read_volume_error(i, &verror, &verrormethod);
    else if (strcmp(buf, "fe") == 0)	// fit error
      i >> fit_residual;
    else if (strcmp(buf, "lw") == 0)	// linewidth
      read_linewidth(i, sp, &lw, &lwtype);
    else if (strcmp(buf, "gd") == 0)	// goodness -- obsolete
      i.getline(buf, sizeof buf);

    /*
     * We use "rs" and not "ri" because we can be copying
     * to a spectrum that has not yet seen these assignments
     * and we want to create them if they are currently absent.
     */
    else if (strcmp(buf, "rs") == 0) {
      i.getline(buf, sizeof buf);
      assign = buf;
    }

    else if (strcmp(buf, "pl") == 0) {	// No longer used.
      i.getline(buf, sizeof buf);	// Throw away per peak
					// extra planes info.
    }

    else if (strcmp(buf, "xr") == 0) {	// cross reference
      i.getline(buf, sizeof buf);	// read rest of line
      Ornament *op = FindXRef(proj, buf, ids);
      if (op) {
	if (op->type() == ::peakgroup)
	  pg = (PeakGp *) op;
	else if (op->type() == ::label)
	  label = (Label *) op;
      }
    }
    else if (strcmp(buf, "[") == 0) {	// embedded ornament
      Ornament *op = restore_embedded_ornament_for_undo(s, sp, i, ids);
      if (op) {
	if (op->type() == ::label)
	  label = (Label *) op;
      }
    }

    else if (strcmp(buf, "<>") == 0)
      break;
    else
      i.getline(buf, sizeof buf);	// skip rest of line
  }

  SPoint pos = freq - alias;
  if (pk == NULL)
    pk = new Peak(sp, pos);
  else
    pk->set_position(pos);

  if (itype == INTEGRATE_BOX || itype == INTEGRATE_ELLIPSE)
    pk->box_volume(itype, volume);
  else if (itype == INTEGRATE_GAUSSIAN || itype == INTEGRATE_LORENTZIAN)
    pk->fit_volume(itype, volume, fit_residual);
  else if (itype == INTEGRATE_MANUAL)
    pk->manual_volume(volume);
  else
    pk->no_volume();
  if (verror > 0)
    pk->set_volume_error(verror, verrormethod);
  pk->FitHeight(fheight);
  if (havedh)
    pk->DataHeight(dheight);
  pk->linewidth(lwtype, lw);
  RestoreAssignment(pk, assign);
  pk->set_alias(alias);
  if (label && pk->label() == NULL)
    label->attach(pk);
  if (pg && pk->peakgp() == NULL)
    pg->add(*pk);

  return pk;
}

static void read_volume(istream &strm, double *volume,
			Integration_Method *itype)
{
  char	nm[MAX_LINE_LENGTH];

  strm >> *volume;
  strm.get(nm, sizeof nm, '\n');
  *itype = integration_type(nm);
}

static void read_volume_error(istream &strm, double *verror, Stringy *method)
{
  strm >> *verror;
  stream_line(strm, method);
}

static void read_linewidth(istream &strm, Spectrum *sp,
			   SPoint *lw, Stringy *lwtype)
{
  char	nm[MAX_LINE_LENGTH];

  *lw = sp->scale(read_point(strm), HZ, PPM);
  strm.get(nm, sizeof nm, '\n');
  *lwtype = skip_white(nm);
  if (*lwtype == "none")
    *lwtype = "";
}

//
// Save the Peak to a save file in old (version 1) format.
//
static void save_peakgroup_v1(FILE *fp, PeakGp *pg)
{
  if (!pg->user_name().is_empty())
    fprintf(fp, "label %s\n", pg->user_name().cstring());

  fprintf(fp, "id %d\n", nextid++);		// For Sun compatibility.

  fprintf(fp, "pos %s\n", point_format(pg->position(), "%.3f", false));
  if (pg->IsAliased())
    fprintf(fp, "fq %s\n", point_format(pg->frequency(), "%.3f", false));

  double volume;
  if (pg->volume(&volume))
    {
      fprintf(fp,"integral %.4e\n", volume);

      double verror;
      Stringy method;
      if (pg->volume_error(&verror, &method))
	fprintf(fp, "ve %.3f %s\n", verror, method.cstring());
    }

  //
  // Save resonance information
  //
  SaveAssignment(pg, fp);

  //
  // Save any peaklets as embedded peaklets
  //
  const List &peaklets = pg->peaklets();
  for (int pi = 0 ; pi < peaklets.size() ; ++pi)
    {
      fputs("[\n", fp);
      ornament_save_v1((Peak *) peaklets[pi], fp);
      fputs("]\n", fp);
    }

  //
  // If I have a label save the embedded label
  //
  if (pg->label()) {
    fputs("[\n", fp);
    ornament_save_v1(pg->label(), fp);
    fputs("]\n", fp);
  }
}

//
// Restore a version 1 save file format PeakGp.
//
static PeakGp *restore_peakgroup_v1(FILE *fp, Spectrum *sp)
{
  char	buf[MAX_LINE_LENGTH], *cp, *rest;
  int	status;
  Ornament_Type	next_type;

  Ornament_Properties oprops;
  oprops.type = peakgroup;

  Stringy assign;
  SPoint pos(sp->dimension()), freq(sp->dimension());
  double verror = 0;
  Stringy verrormethod;
  Label *label = NULL;
  List peaklets;

  while (ornament_get_line_v1(oprops, buf, sizeof buf, fp,
			      &cp, &rest, &status, &next_type)) {
    if (strcmp(cp, "pos") == 0) {
      pos = read_point(next_line(&rest), NULL);
      freq = pos;
    }

    else if (strcmp(cp, "fq") == 0) {
      freq = read_point(next_line(&rest), NULL);
    }

    //
    // label becomes user label
    //
    else if (strcmp(cp, "label") == 0) {
      cp = next_line(&rest);		// Not used.
    }


    else if (strcmp(cp, "id") == 0)
				  ;  // Obsolete id number.

    else if (strcmp(cp, "ri") == 0)
				  ;	// Obsolete assignment resonance indexes

    else if (strcmp(cp, "rs") == 0) {
      assign = next_line(&rest);
    }

    else if (strcmp(cp, "pl") == 0) 
				  ; // Per peak extra planes info no longer used.
    else if (strcmp(cp, "ve") == 0) 
      restore_volume_error(next_line(&rest), &verror, &verrormethod);

    //
    // Watch for start of embedded ornament
    //
    else if (strcmp(cp, "[") == 0) {
      Ornament_Type next_type;
      Ornament_Properties oprops2;
      ornament_get_line_v1(oprops2, buf, sizeof buf, fp,
			   &cp, &rest, &status, &next_type);

      //
      // If status == RESTORE_ORNAMENT_CONTINUE the type
      // of embedded ornament to restored is next_type.
      //
      if (status == RESTORE_ORNAMENT_CONTINUE) {
	if (next_type == ::label) {
	  label = (Label *) load_ornament(::label, sp, fp);
	}
	else if (next_type == ::peak) {
	  Peak *pk = (Peak *) load_ornament(::peak, sp, fp);
	  peaklets.append(pk);
	}
      }
    }
  }

  PeakGp *pg = NULL;
  if (status)
    {
      pg = new PeakGp(sp, pos);
      RestoreAssignment(pg, assign);
      pg->set_alias(freq - pos);
      if (label)
	label->attach(pg);
      for (int pi = 0 ; pi < peaklets.size() ; ++pi)
	pg->add(*(Peak *) peaklets[pi]);
      set_ornament_properties(pg, oprops);
      if (verror > 0)
	pg->set_volume_error(verror, verrormethod);
    }

  return pg;
}

/*
 * Save state information out to stream <o>. The state information can be used
 * to restore the PeakGp back to its initial state. This function is used
 * when you do an "Edit Copy" or "Edit Cut".
 */
static void save_peakgroup_for_undo(ostream &o, PeakGp *pg, Object_Table &ids)
{
  if (!pg->user_name().is_empty())
    o << "nm " << pg->user_name() << endl;

  if (pg->IsAliased())
    o << "al " << point_format(pg->alias(), "%.3f", true) << endl;

  double verror;
  Stringy verrormethod;
  if (pg->volume() && pg->volume_error(&verror, &verrormethod))
    o << "ve " << verror << ' ' << verrormethod << endl;

  Label *label = pg->label();
  if (label)
    SaveXRef(label, ids, o);

  SaveAssignment(pg, o);

  const List &peaklets = pg->peaklets();
  for (int pi = 0 ; pi < peaklets.size() ; ++pi)
    SaveXRef((Peak *) peaklets[pi], ids, o);
}

/*
 * Restore this PeakGp's state information from stream <i>. This function is
 * used when you do an "Edit Paste" and sometimes when you do "Edit Undo".
 */
static PeakGp *restore_peakgroup_for_undo(Session &s, istream &i,
					  Spectrum *sp, PeakGp *pg,
					  Object_Table &ids)
{
  char		buf[MAX_LINE_LENGTH];

  Stringy assign;
  SPoint alias(sp->dimension());
  double verror = 0;
  Stringy verrormethod;
  Label *label = NULL;
  List peaklets;
  Project &proj = s.project();

  for ( ;; ) {
    i >> buf;
    if (strcmp(buf, "nm") == 0) {	// name
      i.getline(buf, sizeof buf);	// Not used.
    }
    else if (strcmp(buf, "of") == 0) {	// frequency
      Reporter &rr = s.reporter();
      rr.message("Peakgroup frequency override is no longer supported\n");
    }

    else if (strcmp(buf, "al") == 0) {	// alias
      i.getline(buf, sizeof buf);
      alias = read_point(buf, NULL);
    }

    else if (strcmp(buf, "ve") == 0)	// volume error estimate
      read_volume_error(i, &verror, &verrormethod);

    /*
     * We use "rs" and not "ri" because we can be copying
     * to a spectrum that has not yet seen these assignments
     * and we want to create them if they are currently absent.
     */
    else if (strcmp(buf, "rs") == 0) {
      i.getline(buf, sizeof buf);
      assign = buf;
    }
    else if (strcmp(buf, "pl") == 0) {	// No longer used.
      i.getline(buf, sizeof buf);	// Throw away per peak
      // extra planes info.
    }
    else if (strcmp(buf, "xr") == 0) {	// cross reference
      i.getline(buf, sizeof buf);	// read rest of line
      Ornament *op = FindXRef(proj, buf, ids);
      if (op) {
	if (op->type() == ::label)
	  label = (Label *) op;
	else if (op->type() == ::peak)
	  peaklets.append((Peak *) op);
      }
    }
    else if (strcmp(buf, "[") == 0) {	// embedded ornament
      Ornament *op = restore_embedded_ornament_for_undo(s, sp, i, ids);
      if (op) {
	if (op->type() == ::label)
	  label = (Label *) op;
	else if (op->type() == ::peak)
	  peaklets.append((Peak *) op);
      }
    }
    else if (strcmp(buf, "<>") == 0)
      break;
    else
      i.getline(buf, sizeof buf);	// skip rest of line
  }

  if (pg == NULL)
    pg = new PeakGp(sp, SPoint(sp->dimension()));
  RestoreAssignment(pg, assign);
  pg->set_alias(alias);
  if (verror > 0)
    pg->set_volume_error(verror, verrormethod);
  if (label && pg->label() == NULL)
    label->attach(pg);
  for (int pi = 0 ; pi < peaklets.size() ; ++pi)
    pg->add(*(Peak *) peaklets[pi]);

  return pg;
}

//
// Read a line from a version 1 save file. This routine handles generic
// arguments such as the note, color, size and flags. _getLineV1
// doesn't return until it reads a line it doesn't handle.
//
// If the line is valid and is part of this ornament, _getLineV1 returns true.
// Otherwise, _getLineV1 returns false and sets the status to either:
//
//	RESTORE_ORNAMENT_SUCCESS	- finished all ornaments
//	RESTORE_ORNAMENT_CONTINUE	- found the start of a new ornament
//	RESTORE_ORNAMENT_FAILED		- hit end of file.
//

static Ornament_Type last_RestoreV1Type;	// state variable

static bool ornament_get_line_v1(Ornament_Properties &oprops,
				 char *buf, int bufsiz, FILE *fp,
				 char **tokp, char **rest, int *statusp,
				 Ornament_Type *next_type)
{
	char	*cp;

	*statusp = RESTORE_ORNAMENT_SUCCESS;
	while (fgets(buf, bufsiz, fp)) {
		if (strcmp(buf, SAVE_ENDORNAMENT) == 0) {
			*statusp = RESTORE_ORNAMENT_SUCCESS;
			last_RestoreV1Type = (Ornament_Type) -1;
			return false;
		}

		//
		// Ignore blank lines
		//
		*rest = buf;
		*tokp = next_token(rest);
		if (*tokp == NULL)
			continue;

		cp = *tokp;

		//
		// The "type" starts a new ornament.
		//
		if (strcmp(cp, "type") == 0) {
			*next_type = ornament_name2type(next_token(rest));
			last_RestoreV1Type = *next_type;
			*statusp = RESTORE_ORNAMENT_CONTINUE;
			return false;
		}

		//
		// Handle generic ornament options.
		//
		if (strcmp(cp, "note") == 0)
		  oprops.note = ornament_get_note_v1(rest, fp);
		else if (strcmp(cp, "color") == 0)
		  oprops.color = colorread(next_line(rest), oprops.type);
		else if (strcmp(cp, "size") == 0)
		  next_line(rest);		// Obsolete mSize
		else if (strcmp(cp, "flags") == 0) {
		  cp = next_token(rest);
		  if (cp) {
		    oprops.select = atoi(cp);

		    cp = next_token(rest);
		    if (cp) {
		      oprops.lock = atoi(cp);

		      cp = next_token(rest);
		      if (cp) {
			// deftranspose = atoi(cp);	// skipped
			cp = next_token(rest);	// Obsolete HasPointer
		      }
		    }
		  }
		}

		//
		// Handle end of embedded ornament
		//
		else if (strcmp(cp, "]") == 0)
			return false;

		//
		// Every other type of line we pass.
		//
		else
			return true;
	}

	//
	// Hit end of file.
	//
	*statusp = RESTORE_ORNAMENT_FAILED;
	return false;
}

// ----------------------------------------------------------------------------
//
static void set_ornament_properties(Ornament *op, Ornament_Properties &oprops)
{
  op->SetNote(oprops.note);
  op->SetColor(oprops.color);
  op->select(oprops.select);
  op->lock(oprops.lock);
}

//
// Restore a version 1 save file note from a file.
//
static Stringy ornament_get_note_v1(char **rest, FILE *fp)
{
	char	*cp;
	char	buf[MAX_LINE_LENGTH];
	char	notebuf[MAX_LINE_LENGTH], *np;

	/*
	 * handle the first part of the line with next_line()
	 */
  	np = notebuf;
	cp = next_line(rest);
	cp = skip_white(cp);
	cp++;			/* skip opening " */
	do {
		while (*cp) {
			if (cp[0] == '"') {
				if (cp[1] == '"')
					cp++;
				else
					goto done;
			}
			*np++ = *cp++;
		}
	} while (fgets(buf, sizeof buf, fp) && (cp = buf));

done:;
	*np = '\0';
	return notebuf;
}

/*
 * Return the current type of the Version 1 restore
 */
static Ornament_Type RestoreV1Type()
{
  return last_RestoreV1Type;
}


// ----------------------------------------------------------------------------
//
static void SaveAssignment(CrossPeak *xp, FILE *fp)
{
	//
	// Don't save resonances for peaks that belong to a group already.
	//
	if (is_peaklet(xp))
		return;

	//
	// Collect up all resonances. We won't save any assignment information
	// unless at least one resonance is non-null.
	//
	int dim = xp->dimension();
	bool any = false;
	for (int a = 0; a < dim; a++)
	  if (xp->resonance(a))
	    any = true;
	if (!any)
		return;

	/*
	 * Save the resonance names. These will not be required once
	 * the database code has been debugged properly.
	 */
	fprintf(fp, "rs");
	for (int a = 0; a < dim; a++)
	  {
	    Resonance *r = xp->resonance(a);
	    if (r)
	      fprintf(fp, " %s", assignment_name(r).cstring());
	    else
	      fputs(" |?|?|", fp);
	  }
	fputc('\n', fp);
}

// ----------------------------------------------------------------------------
//
static Stringy assignment_name(Resonance *r)
{
  Group *g = r->group();
  Atom *a = r->atom();
  return "|" + g->name() + "|" + a->name() + "|";
}

// ----------------------------------------------------------------------------
//
static void SaveAssignment(CrossPeak *xp, ostream &o)
{
	//
	// Don't save resonances for peaks that belong to a group already.
	//
	if (is_peaklet(xp))
		return;

	//
	// Also don't save any information if there isn't any to save.
	//
	bool any = false;
	for (int i = 0; i < xp->dimension(); i++)
		if (xp->resonance(i))
			any = true;
	if (! any)
		return;

	o << "rs";
	for (int i = 0; i < xp->dimension(); i++) {
		Resonance	*rp = xp->resonance(i);

		if (rp)
			o << ' ' << assignment_name(rp);
		else
			o << " |?|?|";
	}
	o << endl;
}

/*
 * Save the "xr < type id spectrum >" field, used when the ornament
 * crossreferences another ornament. For example, when a label is deleted it
 * crossreferences its parent so that if the label is undeleted the
 * crossreference can be used to reconstruct the connection to the parent
 */
static void SaveXRef(Ornament *op, Object_Table &ids, ostream &o)
{
	o	<< "xr ";
	ornament_save_identifier(op, ids.id(op, "ornament"), o);
}

/*
 * Return the ornament in this spectrum that is cross referenced by <xRef>
 */
static Ornament *FindXRef(Project &proj, const char *xRef, Object_Table &ids)
{
	Spectrum	*sp;
	Ornament_Type		otype;
	int		oid;

	return (ParseIdentifier(proj, &otype, &oid, &sp, xRef) ?
		FindOrnament(sp, otype, oid, ids) : NULL);
}

/*
 * Return the Ornament of type <t> with DB ID <id>, or NULL if not found
 */
static Ornament *FindOrnament(Spectrum *sp, Ornament_Type t, int id,
			      Object_Table &ids)
{
  Stringy type;
  Ornament *op = (Ornament *) ids.object(id, &type);

  return (op && type == "ornament"
	  && op->type() == t && op->spectrum() == sp ?
	  op : NULL);
}

/*
 * Parse the object identifier in <buf>, a line of the form:
 *
 *	< type id spectrum_full_name >
 *
 * into a Spectrum, ornament type, and ornament id.
 */
static bool ParseIdentifier(Project &proj, Ornament_Type *typep, int *idp,
			    Spectrum **spp, const char *buf)
{
	buf = skip_white(buf);
	if (buf[0] != '<')
		return false;

	char		speName[SPECTRUM_NAME_SIZE];
	const char	*bp = buf + 1;
	char		*cp = speName;

	/*
	 * Copy the type name into <speName>, then evaluate it.
	 */
	bp = skip_white(bp);
	if (*bp == '\0')
		return false;
	while (*bp && !isspace(*bp))
		*cp++ = *bp++;
	*cp = '\0';
	*typep = ornament_name2type(speName);

	/*
	 * Get the ornament ID.
	 */
	bp = skip_white(bp);
	if (*bp == '\0')
		return false;
	*idp = atoi(bp);
	bp = skip_nonwhite(bp);
	bp = skip_white(bp);
	if (*bp == '\0')
		return false;

	/*
	 * Copy the spectrum into <speName>
	 */
	strcpy(speName, bp);

	/*
	 * Trim backwards, removing trailing whitespace
	 */
	cp = strrchr(speName, '>');
	if (cp) {
		*cp = ' ';
		while (cp >= speName && isspace(*cp))
			*cp-- = '\0';
	}
	Spectrum *sp = proj.find_spectrum(speName);
	*spp = sp;

	return sp != NULL;
}

/*
 * Read a color from string <str> and return it.
 */
static Stringy colorread(const char *str, Ornament_Type type)
{
  int text, fill, edge, last;

  if (str)
    {
      int count = sscanf(str, "%d %d %d %n", &text, &fill, &edge, &last);
      if (count == 3)
	{
	  const char *colorname = skip_white(str + last);
	  if (strlen(colorname) > 0)
	    return colorname;

	  switch (type)
	    {
	    case label:	return old_file_color(text);
	    case line:	return old_file_color(fill);
	    case grid:	return old_file_color(fill);
	    case peak:	return old_file_color(fill);
	    case peakgroup:	return old_file_color(fill);
	    default:
	      break;
	    }
	}
    }

  return "white";
}

/*
 * Format COLOR <c> into a string and return the string.
 */
static Stringy colorformat(const Color &c, Ornament_Type t)
{
  Stringy s;

  if (t == label)
    s = formatted_string("%d %d %d %s",
			 old_file_color_index(c), 0, 0, c.name().cstring());
  else
    s = formatted_string("%d %d %d %s",
			 0, old_file_color_index(c), 0, c.name().cstring());

  return s;
}

/*
 * Save ornaments to file.
 */
static void save_ornaments(Session &s, Spectrum *sp, FILE *fp)
{
	fprintf(fp, SAVE_BEGINORNAMENT);
	nextid = 1;
	int pcount = 0;
	const List &olist = sp->ornaments();
	for (int oi = 0 ; oi < olist.size() ; ++oi)
	  {
	    Ornament *op = (Ornament *) olist[oi];
	    if (!has_owner(op))
	      {
		ornament_save_v1(op, fp);

		if (is_crosspeak(op))
		  pcount += 1;
		if (pcount % 100 == 0)
		  status_line(s, "Saved %d peaks of %s",
			      pcount, sp->name().cstring());
	      }
	  }
	fprintf(fp, SAVE_ENDORNAMENT);
	status_line(s, "");
}

// ----------------------------------------------------------------------------
//
static bool has_owner(Ornament *orn)
{
  return ((orn->type() == label && ((Label *)orn)->attached() != NULL) ||
	  (orn->type() == peak && ((Peak *)orn)->peakgp() != NULL));
}

/*
 * Restore an ornament from file. The <SAVE_BEGINORNAMENT> string
 * starts the list of ornaments. A "type" statement starts each
 * ornament and after the first ornament, a "type" statement is used
 * to indicate the beginning of the next ornament. Thus, after the
 * first "type" statement we just loop until the return status of
 * the ornament restore is no longer RESTORE_ORNAMENT_CONTINUE.
 */
static bool load_ornaments(Session &s, Spectrum *sp, FILE *fp)
{
	char	buf[MAX_LINE_LENGTH], *cp;
	Ornament_Type	type = (Ornament_Type) -1;

	while (fgets(buf, sizeof buf, fp)) {
		if (strcmp(buf, SAVE_ENDORNAMENT) == 0)
			break;

		char *rest = buf;
		cp = next_token(&rest);
		if (cp &&  strcmp(cp, "type") == 0 &&
		    (cp = next_token(&rest))) {

			/*
			 * The type is now saved as a name, but was
			 * at one time saved as a number.
			 */
			if (isalpha(*cp))
				type = ornament_name2type(cp);
			else
				type = (Ornament_Type) atoi(cp);


			/*
			 * The "type xxx" statement both terminates an
			 * ornament and starts the next ornament. Since
			 * the line has already been read, we use a global
			 * to pass back the type of ornament that is next.
			 */
			while (type != -1) {
				int pcount = sp->crosspeaks().size();

				Ornament *op = load_ornament(type, sp, fp);
				if (op == NULL)
				  break;
				type = RestoreV1Type();

				int new_pcount = sp->crosspeaks().size();
				if (new_pcount / 100 != pcount / 100)
				  status_line(s,
					      "Loaded %d peaks of %s",
					      new_pcount,
					      sp->name().cstring());
			}
			break;
		}
	}
	status_line(s, "");
	return type == (Ornament_Type) -1;
}


/*
 * Restore an ornament of the specified type.
 */
static Ornament *load_ornament(Ornament_Type type, Spectrum *sp, FILE *fp)
{
  Ornament *op = NULL;

  switch (type)
    {
    case grid: op = restore_grid_v1(fp, sp);      break;
    case line: op = restore_line_v1(fp, sp);      break;
    case label: op = restore_label_v1(fp, sp);      break;
    case peak: op = restore_peak_v1(fp, sp);      break;
    case peakgroup: op = restore_peakgroup_v1(fp, sp);      break;
    default:
      warn("load_ornament(): Bad ornament type.\n");
    }

  return op;
}

// ----------------------------------------------------------------------------
//
static void peak_list_option(char *line, Peak_List_Options *options)
{
	char		*which, *arg;

	char *rest = line;
	if (line == NULL || (which = next_token(&rest)) == NULL)
		return;

	if (strcmp(which, "nameType") == 0) {
		arg = next_token(&rest);
		if (strcmp(arg, "user") == 0) {
		  options->fields[USER_NAME_FIELD] = true;
		  options->fields[ASSIGNMENT_NAME_FIELD] = false;
		  options->sort_type = USER_NAME_SORT;
		} else if (strcmp(arg, "assignment") == 0) {
		  options->fields[ASSIGNMENT_NAME_FIELD] = true;
		  options->fields[USER_NAME_FIELD] = false;
		  options->sort_type = RESONANCE_NAME_SORT;
		}
	}
	else if (strcmp(which, "sortBy") == 0) {
		arg = next_token(&rest);
		if (arg) {
		  if (strcmp(arg, "frequency") == 0)
		    options->sort_type = FREQUENCY_SORT;
		  else if (strcmp(arg, "volume") == 0)
		    options->sort_type = VOLUME_SORT;
		  else if (strcmp(arg, "assignment-distance") == 0)
		    options->sort_type = ASSIGNMENT_DISTANCE_SORT;
		  else if (strcmp(arg, "none") == 0)
		    options->sort_type = NO_SORT;
		}
	}
	else if (strcmp(which, "sortAxis") == 0) {
		arg = next_token(&rest);
		if (arg) {
			options->sort_axis = label2axis(arg);
		}
	}
	else if (strcmp(which, "sortPairCrossdiagonal") == 0) {
	  options->sort_pairx = true;
	}
	else if (strcmp(which, "assignmentFormat") == 0) {
	  options->assignment_format = next_line(&rest);
	}
  	else if (strcmp(which, "showFlags") == 0) {
	  while ((arg = next_token(&rest), arg != NULL))
	    if (strcmp(arg, "frequency") == 0)
	      options->fields[FREQUENCY_FIELD] = true;
	    else if (strcmp(arg, "frequency-hz") == 0)
	      options->fields[HZ_FREQUENCY_FIELD] = true;
	    else if (strcmp(arg, "resonance-dev") == 0)
	      options->fields[RESONANCE_DEV_FIELD] = true;
	    else if (strcmp(arg, "linewidth") == 0)
	      options->fields[LINEWIDTH_FIELD] = true;
	    else if (strcmp(arg, "volume") == 0)
	      options->fields[VOLUME_FIELD] = true;
	    else if (strcmp(arg, "volume-error") == 0)
	      options->fields[VOLUME_ERROR_FIELD] = true;
	    else if (strcmp(arg, "trans-volume") == 0)
	      options->fields[TRANSPOSED_VOLUME_FIELD] = true;
	    else if (strcmp(arg, "fit-residual") == 0)
	      options->fields[FIT_RESIDUAL_FIELD] = true;
	    else if (strcmp(arg, "fit-height") == 0)
	      options->fields[FIT_HEIGHT_FIELD] = true;
	    else if (strcmp(arg, "data-height") == 0)
	      options->fields[DATA_HEIGHT_FIELD] = true;
	    else if (strcmp(arg, "resonance-freq") == 0)
	      options->fields[RES_FREQ_FIELD] = true;
	    else if (strcmp(arg, "res-axis-dev") == 0)
	      options->fields[RES_DEV_FIELD] = true;
	    else if (strcmp(arg, "spectrum-name") == 0)
	      options->fields[SPECTRUM_NAME_FIELD] = true;
	    else if (strcmp(arg, "assignment-distance") == 0)
	      options->fields[ASSIGNMENT_DISTANCE_FIELD] = true;
	    else if (strcmp(arg, "note") == 0)
	      options->fields[NOTE_FIELD] = true;
	    else if (strcmp(arg, "S/N") == 0)
	      options->fields[NOISE_FIELD] = true;
	    else if (strcmp(arg, "mardigras") == 0)
	      options->fields[MARDIGRAS_FIELD] = true;
	    else if (strcmp(arg, "diana") == 0)
	      options->fields[DIANA_FIELD] = true;
	}

}

// ----------------------------------------------------------------------------
//
static void save_peak_list_options(FILE *fp, Peak_List_Options *options)
{
  switch (options->sort_type)
    {
    case FREQUENCY_SORT:
      fprintf(fp, "listTool sortBy frequency\nlistTool sortAxis w%d\n",
	      options->sort_axis + 1);
      break;
    case RESONANCE_NAME_SORT:
      fprintf(fp, "listTool sortBy label\n"
	      "listTool nameType assignment\n"
	      "listTool sortAxis w%d\n",
	      options->sort_axis + 1);
      break;
    case USER_NAME_SORT:
      fprintf(fp, "listTool sortBy label\nlistTool nameType user\n"); break;
    case VOLUME_SORT:
      fprintf(fp, "listTool sortBy volume\n"); break;
    case ASSIGNMENT_DISTANCE_SORT:
      fprintf(fp, "listTool sortBy assignment-distance\n"); break;
    case NO_SORT:
      fprintf(fp, "listTool sortBy none\n"); break;
    }

  if (options->sort_pairx)
    fprintf(fp, "listTool sortPairCrossdiagonal\n");

  if (!options->assignment_format.is_empty())
    fprintf(fp, "listTool assignmentFormat %s\n",
	    options->assignment_format.cstring());

  fprintf(fp, "listTool showFlags");
  if (options->fields[FREQUENCY_FIELD])		fprintf(fp, " frequency");
  if (options->fields[HZ_FREQUENCY_FIELD])	fprintf(fp, " frequency-hz");
  if (options->fields[RESONANCE_DEV_FIELD])	fprintf(fp, " resonance-dev");
  if (options->fields[LINEWIDTH_FIELD])		fprintf(fp, " linewidth");
  if (options->fields[VOLUME_FIELD])		fprintf(fp, " volume");
  if (options->fields[VOLUME_ERROR_FIELD])	fprintf(fp, " volume-error");
  if (options->fields[TRANSPOSED_VOLUME_FIELD])	fprintf(fp, " trans-volume");
  if (options->fields[FIT_RESIDUAL_FIELD])	fprintf(fp, " fit-residual");
  if (options->fields[FIT_HEIGHT_FIELD])	fprintf(fp, " fit-height");
  if (options->fields[DATA_HEIGHT_FIELD])	fprintf(fp, " data-height");
  if (options->fields[RES_FREQ_FIELD])		fprintf(fp, "resonance-freq");
  if (options->fields[RES_DEV_FIELD])		fprintf(fp, "res-axis-dev");
  if (options->fields[SPECTRUM_NAME_FIELD])	fprintf(fp, "spectrum-name");
  if (options->fields[ASSIGNMENT_DISTANCE_FIELD])
    fprintf(fp, " assignment-distance");
  if (options->fields[NOTE_FIELD])	fprintf(fp, " note");
  if (options->fields[NOISE_FIELD])	fprintf(fp, " S/N");
  if (options->fields[MARDIGRAS_FIELD])	fprintf(fp, " mardigras");
  if (options->fields[DIANA_FIELD])	fprintf(fp, " diana");
  fprintf(fp, "\n");
}

/*
 * Save the mode to file pointer <fp>
 */
static void mode_save(Session &s, FILE *fp)
{
  int m = (int) pointer_mode(s);
  if (m > 7)
    m += 1;
  fprintf(fp, "set mode %d\n", m);
}

/*
 * Save the view's current parameters.
 */
static void save_view_params(View *view, const View_Settings &s, FILE *fp)
{
	Spectrum *sp = view->spectrum();
	int	dimension = sp->dimension();
	Rectangle vr = view->view_rectangle();

	fprintf(fp, SAVE_BEGINPARAMS);

	fprintf(fp, "orientation");
	for (int i = 0; i < dimension; i++)
		fprintf(fp, " %d", view->axis_name(i));
	putc('\n', fp);

	fprintf(fp, "location %d %d\n", s.x, s.y);
	fprintf(fp, "size %d %d\n", s.width, s.height);

	fprintf(fp, "offset");
	for (int i = 0; i < dimension; i++)
	  if (i == view->axis(X))
	    fprintf(fp, " %f", view->map(vr.max(X), X, PPM, INDEX));
	  else if (i == view->axis(Y))
	    fprintf(fp, " %f", view->map(vr.max(Y), Y, PPM, INDEX));
	  else
	    fprintf(fp, " %f",
		    sp->map(s.center[i], i, PPM, INDEX));
	putc('\n', fp);

	fprintf(fp, "scale");
	double pz0 = sp->scale(s.pixel_size[0], 0, PPM, INDEX);
	for (int i = 0; i < dimension; i++)
	  {
	    double pzi = sp->scale(s.pixel_size[i], i, PPM, INDEX);
	    fprintf(fp, " %f", pz0 / pzi);
	  }
	putc('\n', fp);

	double z = 1 / sp->scale(s.pixel_size[0], 0, PPM, INDEX);
	fprintf(fp, "zoom %f\n", z);
	fprintf(fp, "flags %d\n", 0);
	fprintf(fp, SAVE_ENDPARAMS);
}

/*
 *	Save information about this view in the save file.
 */
static void save_view(View *view, FILE *fp)
{
	View_Settings s = view->settings();

	fprintf(fp, SAVE_BEGINVIEW);
	fprintf(fp, "name %s\n", s.view_name.cstring());
	fprintf(fp, "precision %d\n", 0);
	fprintf(fp, "precision_by_units %d %d %d\n", 0, 0, 0);

	fprintf(fp, "viewmode %d\n", 0);

	/*
	 * Save the shown ornament state.
	 */
	fprintf(fp, "show %d", s.show_ornaments);
	for (int t = 0 ; t < ORNAMENT_TYPE_COUNT ; ++t)
	  if (s.show_ornament((Ornament_Type) t))
	    fprintf(fp, " %s", index_name(Ornament_Type_Names, t).cstring());
	fputc('\n', fp);

	fprintf(fp, "axistype %d\n", s.scale_units);
	fprintf(fp, "flags");
	if (!s.show_view)
	  fprintf(fp, " hidden");	// uiview.hidden
	if (s.show_crosshair)
	  fprintf(fp, " crosshair");	// showing a crosshair
	if (s.show_crosshair_from_other_views)
	  fprintf(fp, " crosshair2");	// showing other view's crosshair
	if (s.show_transpose_crosshair)
	  fprintf(fp, " crosshairx");	// showing cross-diagonal crosshair
	if (!s.show_scales)
	  fprintf(fp, " noscales");	// showing axis scales
	if (!s.show_scrollbars)
	  fprintf(fp, " noscrollbars");	// showing scrollbars
	if (s.show_slices)
	  fprintf(fp, " slices");	// showing slice panels
	if (!s.slice_auto_scale)
	  fprintf(fp, " fixedslices");	// fixed or auto scaling for slices
	if (s.slice_subtract_peaks)
	  fprintf(fp, " slicesubtract");// show peak subtracted slice trace
	if (s.show_resonance_panels)
	  fprintf(fp, " resonance");	// showing resonance panels
	if (s.filter_resonances)
	  fprintf(fp, " resfilter");	// filter resonances w/ assign guesses
	if (s.show_peak_info)
	  fprintf(fp, " peakinfo");	// showing peak panel
	if (s.show_contour_scale)
	  fprintf(fp, " contourscale");	// showing contour scale
	if (s.show_nucleus_names)
	  fprintf(fp, " atomaxes");	// axis shows atom names
	if (s.subtract_fit_peaks)
	  fprintf(fp, " subractpeaks");	// subtract fit peaks before contouring
	fputc('\n', fp);

	save_contour_parameters(fp, s.pos_contours, s.neg_contours);
	save_view_params(view, s, fp);	// Formerly home params.
	save_view_params(view, s, fp);	// Formerly last params.
	save_view_params(view, s, fp);	// Current params.

	fprintf(fp, SAVE_ENDVIEW);
}

/*
 * Save the contour parameters for View <view> to file <fp>
 */
static void save_contour_parameters(FILE *fp, const Contour_Parameters &pos,
				    const Contour_Parameters &neg)
{
	fprintf(fp, "contour.pos %d %g %f %f %s\n",
		pos.levels.levels,
		pos.levels.lowest,
		pos.levels.factor,
		0.0,
		pos.color_scheme().cstring());
	fprintf(fp, "contour.neg %d %g %f %f %s\n",
		neg.levels.levels,
		neg.levels.lowest,
		neg.levels.factor,
		0.0,
		neg.color_scheme().cstring());
}

/*
 * Restore the contour parameters for View <view> from line <line>,
 * where <which> is "pos" or "neg".
 */
static void read_contour_parameters(char *which, char *line,
				    Contour_Parameters *pos,
				    Contour_Parameters *neg)
{
  char *tok, *rest = line;
  tok = next_token(&rest);
  int levels = (tok ? atoi(tok) : 0);
  tok = next_token(&rest);
  double lowest = (tok ? atof(tok) : 0.0);
  tok = next_token(&rest);
  double factor = (tok ? atof(tok) : 0.0);
  tok = next_token(&rest);
  Stringy colorname = trim_white(rest);

  if (strcmp(which, "pos") == 0) {
    pos->levels.levels = levels;
    pos->levels.lowest = lowest;
    pos->levels.factor = factor;
    pos->set_color_scheme(colorname.is_empty() ? Stringy("red") : colorname);
  }
  else if (strcmp(which, "neg") == 0) {
    neg->levels.levels = levels;
    neg->levels.lowest = lowest;
    neg->levels.factor = factor;
    neg->set_color_scheme(colorname.is_empty() ? Stringy("green") : colorname);
  }
}

/*
 * Flags for fixing save files from previous versions.
 */
static int	fix_1_6;		// state variable

/*
 * Restore from file <pathName>
 */
static Spectrum *restore_savefile(Session &s, const Stringy &path,
				  List &views_to_show, Stringy *error_msg)
{
	int	version, release;

	if (!check_file_type(path, SAVEFILE_SPECTRUM, &version, &release))
	  {
	    *error_msg = formatted_string("Couldn't read save file %s",
					  path.cstring());
	    return NULL;
	  }

	/*
	 * <= 1.06 save file.had coordinates that were stored as
	 * X Y, regardless of which omega was along which axis.
	 * Version 1.07 and above file.have coordinates stored as
	 * W1, W2, ... Also, the defaults for 1.06 files where in
	 * ppm, while for 1.07 and above they are in some normalized
	 * number.
	 */
	fix_1_6 = (version == 1 && release <= 6);

	Spectrum *sp = NULL;
	SaveFile sf(s, SAVEFILE_SPECTRUM, true);
	sf.set_path(path);
	FILE *fp = sf.OpenForReading();
	if (fp == NULL)
	  {
	    *error_msg = "Couldn't open " + sf.path();
	    return NULL;
	  }

	int	which = 0;
	char	buf[MAX_LINE_LENGTH];
	bool cancelled = false;

	while (fgets(buf, sizeof buf, fp)) {
	      if (isemptyline(buf))
		continue;
	      if (strcmp(buf, SAVE_BEGINUSER) == 0)
		user_restore(s, fp);
	      else if (strcmp(buf, SAVE_BEGINSPECTRUM) == 0) {

		sp = spectrum_restore_one(s, fp, sf, ++which, views_to_show,
					  &cancelled, error_msg);
		if (sp == NULL && !cancelled)
		  {
		    Reporter &rr = s.reporter();
		    rr.message(error_msg->cstring());
		    break;				// Error reading file
		  }

		if (!cancelled)
		  {
		    Reporter &rr = s.reporter();
		    rr.message("Loaded %-20s",	sf.ShortPath().cstring());
		    if (sp->crosspeaks().size() > 0)
		      rr.message(" %5d peaks\n", sp->crosspeaks().size());
		    else
		      rr.message("\n");
		  }
	      }
	      else if (strcmp(buf, SAVE_BEGINSYNC) == 0)
		skip_past_line(fp, SAVE_ENDSYNC);	// Obsolete view syncs
	      
	      else if (strcmp(buf, SAVE_BEGINSET) == 0
		       ||       strcmp(buf, "<strips>\n") == 0)
		set_restore(fp);
	}
	sf.EndRead(fp);

	return sp;
}

// ----------------------------------------------------------------------------
//
static bool skip_past_line(FILE *fp, const char *line)
{
  char	buf[MAX_LINE_LENGTH];

  while (fgets(buf, sizeof buf, fp))
    if (strcmp(buf, line) == 0)
      return true;

  return false;
}

/*
 * Restore the spectrum from a save file opened as <fp>, from file
 * <pathName>. The <whichSpectrum> is used for backward compatibility
 * to Sparky 2.xx, where save files could contain more than one spectrum.
 * Sparky 3.00 requires save files to contain exactly one spectrum.
 *
 * The save file holds the Spectrum's complete information. New to Sparky
 * 3.00 is the idea of a Database. The database allows information to be
 * shared between Spectra of the same Condition of the same Molecule.
 *
 * The Sparky DB was started with Sparky version 2.00 but did not come
 * to full file-based fruition until v3.00. This file-based DB should be
 * replaced some day with a real DB
 */
static Spectrum *spectrum_restore_one(Session &s, FILE *fp, SaveFile &sf,
				      int whichSpectrum,  List &views_to_show,
				      bool *cancelled, Stringy *error_msg)
{
  int molecule_id, condition_id, spectrum_id, dim;
  Stringy db_path, molecule_name, condition_name, spectrum_name, data_path;

  if (!db_stuff_from_save_file(sf, &db_path,
			       &molecule_id, &molecule_name,
			       &condition_id, &condition_name,
			       &spectrum_id, &spectrum_name,
			       &data_path, &dim))
    {
      *error_msg = formatted_string("Couldn't open file %s\n",
				    sf.path().cstring());
      return NULL;
    }

  Stringy sf_path = sf.path();
  if (whichSpectrum > 1)
    {
      /*
       * Generate a save file name by appending the
       * spectrum name to the file
       */
      sf_path = formatted_string("%s.%d", sf.path().cstring(),
				 whichSpectrum);
      Reporter &rr = s.reporter();
      rr.warning("There are multiple spectra in save file\n"
		 "  %s\n"
		 "For spectrum %d I will create the save file\n"
		 "  %s\n"
		 "You can change this name with File/Save As...",
		 sf.path().cstring(), whichSpectrum, sf_path.cstring());
    }

  *cancelled = false;
  Project &proj = s.project();
  Spectrum *loaded_sp = proj.find_savefile_spectrum(sf_path);
  if (loaded_sp)
    {
      Stringy msg = formatted_string("Spectrum %s is already loaded.\n",
				     loaded_sp->fullname().cstring());
      if (query(s, msg, "Cancel spectrum load",
		"Unload spectrum, then reload it") != 2)
	{
	  *cancelled = true;
	  return NULL;
	}
      remove_spectrum_view_params(loaded_sp, views_to_show);
      delete loaded_sp;
    }

  Spectrum *sp = NULL;
  Condition *c = proj.define_condition(molecule_name, condition_name);
  merge_db_data(db_path, molecule_id, condition_id, c);
  NMR_Data *nmr_data = open_nmr_data(s, data_path);
  if (nmr_data == NULL)
    return NULL;

  sp = new Spectrum(s, spectrum_name, c, sf_path, nmr_data);
  sp->set_save_file(sf);

  if (!restore_spectrum(s, sp, fp, views_to_show))
    {
      remove_spectrum_view_params(sp, views_to_show);
      delete sp;
      *error_msg = "Error parsing " + sf_path;
      return NULL;
    }

  return sp;
}

// ----------------------------------------------------------------------------
// Delete View_Parameters from list if they refer to the specified spectrum.
//
static void remove_spectrum_view_params(Spectrum *sp, List &views_to_show)
{
  List vplist;
  for (int vi = 0 ; vi < views_to_show.size() ; ++vi)
    {
      View_Parameters *v = (View_Parameters *) views_to_show[vi];
      if (v->sp == sp)
	delete v;
      else
	vplist.append(v);
    }
  views_to_show = vplist;
}

/*
 * Restore this Spectrum from save file pointed to by <fp>
 */
static bool restore_spectrum(Session &s, Spectrum *sp, FILE *fp,
			     List &views_to_show)
{
	char		buf[MAX_LINE_LENGTH], *cp;
	int	i;
	int	dim = sp->dimension();
	int	guess_bits = 0;

	SPoint vis_depth(dim);
	for (int a = 0 ; a < dim ; ++a)
	  vis_depth[a] = sp->scale(1.0, a, INDEX, PPM);

	Guess_Assignments ga = sp->GetAssignGuessOptions();
	for (int a = 0 ; a < dim ; ++a)
	  ga.ppm_range[a] = (sp->nucleus_type(a) == "1H" ? .1 : .5);
	sp->SetAssignGuessOptions(ga);

	while (fgets(buf, sizeof buf, fp)) {
		if (strcmp(buf, SAVE_ENDSPECTRUM) == 0) {

			sp->save_file().NeedsSaving(false);

			return true;
		}

		char *rest = buf;
		if (strcmp(buf, SAVE_BEGINVIEW) == 0) {
			View_Parameters *v = read_view_parameters(sp, fp);
			if (v)
			  {
			    v->settings.visible_depth = vis_depth;
			    views_to_show.append(v);
			  }
		}
		else if (strcmp(buf, SAVE_BEGINORNAMENT) == 0) {
			load_ornaments(s, sp, fp);
		}
		else if (strcmp(buf, SAVE_BEGINLINKS) == 0)
			ornament_restore_links(fp, sp);
		else if (strcmp(buf, SAVE_BEGINDBGROUPS) == 0)
			skip_past_line(fp, SAVE_ENDDBGROUPS); // Obs grp order

		else if (strcmp(buf, SAVE_BEGINATTACHEDDATA) == 0)
			restore_attached_data(sp->saved_values(), fp);

		/*
		 * All other textual parameters. If there is nothing in
		 * the buffer, we ignore blank lines.
		 */
		else if ((cp = next_token(&rest))) {

			/*
			 * The bitmap file.
			 */
			if (strcmp(cp, "bitmap") == 0) {
			  (void) next_token(&rest); // Throw away file name.
			}

			/*
			 * Spectrum parameters
			 */
			else if (strcmp(cp, "shift") == 0) {
				SPoint ppmshifts(dim);
				for (i = 0; i < dim; i++)
				  ppmshifts[i] = next_double(&rest);
				sp->set_ppm_shift(ppmshifts);
			}
			else if (strcmp(cp, "points") == 0) {
				for (i = 0; i < dim; i++)
				  next_int(&rest); // Obsolete npoints_saved
			}
			else if (strcmp(cp, "sweepwidth") == 0) {
				SPoint sweep_width(dim);
				for (i = 0; i < dim; i++)
				  sweep_width[i] = next_double(&rest);
				sp->set_ppm_sweep_width(sweep_width);
			}
			else if (strcmp(cp, "extraPeakPlanes") == 0) {
			    for (i = 0; i < dim; i++)
			      vis_depth[i] = sp->scale(next_int(&rest) + 1,
						       i, INDEX, PPM);
			}
			else if (strcmp(cp, "assignMultiAxisGuess") == 0) {
			    guess_bits = next_int(&rest);
			}

			else if (strcmp(cp, "assignGuessThreshold") == 0) {
			  ; // Not used.
			}

			else if (strcmp(cp, "assignRelation") == 0) {
			    Guess_Assignments ga = sp->GetAssignGuessOptions();
			    int	axis		= next_int(&rest) - 1;
			    int	type		= next_int(&rest);
			    const char *atom	= next_line(&rest);
			    const char *sc = Guess_Assignments::constraint_names[type >= 0 ? type : 0];

			    if (axis >= 0 && axis < dim) {
				if (is_bit_set(guess_bits, axis))
				  ga.sequence_constraint[axis] = sc;
				ga.atom_constraint[axis] = atom;
					
				sp->SetAssignGuessOptions(ga);
			    }
			}

			else if (strcmp(cp, "assignRange") == 0) {
			    Guess_Assignments ga = sp->GetAssignGuessOptions();
			    int	axis		= next_int(&rest) - 1;
			    double range_ppm	= next_double(&rest);

			    if (axis >= 0 && axis < dim) {
				ga.ppm_range[axis] = range_ppm;
				sp->SetAssignGuessOptions(ga);
			    }
			}

			else if (strcmp(cp, "assignFilter") == 0) {
			  ; // Not used.
			}

			else if (strcmp(cp, "assignFormat") == 0)
			    sp->set_assignment_format(next_token(&rest));

			/*
			 * This inefficient scheme has the static ListTool
			 * ParseDefaults function fill in the (global) options,
			 * which are then used to fill this spectrum's options.
			 */
			else if (strcmp(cp, "listTool") == 0) {
			  peak_list_option(next_line(&rest),
					   &sp->mPeakListOptions);
			}

			else if (strcmp(cp, "contourFile") == 0)
			  ;	// Obsolete filename

			else if (strcmp(cp, "contourCacheName") == 0)
			  ;	// obsolete
			else if (strcmp(cp, "contourCacheSize") == 0) {
			  ;  // Obsolete # contour planes
			}
			else if (strcmp(cp, "contourCacheSaveToFile") == 0)
			 ; // obsolete
			else if (strcmp(cp, "contourTemp") == 0)
			 ; // obsolete
			else if (strcmp(cp, "contourCache") == 0)
			 ; // obsolete

			/*
			 * Other options
			 */
			else if (strncmp(cp, "colormap_", 9) == 0
			||       strncmp(cp, "colormap.", 9) == 0)
				; // Obsolete line of color indices.
			else if (strncmp(cp, "integrate.", 10) == 0)
			  LoadIntegrate(sp, sp->mIntegrate, cp + 10, &rest);
			else if (strncmp(cp, "peak.", 5) == 0)
			  LoadPeakPick(sp->mPeakPick, cp + 5, &rest);
			else if (strncmp(cp, "noise.", 6) == 0)
			  LoadNoise(sp, cp + 6, &rest);
			else if (strncmp(cp, "ornament.", 9) == 0)
			  LoadOrnamentSizes(sp, sp->mSizes, &sp->mSelectsize,
					    &sp->mPointersize,
					    &sp->mLineendsize,
					    cp + 9, next_line(&rest));
		}
	}
	return false;
}

/*
 * Restore integration parameters from this line
 */

#define OVERLAP_BY_CONTOUR	0x1
#define OVERLAP_BY_DISTANCE	0x2

static void LoadIntegrate(Spectrum *sp, Integration_Parameters &ip,
			  char *cp, char **rest)
{
	if (strcmp(cp, "overlapped_sep") == 0) {
	  SPoint d = read_point(*rest, rest);	// In Hz
	  if (d.dimension() != sp->dimension())
	    {
	      d = SPoint(sp->dimension());
	      for (int a = 0 ; a < sp->dimension() ; ++a)
		d[a] = 30;
	    }
	  ip.grouping_dist = sp->scale(d, HZ, PPM);
	}
	else if (strcmp(cp, "methods") == 0) {
	  Integration_Method method = (Integration_Method) next_int(rest);
	  ip.integration_method =
	    (method == INTEGRATE_NONE ? INTEGRATE_GAUSSIAN : method);

	  next_int(rest);		// Obsolete overlap method
	  cp = next_token(rest);
	  if (cp)
	    {
	      int grouping_method = atoi(cp);
	      ip.contour_grouping = grouping_method & OVERLAP_BY_CONTOUR;
	      ip.distance_grouping = grouping_method & OVERLAP_BY_DISTANCE;
	    }
	}
	else if (strcmp(cp, "allow_motion") == 0)
	  ip.allow_motion = next_int(rest);
	else if (strcmp(cp, "motion_range") == 0)
	  ip.motion_range = read_point(*rest, rest);
	else if (strcmp(cp, "adjust_linewidths") == 0)
	  ip.adjust_linewidths = next_int(rest);
	else if (strcmp(cp, "min_linewidth") == 0)
	  ip.linewidth_range.min = read_point(*rest, rest);
	else if (strcmp(cp, "max_linewidth") == 0)
	  ip.linewidth_range.max = read_point(*rest, rest);
	else if (strcmp(cp, "fit_baseline") == 0)
	  ip.fit_baseline = next_int(rest);
	else if (strcmp(cp, "subtract_peaks") == 0)
	  ip.subtract_peaks = next_int(rest);
	else if (strcmp(cp, "contoured_data") == 0)
	  ip.contoured_data = next_int(rest);
	else if (strcmp(cp, "rectangle_data") == 0)
	  ip.rectangle_data = next_int(rest);
	else if (strcmp(cp, "maxiterations") == 0) {
		ip.maxiterations = next_int(rest);
		if (ip.maxiterations == 0)
		  ip.maxiterations = 20000;
	}
	else if (strcmp(cp, "tolerance") == 0)
	  ip.tolerance = next_double(rest);
	else if (strcmp(cp, "notify") == 0) ;		// obsolete
	else if (strcmp(cp, "estimate_method") == 0) ;	// obsolete
	else if (strcmp(cp, "estimate") == 0) ;		// obsolete
}

/*
 * Save integration parameters to file.
 */
static void SaveIntegrate(Spectrum *sp, Integration_Parameters &ip, FILE *fp)
{
	SPoint gd_hz = sp->scale(ip.grouping_dist, PPM, HZ);
	fprintf(fp, "integrate.overlapped_sep %s\n",
		point_format(gd_hz, "%.3f", false));
	int cgflag = (ip.contour_grouping ? OVERLAP_BY_CONTOUR : 0);
	int dgflag = (ip.distance_grouping ? OVERLAP_BY_DISTANCE : 0);
	int grouping_method = (cgflag | dgflag);
	fprintf(fp, "integrate.methods %d %d %d\n",
		ip.integration_method, 0, grouping_method);
	fprintf(fp, "integrate.allow_motion %d\n", ip.allow_motion);
	fprintf(fp, "integrate.adjust_linewidths %d\n", ip.adjust_linewidths);
	fprintf(fp, "integrate.motion_range %s\n",
		point_format(ip.motion_range, "%.3f", false));
	fprintf(fp, "integrate.min_linewidth %s\n",
		point_format(ip.linewidth_range.min, "%.4f", false));
	fprintf(fp, "integrate.max_linewidth %s\n",
		point_format(ip.linewidth_range.max, "%.4f", false));
	fprintf(fp, "integrate.fit_baseline %d\n", ip.fit_baseline);
	fprintf(fp, "integrate.subtract_peaks %d\n", ip.subtract_peaks);
	fprintf(fp, "integrate.contoured_data %d\n", ip.contoured_data);
	fprintf(fp, "integrate.rectangle_data %d\n", ip.rectangle_data);
	fprintf(fp, "integrate.maxiterations %d\n", ip.maxiterations);
	fprintf(fp, "integrate.tolerance %f\n", ip.tolerance);
}

/*
 * Restore integration parameters from this line
 */
static void LoadNoise(Spectrum *sp, char *cp, char **rest)
{
	if (strcmp(cp, "sigma") == 0) {
		double noise = next_double(rest);
		if (noise > 0)
		  sp->set_noise_level(noise);
	}
	else if (strcmp(cp, "sigma_threshold") == 0)
		next_double(rest);  // obsolete acceptance probability
}

/*
 * Save integration parameters to file.
 */
static void SaveNoise(Spectrum *sp, FILE *fp)
{
	if (sp->have_noise_level()) {
		fprintf(fp, "noise.sigma %f\n", sp->noise_level());
	}
}

/*
 * Load ornament sizes.
 */
static void LoadOrnamentSizes(Spectrum *sp, double *ornament_sizes,
			      float *select_size, float *pointer_size,
			      float *line_end_size, char *which, char *value)
{
	char	*cp;
	Ornament_Type	type;

	/*
	 * which can have the values:
	 *	<type>.size, selectsize, pointersize or lineendsize.
	 */
	cp = strchr(which, '.');
	if (cp) {
		*cp++ = '\0';
		type = ornament_name2type(which);
		if (valid_otype(type) && strcmp(cp, "size") == 0)
		  ornament_sizes[(int) type] =
		    sp->scale(atof(value), 0, INDEX, PPM);
	}
	else if (strcmp(which, "selectsize") == 0) {
		/*
		 * The pointersize was the same as the select size, but
		 * this is no longer the case. However, for old save files
		 * set the pointersize when the select size is set so that
		 * the old save files will work the same way. New save files
		 * will use a separate line (which will appear after the
		 * selectsize line) to set the pointersize properly.
		 */
		*select_size = *pointer_size =
		  sp->scale(atof(value), 0, INDEX, PPM);
	}
	else if (strcmp(which, "pointersize") == 0)
	  *pointer_size = sp->scale(atof(value), 0, INDEX, PPM);
	else if (strcmp(which, "lineendsize") == 0)
	  *line_end_size = sp->scale(atof(value), 0, INDEX, PPM);
}

/*
 * Save ornament sizes.
 */
static void SaveOrnamentSizes(Spectrum *sp, double *ornament_sizes,
			      float select_size, float pointer_size,
			      float line_end_size, FILE *fp)
{
	for (int i = 0; i < ORNAMENT_TYPE_COUNT; i++)
	  fprintf(fp, "ornament.%s.size %f\n",
		  index_name(Ornament_Type_Names, i).cstring(),
		  sp->scale(ornament_sizes[i], 0, PPM, INDEX));
	fprintf(fp, "ornament.selectsize %f\n",
		sp->scale(select_size, 0, PPM, INDEX));
	fprintf(fp, "ornament.pointersize %f\n",
		sp->scale(pointer_size, 0, PPM, INDEX));
	fprintf(fp, "ornament.lineendsize %f\n",
		sp->scale(line_end_size, 0, PPM, INDEX));
}

/*
 * Restore peak pick parameters from this line
 */
static void LoadPeakPick(Peak_Pick_Parameters &pp, char *cp, char **rest)
{
  if (strcmp(cp, "pick") == 0)
    ; // obsolete minPosHt, minPosVol, minNegHt, minNegVol, assignedPeaksOnly
  else if (strcmp(cp, "pick-minimum-linewidth") == 0)
    pp.minimum_linewidth = read_point(*rest, rest);
  else if (strcmp(cp, "pick-minimum-dropoff") == 0)
    pp.minimum_drop_factor = atof(next_token(rest));
}

/*
 * Save peak pick parameters to file.
 */
static void SavePeakPick(Peak_Pick_Parameters &pp, FILE *fp)
{
  double minPosHt = 0, minNegHt = 0;
  double minPosVol = 0, minNegVol = 0;
  bool assignedPeaksOnly = false;
  fprintf(fp, "peak.pick %f %f %f %f %d\n",
	  minPosHt, minPosVol, minNegHt, minNegVol,
	  (int) assignedPeaksOnly);
  fprintf(fp, "peak.pick-minimum-linewidth %s\n",
	  point_format(pp.minimum_linewidth, "%.6f", false));
  fprintf(fp, "peak.pick-minimum-dropoff %.2f\n", pp.minimum_drop_factor);
}

// ----------------------------------------------------------------------------
//

/*
 * When we save the spectrum we actually have to save its state in two
 * places -- the DB and the save file. Eventually this should be changed
 * so that View information goes into the project file and database information
 * goes into the DB and there is no save file. But I can't change things
 * at this state.
 */

bool save_spectrum(Session &s, Spectrum *sp, FILE *fp)
{
		int		failed = false;
		int		i;

		user_save(s, fp);

		fprintf(fp, SAVE_BEGINSPECTRUM);

		/*
		 * Save the <non-default> molecule and condition.
		 */
		Condition	*cp = sp->condition();
		Molecule	*mp = cp ? cp->molecule() : NULL;

		if (mp && !mp->name().is_empty()) {
			fprintf(fp, "molecule %s\n", mp->name().cstring());
		}
		if (cp && !cp->name().is_empty()) {
			fprintf(fp, "condition %s\n", cp->name().cstring());
		}

		fprintf(fp, "name %s\n", sp->name().cstring());
		Stringy datapath =
		  relative_path(sp->data_path(), sp->save_file().directory());
		datapath = to_standard_path_separator(datapath);
		fprintf(fp, "pathname %s\n", datapath.cstring());

		fprintf(fp, "dimension %d\n", sp->dimension());

		fprintf(fp, "shift");
		for (i = 0; i < sp->dimension(); i++)
			fprintf(fp, " %f", sp->ppm_shift()[i]);
		putc('\n', fp);

		fprintf(fp, "points");
		for (i = 0; i < sp->dimension(); i++)
			fprintf(fp, " %d", sp->index_range().size(i));
		fputc('\n', fp);

		if (sp->ppm_sweep_width() != sp->ppm_spectrum_width())
		  {
		    fprintf(fp, "sweepwidth");
		    for (i = 0; i < sp->dimension(); i++)
		      fprintf(fp, " %f", sp->ppm_sweep_width()[i]);
		    fputc('\n', fp);
		  }

		/*
		 * Save the number of peak planes
		 */
		if (sp->dimension() > 2) {
		  View *v = s.project().spectrum_view(sp);
		    fprintf(fp, "extraPeakPlanes");
		    for (i = 0; i < sp->dimension(); i++) {
		      double d = sp->scale(v->visible_depth(i), i, PPM, INDEX);
		      fprintf(fp, " %d", max(0, round(d-1)));
		    }
		    fputc('\n', fp);
		}

		save_guessing_options(sp, fp);

		fprintf(fp, "assignFormat %s\n",
			sp->assignment_format().cstring());

		save_peak_list_options(fp, &sp->mPeakListOptions);

		SaveIntegrate(sp, sp->mIntegrate, fp);
		SavePeakPick(sp->mPeakPick, fp);
		SaveNoise(sp, fp);
		SaveOrnamentSizes(sp, sp->mSizes, sp->mSelectsize,
				  sp->mPointersize, sp->mLineendsize, fp);

		save_attached_data(sp->saved_values(), fp);

		/*
		 * Save all the views
		 */
		List vlist = s.project().view_list(sp);
		for (int vi = 0 ; vi < vlist.size() ; ++vi)
		  {
		    View *v = (View *) vlist[vi];
		    if (v->is_top_level_window())
		      save_view(v, fp);
		  }

		/*
		 * And the ornaments
		 */
		save_ornaments(s, sp, fp);

		fprintf(fp, SAVE_ENDSPECTRUM);

	return !failed;
}

// ----------------------------------------------------------------------------
// Save assignment guessing options.
//
static void save_guessing_options(Spectrum *sp, FILE *fp)
{
  const Guess_Assignments &ga = sp->GetAssignGuessOptions();
  int bits = 0;
  for (int i = 0; i < sp->dimension(); i++) {
    bits = set_bit(bits, i, (ga.sequence_constraint[i]
			     != Guess_Assignments::no_group));
  }
  fprintf(fp, "assignMultiAxisGuess %d\n", bits);
  fprintf(fp, "assignGuessThreshold %f\n", 0.0);
  for (int i = 0; i < sp->dimension(); i++) {
    Stringy sc_name = ga.sequence_constraint[i];
    int sc = name_index(Guess_Assignments::constraint_names, sc_name);
    fprintf(fp, "assignRelation %d %d %s\n",
	    i + 1, (sc >= 0 ? sc : 0), ga.atom_constraint[i].cstring());
  }
  for (int i = 0; i < sp->dimension(); i++) {
    fprintf(fp, "assignRange %d %f\n", i + 1, ga.ppm_range[i]);
  }
}

// ----------------------------------------------------------------------------
// Replace path separator with / when saving paths in files.
// This is so files will be portable between Unix and MS Windows.
//
static Stringy to_standard_path_separator(const Stringy &path)
{
  char sep = path_separator()[0];
  return replace_character(path, sep, '/');
}

// ----------------------------------------------------------------------------
// Replace / with path separator when reading paths in files.
// This is so files will be portable between Unix and MS Windows.
//
static Stringy from_standard_path_separator(const Stringy &path)
{
  char sep = path_separator()[0];
  return replace_character(path, '/', sep);
}

// ----------------------------------------------------------------------------
// If path is base-path/xxx/file and the directory is base-path/yyy then
// return the relative path ../xxx/file.  Otherwise return path.
// Symbolic links are expanded out before the comparisons are made because
// the file reached by the relative ../xxx/file path depends on how the
// directory path is written (ie /symlink/../xxx/file is different from
// /somewhere/else/../xxx/file even when /symlink points to /somewhere/else)
//
// This routine is used so that your Sparky directory containing Save file,
// Project file, and spectra subdirectories can be moved without fixing
// absolute paths in the files.
//
static Stringy relative_path(const Stringy &path, const Stringy &directory)
{
  Stringy p = resolve_links(path);
  Stringy d = resolve_links(directory);
  Stringy s = file_directory(d);
  if (s.length() < p.length() && s == p.substring(0, s.length()))
    return "../" + p.substring(s.length() + 1);
  return path;
}

// ----------------------------------------------------------------------------
//
static Stringy absolute_path(const Stringy &path, const Stringy &directory)
{
  if (is_absolute_path(path))
    return dedotify_path(path);

  //
  // Resolve relative paths relative to the given directory.
  // For backwards compatibility this currently only applies to paths
  // starting with '..'.  Other relative paths are resolved relative
  // to the current working directory.  Ick.
  //
  Stringy d = ((path.length() >= 2 && path[0] == '.' && path[1] == '.') ?
	       resolve_links(directory) :
	       current_directory());

  return dedotify_path(file_path(d, path));
}

/*
 * Construct the spectrum name by stripping any path components
 * and common extensions.
 */
static Stringy spectrum_name_from_data_path(const Stringy &path)
{
	Stringy file = file_name(path);
	const char *first = file.cstring();
	const char *last = first + strlen(first);

	/*
	 * Strip off common extensions
	 */
	const char *cp = strrchr(first, '.');
	if (cp) {
		if (strcmp(cp, ".w1") == 0
		||  strcmp(cp, ".w2") == 0
		||  strcmp(cp, ".w3") == 0
		||  strcmp(cp, ".ucsf") == 0
		||  strcmp(cp, ".trim") == 0
		||  strcmp(cp, ".dat") == 0)
			last = cp;
	}

	return substring(first, last);
}

// ----------------------------------------------------------------------------
//
static Spectrum *open_new_spectrum(Session &s, const Stringy &data_path,
				   Stringy *error_msg)
{
  Stringy savefile_path = "";
  Stringy spectrum_name = spectrum_name_from_data_path(data_path);
  Project &proj = s.project();
  Spectrum *sp = NULL;
  NMR_Data *nmr_data = open_nmr_data(s, data_path);

  if (nmr_data == NULL)
    {
      *error_msg = formatted_string("Can't read data file %s\n",
				    data_path.cstring());
      return NULL;
    }

  Condition *c = proj.default_condition();
  sp = new Spectrum(s, spectrum_name, c, savefile_path, nmr_data);

  View_Settings settings(sp);
  (void) new View(s, NULL, sp, settings);

  return sp;
}

// ----------------------------------------------------------------------------
//
static NMR_Data *open_nmr_data(Session &s, const Stringy &data_path)
{
  Stringy path = data_path;

  Memory_Cache *mcache = s.project().memory_cache();
  NMR_Data *nmr_data = read_nmr_data(path, mcache);

  while (nmr_data == NULL)
    {
      path = request_new_data_location(s, path);
      if (path.is_empty())
	break;
      else
	nmr_data = read_nmr_data(path, mcache);
    }

  return nmr_data;
}

// ----------------------------------------------------------------------------
//
static bool is_nmr_data(const Stringy &path)
{
  return (is_ucsf_nmr_data(path) ||
	  is_felix_nmr_data(path) ||
	  is_bruker_nmr_data(path));
}

// ----------------------------------------------------------------------------
//
static NMR_Data *read_nmr_data(const Stringy &path, Memory_Cache *mcache)
{
  Stringy error_msg;
  if (is_ucsf_nmr_data(path))
    return ucsf_nmr_data(path, mcache);
  else if (is_felix_nmr_data(path))
    return felix_nmr_data(path, mcache);
  else if (is_bruker_nmr_data(path))
    return bruker_nmr_data(path, &error_msg, mcache);

  return NULL;
}

// ----------------------------------------------------------------------------
//
static void read_view_geometry(FILE *fp, Spectrum *sp,
				 IPoint *axis_order, int *x, int *y,
				 int *xsize, int *ysize,
				 SPoint *center, SPoint *pixel_size,
				 SPoint *depth)
{
	char		buf[MAX_LINE_LENGTH], *cp;
	double		zoom = 1;
	int		dim = sp->dimension();
	SPoint		offsets(dim), scales(dim);
	int		swapxy = fix_1_6; // && view->axis(X) != W1;

	while (fgets(buf, sizeof buf, fp)) {
		if (strcmp(buf, SAVE_ENDPARAMS) == 0)
			break;
		char *rest = buf;
		cp = next_token(&rest);
		if (strcmp(cp, "orientation") == 0) {
			int za = 2;
			for (int i = 0; i < dim; i++) {
				Axis a = (Axis) next_int(&rest);
				if (a == X) (*axis_order)[0] = i;
				else if (a == Y) (*axis_order)[1] = i;
				else (*axis_order)[za++] = i;
			}
		}
		else if (strcmp(cp, "location") == 0) {
			*x = next_int(&rest);		// Read x
			*y = next_int(&rest);		// Read y
		}
		else if (strcmp(cp, "size") == 0) {
			*xsize = next_int(&rest);	// Read width
			*ysize = next_int(&rest);	// Read height
		}

		/*
		 * Since the offset is measured in data points, it will have
		 * to be adjusted if size of the spectrum (when saved) differs
		 * from the size when restored (because of reprocessing)
		 */
		else if (strcmp(cp, "offset") == 0) {
			for (int i = 0; i < dim; i++)
				offsets[i] = atof(next_token(&rest));
		}

		else if (strcmp(cp, "zoom") == 0) {
			zoom = atof(next_token(&rest));

			if (swapxy)
				zoom = atof(next_token(&rest));
		}

		/*
		 * Since the scale is a mapping of data points to screen
		 * units, it will have to be adjusted if size of the spectrum
		 * (when saved) differs from the size when restored (because
		 * of reprocessing)
		 */
		else if (strcmp(cp, "scale") == 0) {
			for (int i = 0; i < dim; i++)
				scales[i] = atof(next_token(&rest));
		}
		else if (strcmp(cp, "flags") == 0) {
			(void) next_int(&rest);	// Read transpose flag
		}
	}

  for (int a = 0 ; a < dim ; ++a)
    {
      (*pixel_size)[a] = 1 / (zoom * scales[a]);
      (*depth)[a] = 1.0;
    }

  int xaxis = (*axis_order)[0];
  int yaxis = (*axis_order)[1];

  (*center)[xaxis] = offsets[xaxis] + *xsize * (*pixel_size)[xaxis] / 2;
  (*center)[yaxis] = offsets[yaxis] + *ysize * (*pixel_size)[yaxis] / 2;

  for (int a = 0 ; a < dim ; ++a)
    if (a != xaxis && a != yaxis)
	(*center)[a] = offsets[a];

  *center = sp->map(*center, INDEX, PPM);
  *pixel_size = sp->scale(*pixel_size, INDEX, PPM);
  *depth = sp->scale(*depth, INDEX, PPM);
}

// ----------------------------------------------------------------------------
//
View_Parameters::View_Parameters(Spectrum *sp, const View_Settings &s)
  : settings(s)
{
  this->sp = sp;
}

// ----------------------------------------------------------------------------
//
View *View_Parameters::create_view(Session &s)
{
  return new View(s, NULL, sp, settings);
}

/* 
 *	Restore the view from file <fp>
 */
static View_Parameters *read_view_parameters(Spectrum *sp, FILE *fp)
{
  char	buf[MAX_LINE_LENGTH], *cp;

  View_Settings s(sp);
  s.show_crosshair = false;
  s.show_crosshair_from_other_views = false;

  while (fgets(buf, sizeof buf, fp))
    {
      if (strcmp(buf, SAVE_ENDVIEW) == 0)
	return new View_Parameters(sp, s);

      /*
       * The parameters are stored as the home params, last params,
       * and current params. To restore these from file we copy copy
       * the last params into home params and the current params
       * into last params, then restore the current params. These
       * operations restore all the parameters properly.
       */
      if (strcmp(buf, SAVE_BEGINPARAMS) == 0)
	{
	  read_view_geometry(fp, sp,
			     &s.axis_order, &s.x, &s.y, &s.width, &s.height,
			     &s.center, &s.pixel_size, &s.visible_depth);
	  continue;
	}
      char *rest = buf;
      cp = next_token(&rest);

      /*
       * The view name
       */
      if (strcmp(cp, "name") == 0)
	s.view_name = next_line(&rest);

      /*
       * The view number -- we set it from the name
       */
      else if (strcmp(cp, "number") == 0)
	; // Obsolete

      /*
       * The precision for display along this axis
       */
      else if (strcmp(cp, "precision") == 0)
	(void) next_int(&rest);	// Read precision.

      /*
       * The precision for display along this axis, by axis type
       */
      else if (strcmp(cp, "precision_by_type") == 0)
	{
	  (void) next_int(&rest);	// Read data precision
	  (void) next_int(&rest);	// Read ppm precision
	  (void) next_int(&rest);	// Read hz precision
	}

      /*
       * The method of data display
       */
      else if (strcmp(cp, "viewmode") == 0)
	next_int(&rest);		// Obsolete

      /*
       * The "show ornament" states
       */
      else if (strcmp(cp, "show") == 0)
	{
	  s.show_ornaments = next_int(&rest);

	  int i;
	  while ((cp = next_token(&rest)))
	    if ((i = ornament_name2type(cp)) != -1)
	      s.show_ornament((Ornament_Type) i, true);
	}

      /*
       * The axis units (ppm, data, HZ)
       */
      else if (strcmp(cp, "axistype") == 0)
	s.scale_units = (Units) next_int(&rest);

      /*
       * The icon's location
       */
      else if (strcmp(cp, "iconlocation") == 0)
	{
	  (void) next_int(&rest);	// Obsolete iconloc[0]
	  (void) next_int(&rest);	// Obsolete iconloc[1]
	}

      /*
       * The contour parameters
       */
      else if (strncmp(cp, "contour.", 8) == 0)
	read_contour_parameters(cp + 8, next_line(&rest),
				&s.pos_contours, &s.neg_contours);

      /*
       * The other named states
       */
      else if (strcmp(cp, "flags") == 0)
	{
	  while ((cp = next_token(&rest)))
	    {
	      if (strcmp(cp, "closed") == 0)
		; // Obsolete

	      /*
	       * When created, hidden views start full
	       * size, just like closed views.
	       */
	      else if (strcmp(cp, "hidden") == 0)
		s.show_view = false;
	      else if (strcmp(cp, "noscales") == 0)
		s.show_scales = false;
	      else if (strcmp(cp, "noscrollbars") == 0)
		s.show_scrollbars = false;
	      else if (strcmp(cp, "slices") == 0)
		s.show_slices = true;
	      else if (strcmp(cp, "fixedslices") == 0)
		s.slice_auto_scale = false;
	      else if (strcmp(cp, "slicesubtract") == 0)
		s.slice_subtract_peaks = true;
	      else if (strcmp(cp, "resonance") == 0)
		s.show_resonance_panels = true;
	      else if (strcmp(cp, "resfilter") == 0)
		s.filter_resonances = true;
	      else if (strcmp(cp, "peakinfo") == 0)
		s.show_peak_info = true;
	      else if (strcmp(cp, "contourscale") == 0)
		s.show_contour_scale = true;
	      else if (strcmp(cp, "nocursor") == 0)
		; // Obsolete
	      else if (strcmp(cp, "crosshair") == 0)
		s.show_crosshair = true;
	      else if (strcmp(cp, "crosshair2") == 0)
		s.show_crosshair_from_other_views = true;
	      else if (strcmp(cp, "crosshairx") == 0)
		s.show_transpose_crosshair = true;
	      else if (strcmp(cp, "atomaxes") == 0)
		s.show_nucleus_names = true;
	      else if (strcmp(cp, "subtractpeaks") == 0)
		s.subtract_fit_peaks = true;
	    }
	}
    }

  return NULL;
}

// ----------------------------------------------------------------------------
//
static bool merge_db_data(const Stringy &db_path,
			  int mol_id, int cond_id, Condition *c)
{
  if (db_path.is_empty() || mol_id != NO_ID || cond_id != NO_ID)
    return false;

  return (merge_molecule_db(db_path, mol_id, c->molecule()) &&
	  merge_condition_db(db_path, mol_id, cond_id, c));
}

// ----------------------------------------------------------------------------
//
static bool merge_molecule_db(const Stringy db_path, int mol_id, Molecule *m)
{
  Stringy db_dir = file_path(db_path, formatted_string("%d", mol_id));
  Stringy db_file = file_path(db_dir, DBFILENAME);

  return merge_groups_atoms_sequence(db_file, m);
}

// ----------------------------------------------------------------------------
//
static bool merge_groups_atoms_sequence(const Stringy &db_file, Molecule *m)
{
  FILE *fp = fopen(db_file.cstring(), "r");

  if (fp == NULL)
    return false;

  char buf[MAX_LINE_LENGTH];
  while (fgets(buf, sizeof buf, fp))
    {
      char	*rest = buf;
      char	*cp = next_token(&rest);

      if (cp == NULL)
	continue;

      if (strcmp(cp, "<seq>") == 0) {
	List seq;

	while (fgets(buf, sizeof buf, fp))
	  {
	    char *rest = buf;
	    cp = next_line(&rest);

	    if (cp == NULL)	// Blank line is a break
	      seq.append(NULL);
	    else if (strcmp(cp, "</seq>") == 0)
	      break;
	    else
	      seq.append(m->define_group(cp));
	  }
      }

      else if (strcmp(cp, "<groups>") == 0) {
	while (fgets(buf, sizeof buf, fp))
	  {
	    int	idThis, idNext;
	    char	nmThis[ANAMESIZE], nmNext[ANAMESIZE];

	    char *rest = buf;
	    cp = next_line(&rest);
	    if (cp == NULL)
	      continue;
	    if (strcmp(cp, "</groups>") == 0)
	      break;
	    /*
	     * Create groups for:
	     *	<id> <name> -or-
	     *	<id> <name> <idNext> <nameNext>
	     */
	    switch (sscanf(cp, "%d %s %d %s",
			   &idThis, nmThis, &idNext, nmNext)) {
	    case 2:
	    case 4:
	      m->define_group(nmThis);
	      break;
	    }
	  }
      }

      else if (strcmp(cp, "<atoms>") == 0) {
	//
	// Skip over obsolete atom specifications
	//
	while (fgets(buf, sizeof buf, fp))
	  if (strcmp(buf, "</atoms>") == 0)
	    break;
      }
    }

  fclose(fp);

  return true;
}

// ----------------------------------------------------------------------------
//
static bool merge_condition_db(const Stringy &db_path, int mol_id, int cond_id,
			       Condition *c)
{
  Stringy db_dir = file_path(db_path, formatted_string("%d", mol_id));
  db_dir = file_path(db_dir, formatted_string("%d", cond_id));
  Stringy db_file = file_path(db_dir, DBFILENAME);
  
  return merge_resonances(db_file, c);
}

// ----------------------------------------------------------------------------
//
static bool merge_resonances(const Stringy &db_file, Condition *c)
{
  FILE *fp = fopen(db_file.cstring(), "r");

  if (fp == NULL)
    return false;

  char buf[MAX_LINE_LENGTH];
  while (fgets(buf, sizeof buf, fp))
    {
      char	*rest = buf;
      char	*cp = next_token(&rest);

      if (cp && strcmp(cp, "<resonances>") == 0) {

	while (fgets(buf, sizeof buf, fp))
	  {
	    char	*rest = buf;
	    cp = next_line(&rest);
	    if (strcmp(cp, "</resonances>") == 0)
	      break;

	    /*
	     * Expect:
	     *	<id> <name> <frequency> [*]
	     * We ignore the name but use the <frequency>
	     * to set the initial, user-set frequency
	     * This is used as the resonance frequency if
	     * the resonance has no assignmented peaks.
	     * The trailing '*', if present, is used to
	     * show that the resonance has not peaks but
	     * is only used for output purpose.
	     */

	    int id;
	    char rname[ANAMESIZE];
	    double freq_ppm;
	    Stringy group, atom;
	    if (sscanf(cp, "%d %s %lf", &id, rname, &freq_ppm) == 3 &&
		parse_group_atom(rname, &group, &atom, NULL))
	      {
		Resonance *rp = c->define_resonance(group, atom,
						    guess_nucleus(atom));
		if (rp)
		  rp->set_frequency(freq_ppm);
	      }
	}
      }
    }
  fclose(fp);

  return true;
}

// ----------------------------------------------------------------------------
//
static bool db_stuff_from_save_file(SaveFile &sf, Stringy *db_path,
				    int *molecule_id, Stringy *molecule_name,
				    int *condition_id, Stringy *condition_name,
				    int *spectrum_id, Stringy *spectrum_name,
				    Stringy *data_path, int *dim)
{
  *db_path = "";
  *molecule_id = NO_ID;
  *molecule_name = "";
  *condition_id = NO_ID;
  *condition_name = "";
  *spectrum_id = NO_ID;
  *spectrum_name = "";
  *data_path = "";
  *dim = 0;

  FILE *fp = sf.OpenForReading();
  if (fp == NULL)
    return false;

  char buf[MAX_LINE_LENGTH];
  while (fgets(buf, sizeof buf, fp))
    {
      char *rest = buf;
      char *tag = next_token(&rest);
      if (strcmp(tag, "dbdir") == 0)
	*db_path = trim_white(rest);
      else if (strcmp(tag, "molecule_id") == 0)
	*molecule_id = next_int(&rest);
      else if (strcmp(tag, "molecule") == 0)
	*molecule_name = next_line(&rest);
      else if (strcmp(tag, "condition_id") == 0)
	*condition_id = next_int(&rest);
      else if (strcmp(tag, "condition") == 0)
	*condition_name = next_line(&rest);
      else if (strcmp(tag, "id") == 0)
	*spectrum_id = next_int(&rest);
      else if (strcmp(tag, "name") == 0)
	*spectrum_name = next_line(&rest);
      else if (strcmp(tag, "pathname") == 0)
	{
	  Stringy path = trim_white(rest);
	  path = from_standard_path_separator(path);
	  *data_path = absolute_path(path, sf.directory());
	}
      else if (strcmp(tag, "dimension") == 0)
	{
	  *dim = next_int(&rest);
	  break;		// dimension appears last in file
				// must break because other id lines follow.
	}
    }
  sf.EndRead(fp);

  return true;
}

/*
 * Save the user preferences to Spectrum save file opened as <fp>.
 */
static void user_save(Session &s, FILE *fp)
{
  Project &proj = s.project();
  Preferences &pref = proj.preferences;

	fprintf(fp, SAVE_BEGINUSER);
	mode_save(s, fp);
	fprintf(fp, "set saveprompt %d\n", pref.prompt_before_overwrite);
	fprintf(fp, "set saveinterval %d\n", pref.auto_backup_interval);
	fprintf(fp, "set resizeViews %d\n", pref.resize_views);
	fprintf(fp, "set keytimeout %d\n", pref.key_timeout_interval);
	fprintf(fp, "set cachesize %d\n", pref.memory_cache_size);
	fprintf(fp, "set contourgraying %d\n", pref.contour_graying);

	save_regions(proj.view_regions.region_list(), fp);
	print_save(proj.print_options, fp);
	fprintf(fp, SAVE_ENDUSER);
}

/*
 * Restore the user preferences from Spectrum save file opened as <fp>.
 *
 */
static void user_restore(Session &s, FILE *fp)
{
	char		buf[MAX_LINE_LENGTH];

	/*
	 * Since user preferences are stored in Sparky command-language
	 * format they can be restored simply by parsing the commands
	 * until the end of the user section is reached.
	 */
	while (fgets(buf, sizeof buf, fp)) {
		if (strcmp(buf, SAVE_ENDUSER) == 0)
			break;
		(void) parse_line(s, buf);
	}
}

/*
 * Save-to-file preferences are set by the following .sparky-init
 * or savefile commands:
 *
 *	set saveprompt		0 | 1
 *	set saveinterval	0 | time_in_seconds
 *
 * The defaults are "0" (no prompt if overwriting a savefile) and "0"
 * (no autosave).
 *
 * These preferences are also recorded in the savefile.
 */
static bool user_saveauto_set(Session &s, char *str)
  { s.project().set_auto_backup(atoi(str)); return true; }
static bool user_saveprompt_set(Session &s, char *str)
  { s.project().preferences.prompt_before_overwrite = YorN(str); return true; }
static bool user_resize_views_set(Session &s, char *str)
  { s.project().preferences.resize_views = YorN(str); return true; }
static bool user_keytimeout_set(Session &s, char *str)
  { s.project().preferences.key_timeout_interval = atoi(str); return true; }
static bool user_cachesize_set(Session &s, char *str)
  { s.project().preferences.memory_cache_size = atoi(str); return true; }
static bool user_contour_graying_set(Session &s, char *str)
  { s.project().preferences.contour_graying = YorN(str); return true; }

/*
 * Restore a view set from file <fp>
 */
static void set_restore(FILE *fp)
{
	char	buf[MAX_LINE_LENGTH];

	while (fgets(buf, sizeof buf, fp)) {
		if (strcmp(buf, SAVE_ENDSET) == 0
		||  strcmp(buf, "<end strips>\n") == 0)
			return;
	}
}

// ----------------------------------------------------------------------------
//
static void save_regions(const List &rlist, FILE *fp)
{
  for (int ri = 0 ; ri < rlist.size() ; ++ri)
    {
      View_Region *vr = (View_Region *) rlist[ri];
      fprintf(fp, "define region ");
      fputstring(fp, vr->name.cstring());
      fprintf(fp, " ");
      fputstring(fp, vr->accelerator.cstring());
      fprintf(fp, " %s", point_format(vr->ppm_region.max, "%.3f", true));
      fprintf(fp, " %s", point_format(vr->ppm_region.min, "%.3f", true));
      fprintf(fp, " %s\n", "contour");
    }
}

// ----------------------------------------------------------------------------
// Define a region from string.
//
#define REGION_ACCELERATOR_SIZE		2
static bool read_region(Session &s, char *cp)
{
  char		*cp1, *name;
  char		accel[REGION_ACCELERATOR_SIZE + 1];

  cp1 = sgetstring(&cp);
  if (cp1) {
    name = cp1;
    cp1 = sgetstring(&cp);
    if (cp1) {
      strncpy_terminate(accel, cp1, sizeof accel);

      /*
       * Old-style define had dfX, ufX, dfY, ufY. New
       * style name has [ dfW1 dfW2 ... ] [ ufW1 ufW2 ... ]
       */
      Project &proj = s.project();
      cp = skip_white(cp);
      if (cp[0] == '[') {
	SPoint ppm_max = read_point(cp, &cp);
	SPoint ppm_min = read_point(cp, &cp);
	SRegion ppm_region(ppm_min, ppm_max);
	View_Region vr(name, accel, ppm_region);
	proj.view_regions.add_region(vr);
      }
      else {
	double uf[DIM], df[DIM];
	sscanf(cp, "%lf %lf %lf %lf",
	       &df[0], &uf[0], &df[1], &uf[1]);
	SPoint ppm_max(2, df);
	SPoint ppm_min(2, uf);
	SRegion ppm_region(ppm_min, ppm_max);
	View_Region vr(name, accel, ppm_region);
	proj.view_regions.add_region(vr);
      }
      return true;
    }
  }
  return false;
}


//
// Save the print defaults to the sparky save file.
//

#define OPTIONS_PORTRAIT	0x1	// portrait mode
#define OPTIONS_BANNER		0x2	// if a title is printed
#define OPTIONS_GRIDS		0x4	// use grids instead of ticks
#define OPTIONS_PEAKGROUPS	0x8	// print peakgroups
#define OPTIONS_PEAKS		0x10	// print peaks
#define OPTIONS_FILLPOS		0x20	// fill positive contour levels
#define OPTIONS_FILLNEG		0x40	// fill negative contour levels
#define OPTIONS_BLACKLABELS	0x80	// black labels with clear background
#define OPTIONS_FIXEDSCALE	0x100	// fixed value scale
#define OPTIONS_CROSSHAIR	0x200	// print current crosshair location

static void print_save(const Print_Options &op, FILE *fp)
{
	fprintf(fp, "default print command %s\n", op.print_command.cstring());
	if (op.print_to_file)
	  fprintf(fp, "default print file %s\n", op.print_file.cstring());

	unsigned int options = 0;
	if (!op.landscape)	options |= OPTIONS_PORTRAIT; 
	if (op.show_banner)	options |= OPTIONS_BANNER;
	if (op.show_grids)	options |= OPTIONS_GRIDS;
	if (op.show_peakgroups) options |= OPTIONS_PEAKGROUPS;
	if (op.show_peaks)	options |= OPTIONS_PEAKS;
	if (op.black_labels)	options |= OPTIONS_BLACKLABELS;
	if (op.fixed_scale)	options |= OPTIONS_FIXEDSCALE;

	fprintf(fp, "default print options %d %d %f %f\n",
		PPM, options, op.fixed_scales[0], op.fixed_scales[1]);
	if (!op.title.is_empty())
		fprintf(fp, "default print title %s\n", op.title.cstring());
}

// ----------------------------------------------------------------------------
// Restore view axis synchronizations.
//
static bool restore_view_axis_syncs(Session &s, FILE *fp)
{
	char	buf[MAX_LINE_LENGTH];

	Project &proj = s.project();
	while (fgets(buf, sizeof buf, fp)) {

		if (strcmp(buf, SAVE_ENDSYNCAXES) == 0)
			return true;

		char *rest = buf;
		char *v1_name = next_token(&rest);
		int v1_axis = next_int(&rest);
		char *v2_name = next_token(&rest);
		int v2_axis = next_int(&rest);
		if (v1_name && v2_name) {
		  View *v1 = proj.find_view(v1_name);
		  View *v2 = proj.find_view(v2_name);
		  if (v1 && v2)
		    proj.synchronize_view_axes(view_axis(v1, v1_axis),
						view_axis(v2, v2_axis));

		  /*
		   * Otherwise forget this synchronization.
		   */
		  else
		    {
		      Reporter &rr = s.reporter();
		      rr.message("Ignoring view sync %s %d %s %d\n",
				 v1_name, v1_axis, v2_name, v2_axis);
		    }
		}
	}

	return false;
}

// ----------------------------------------------------------------------------
// Save view axis synchronizations.
//
static void save_view_axis_syncs(Project &proj, FILE *fp)
{
  bool didone = false;

  const List &vasets = proj.synchronized_view_axis_sets();
  for (int vas = 0 ; vas < vasets.size() ; ++vas)
    {
      List *valist = (List *) vasets[vas];

      if (valist->size() >= 2)
	{
	  view_axis *va1 = (view_axis *) (*valist)[0];
	  for (int k = 1 ; k < valist->size() ; ++k)
	    {
	      view_axis *va2 = (view_axis *) (*valist)[k];
	      if (!didone)
		{
		  fprintf(fp, SAVE_BEGINSYNCAXES);
		  didone = true;
		}
	      fprintf(fp, "%s %d %s %d\n",
		      va1->view()->name().cstring(), va1->axis(),
		      va2->view()->name().cstring(), va2->axis());
	    }
	}
    }

  if (didone)
    fprintf(fp, SAVE_ENDSYNCAXES);
}

// ----------------------------------------------------------------------------
//
static void save_view_overlays(Project &proj, FILE *fp)
{
  fprintf(fp, SAVE_BEGINOVERLAYS);
  List vlist = proj.view_list();
  for (int vi = 0 ; vi < vlist.size() ; ++vi)
    {
      View *onto = (View *) vlist[vi];
      List froms = proj.overlaid_views(onto);
      for (int fvi = 0 ; fvi < froms.size() ; ++fvi)
	fprintf(fp, "overlay %s %s\n",
		((View *) froms[fvi])->name().cstring(),
		onto->name().cstring());
    }
  fprintf(fp, SAVE_ENDOVERLAYS);
}

// ----------------------------------------------------------------------------
//
static bool restore_view_overlays(Project &proj, FILE *fp)
{
  Stringy line, rest;

  while (get_line(fp, &line))
    if (line == SAVE_ENDOVERLAYS)
      return true;
    else if (line_tag(line, &rest) == "overlay")
      {
	View *from = proj.find_view(first_token(rest, &rest));
	View *onto = proj.find_view(first_token(rest, &rest));
	if (from && onto)
	  proj.add_overlay(from, onto);
      }

  return false;
}

/*
 * Restore the project from project file <nm>
 */
static bool restore_project(Session &s, const Stringy &path,
			    Stringy *error_msg)
{
  Project &proj = s.project();
  Stringy old_path = proj.save_file().path();
  proj.save_file().set_path(path);
  FILE *fp = proj.save_file().OpenForReading();
  if (fp == NULL)
    {
      proj.save_file().set_path(old_path);
      *error_msg = formatted_string("Could not open project file\n"
				    "  %s\n", path.cstring());
      return false;
    }

  //
  // Ask about unloading current project if it has not been saved.
  //
  if (! proj.unload_query())
    {
      proj.save_file().EndRead(fp);
      proj.save_file().set_path(old_path);
      return false;
    }

  proj.save_file().set_path(path);	// Unload clears path, so set it again

  Reporter &rr = s.reporter();
  rr.message("\nLoad project %s\n", proj.save_file().ShortPath().cstring());

  //
  // First read, then restore any spectrum and save files
  //
  bool cancelled = false;
  List savefiles;
  bool success = read_project_entries(proj.save_file(), &savefiles);
  if (success)
    {
      List views_to_show;
      for (int sf = 0 ; sf < savefiles.size() ; ++sf)
	{
	  Stringy *savefile = (Stringy *) savefiles[sf];
	  Stringy error;
	  Spectrum *sp = restore_savefile(s, *savefile, views_to_show, &error);
	  if (sp == NULL &&
	      query(s, error + "\nStop project load?",
		    "Yes, stop load", "No, continue loading") == 1)
	    {
	      success = false;
	      cancelled = true;
	      break;
	    }
	}
      create_views(s, views_to_show);
      free_string_list_entries(savefiles);
    }

  if (success && !cancelled) {

    char	buf[MAX_LINE_LENGTH];
    while (fgets(buf, sizeof buf, fp))
      {
	if (isemptyline(buf))
	  continue;

	//
	// Read options
	//
	if (strcmp(buf, "<options>\n") == 0) {
	  if (! restore_project_options(proj, fp)) {
	    success = false;
	  }
	}

	//
	// Then inter-spectrum axis maps
	//
	else if (strcmp(buf, SAVE_BEGINAXISMAP) == 0) {
	  if (! skip_past_line(fp, SAVE_ENDAXISMAP))
	    success = false;
	}

	//
	// Also restore any broadening.
	//
	else if (strcmp(buf, "<broadening>\n") == 0) {
	  if (! restore_broadening_options(proj, fp))
	    success = false;
	}

	//
	// Then inter-spectrum synchronizations
	//
	else if (strcmp(buf, SAVE_BEGINSYNC) == 0) {
	  if (!skip_past_line(fp, SAVE_ENDSYNC))
	    success = false;
	}

	//
	// Then view axis synchronizations
	//
	else if (strcmp(buf, SAVE_BEGINSYNCAXES) == 0) {
	  if (! restore_view_axis_syncs(s, fp))
	    success = false;
	}

	else if (strcmp(buf, SAVE_BEGINOVERLAYS) == 0) {
	  if (! restore_view_overlays(proj, fp))
	    success = false;
	}

	else if (strcmp(buf, SAVE_BEGINATTACHEDDATA) == 0) {
	  if (! restore_attached_data(proj.saved_values(), fp))
	    success = false;
	}

	else if (strcmp(buf, SAVE_BEGINMOLECULE) == 0) {
	  if (! restore_molecule(proj, fp))
	    success = false;
	}
      }
  }

  proj.save_file().EndRead(fp);
  proj.save_file().NeedsSaving(false);

  const char *status = (success ? "complete" :
			(cancelled ? "stopped" : "failed"));
  rr.message("Project load %s\n", status);

  return success;
}

// ----------------------------------------------------------------------------
//
static void create_views(Session &s, List &view_params)
{
  for (int vi = 0 ; vi < view_params.size() ; ++vi)
    {
      View_Parameters *v = (View_Parameters *) view_params[vi];
      v->create_view(s);
      delete v;
    }
  view_params.erase();
}

/*
 * Read the project entries of the project file named <projFile>.
 *
 */
static bool read_project_entries(SaveFile &sf, List *savefiles)
{
	FILE *fp = sf.OpenForReading();
	if (fp) {
		int	inSaveFiles = false;
		int	inSpectra = false;

		char	buf[MAX_LINE_LENGTH];
		while (fgets(buf, sizeof buf, fp)) {
			if (strncmp(buf, "<DB", 3) == 0)
			  ; // Obsolete database path
			else if (strcmp(buf, "<spectra>\n") == 0)
				inSpectra = true;
			else if (strcmp(buf, "<savefiles>\n") == 0)
				inSaveFiles = true;
			else if (inSpectra) {
				if (! strcmp(buf, "<end spectra>\n")) {
					inSpectra = false;
					continue;
				}
				//
				// Don't read database entries from project.
				//
			}
			else if (inSaveFiles) {
				if (! strcmp(buf, "<end savefiles>\n"))
					break;
				char *rest = buf;
				Stringy path = trim_white(rest);
				path = from_standard_path_separator(path);
				path = absolute_path(path, sf.directory());
				savefiles->append(new Stringy(path));
			}
		}
		sf.EndRead(fp);

		return true;
	}

	free_string_list_entries(*savefiles);

	return false;
}


/*
 * Restore the Project options from file pointer <fp>, returning true
 * it the options are restored properly.
 */
static bool restore_project_options(Project &proj, FILE *fp)
{
	bool status = true;
	char buf[MAX_LINE_LENGTH], *cp;

	while (fgets(buf, sizeof buf, fp)) {
		if (strcmp(buf, "<end options>\n") == 0)
			return true;

		char *rest = buf;
		if ((cp = next_token(&rest)) == NULL)
			continue;

		if (strcmp(cp, "assignmentCopyFrom") == 0) {
		  if ((cp = next_line(&rest))) {
		    Spectrum *sp = proj.find_spectrum(cp);
		    proj.set_assignment_copy_from_spectrum(sp);
		  }
		}

		/*
		 * v3.00 style, long spectrum names
		 */
		else if (strcmp(cp, "broadeningFrom") == 0) {
			if ((cp = next_line(&rest)))
			  ;  // Obsolete from spectrum name.
		}
		else if (strcmp(cp, "broadeningTo") == 0) {
			if ((cp = next_line(&rest)))
			  ;  // Obsolete to spectrum name.
		}

		/*
		 * v2.06 style, short spectrum names
		 */
		else if (strcmp(cp, "broadeningCalc") == 0) {
			if ((cp = next_token(&rest)))
			  ;  // Obsolete from spectrum name.
			if ((cp = next_line(&rest)))
			  ;  // Obsolete to spectrum name.
		}
	}

	return status;
}

/*
 * Restore the Project broadening information from file pointer <fp>
 */
static bool restore_broadening_options(Project &, FILE *fp)
{
	char	buf[MAX_LINE_LENGTH];

	/*
	 * The first line is <from> fromSpectrum <to> toSpectrum.
	 */
	while (fgets(buf, sizeof buf, fp)) {
		if (isemptyline(buf))
			continue;

		if (strcmp(buf, "<end broadening>") == 0)
			return true;

		// Obsolete broadening set data.
	}
	return false;
}

// ----------------------------------------------------------------------------
//
static bool is_project_file(const Stringy &path)
{
  return check_file_type(path, SAVEFILE_PROJECT);
}

// ----------------------------------------------------------------------------
//
bool save_project(Session &s, bool backup)
{
  Project &proj = s.project();
  save_spectra(s, proj.spectrum_list(), backup);

  if (proj.save_file().path().is_empty())
    return false;

  if (!backup)
    proj.save_file().NeedsSaving(true);	// Force a project save.

  FILE *fp = proj.save_file().OpenForSaving(backup);
  if (fp == NULL)
    return false;

  bool success = save_project(s, fp);
  success = proj.save_file().EndSave(fp, backup, success);

  if (!success)
    s.reporter().warning("Project %s failed.", backup ? "backup" : "save");

  return success;
}

// ----------------------------------------------------------------------------
//
static bool save_project(Session &s, FILE *fp)
{
	int		failed = false;

	/*
	 * Write out *all* spectrum save files. We do this so
	 * that sparky can read this project file.
	 */

	Project &proj = s.project();
	fprintf(fp, "<savefiles>\n");
	const List &slist = proj.spectrum_list();
	for (int si = 0 ; si < slist.size() ; ++si) {
	  Spectrum *sp = (Spectrum *) slist[si];
	  Stringy path = relative_path(sp->save_file().path(),
				       proj.save_file().directory());
	  path = to_standard_path_separator(path);
	  fprintf(fp, "%s\n", path.cstring());
	}
	fprintf(fp, "<end savefiles>\n");

	save_project_options(proj, fp);
	save_view_axis_syncs(proj, fp);
	save_view_overlays(proj, fp);
	save_attached_data(proj.saved_values(), fp);
	save_molecules(proj, fp);

	return !failed;
}

/*
 * Save the Project options to file pointer <fp>
 */
static void save_project_options(Project &proj, FILE *fp)
{
  fputs("<options>\n", fp);

  if (proj.assignment_copy_from_spectrum())
    fprintf(fp, "assignmentCopyFrom %s\n",
	    proj.assignment_copy_from_spectrum()->fullname().cstring());

  fputs("<end options>\n", fp);
}

// ----------------------------------------------------------------------------
//
static void save_molecules(Project &proj, FILE *fp)
{
  List mlist = proj.molecule_list();
  for (int mi = 0 ; mi < mlist.size() ; ++mi)
    {
      Molecule *m = (Molecule *) mlist[mi];
      if (some_spectrum_uses_molecule(m, proj.spectrum_list()))
	{
	  fprintf(fp, SAVE_BEGINMOLECULE);
	  fprintf(fp, "name %s\n", m->name().cstring());
	  save_attached_data(m->saved_values(), fp);
	  save_conditions(proj, m, fp);
	  fprintf(fp, SAVE_ENDMOLECULE);
	}
    }
}

// ----------------------------------------------------------------------------
//
static bool some_spectrum_uses_molecule(Molecule *m, const List &slist)
{
  for (int si = 0 ; si < slist.size() ; ++si)
    if (((Spectrum *) slist[si])->molecule() == m)
      return true;

  return false;
}

// ----------------------------------------------------------------------------
//
static bool some_spectrum_uses_condition(Condition *c, const List &slist)
{
  for (int si = 0 ; si < slist.size() ; ++si)
    if (((Spectrum *) slist[si])->condition() == c)
      return true;

  return false;
}

// ----------------------------------------------------------------------------
//
static bool restore_molecule(Project &proj, FILE *fp)
{
  Stringy line, rest;
  Molecule *m = NULL;

  while (get_line(fp, &line))
    if (line == SAVE_BEGINSEQUENCE)
      {
	if (m == NULL || !restore_sequence(m, fp))
	  break;
      }
    else if (line == SAVE_BEGINCONDITION)
      {
	if (m == NULL || !restore_condition(m, fp))
	  break;
      }
    else if (line == SAVE_BEGINATTACHEDDATA)
      {
	if (m == NULL || !restore_attached_data(m->saved_values(), fp))
	  break;
      }
    else if (line == SAVE_ENDMOLECULE)
      return true;
    else if (line_tag(line, &rest) == "name")
      m = proj.define_molecule(trim_white(rest));

  skip_past_line(fp, SAVE_ENDMOLECULE);
  return false;
}

// ----------------------------------------------------------------------------
//
static bool get_line(FILE *fp, Stringy *line)
{
  char buf[MAX_LINE_LENGTH];

  if (fgets(buf, sizeof(buf), fp))
    {
      *line = buf;
      return true;
    }

  return false;
}

// ----------------------------------------------------------------------------
//
static Stringy line_tag(const Stringy &line, Stringy *rest)
{
  const char *start = line.cstring();
  start = skip_white(start);
  const char *end = skip_nonwhite(start);
  *rest = end;
  return substring(start, end);
}

// ----------------------------------------------------------------------------
//
static bool restore_sequence(Molecule *m, FILE *fp)
{
  Stringy line;
  List seq;

  while (get_line(fp, &line))
    if (line == SAVE_ENDSEQUENCE)
      return true;
    else if (line.is_empty())
      seq.append(NULL);
    else
      seq.append(m->define_group(line));

  skip_past_line(fp, SAVE_ENDSEQUENCE);
  return false;
}

// ----------------------------------------------------------------------------
//
static void save_conditions(Project &proj, Molecule *m, FILE *fp)
{
  List clist = m->condition_list();
  for (int ci = 0 ; ci < clist.size() ; ++ci)
    {
      Condition *c = (Condition *) clist[ci];
      if (some_spectrum_uses_condition(c, proj.spectrum_list()))
	{
	  fprintf(fp, SAVE_BEGINCONDITION);
	  fprintf(fp, "name %s\n", c->name().cstring());
	  fprintf(fp, SAVE_BEGINRESONANCES);
	  List rlist = c->resonance_list();
	  rlist.sort(compare_resonance_names);
	  for (int ri = 0 ; ri < rlist.size() ; ++ri)
	    {
	      Resonance *r = (Resonance *) rlist[ri];
	      fprintf(fp, "%-12s %-12.6g %3s\n",
		      assignment_name(r).cstring(),
		      r->frequency(),
		      r->atom()->nucleus().cstring());
	    }
	  fprintf(fp, SAVE_ENDRESONANCES);
	  fprintf(fp, SAVE_ENDCONDITION);
	}
    }
}

// ----------------------------------------------------------------------------
//
static bool restore_condition(Molecule *m, FILE *fp)
{
  Stringy line, rest;
  Condition *c = NULL;

  while (get_line(fp, &line))
    if (line == SAVE_BEGINRESONANCES)
      {
	if (c == NULL || !restore_resonances(c, fp))
	  break;
      }
    else if (line == SAVE_ENDCONDITION)
      return true;
    else if (line_tag(line, &rest) == "name")
      c = m->define_condition(trim_white(rest));

  skip_past_line(fp, SAVE_ENDCONDITION);
  return false;
}

// ----------------------------------------------------------------------------
//
static bool restore_resonances(Condition *c, FILE *fp)
{
  Stringy line;

  while (get_line(fp, &line))
    if (line == SAVE_ENDRESONANCES)
      return true;
    else
      {
	Stringy group, atom, rest;
	if (parse_group_atom(line, &group, &atom, &rest))
	  {
	    Stringy freq = first_token(rest, &rest);
	    double freq_ppm;
	    if (sscanf(freq.cstring(), "%lf", &freq_ppm) == 1)
	      {
		Stringy nucleus = first_token(rest, &rest);
		if (nucleus.is_empty())
		  nucleus = guess_nucleus(atom);
		Resonance *r = c->define_resonance(group, atom, nucleus);
		if (r)
		  r->set_frequency(freq_ppm);
	      }
	  }
      }

  skip_past_line(fp, SAVE_ENDRESONANCES);
  return false;
}

// ----------------------------------------------------------------------------
//
static Stringy guess_nucleus(const Stringy &atom_name)
{
  if (atom_name.is_empty())
    return "1H";

  char first_char = atom_name[0];
  if (first_char == 'C' || first_char == 'c')
    return "13C";
  else if (first_char == 'N' || first_char == 'n')
    return "15N";

  return "1H";
}

// ----------------------------------------------------------------------------
//
static bool restore_attached_data(AttachedData &ad, FILE *fp)
{
  Stringy line;

  while (get_line(fp, &line))
    if (line == SAVE_ENDATTACHEDDATA)
      return true;
    else
      {
	Stringy rest;
	Stringy key = first_token(line, &rest);
	if (rest.length() >= 2)
	  {
	    // remove leading space and trailing newline to get value
	    Stringy value = rest.substring(1, rest.length()-1);
	    ad.save_value(key, value);
	  }
      }

  skip_past_line(fp, SAVE_ENDATTACHEDDATA);
  return false;
}

// ----------------------------------------------------------------------------
//
static void save_attached_data(AttachedData &ad, FILE *fp)
{
  fprintf(fp, SAVE_BEGINATTACHEDDATA);
  List keys = ad.keys();
  for (int ki = 0 ; ki < keys.size() ; ++ki)
    {
      Stringy key = *(Stringy *) keys[ki];
      Stringy value;
      ad.saved_value(key, &value);
      fprintf(fp, "%s %s\n", key.cstring(), value.cstring());
    }
  fprintf(fp, SAVE_ENDATTACHEDDATA);
}

// ----------------------------------------------------------------------------
//
static bool save_spectra(Session &s, const List &spectra, bool backup)
{
  bool success = true;
  for (int si = 0 ; si < spectra.size() ; ++si)
    {
      Spectrum *sp = (Spectrum *) spectra[si];
      success = save_spectrum(s, sp, backup) && success;
    }

  return success;
}

// ----------------------------------------------------------------------------
//
static bool is_save_file(const Stringy &path)
{
  return check_file_type(path, SAVEFILE_SPECTRUM);
}

// ----------------------------------------------------------------------------
//
bool save_spectrum(Session &s, Spectrum *sp, bool backup)
{
  if (!backup && sp->save_file().path().is_empty())
    sp->save_file().set_path(s.project().new_save_file_path(sp->name()));

  if (!backup)
    sp->save_file().NeedsSaving(true);	// Force a spectrum save.

  FILE	*fp = sp->save_file().OpenForSaving(backup);
  if (fp == NULL)
    return false;

  bool success = save_spectrum(s, sp, fp);
  success = sp->save_file().EndSave(fp, backup, success);

  if (!success)
    s.reporter().warning("Spectrum %s failed.\n"
			 "  %s\n",
			 (backup ? "backup" : "save"),
			 sp->save_file().path().cstring());

  return success;
}

// ----------------------------------------------------------------------------
// Parse |group|atom|
//
static bool parse_group_atom(const Stringy &line,
			     Stringy *group, Stringy *atom, Stringy *rest)
{
  char *p1 = strchr(line.cstring(), '|');
  char *p2 = (p1 ? strchr(p1+1, '|') : NULL);
  char *p3 = (p2 ? strchr(p2+1, '|') : NULL);

  if (p3 == NULL)
    {
      *group = "";
      *atom = "";
      if (rest)
	*rest = line;
      return false;
    }

  *group = substring(p1+1, p2);
  *atom = substring(p2+1, p3);
  if (rest)
    *rest = p3 + 1;
  return true;
}

/*
 * Open any of the types of files sparky is presented with: spectrum,
 * save, or project. This returns true if successful, false if it fails.
 */
bool open_sparky_file(Session &s, const Stringy &path, Stringy *error_msg)
{
  Stringy epath = tilde_expand(path);

  if (is_project_file(epath))
    return restore_project(s, epath, error_msg);
  else if (open_spectrum(s, epath, error_msg) != NULL)
    return true;

  return false;
}

// ----------------------------------------------------------------------------
// Open an nmr data file or Sparky save file.
//
Spectrum *open_spectrum(Session &s, const Stringy &path, Stringy *error_msg)
{
  if (is_nmr_data(path))
    return open_new_spectrum(s, path, error_msg);
  else if (is_save_file(path))
    {
      List views_to_show;
      Spectrum *sp = restore_savefile(s, path, views_to_show, error_msg);
      create_views(s, views_to_show);
      return sp;
    }

  *error_msg = "Unrecognized file type " + path;
  return NULL;
}

#define COMMANDSEP	';'

//
// Read the default print file from the string 'cp'
//
static bool print_command_default(Session &s, char *cp)
{
  Print_Options &po = s.project().print_options;
  po.print_command = next_line(&cp);
  return true;
}

//
// Read the default print file from the string 'cp'
//
static bool print_file_default(Session &s, char *cp)
{
  Print_Options &po = s.project().print_options;
  po.preview = false;
  po.print_to_file = true;
  po.print_file = next_line(&cp);
  return true;
}

//
// Read the default print options from the string 'cp'
//
static bool print_options_default(Session &s, char *cp)
{
  Print_Options &po = s.project().print_options;

  char *rest = cp;

  Units fixed_scale_units = (Units) next_int(&rest);

  unsigned int options = next_int(&rest);
  po.landscape = !(options & OPTIONS_PORTRAIT);
  po.show_banner = (options & OPTIONS_BANNER);
  po.show_grids = (options & OPTIONS_GRIDS);
  po.show_peakgroups = (options & OPTIONS_PEAKGROUPS);
  po.show_peaks = (options & OPTIONS_PEAKS);
  po.black_labels = (options & OPTIONS_BLACKLABELS);
  po.fixed_scale = (options & OPTIONS_FIXEDSCALE);

  po.fixed_scales[0] = next_double(&rest);
  po.fixed_scales[1] = next_double(&rest);

  //
  // Turn off fixed scale if units are not ppm / cm.
  //
  if (fixed_scale_units != PPM ||
      po.fixed_scales[0] == 0 ||
      po.fixed_scales[1] == 0) {
    po.fixed_scale = false;
    po.fixed_scales[0] = 1.0;
    po.fixed_scales[1] = 1.0;
  }

  return true;
}

//
// Read the default title from the string 'cp'
//
static bool print_title_default(Session &s, char *cp)
{
  Print_Options &po = s.project().print_options;
  po.title = next_line(&cp);
  return true;
}

/*
 * A command string may hold multiple commands, separated by ";"
 *
 * Each command is parsed one whitespace-separated token at a time.
 *
 * The first token is looked up in the <table> called <_genTable>.
 * The following logic is followed:
 *
 * general_functions_loop:
 *
 *   If the token matches an element of the current <table> then one of
 *   two things happens:
 *	a) the <table> entry names a <sub-table>. In this case the
 *	   rest of the command is reparsed, with the <sub-table>
 *	   becoming the <table> and the token being advanced to the
 *	   next token. To back to <general_functions_loop>.
 *
 *	b) the <table> entry names a function. In this case the rest
 *	   of the command is passed as a single line to the function
 *	   and the return status of the function is used to determine
 *	   if the command was successful or not.
 *
 * If the first token does not match any elements of <_genTable> then
 * the lookup table used is <_viewTable>. Again, a similar logic applies
 * except when the function dispatch occurs. In this case the function
 * and remainder of the command are passed to a dispatch function
 * which dispatches the function with a pointer to the current VIEW
 * and the remainder of the command.
 *
 * At one time the command parser was going to allow overriding the
 * destination of command so that it didn't have to go to the current
 * view, but this never became necessary.
 */
typedef struct fntbl FNTBL;
struct fntbl {
	const char	*fname;			// entry name
	FNTBL		*tbl;			// sub-table entry
	bool		(*fnp)(Session &, char *);	// function to call
};

/*
 * Table for:	define ....
 */
static FNTBL g_define[] = {
	{"region",	NULL,		read_region},
	{NULL,		NULL,		NULL}
};

/*
 * Table for:	set ....
 */
static FNTBL g_set[] = {
	{"mode",	NULL,		mode_set_mode},
	{"resizeViews",	NULL,		user_resize_views_set},
	{"saveinterval", NULL,		user_saveauto_set},
	{"saveprompt",	NULL,		user_saveprompt_set},
	{"keytimeout", NULL,		user_keytimeout_set},
	{"cachesize", NULL,		user_cachesize_set},
	{"contourgraying",NULL,		user_contour_graying_set},
	{NULL,		NULL,		NULL}
};

/*
 * Table for:	default print ....
 */
static FNTBL g_default_print[] = {
	{"command",	NULL,		print_command_default},
	{"file",	NULL,		print_file_default},
	{"title",	NULL,		print_title_default},
	{"options",	NULL,		print_options_default},
	{NULL,		NULL,		NULL}
};

/*
 * Table for:	default ....
 */
static FNTBL g_default[] = {
	{"print",	g_default_print,	NULL},
	{NULL,		NULL,		NULL}
};

/*
 * General functions:
 *
 * All matched functions are called with (char *args)
 */
static FNTBL _genTable[] = {
	{"default",	g_default,	NULL},
	{"define",	g_define,	NULL},
	{"set",		g_set,		NULL},
	{NULL,		NULL,		NULL}
};


/*
 * Look up the function name in the table FNTLB and return the function
 * pointer associated with this name. Update <*linep> to point to the
 * start of the first word after the function name.
 */
static bool (* lookup_parse_function(FNTBL fntable[], char **linep))(Session &, char *)
{
	char		*cp, *fname, *end, savech;
	int		i;
	bool		(*fnp)(Session &, char *) = NULL;

	cp = fname = *linep;
	cp = skip_nonwhite(cp);
	end = cp;
	if ((savech = *cp) != '\0')
		*cp++ = '\0';
	cp = skip_white(cp);
	*linep = cp;

	/*
	 * Look for the function in the appropriate table. If the table
	 * entry has a subtable, then call this function recursively
	 * to lookup the next keyword.
	 */
	for (i = 0; fntable[i].fname != NULL; i++) {
		if (strcmp(fntable[i].fname, fname) == 0) {
			if (fntable[i].tbl != NULL)
			  fnp = lookup_parse_function(fntable[i].tbl, linep);
			else
			  fnp  = fntable[i].fnp;
			break;
		}
	}
	if (savech != '\0')
		*end = savech;

	return fnp;
}


/*
 * _parseCommand:
 *
 * Take the command string <cmd> and look up the function that should be
 * called, then call the function with the arguments on the command string.
 */
static bool _parseCommand(Session &s, char *cmd)
{
	bool		(*fnp)(Session &, char *);
	char		*cp, *data;
	bool		status;

	/*
	 * Strip leading and trailing whitespace
	 */
	cp = cmd;
	cp = skip_white(cp);
	data = cmd = cp;
	cp = data + strlen(cp) - 1;
	while (cp > data && isspace(*cp))
		*cp-- = '\0';

	/* Ignore blank and comment lines
	 */
	if (cp <= cmd || *data == COMMENT)
		return true;

	fnp = lookup_parse_function(_genTable, &data);
	if (fnp) {
		(*fnp)(s, data);
		status = true;
	} else {
		status = false;
	}

	return status;
}

static void parse_line_no_preempt_check(Session &s, const char *cmd)
{
	char	*cp, buf[MAX_LINE_LENGTH];

	/*
	 * Remove leading whitespace and skip null commands
	 */
	if (cmd)
		cmd = skip_white(cmd);
	if (cmd == NULL || *cmd == '\0')
		return;

	/*
	 * Make a copy of the command
	 */
	strncpy_terminate(buf, cmd, sizeof buf);
	cp = strtokquote(buf, COMMANDSEP);
	do {
		_parseCommand(s, cp);
	} while ((cp = strtokquote(NULL, COMMANDSEP)));
}


/*
 */
static bool parse_line(Session &s, const char *cmd)
{
	parse_line_no_preempt_check(s, cmd);

	return true;
}

//
// Index 0 used to be "clear".
//
static const char *old_file_colors[] =
{"white", "black", "white", "gray", "red", "green", "blue", "magenta", "cyan"};

// ----------------------------------------------------------------------------
//
static int old_file_color_index(const Color &color)
{
  for (int c = 1 ; c < array_size(old_file_colors) ; ++c)
    if (color.name() == old_file_colors[c])
      return c;

  return 2;	// Return white if color not found.
}

// ----------------------------------------------------------------------------
//
static Stringy old_file_color(int index)
{
  if (index < 0 || index > array_size(old_file_colors))
    fatal_error("old_file_color(): Bad color index %d\n", INDEX);

  return old_file_colors[index];
}

/*
 * Set the mode to <cp>
 */
static bool mode_set_mode(Session &s, char *cp)
{
	if (cp) {
	  int m = atoi(cp);
	  if (m > 7)
	    m -= 1;
	  set_pointer_mode(s, (Pointer_Mode) m);
	}
	return true;
}

/*
 * Given an ornament type <name>, return the ornament type.
 */
static Ornament_Type ornament_name2type(const char *name)
{
  if (name) {
    for (int i = 0; Ornament_Type_Names[i]; i++)
      if (strcmp(Ornament_Type_Names[i], name) == 0)
	return (Ornament_Type) i;
    if (strcmp("xpeak", name) == 0)
      return peakgroup;
  }
  return (Ornament_Type) -1;
}

/*
 * Ornament links were the way Sparky showed ownership of ornaments in
 * version 1.96 and before. This is no longer used but is still supported
 * in pre-version 1.96 save files.
 */
#define IGNORE_LINKS "# Note: ornament links unused by Sparky >1.96\n"

static bool ornament_restore_links(FILE *fp, Spectrum *sp)
{
	Ornament	*parent;
	Ornament	*child;
	char		buf[MAX_LINE_LENGTH];
	Ornament_Type		type;
	int		id;
	int		ignoring = false;

	while (fgets(buf, sizeof buf, fp)) {
		if (strcmp(buf, SAVE_ENDLINKS) == 0)
			return true;

		if (ignoring)
			continue;

		if (buf[0] == '#' && strcmp(buf, IGNORE_LINKS) == 0) {
			ignoring = true;
			continue;
		}

		char *rest = buf;
		type = ornament_name2type(next_token(&rest));
		if (! valid_otype(type))
			continue;

		id = atoi(next_token(&rest));
		parent = sp->nth_ornament(type, id);
		if (parent == NULL)
		  continue;

		type = ornament_name2type(next_token(&rest));
		if (! valid_otype(type))
			continue;

		id = atoi(next_token(&rest));
		child = sp->nth_ornament(type, id);
		if (child == NULL)
		  continue;

		if (parent->type() == peakgroup && child->type() == peak)
			((PeakGp *) parent)->add(* (Peak *) child);
		else if (parent->type() == peakgroup && child->type() == label)
			((Label *)child)->attach((CrossPeak *) parent);
		else if (parent->type() == peak && child->type() == label)
			((Label *)child)->attach((CrossPeak *) parent);
	}
	return false;
}

// ----------------------------------------------------------------------------
//
bool load_resonances(const Stringy &path, Condition *c)
{
  FILE *fp = fopen(path.cstring(), "r");
  if (fp == NULL)
    return false;

  char buf[MAX_LINE_LENGTH];
  while (fgets(buf, sizeof buf, fp))
    {
      Stringy rest = buf;
      Stringy group = first_token(rest, &rest);
      Stringy atom = first_token(rest, &rest);
      Stringy nucleus = first_token(rest, &rest);
      nucleus = standard_nucleus_name(nucleus);
      Stringy ftext = first_token(rest, &rest);
      double freq = atof(ftext.cstring());
      Resonance *r = c->define_resonance(group, atom, nucleus);
      if (r && r->assignment_count() == 0)
	r->set_frequency(freq);
    }

  return true;
}
