/*
 * Label.C:		Implementation of the Label ornament class.
 *
 * A label is an ornament whose text can be changed either by the
 * user (if this is a user label) or automatically (if this is
 * a label showing an assignment).
 *
 */
#include <math.h>		// Use atan2()
#include <string.h>		// Use strcmp()

#include "axismap.h"		// Use Axis_Map
#include "color.h"
#include "label.h"
#include "mathconst.h"		// use MC_PI
#include "memalloc.h"		// use new()
#include "notifier.h"	// use Notifier
#include "num.h"
#include "peakgp.h"
#include "peak.h"
#include "print.h"		// Use print_options()
#include "spectrum.h"
#include "spoint.h"		// Use SPoint


#define LABEL_X_OFFSET	.1
#define LABEL_Y_OFFSET	.1

static LABELPOS offset_label_positions(Spectrum *sp, const SPoint &pos);

// ---------------------------------------------------------------------------
//  Floating label
//
Label::Label(Spectrum *sp, Stringy text, const SPoint &position) : Ornament(sp)
{
  initialize(sp, text, false, position, NULL);
}

// ---------------------------------------------------------------------------
//  Assignment label
//
Label::Label(CrossPeak *xp) : Ornament(xp->spectrum())
{
  Spectrum *sp = xp->spectrum();
  initialize(sp, xp->assignment_name(sp->assignment_format()),
	     true, xp->position(), xp);
}

// ---------------------------------------------------------------------------
//  User label
//
Label::Label(CrossPeak *xp, Stringy text) : Ornament(xp->spectrum())
{
  initialize(xp->spectrum(), text, false, xp->position(), xp);
}

// ---------------------------------------------------------------------------
//
void Label::initialize(Spectrum *sp, const Stringy &text, bool assignment,
		       const SPoint &pos, CrossPeak *parent)
{
	mAssignment	= assignment;
	mString		= text;
	mParent		= parent;
	mCenter		= pos;
	mLabelPos	= (parent ?
			   offset_label_positions(sp, parent->position()) :
			   label_positions(pos));

	if (parent)
	  parent->label(this);

	sp->notifier().send_notice(nt_created_ornament, (Ornament *) this);
}

//
// ~Label:
//
// Deallocate any allocated information, and if there is a parent, inform
// the parent that it no longer has a label.
//
Label::~Label()
{
  Notifier &n = spectrum()->notifier();
  n.send_notice(nt_will_delete_ornament, (Ornament *) this);
  n.send_notice(nt_will_delete_label, this);

	if (attached())
	  attached()->label(NULL);
}

// ----------------------------------------------------------------------------
//
Ornament *Label::copy(Spectrum *to, const Axis_Map &axismap)
{
  Spectrum *from = spectrum();
  SPoint pos = axismap.map(location());
  Label *lbl = new Label(to, GetString(), pos);

  int dim = from->dimension();
  for (int ax = 0 ; ax < dim ; ++ax)
    {
      int to_ax = axismap.map(ax);
      for (int ay = 0 ; ay < dim ; ++ay)
	if (ay != ax)
	  {
	    int to_ay = axismap.map(ay);

	    double to_x = xPosition(ax, ay);
	    double to_y = yPosition(ax, ay);
	    
	    lbl->xPosition(to_ax, to_ay) = to_x;
	    lbl->yPosition(to_ax, to_ay) = to_y;
	  }
    }

  lbl->lock(IsLocked());
  lbl->SetNote(GetNote());
  lbl->SetColor(GetColor());

  return lbl;
}

// ----------------------------------------------------------------------------
//
Ornament_Type Label::type() const { return ::label; }
const char *Label::type_name() const { return "label"; }

/*
 * Make <text> be the new value of the label.
 */
void Label::SetString(const Stringy &text)
{
  Notifier &n = spectrum()->notifier();
  n.send_notice(nt_will_change_ornament, (Ornament *) this);
  mString = trim_white(text);
  n.send_notice(nt_changed_ornament, (Ornament *) this);
}
Stringy Label::GetString() const
  { return mString; }

//
// Set the parent of this label.
//
void Label::attach(CrossPeak *xp)
{
  Notifier &n = spectrum()->notifier();
  n.send_notice(nt_will_change_ornament, (Ornament *) this);
  mParent = xp;
  if (xp)
    xp->label(this);
  n.send_notice(nt_changed_ornament, (Ornament *) this);
}

// ----------------------------------------------------------------------------
//
SRegion Label::selection_region(ODraw &dr) const
{
  Rectangle tbox = text_box(dr);

  SRegion sel = SRegion(dr.axis(X), tbox.min(X), tbox.max(X),
			dr.axis(Y), tbox.min(Y), tbox.max(Y),
			location());

  return sel;
}

// ----------------------------------------------------------------------------
//
SRegion Label::erasure_region(ODraw &dr) const
{
  Rectangle tbox = text_box(dr);

  SRegion r = SRegion(dr.axis(X), tbox.min(X), tbox.max(X),
		      dr.axis(Y), tbox.min(Y), tbox.max(Y),
		      location());

  CrossPeak *xp = attached();
  if (xp)
    r.encompass(xp->position());

  add_selection_padding(&r, dr);

  return r;
}

// ----------------------------------------------------------------------------
//
Rectangle Label::text_box(ODraw &dr) const
{
	double		x, y, w, h;

	center(dr, &x, &y);
	size(dr, &w, &h);

	return Rectangle(x - w/2, y - h/2, x + w/2, y + h/2);
}

// ----------------------------------------------------------------------------
// Return Label center in data units.
//
void Label::center(ODraw &dr, double *cx, double *cy) const
{
  int	ox = dr.axis(X);
  int	oy = dr.axis(Y);

  *cx = mLabelPos.x[ox][oy];
  *cy = mLabelPos.y[ox][oy];
}

// ----------------------------------------------------------------------------
// Return text size plus some padding so when the text is outlined by a
// selection box it is still legible.
//
void Label::size(ODraw &dr, double *w, double *h) const
{
  double width, ascent, descent;
  dr.text_size(mString, &width, &ascent, &descent);
  double ypadding = .1 * ascent;
  double xpadding = ypadding / dr.aspect();
  
  *w = width + 2 * xpadding;
  *h = ascent + descent + 2 * ypadding;
}

// ----------------------------------------------------------------------------
//
SPoint Label::location() const
{
  CrossPeak *xp = attached();
  return (xp ? xp->position() : mCenter);
}

/*
 * This IncrementLocation() is called when the label is moved manually
 * and hence only in the 'X' and 'Y' dimension on screen.
 */
void Label::IncrementLocation(double dx, double dy, int xaxis, int yaxis)
{
  LABELPOS pos = mLabelPos;

  pos.x[xaxis][yaxis] += dx;
  pos.y[xaxis][yaxis] += dy;
  set_positions(mCenter, pos);
}

/*
 * This IncrementLocation() is called when the label is being moved because
 * the origin is being shifted. Thus all <mLabelPos>s must be updated.
 */
void Label::IncrementLocation(const SPoint &dr)
{
  LABELPOS pos = mLabelPos;
  int	dim = dr.dimension();
  for (int wa = 0; wa < dim; wa++) {
    for (int wb = 0; wb < dim; wb++) {
      if (wa != wb) {
	pos.x[wa][wb] += dr[wa];
	pos.y[wa][wb] += dr[wb];
      }
    }
  }
  set_positions(mCenter + dr, pos);
}

// ----------------------------------------------------------------------------
//
void Label::set_positions(const SPoint &center, const LABELPOS &pos)
{
  Notifier &n = spectrum()->notifier();
  n.send_notice(nt_will_change_ornament, (Ornament *) this);
  mCenter = center;
  mLabelPos = pos;
  n.send_notice(nt_changed_ornament, (Ornament *) this);
}

//
//
// The newly created location needs fixing up in the orientations other
// than the current orientation. To do the fixup we place the label
// in an "above" or "below" position, depending on whether the label
// location is above or below the midpoint of the spectrum.
//

#define SHIFTFACTOR	5

static LABELPOS offset_label_positions(Spectrum *sp, const SPoint &pos)
{
  LABELPOS labelpos;

  for (int x = 0; x < sp->dimension(); x++) {
    for (int y = 0; y < sp->dimension(); y++) {
      double f =  sp->nucleus_ppm_factor(y) / sp->nucleus_ppm_factor(0);
      double offset = f * sp->ornament_size(peak);
      labelpos.x[x][y] = pos[x];
      labelpos.y[x][y] = pos[y] - offset * SHIFTFACTOR;
    }
  }

  return labelpos;
}

// ----------------------------------------------------------------------------
// pos must be in ppm units.
//

LABELPOS label_positions(const SPoint &pos)
{
	LABELPOS labelpos;

	int dim = pos.dimension();
	for (int i = 0; i < dim; i++)
	    for (int j = 0; j < dim; j++)
	      {
		labelpos.x[i][j] = pos[i];
		labelpos.y[i][j] = pos[j];
	      }

	return labelpos;
}

// ----------------------------------------------------------------------------
//
void Label::assignment(bool assign)
{
  mAssignment = assign;
  if (assign)
    AssignmentChanged();
}

// ----------------------------------------------------------------------------
//
bool Label::is_assignment() const
{
  return mAssignment;
}

/*
 * Called when the label's parent is changed.
 *
 * When the label is showing the computed name of the parent, update this label
 */
void Label::AssignmentChanged()
{
  if (attached() && is_assignment())
    SetString(attached()->assignment_name(spectrum()->assignment_format()));
}

// ----------------------------------------------------------------------------
// Draw a label.
//
void Label::display(ODraw &dr) const
{
  double center_x, center_y;
  center(dr, &center_x, &center_y);

  dr.set_drawing_color(GetColor());
  dr.draw_text(center_x, center_y, display_string(mString).cstring());

  draw_pointer(dr);

  Ornament::display(dr);	// Handle selection highlighting.
}

// ----------------------------------------------------------------------------
// Remove back slashes used to indicate Greek letters.
//
Stringy display_string(const Stringy &text)
{
  const char *s = text.cstring();
  const char *escape = strchr(s, '\\');
  if (escape)
    return substring(s, escape) + display_string(escape + 1);

  return text;
}

// ----------------------------------------------------------------------------
//
void Label::draw_pointer(ODraw &dr) const
{
  double x1, y1, x2, y2, x3, y3;
  if (pointer_triangle(dr, 1 / dr.aspect(), &x1, &y1, &x2, &y2, &x3, &y3))
    dr.draw_triangle(x1, y1, x2, y2, x3, y3);
}

// ----------------------------------------------------------------------------
// Send a Postscript command to draw a label.
//
void Label::print(ODraw &dr, FILE *fp,
		  double xsc, double ysc, Rectangle r) const
{
	if (!dr.is_visible(this, r))
		return;

	Color	c = GetColor();
	if (dr.black_labels())
		c = "black";
	else if (c == "white")	// Change white to black
		c = "black";

	Stringy pp = postscript_pointer(dr, c, xsc, ysc, r.max(X), r.max(Y));

	double cx, cy;
	center(dr, &cx, &cy);
	double center_x = (r.max(X) - cx) * xsc;
	double center_y = (r.max(Y) - cy) * ysc;

	double width, ascent, descent;
	dr.text_size(GetString(), &width, &ascent, &descent);
	double w = width * xsc;
	double h = (ascent + descent) * ysc;
	double xoffset = -w/2;
	double yoffset = -h / 2 + descent * ysc;

	if (!GetString().is_empty())
	  fprintf(fp,
		  "{gsave %f %f rmoveto 1 1 scale %d setpixel %s grestore} "
		  "{%s} "
		  "%d %d %.0f %.0f %.0f %.0f PA\n",
		xoffset, yoffset,
		color_index(c),		// text color
		postscript_string(GetString()).cstring(),
		pp.cstring(),
		-1,				// edge color
		-1,				// fill color
		center_x, center_y, w, h);

}

// ----------------------------------------------------------------------------
// Return a postscript command to draw the specified string.
// Back slash escaped characters are drawn in the symbol font.
// This produces Greek letters for \a, \b, \g ...
//
Stringy postscript_string(const Stringy &text)
{
  Stringy ps;

  if (!text.is_empty())
    {
      if (text[0] == '\\' && text.length() >= 2)
	{
	  // Use symbol font.
	  ps = (formatted_string("XfS (%c) show XfR", text[1]) + " " +
		postscript_string(text.cstring() + 2));
	}
      else
	{
	  const char *s = text.cstring();
	  const char *escape = strchr(s, '\\');

	  if (escape == NULL)
	    ps = formatted_string("(%s) show", s);
	  else
	    ps = (postscript_string(substring(s, escape)) + " " +
		  postscript_string(escape));
	}
    }

  return ps;
}

/*
 * 
 *
 * Compute the pointer arguments from the label to the parent of the label.
 * The <buf> will be filled in with an executable PostScript string
 * that draws the pointer.
 */
bool Label::pointer_triangle(ODraw &dr, double aspect,
			     double *x1, double *y1,
			     double *x2, double *y2,
			     double *x3, double *y3) const
{
	if (attached() == NULL)
		return false;

	/*
	 * Compute the boxes for both objects
	 */
	Rectangle	rto = attached()->marker_square(dr);
	Rectangle	rfrom = text_box(dr);

	double labelWidth =	rfrom.size(X);
	double labelHeight =	rfrom.size(Y);
	double labelLeft =	rfrom.min(X);
	double labelRight =	rfrom.min(X) + rfrom.size(X);
	double labelBottom =	rfrom.min(Y);
	double labelTop =	rfrom.min(Y) + rfrom.size(Y);
	double peakWidth =	rto.size(X);
	double peakHeight =	rto.size(Y);
	double peakLeft =	rto.min(X);
	double peakRight =	rto.min(X) + rto.size(X);
	double peakBottom =	rto.min(Y);
	double peakTop =	rto.min(Y) + rto.size(Y);


	/*
	 * Compute each center.
	 */
	double peakCenterX = peakLeft + peakWidth / 2;
	double peakCenterY = peakBottom + peakHeight / 2;
	double labelCenterX = labelLeft + labelWidth / 2;
	double labelCenterY = labelBottom + labelHeight / 2;


	/*
	 * We treat the relative positions of the peak and label as 4
	 * separate cases. The area around the peak is divided into 4
	 * quadrants with angular limits A(-45, 45), B(45, 135), C(135, -135),
	 * and D(-135, -45). For A (C) the pointer is from the left (right)
	 * side of the label to the right (left) side of the peak. For B (D)
	 * the pointer is from the bottom (top) of the label to the top
	 * (bottom) of the peak:
	 *
	 *    \  B   /
	 *     \    /			TL		TR
	 *	+--+			+---------------+
	 *   C	|O |  A			|     label	|
	 *	+--+			+---------------+
	 *     /    \			BL		BR
	 *    /  D   \
	 *
	 * Determining which quadrant the label falls in is not exactly
	 * trivial. We can compute the vector from the peak to the label
	 * but we have to pick points on the peak and the label for the
	 * angle. The peak is square-symmetric, so we can choose the
	 * peak center as one of the points of the vector.
	 *
	 * For the label we will use appropriate corners. Assume the label
	 * center is above the X-line. If the angle from the peak
	 * center O to BL (aOBL) falls in angular limit A and the angle from
	 * peak center O to BR (aOBR) falls in angular limit A, then we
	 * have case A. If aOBL or aOBR falls in angular limit B we have
	 * case B. if aOBL and aOBR falls in angular limit C we have case C.
	 *
	 * Likewise if the label center is below the X line we use the TL
	 * and TR points and the A, D, and C  ranges.
	 */
	double	aOL, aOR;

	/* 
	 * If label and peak are at exactly the same height, angles to
	 * the corners are considered to be exactly 0 or 180.
	 */
	if (labelCenterY == peakCenterY) {
		aOL = labelLeft  < peakCenterX ? 180 : 0;
		aOR = labelRight < peakCenterX ? 180 : 0;
	}
	else {
		double labelY = (labelCenterY > peakCenterY)
			? labelBottom : labelTop;
		
		aOL = atan2((labelY - peakCenterY) * aspect,
			    (labelLeft - peakCenterX)) * 180.0 / MC_PI;
		aOR = atan2((labelY - peakCenterY) * aspect,
			    (labelRight - peakCenterX)) * 180.0 / MC_PI;
	}


	/*
	 * The point of the pointer is positioned on the peak by starting at
	 * the middle of the peak and moving by (factor * peakhalfwidth).
	 * Likewise, the base of the pointer is positioned on the label by
	 * starting at the middle of the label plus/minus the base halfwidth
	 * and moving by (factor * (labelhalfwidth - basehalfwidth)). The
	 * base is started either to the left or right, depending on if
	 * it is pointing left or right.
	 */
	double  deltaX = .15 * labelHeight * aspect;
	double  deltaY = .15 * labelHeight;
	double	gapX = .25 * peakWidth;
	double	gapY = .25 * peakHeight; 
	double	peakXDelta = peakWidth / 2 + gapX;
	double	peakYDelta = peakHeight / 2 + gapY;
	double	labelXDelta = max(labelWidth  / 2 - deltaX, 0.0);
	double	labelYDelta = max(labelHeight / 2 - deltaY, 0.0);
	double	factor;	
	double	pointerPeakX;
	double	pointerPeakY;
	double	pointerLab1X;
	double	pointerLab1Y;
	double	pointerLab2X;
	double	pointerLab2Y;

	/*
	 * Now treat the 4 cases
	 */
	if (is_between(aOL, -45, 45) && is_between(aOR, -45, 45)) {
		/*
		 * case A, label right of peak
		 */
		factor = aOL / 45;
		factor = labelCenterY > peakCenterY
			? max(factor, 0.0) : min(factor, 0.0);

		pointerPeakX = peakRight + gapX;
		pointerPeakY = peakCenterY + peakYDelta * factor;
		pointerLab1X =
		pointerLab2X = labelLeft - gapX;
		pointerLab1Y = labelCenterY + deltaY - labelYDelta * factor;
		pointerLab2Y = labelCenterY - deltaY - labelYDelta * factor;
	}
	else if (is_between(abs(aOL), 135, 180)
		 && is_between(abs(aOR), 135, 180)) {
		/*
		 * case C, label left of peak
		 */
		factor = (aOR / abs(aOR)) * (180 - abs(aOR)) / 45;
		factor = labelCenterY > peakCenterY
			? max(factor, 0.0) : min(factor, 0.0);

		pointerPeakX = peakLeft - gapX;
		pointerPeakY = peakCenterY + peakYDelta * factor;
		pointerLab1X =
		pointerLab2X = labelRight + gapX;
		pointerLab1Y = labelCenterY + deltaY - labelYDelta * factor;
		pointerLab2Y = labelCenterY - deltaY - labelYDelta * factor;
	}
	else if (labelBottom > peakTop) {
		
		/*
		 * Call the factor when the labelLeft corner is at 45 degrees
		 * a factor of 1 and the factor when the labelRight corner is
		 * at 135 degrees a factor of -1. This is easiest to compute
		 * by finding the position of the labelLeft corner at each
		 * of these extremes and using linear interpolation.
		 */
		double	posLeft = peakCenterX - labelWidth
			- (labelBottom - peakCenterY) * aspect;
		double	posRight = peakCenterX
			+ (labelBottom - peakCenterY) * aspect;
		factor = -1.1+2.2*(labelLeft - posLeft) / (posRight - posLeft);
		factor = bounded(-1., factor, 1.);

		pointerPeakX = peakCenterX + peakXDelta * factor;
		pointerPeakY = peakTop + gapY;
		pointerLab1X = labelCenterX - deltaX - labelXDelta * factor;
		pointerLab2X = labelCenterX + deltaX - labelXDelta * factor;
		pointerLab1Y = 
		pointerLab2Y = labelBottom - gapY;
	}
	else if (labelTop < peakBottom) {

		double	posLeft = peakCenterX - labelWidth
			- (peakCenterY - labelTop) * aspect;
		double	posRight = peakCenterX
			+ (peakCenterY - labelTop) * aspect;
		factor = -1.1+2.2*(labelLeft - posLeft) / (posRight - posLeft);
		factor = bounded(-1., factor, 1.);

		pointerPeakX = peakCenterX + peakXDelta * factor;
		pointerPeakY = peakBottom - gapY;
		pointerLab1X = labelCenterX - deltaX - labelXDelta * factor;
		pointerLab2X = labelCenterX + deltaX - labelXDelta * factor;
		pointerLab1Y = 
		pointerLab2Y = labelTop + gapY;
	}
	else {
		/*
		 * Label too close to peak for good pointer.
		 */
		return false;
	}

	*x1 = pointerLab1X;
	*y1 = pointerLab1Y;
	*x2 = pointerPeakX;
	*y2 = pointerPeakY;
	*x3 = pointerLab2X;
	*y3 = pointerLab2Y;

	return true;
}

/*
 * Compute the pointer arguments from the label to the parent of the label.
 * The <buf> will be filled in with an executable PostScript string
 * that draws the pointer.
 */
Stringy Label::postscript_pointer(ODraw &	dr,
				 Color		c,
				 double		xscale,
				 double		yscale,
				 double		xoff,
				 double		yoff) const
{
  double x1, y1, x2, y2, x3, y3;

  if (pointer_triangle(dr, yscale/xscale, &x1, &y1, &x2, &y2, &x3, &y3))
    return formatted_string("%d %d %.3f %.3f %.3f %.3f %.3f %.3f PaintPointer",
			    color_index(c),	// fill color
			    color_index(c),	// edge color
			    (xoff - x1) * xscale, (yoff - y1) * yscale,
			    (xoff - x2) * xscale, (yoff - y2) * yscale,
			    (xoff - x3) * xscale, (yoff - y3) * yscale);
  return "";
}

// ----------------------------------------------------------------------------
// Add an assignment label to peaks if they do not yet have a label.
//
void label_peaks(const List &xplist)
{
  for (int pi = 0 ; pi < xplist.size() ; ++pi)
    {
      CrossPeak *xp = (CrossPeak *) xplist[pi];
      Label *label = xp->label();
      if (label == NULL)
	(void) new Label(xp);
    }
}

// ----------------------------------------------------------------------------
// Remove labels from peaks.
//
void unlabel_peaks(const List &xplist)
{
  for (int pi = 0 ; pi < xplist.size() ; ++pi)
    {
      CrossPeak *xp = (CrossPeak *) xplist[pi];
      Label *lbl = xp->label();
      if (lbl)
	delete lbl;
    }
}

