/*
 * label.h:		Interface to the Label ornament class.
 *
 * A label is an ornament whose text can be changed either by the
 * user (if this is a user label) or automatically (if this is
 * a label showing an assignment).
 *
 */
#ifndef _Label_h
#define _Label_h

class Label;

#include "ornament.h"		// Use Ornament
#include "spoint.h"		// Use SPoint
#include "stringc.h"		// use Stringy

struct Axis_Map;
class Color;
class CrossPeak;
class Rectangle;

//
// With multiple dimensional spectra, the labels are very hard to
// position. It is easier to define pairs of x,y positions for each plane.
//
struct LABELPOS { float	x[DIM][DIM], y[DIM][DIM]; };

//
// class Label
//
// A Label is a text string that can be associated with another ornament.
// Labels contain a string, which is compiled into a PostScript display
// string. Labels also may have a parent, which is the ornament this
// label is associated with.
//
class Label : public Ornament {

public:

// Public constructors and destructors

	Label(Spectrum *sp, Stringy text, const SPoint &position);
	Label(CrossPeak *xp);			// Assignment label
	Label(CrossPeak *xp, Stringy text);	// User label

	virtual ~Label();


// Public methods

	virtual Ornament *copy(Spectrum *to, const Axis_Map &axismap);

	//
	// Routines for ornament display
	//
	virtual void		display(ODraw &) const;
	virtual void		print(ODraw &, FILE *fp,
				      double xsc, double ysc,
				      Rectangle r) const;
	virtual SRegion		erasure_region(ODraw &) const;
	virtual	SRegion		selection_region(ODraw &) const;

	virtual Ornament_Type		type() const;
	virtual const char	*type_name() const;

	void			SetString(const Stringy &str);
	Stringy			GetString() const;

	//
	// The parent is the ornament the label is attached to.
	//
	CrossPeak	*attached() const { return mParent; }
	void		attach(CrossPeak *xp);

	//
	// Change the location incrementally
	//
	SPoint		location() const;
	virtual void	IncrementLocation(const SPoint &);
	void		IncrementLocation(double dx, double dy,
					  int xaxis, int yaxis);
	void		set_positions(const SPoint &center, const LABELPOS &);

	//
	// Return the type of the label
	//
	void		assignment(bool assign);
	bool		is_assignment() const;

	//
	// Perhaps do something when the thing pointed to changes
	//
	void			AssignmentChanged();

	float			&xPosition(int o1, int o2)
					{ return mLabelPos.x[o1][o2]; }
	float			&yPosition(int o1, int o2)
					{ return mLabelPos.y[o1][o2]; }

private:

// Private data
//

	CrossPeak		*mParent;	// the associated ornament
	SPoint			mCenter;
	bool			mAssignment;	// is this an assignment label
	Stringy			mString;	// what is displayed.
	LABELPOS		mLabelPos;

// Private methods
//
	void		initialize(Spectrum *sp, const Stringy &text,
				   bool assignment, const SPoint &pos,
				   CrossPeak *parent);

	Rectangle	text_box(ODraw &) const;
	void		center(ODraw &, double *cx, double *cy) const;
	void		size(ODraw &, double *w, double *h) const;
	void		draw_pointer(ODraw &) const;
	bool		pointer_triangle(ODraw &, double aspect,
					 double *x1, double *y1,
					 double *x2, double *y2,
					 double *x3, double *y3) const;
	Stringy		postscript_pointer(ODraw &, Color c,
					   double xscale, double yscale,
					   double xoff, double yoff) const;
};

Stringy postscript_string(const Stringy &text);
Stringy display_string(const Stringy &text);
LABELPOS label_positions(const SPoint &pos);
void label_peaks(const List &xplist);
void unlabel_peaks(const List &xplist);


#endif
