/*
 * Line.C:	Implementation of the Line ornament class.
 *
 * A line is an ornament drawn between 2 points, either horizontally
 * or vertically.
 */
#include <stdlib.h>		// Use atoi()
#include <math.h>		// use fabs()

#include "axismap.h"		// Use Axis_Map
#include "color.h"
#include "line.h"
#include "memalloc.h"		// use new()
#include "notifier.h"		// use Notifier
#include "num.h"
#include "print.h"		// use color_index()
#include "spectrum.h"		// Use Spectrum

//
// Line:
//
// Call the superclass constructor, then set the "ends" to 0, to show
// the Line currently has no arrow ends.
//
Line::Line(Spectrum *sp, const SPoint &r1, const SPoint &r2)
	: Ornament(sp)
{
  int long_axis = 0;

  SPoint delta = r1 - r2;
  for (int a = 1 ; a < sp->dimension() ; ++a)
    if (abs(delta[a]) > abs(delta[long_axis]))
      long_axis = a;

  SPoint center = (r1 + r2) * .5;
  SPoint p1 = center, p2 = center;
  p1[long_axis] = r1[long_axis];
  p2[long_axis] = r2[long_axis];

  this->bl = p1;
  this->tr = p2;
  this->axis = long_axis;
  this->ends = 0;
  SetColor("white");

  sp->notifier().send_notice(nt_created_ornament, (Ornament *) this);
}

// ----------------------------------------------------------------------------
//
Line::~Line()
{
  Notifier &n = spectrum()->notifier();
  n.send_notice(nt_will_delete_ornament, (Ornament *) this);
  n.send_notice(nt_will_delete_line, this);
}

// ----------------------------------------------------------------------------
//
Ornament *Line::copy(Spectrum *to, const Axis_Map &axismap)
{
  SPoint p1 = axismap.map(bl);
  SPoint p2 = axismap.map(tr);
  Line *line = new Line(to, p1, p2);

  line->lock(IsLocked());
  line->SetNote(GetNote());
  line->SetColor(GetColor());

  return line;
}

// ----------------------------------------------------------------------------
//
Ornament_Type Line::type() const { return ::line; }
const char *Line::type_name() const { return "line"; }

//
//
SRegion Line::erasure_region(ODraw &dr) const
{
  double x1, y1, x2, y2;

  dr.xy_point(bl, &x1, &y1);
  dr.xy_point(tr, &x2, &y2);

  SRegion r(dr.axis(X), min(x1, x2), max(x1, x2),
	    dr.axis(Y), min(y1, y2), max(y1, y2),
	    bl);

  add_selection_padding(&r, dr);

  return r;
}

//
// Move the line by moving both endpoints by dx,dy
//
void Line::IncrementLocation(const SPoint &dr)
{
  Notifier &n = spectrum()->notifier();
  n.send_notice(nt_will_change_ornament, (Ornament *) this);
  bl += dr;
  tr += dr;
  n.send_notice(nt_changed_ornament, (Ornament *) this);
}

//
// Draw an arrow.
//
void
Line::display(ODraw &dr) const
{
  double x1, y1, x2, y2;

  dr.xy_point(bl, &x1, &y1);
  dr.xy_point(tr, &x2, &y2);

  dr.set_drawing_color(GetColor());
  dr.draw_line(x1, y1, x2, y2);

  Ornament::display(dr);	// Handle selection highlighting.
}

// ----------------------------------------------------------------------------
// Send a Postscript command to draw a line.
//
void Line::print(ODraw &dr, FILE *fp,
		 double xsc, double ysc, Rectangle r) const
{
  if (!dr.is_visible(this, r))
    return;

  double x1, y1, x2, y2, x, y, w, h;

  dr.xy_point(bl, &x1, &y1);
  dr.xy_point(tr, &x2, &y2);

  x = (r.max(X) - max(x1, x2)) * xsc;
  y = (r.max(Y) - max(y1, y2)) * ysc;
  w = (fabs(x2 - x1) + dr.ornament_size(line, dr.axis(X))) * xsc;
  h = (fabs(y2 - y1) + dr.ornament_size(line, dr.axis(Y))) * ysc;

  fprintf(fp, "%d %d %d %.0f %.0f %.0f %.0f PL\n",
	  (ends + (axis == dr.axis(Y) ? 3 : 0)),
	  color_index(GetColor()),	// edge color
	  color_index(GetColor()),	// fill color
	  x, y, w, h);
}
