// ----------------------------------------------------------------------------
//

#include "list.h"		// use List
#include "memcache.h"		// use Memory_Cache
#include "table.h"		// use Table

// ----------------------------------------------------------------------------
//
class Cache_Entry
{
public:
  Cache_Entry(void *object,
	      size_t size,
	      void (*free_cb)(void *object, void *cb_data),
	      void *cb_data);
  ~Cache_Entry();

  void *object();
  size_t size();
  bool is_locked();
  void lock();
  void unlock();
  void insert_before(Cache_Entry *e);
  void free_object();
  Cache_Entry *previous();
  Cache_Entry *next();

private:
  void *obj;
  size_t sz;
  void (*free_cb)(void *object, void *cb_data);
  void *cb_data;
  int lock_count;
  Cache_Entry *prev, *nxt;		// doubly linked list

  void remove_from_list();
};

// ----------------------------------------------------------------------------
//
Memory_Cache::Memory_Cache(size_t max_size)
{
  this->max_size = max_size;
  this->current_size = 0;
  this->newest = NULL;
  this->oldest = NULL;
}

// ----------------------------------------------------------------------------
//
Memory_Cache::~Memory_Cache()
{
  Cache_Entry *next;
  for (Cache_Entry *e = newest ; e != NULL ; e = next)
    {
      next = e->next();
      delete e;
    }
}

// ----------------------------------------------------------------------------
//
void Memory_Cache::resize(size_t bytes)
{
  max_size = bytes;
  if (current_size > max_size)
    make_room();
}

// ----------------------------------------------------------------------------
//
void Memory_Cache::add_to_cache(void *object,
				size_t size,
				void (*free_cb)(void *object, void *cb_data),
				void *cb_data)
{
  Cache_Entry *e = new Cache_Entry(object, size, free_cb, cb_data);
  if (newest == NULL)
    {
      newest = e;
      oldest = e;
    }
  else
    {
      e->insert_before(newest);
      newest = e;
    }
  object_to_entry.insert((TableKey) object, (TableData) e);
  current_size = current_size + size;
  if (current_size > max_size)
    {
      e->lock();
      make_room();
      e->unlock();
    }
}

// ----------------------------------------------------------------------------
//
bool Memory_Cache::remove_from_cache(void *object)
{
  TableData entry;
  if (!object_to_entry.find((TableKey) object, &entry))
    return false;

  Cache_Entry *e = (Cache_Entry *) entry;
  remove_from_cache(e);
  delete e;

  return true;
}

// ----------------------------------------------------------------------------
//
void Memory_Cache::remove_from_cache(Cache_Entry *e)
{
  current_size = current_size - e->size();

  if (e == oldest)
    oldest = e->previous();
  if (e == newest)
    newest = e->next();

  object_to_entry.remove((TableKey) e->object());
}

// ----------------------------------------------------------------------------
//
void Memory_Cache::lock(void *object)
{
  TableData entry;
  if (object_to_entry.find((TableKey) object, &entry))
    ((Cache_Entry *) entry)->lock();
}

// ----------------------------------------------------------------------------
//
void Memory_Cache::unlock(void *object)
{
  TableData entry;
  if (object_to_entry.find((TableKey) object, &entry))
    ((Cache_Entry *) entry)->unlock();
}

// ----------------------------------------------------------------------------
//
void Memory_Cache::free_all()
{
  List elist = object_to_entry.values();
  for (int ei = 0 ; ei < elist.size() ; ++ei)
    {
      Cache_Entry *e = (Cache_Entry *) elist[ei];
      remove_from_cache(e);
      e->free_object();
      delete e;
    }
}

// ----------------------------------------------------------------------------
//
void Memory_Cache::make_room()
{
  size_t threshold = (7 * max_size) / 10;
  for (Cache_Entry *e = oldest ; e != NULL && current_size > threshold ; )
    {
      Cache_Entry *prev = e->previous();
      if (!e->is_locked())
	{
	  remove_from_cache(e);
	  e->free_object();
	  delete e;
	}
      e = prev;
    }
}


// ----------------------------------------------------------------------------
//
Cache_Entry::Cache_Entry(void *object,
			 size_t size,
			 void (*free_cb)(void *object, void *cb_data),
			 void *cb_data)
{
  this->obj = object;
  this->sz = size;
  this->free_cb = free_cb;
  this->cb_data = cb_data;
  this->lock_count = 0;
  this->prev = NULL;
  this->nxt = NULL;
}

// ----------------------------------------------------------------------------
//
Cache_Entry::~Cache_Entry()
{
  remove_from_list();
}


// ----------------------------------------------------------------------------
//
void *Cache_Entry::object()
  { return obj; }
size_t Cache_Entry::size()
  { return sz; }
bool Cache_Entry::is_locked()
  { return (lock_count > 0); }
void Cache_Entry::lock()
  { lock_count += 1; }
void Cache_Entry::unlock()
  { lock_count -= 1; }
void Cache_Entry::free_object()
  { (*free_cb)(obj, cb_data); }
Cache_Entry *Cache_Entry::previous()
  { return prev; }
Cache_Entry *Cache_Entry::next()
  { return nxt; }

// ----------------------------------------------------------------------------
//
void Cache_Entry::insert_before(Cache_Entry *e)
{
  remove_from_list();

  this->prev = e->prev;
  this->nxt = e;
  if (e->prev)
    e->prev->nxt = this;
  e->prev = this;
}

// ----------------------------------------------------------------------------
//
void Cache_Entry::remove_from_list()
{
  if (this->prev)
    this->prev->nxt = this->nxt;
  if (this->nxt)
    this->nxt->prev = this->prev;
}
