/*
 * Molecule.C:		Implementation of the Molecule class 
 *
 * A Molecule is the object that holds the Atoms and Groups. The Molecule is
 * responsible for creating and destroying Atoms and Groups
 *
 */
#include "atom.h"		// use Atom
#include "condition.h"		// use Condition
#include "constant.h"
#include "group.h"		// use Group
#include "memalloc.h"		// use new()
#include "model.h"		// use Model
#include "molecule.h"		// use Molecule
#include "notifier.h"		// use Notifier
#include "resonance.h"		// use Resonance
#include "session.h"		// use Session
#include "stringc.h"		// use hash_string(), equal_strings()
#include "table.h"		// use Table

// ----------------------------------------------------------------------------
//
class String_Pair
{
public:
  String_Pair(const Stringy *s1, const Stringy *s2, bool copy_strings);
  virtual ~String_Pair();
  static bool equal(const void *sp1, const void *sp2);
  static unsigned long hash(const void *sp);
private:
  const Stringy *s1, *s2;
  bool copy;
};

// ----------------------------------------------------------------------------
//

//
// Construct a molecule in the DB.
//
Molecule::Molecule(Session &s, const Stringy &name)
  : ses(s), name_to_atom(String_Pair::equal, String_Pair::hash)
{
	mName = name;

	session().notifier().send_notice(nt_created_molecule, this);
}

// ----------------------------------------------------------------------------
//
Molecule::~Molecule()
{
  Notifier &n = session().notifier();
  n.send_notice(nt_will_delete_molecule, this);

  List clist = condition_list();
  for (int ci = 0 ; ci < clist.size() ; ++ci)
    delete (Condition *) clist[ci];

  List alist = AtomList();
  for (int ai = 0 ; ai < alist.size() ; ++ai)
    {
      Atom *a = (Atom *) alist[ai];
      n.send_notice(nt_will_delete_atom, a);
      delete a;
    }

  List glist = GroupList();
  for (int gi = 0 ; gi < glist.size() ; ++gi)
    {
      Group *g = (Group *) glist[gi];
      n.send_notice(nt_will_delete_group, g);
      delete g;
    }

  List nlist = name_to_atom.keys();
  for (int ni = 0 ; ni < nlist.size() ; ++ni)
    delete (String_Pair *) nlist[ni];
}

// ----------------------------------------------------------------------------
//
const List &Molecule::condition_list() const
  { return mCondition; }

// ----------------------------------------------------------------------------
//
void Molecule::add_condition(Condition *c)
{
  mCondition.append(c);
  session().notifier().send_notice(nt_added_condition_to_molecule, this);
}

// ----------------------------------------------------------------------------
//
void Molecule::remove_condition(Condition *c)
{
  mCondition.erase(c);
  session().notifier().send_notice(nt_removed_condition_from_molecule, this);
}

// ----------------------------------------------------------------------------
//
Condition *Molecule::define_condition(const Stringy &condition_name)
{
  Condition *c = find_condition(condition_name);
  if (c == NULL)
    c = new Condition(this, condition_name);

  return c;
}

// ----------------------------------------------------------------------------
//
Condition *Molecule::find_condition(const Stringy &condition_name)
{
  const List &clist = condition_list();
  for (int ci = 0 ; ci < clist.size() ; ++ci)
    {
      Condition *c = (Condition *) clist[ci];
      if (c->name() == condition_name)
	return c;
    }

  return NULL;
}

// ----------------------------------------------------------------------------
//
const List &Molecule::AtomList(void) const
  { return mAtom;	}

// ----------------------------------------------------------------------------
//
void Molecule::add_atom(Atom *a)
{
  mAtom.append(a);

  String_Pair *ga = new String_Pair(&a->group()->name(), &a->name(), true);
  name_to_atom.insert(ga, a);
}

// ----------------------------------------------------------------------------
//
void Molecule::remove_atom(Atom *a)
{
  mAtom.erase(a);

  const Stringy &gname = a->group()->name();
  const Stringy &aname = a->name();
  String_Pair ga(&gname, &aname, false);
  String_Pair *key = (String_Pair *) name_to_atom.remove((TableKey) &ga);
  delete key;
}

// ----------------------------------------------------------------------------
//
const List &Molecule::GroupList() const
  { return mGroup; }

// ----------------------------------------------------------------------------
//
void Molecule::add_group(Group *g)
{
	mGroup.append(g);
}

// ----------------------------------------------------------------------------
//
void Molecule::remove_group(Group *g)
{
	mGroup.erase(g);
}

//
// Add a named Atom to this Molecule, returning a pointer to the Atom storage.
//
Atom *Molecule::define_atom(const Stringy &group, const Stringy &atom,
			    const Stringy &nucleus)
{
  Stringy gname = (group.is_white() ? "?" : group.cstring());
  Stringy aname = (atom.is_white() ? "?" : atom.cstring());
	
  Atom *ap = find_atom(gname, aname);
  if (ap == NULL)
    if (aname != "?" || gname != "?")
      {
	Group *gp = define_group(gname);
	ap = new Atom(aname, nucleus, gp);
	add_atom(ap);
      }

  return ap;
}

// ----------------------------------------------------------------------------
//
Atom *Molecule::find_atom(const Stringy &group, const Stringy &atom) const
{
  TableData a;
  String_Pair ga(&group, &atom, false);
  return (name_to_atom.find((TableKey) &ga, &a) ? (Atom *) a : NULL);
}

/*
 * Create a Group with name <nm> (if given) and Group ID <id>.
 *
 * Returns:
 *	A pointer to a new Group or to the existing Group.
 */
Group *Molecule::define_group(const Stringy &nm)
{
  Stringy name = (nm.is_white() ? "?" : nm.cstring());

  Group	*gp = find_group(name);
  if (gp == NULL)
    {
      gp = new Group(name);
      add_group(gp);
    }

  return gp;
}

/*
 * Locate the Group named <nm> in the Molecule.
 *
 * Returns:
 *	A pointer to the existing Group, or NULL if it doesn't exist.
 */
Group *Molecule::find_group(const Stringy &name) const
{
	const List &glist = GroupList();
	int size = glist.size();
	for (int gp = 0 ; gp < size ; ++gp) {
		Group *g = (Group *) glist[gp];
		if (g->name() == name)
			return g;
	}

	return NULL;
}

/*
 * Returns:
 *	A sorted collection of atoms that have Group <gp>.
 */
List Molecule::AtomList(const Group *gp) const
{
	List	galist;

	const List &alist = AtomList();
	for (int ap = 0 ; ap < alist.size() ; ++ap) {
		Atom *a = (Atom *) alist[ap];
		if (a->group() == gp)
			galist.append(a);
	}

	return galist;
}

// ----------------------------------------------------------------------------
//
Stringy Molecule::name() const
  { return mName; }
Session &Molecule::session()
  { return ses; }

// ----------------------------------------------------------------------------
//
const List &Molecule::model_list() const
  { return mModel; }
void Molecule::add_model(Model *m)
  { mModel.append(m); }
void Molecule::remove_model(Model *m)
  { mModel.erase(m); delete m; }
void Molecule::remove_all_models()
{
  const List &models = model_list();
  for (int mi = 0 ; mi < models.size() ; ++mi)
    remove_model((Model *) models[mi]);
}

// ----------------------------------------------------------------------------
//
AttachedData &Molecule::saved_values()
  { return attached_data; }

// ----------------------------------------------------------------------------
//
String_Pair::String_Pair(const Stringy *s1, const Stringy *s2, bool copy)
{
  this->s1 = (copy ? new Stringy(*s1) : s1);
  this->s2 = (copy ? new Stringy(*s2) : s2);
  this->copy = copy;
}

// ----------------------------------------------------------------------------
//
String_Pair::~String_Pair()
{
  if (copy)
    {
      delete s1;
      delete s2;
    }
}

// ----------------------------------------------------------------------------
//
bool String_Pair::equal(const void *spair1, const void *spair2)
{
  String_Pair *sp1 = (String_Pair *) spair1;
  String_Pair *sp2 = (String_Pair *) spair2;
  return *(sp1->s1) == *(sp2->s1) && *(sp1->s2) == *(sp2->s2);
}

// ----------------------------------------------------------------------------
//
unsigned long String_Pair::hash(const void *spair)
{
  String_Pair *sp = (String_Pair *) spair;
  return hash_combine(hash_string(sp->s1), hash_string(sp->s2));
}
