// ----------------------------------------------------------------------------
// NIH NMR data file format
//
#include <stdio.h>		// Use FILE
#include <string.h>		// Use strcmp()

#include "binaryIO.h"		// Use read_f32(), ...
#include "blockfile.h"		// use Block_File
#include "list.h"		// Use List
#include "memalloc.h"		// use new()
#include "nmrdata.h"		// Use NMR_Data
#include "spoint.h"		// Use IPoint, SPoint
#include "stringc.h"		// Use Stringy
#include "system.h"		// Use file_modification_time()
#include "utility.h"		// use fatal_error()

//
// Some NIH format constants
//

#define FDATASIZE          512   /* Length of header in 4-byte float values. */
#define FDIEEECONS   0xeeeeeeee  /* Indicates IEEE floating point format.    */
#define FDVAXCONS    0x11111111  /* Indicates DEC VAX floating point format. */
#define FDORDERCONS       2.345  /* Constant used to determine byte-order.   */
#define FDFMTCONS    FDIEEECONS  /* Floating point format on this computer.  */
#define ZERO_EQUIV       -666.0  /* Used as equivalent for zero.             */

//
// Parameter locations:
//

#define FDMAGIC        0 /* Should be zero in valid NMRi data.               */
#define FDFLTFORMAT    1 /* Constant defining floating point format.         */
#define FDFLTORDER     2 /* Constant defining byte order.                    */
#define FDSIZE        99 /* Number of points in current dim R|I.             */
#define FDREALSIZE    97 /* Number of valid time domain pts in aq dim R|I.   */
#define FDF2APOD      95 /* Number of valid time domain pts in aq dim R|I.   */
#define FDF1APOD     428 /* Number of valid time domain pts in aq+1 dim R|I. */
#define FDSPECNUM    219 /* Number of complex 1D slices in file.             */
#define FDTRANSPOSED 221 /* 1=Transposed relative to aq, 0=Not Transposed.   */
#define FDQUADFLAG   106 /* 0=Complex X or Y 1=Real-Only 2=Pseudo-Quad.      */
#define FDF2QUADFLAG  56 /* Non-standard.                                    */

#define FDMAX        247 
#define FDMIN        248
#define FDSCALEFLAG  250
#define FDDISPMAX    251
#define FDDISPMIN    252

#define FDF2SW       100
#define FDF2OBS      119
#define FDF2ORIG     101
#define FDF2UNITS    152
#define FDF2FTFLAG   220
#define FDF2LB       111
#define FDF2CAR       66 /* Non-standard. */
#define FDF2CENTER    79 /* Non-standard. */

#define FDF1SW       229 
#define FDF1OBS      218 
#define FDF1ORIG     249 
#define FDF1UNITS    234 
#define FDF1FTFLAG   222 
#define FDF1LB       243
#define FDF1QUADFLAG  55 /* Non-standard. */
#define FDF1CAR       67 /* Non-standard. */
#define FDF1CENTER    80 /* Non-standard. */

#define FDPIPEFLAG    57 /* Non-standard. */
#define FDSLICECOUNT 443 /* Non-standard. */
#define FDFILECOUNT  442

#define FDPIPECOUNT   75 /* Non-standard. */
#define FDCARFLAGS    76 /* Non-standard. */
#define FDFIRSTPLANE  77 /* Non-standard. */
#define FDLASTPLANE   78 /* Non-standard. */

#define FDDIMCOUNT     9
#define FDPLANELOC    14

#define FDF2LABEL     16
#define FDF1LABEL     18
#define FDF3LABEL     20
#define FDF4LABEL     22

#define FDDIMORDER    24
#define FDDIMORDER1   24
#define FDDIMORDER2   25 
#define FDDIMORDER3   26
#define FDDIMORDER4   27

#define FDF3OBS       10
#define FDF3SW        11
#define FDF3ORIG      12
#define FDF3FTFLAG    13
#define FDF3SIZE      15
#define FDF3APOD      50 /* Non-standard. */
#define FDF3QUADFLAG  51 /* Non-standard. */
#define FDF3UNITS     58 /* Non-standard. */
#define FDF3P0        60 /* Non-standard. */
#define FDF3P1        61 /* Non-standard. */
#define FDF3CAR       68 /* Non-standard. */
#define FDF3CENTER    81 /* Non-standard. */

#define FDF4OBS       28
#define FDF4SW        29
#define FDF4ORIG      30
#define FDF4FTFLAG    31
#define FDF4SIZE      32
#define FDF4APOD      53 /* Non-standard. */
#define FDF4QUADFLAG  54 /* Non-standard. */
#define FDF4UNITS     59 /* Non-standard. */
#define FDF4P0        62 /* Non-standard. */
#define FDF4P1        63 /* Non-standard. */
#define FDF4CAR       69 /* Non-standard. */
#define FDF4CENTER    82 /* Non-standard. */

#define FDAQSIGN      64 /* Non-standard. */
#define FDPARTITION   65 /* Non-standard. */

#define FDUSER1       70
#define FDUSER2       71
#define FDUSER3       72
#define FDUSER4       73
#define FDUSER5       74

#define FDMCFLAG     135
#define FDF2P0       109
#define FDF2P1       110
#define FDF1P0       245
#define FDF1P1       246

#define FDLASTBLOCK  359
#define FDCONTBLOCK  360
#define FDBASEBLOCK  361
#define FDPEAKBLOCK  362
#define FDBMAPBLOCK  363
#define FDHISTBLOCK  364
#define FD1DBLOCK    365

#define FDMONTH      294
#define FDDAY        295
#define FDYEAR       296

#define FDSRCNAME    286  /* char srcFile[16]  */
#define FDUSERNAME   290  /* char uName[16]    */
#define FDOPERNAME   464  /* char oName[32]    */
#define FDTITLE      297  /* char title[60]    */
#define FDCOMMENT    312  /* char comment[160] */

#define SIZE_NDLABEL    8
#define SIZE_F2LABEL    8
#define SIZE_F1LABEL    8
#define SIZE_F3LABEL    8
#define SIZE_F4LABEL    8

#define SIZE_SRCNAME   16
#define SIZE_USERNAME  16
#define SIZE_OPERNAME  32
#define SIZE_COMMENT  160
#define SIZE_TITLE     60

#define FDNOISE       153
#define FDRANK        180 /* Non-standard. */
#define FDTEMPERATURE 157
#define FD2DVIRGIN    399
#define FD2DPHASE     256
#define FDTAU         199

//
// The following are definitions for generalized ND parameters:
//

#define NDPARM      1000

#define NULL_DIM       0
#define CUR_XDIM       1
#define CUR_YDIM       2
#define CUR_ZDIM       3
#define CUR_ADIM       4

#define ABS_XDIM      -1
#define ABS_YDIM      -2
#define ABS_ZDIM      -3
#define ABS_ADIM      -4

#define CUR_HDIM       9 
#define CUR_VDIM      10
 
#define NDSIZE        (1+NDPARM)
#define NDAPOD        (2+NDPARM)
#define NDSW          (3+NDPARM)
#define NDORIG        (4+NDPARM)
#define NDOBS         (5+NDPARM)
#define NDFTFLAG      (6+NDPARM)
#define NDQUADFLAG    (7+NDPARM)
#define NDUNITS       (8+NDPARM)
#define NDLABEL       (9+NDPARM)
#define NDLABEL1      (9+NDPARM)
#define NDLABEL2     (10+NDPARM)
#define NDP0         (11+NDPARM)
#define NDP1         (12+NDPARM)
#define NDCAR        (13+NDPARM)
#define NDCENTER     (14+NDPARM)

#define FD_SEC       1
#define FD_HZ        2
#define FD_PPM       3
#define FD_PTS       4

#define FD_MAGNITUDE 0
#define FD_TPPI      1
#define FD_STATES    2

#define FD_QUAD       0
#define FD_SINGLATURE 1
#define FD_PSEUDOQUAD 2

// ----------------------------------------------------------------------------
// The axes of an nmr pipe file are labelled XYZA.  Data order is such
// that the X axis varies fastest, next Y, next Z, next A.  The indices
// 0-3 in the header are XYZA.
//
#define MAXNIHDIM 4

struct NIH_Header
{
  int dim;
  bool big_endian;			// file byte order
  int size[MAXNIHDIM];
  int axis_number[MAXNIHDIM];
  bool complex[MAXNIHDIM];
  float spectral_width[MAXNIHDIM];
  float xmtr_freq[MAXNIHDIM];
  float origin_freq[MAXNIHDIM];		// freq at last data point
  float spectrometer_freq[MAXNIHDIM];
  Stringy nucleus[MAXNIHDIM];
};

// ----------------------------------------------------------------------------
//
class NIH_Block_File : public Block_File
{
public:
  NIH_Block_File(const Stringy &path, const NIH_Header &hnih, Memory_Cache *);
  virtual ~NIH_Block_File();

  virtual bool read_block(const IPoint &block, float *values);
  virtual bool write_block(const IPoint &block, float *values);
private:
  FILE *fp;
  bool big_endian;
  IPoint axis_step;

  long block_location(const IPoint &block);
};

static IPoint nih_data_size(const NIH_Header &hnih);
static IPoint nih_block_size(const NIH_Header &hnih);
static SPoint nih_spectrometer_frequency(const NIH_Header &hnih);
static SPoint nih_spectral_width(const NIH_Header &hnih);
static SPoint nih_origin_ppm(const NIH_Header &hnih);
static List nih_axis_labels(const NIH_Header &hnih);
static bool read_nih_header(FILE *nihfile, const Stringy &path,
			    NIH_Header *h, Stringy *error_msg);
static Stringy read_nih_label(int position, FILE *nihfile);
static bool read_nih_float(int position, FILE *nihfile,
			   bool big_endian, float *value);
static bool read_nih_floats(int position, int num, FILE *nihfile,
			    bool big_endian, float *value);

// ----------------------------------------------------------------------------
//
NMR_Data *nih_nmr_data(const Stringy &path, Stringy *error_msg,
		       Memory_Cache *mcache)
{
  FILE *fp = fopen(path.cstring(), "rb");
  if (!fp)
    { *error_msg = "Couldn't open " + path; return NULL; }

  NIH_Header hnih;
  bool readable = read_nih_header(fp, path, &hnih, error_msg);

  fclose(fp);

  if (!readable)
      return NULL;

  Block_File *bf = new NIH_Block_File(path, hnih, mcache);
  NMR_Data *nmr_data = block_file_nmr_data(bf,
					   nih_spectrometer_frequency(hnih),
					   nih_spectral_width(hnih),
					   nih_origin_ppm(hnih),
					   nih_axis_labels(hnih));

  return nmr_data;
}

// ----------------------------------------------------------------------------
//
bool is_nih_nmr_data(const Stringy &path)
{
  FILE *fp = fopen(path.cstring(), "rb");
  if (!fp)
    return false;

  Stringy error_msg;
  NIH_Header hnih;
  bool success = read_nih_header(fp, path, &hnih, &error_msg);
  fclose(fp);

  return success;
}

// ----------------------------------------------------------------------------
//
NIH_Block_File::NIH_Block_File(const Stringy &path, const NIH_Header &hnih,
			       Memory_Cache *mcache) :
  Block_File(path, nih_data_size(hnih), nih_block_size(hnih), mcache)
{
  FILE *fp = fopen(path.cstring(), "rb");
  if (fp == NULL)
    fatal_error("NIH_Block_File::Failed opening: %s", path.cstring());

  IPoint axis_step = IPoint(hnih.dim);
  int bsize = 1;
  for (int d = 0 ; d < hnih.dim ; ++d)
    {
      int a = hnih.axis_number[d];
      axis_step[a] = bsize;
      bsize *= (hnih.complex[d] ? 2 : 1) * hnih.size[d];
    }

  this->fp = fp;
  this->big_endian = hnih.big_endian;
  this->axis_step = axis_step;
}

// ----------------------------------------------------------------------------
//
NIH_Block_File::~NIH_Block_File()
{
  flush_cache();
  fclose(fp);
  fp = NULL;
}

// ----------------------------------------------------------------------------
//
bool NIH_Block_File::read_block(const IPoint &block, float *values)
{
  long position = block_location(block);
  long size = block_volume();

  return read_nih_floats(position, size, this->fp, this->big_endian, values);
}

// ----------------------------------------------------------------------------
//
bool NIH_Block_File::write_block(const IPoint &, float *)
{
  return false;		// Not implemented.
}

// ----------------------------------------------------------------------------
//
long NIH_Block_File::block_location(const IPoint &block)
{
  long offset = FDATASIZE;

  for (int a = 0 ; a < block.dimension() ; ++a)
    offset += axis_step[a] * block[a];
      
  return offset;
}

// ----------------------------------------------------------------------------
//
static IPoint nih_data_size(const NIH_Header &hnih)
{
  IPoint size(hnih.dim);
  for (int d = 0 ; d < size.dimension() ; ++d)
    size[hnih.axis_number[d]] = hnih.size[d];
  return size;
}

// ----------------------------------------------------------------------------
//
static IPoint nih_block_size(const NIH_Header &hnih)
{
  IPoint bsize(hnih.dim);
  bsize[hnih.axis_number[0]] = hnih.size[0];
  for (int d = 1 ; d < bsize.dimension() ; ++d)
    bsize[hnih.axis_number[d]] = 1;
  return bsize;
}

// ----------------------------------------------------------------------------
//
static SPoint nih_spectrometer_frequency(const NIH_Header &hnih)
{
  SPoint freq(hnih.dim);
  for (int d = 0 ; d < freq.dimension() ; ++d)
    freq[hnih.axis_number[d]] = hnih.spectrometer_freq[d];
  return freq;
}

// ----------------------------------------------------------------------------
//
static SPoint nih_spectral_width(const NIH_Header &hnih)
{
  SPoint swidth(hnih.dim);
  for (int d = 0 ; d < swidth.dimension() ; ++d)
    swidth[hnih.axis_number[d]] = hnih.spectral_width[d];
  return swidth;
}

// ----------------------------------------------------------------------------
//
static SPoint nih_origin_ppm(const NIH_Header &hnih)
{
  SPoint origin_ppm(hnih.dim);
  for (int d = 0 ; d < origin_ppm.dimension() ; ++d)
    {
      double hz_per_point = hnih.spectral_width[d]/hnih.size[d];
      origin_ppm[hnih.axis_number[d]] =
	((hnih.origin_freq[d] + hnih.spectral_width[d] - hz_per_point)
	 / hnih.spectrometer_freq[d]);
    }
  return origin_ppm;
}

// ----------------------------------------------------------------------------
//
static List nih_axis_labels(const NIH_Header &hnih)
{
  Stringy clabels[MAXNIHDIM];
  for (int d = 0 ; d < hnih.dim ; ++d)
    clabels[hnih.axis_number[d]] = hnih.nucleus[d];

  List labels;
  for (int a = 0 ; a < hnih.dim ; ++a)
    labels.append(new Stringy(clabels[a]));

  return labels;
}

// ----------------------------------------------------------------------------
//
static int sz[MAXNIHDIM] = {FDSIZE, FDSPECNUM, FDF3SIZE, FDF4SIZE};
static int qf[MAXNIHDIM] = {FDF1QUADFLAG, FDF2QUADFLAG,
			    FDF3QUADFLAG, FDF4QUADFLAG};
static int order[MAXNIHDIM] = {FDDIMORDER1, FDDIMORDER2,
			       FDDIMORDER3, FDDIMORDER4}; 

static int sw[MAXNIHDIM] = {FDF1SW, FDF2SW, FDF3SW, FDF4SW};
static int xf[MAXNIHDIM] = {FDF1CAR, FDF2CAR, FDF3CAR, FDF4CAR};
static int of[MAXNIHDIM] = {FDF1ORIG, FDF2ORIG, FDF3ORIG, FDF4ORIG};
static int sf[MAXNIHDIM] = {FDF1OBS, FDF2OBS, FDF3OBS, FDF4OBS};
static int nuc[MAXNIHDIM] = {FDF1LABEL, FDF2LABEL, FDF3LABEL, FDF4LABEL};

static bool read_nih_header(FILE *nihfile, const Stringy &path,
			    NIH_Header *h, Stringy *error_msg)
{
  Stringy header_error = "Error reading " + path + " header";

  float magic;
  if (!read_nih_float(FDMAGIC, nihfile, true, &magic) || magic != 0)
    { *error_msg = header_error; return false; }

  //
  // Check if file byte order is little endian or big endian.
  //
  char big_endian_order[4] = {0x40, 0x16, 0x14, 0x7b};
  char byte_order[4];
  if (!seek_position(nihfile, 4 * FDFLTORDER, 0) ||
      !read_u8_array(byte_order, 4, nihfile))
    { *error_msg = header_error; return false; }
  bool bigend = (memcmp(byte_order, big_endian_order, 4) == 0);
  h->big_endian = bigend;

  float dimen;
  if (!read_nih_float(FDDIMCOUNT, nihfile, bigend, &dimen))
    { *error_msg = header_error; return false; }
  h->dim = (int) dimen;

  for (int d = 0 ; d < h->dim ; ++d)
    {
      float size, complex;
      if (!read_nih_float(sz[d], nihfile, bigend, &size) ||
	  !read_nih_float(qf[d], nihfile, bigend, &complex))
	{ *error_msg = header_error; return false; }

      h->size[d] = (int) size;
      h->complex[d] = (complex == 0);

      //
      // Correct the sizes to give number of complex data points.
      //
      if ((d == 1 && h->complex[0] && h->complex[1]) ||
	  (d >= 2 && h->complex[d]))
	h->size[d] /= 2;

      float aorder;
      if (!read_nih_float(order[d], nihfile, bigend, &aorder))
	{ *error_msg = header_error; return false; }

      int dnum = (int) aorder - 1;

      //
      // Map the bruk2pipe, var2pipe default order 2134 to descending order.
      // For 2d data to 21, 3d data 321, 4d data 4321.
      //
      h->axis_number[d] = (dnum == 1 ? h->dim - 1 :
			   (dnum == 0 ? h->dim - 2 :
			    h->dim - dnum - 1));

      if (!read_nih_float(sw[dnum], nihfile, bigend, &h->spectral_width[d]) ||
	  !read_nih_float(xf[dnum], nihfile, bigend, &h->xmtr_freq[d]) ||
	  !read_nih_float(of[dnum], nihfile, bigend, &h->origin_freq[d]) ||
	  !read_nih_float(sf[dnum], nihfile, bigend, &h->spectrometer_freq[d]))
	{ *error_msg = header_error; return false; }

      h->nucleus[d] = read_nih_label(nuc[dnum], nihfile);
    }

  //
  // Check if file is too small.
  //
  unsigned long dsize = 1;
  for (int d = 0 ; d < h->dim ; ++d)
    dsize *= h->size[d];
  unsigned long expected_size = 4 * (FDATASIZE + dsize);
  unsigned long size = file_size(path);
  if (size < expected_size)
    {
      *error_msg = formatted_string("%s is smaller (%ld bytes) "
				    "than expected (%ld bytes)",
				    path.cstring(), size, expected_size);
      return false;
    }


  return true;
}

// ----------------------------------------------------------------------------
//
static Stringy read_nih_label(int position, FILE *nihfile)
{
  char label[4];

  if (!seek_position(nihfile, 4 * position, 0) ||
      !read_u8_array(label, 4 * sizeof(char), nihfile))
    return "";

  return standard_nucleus_name(label);
}

// ----------------------------------------------------------------------------
//
static bool read_nih_float(int position, FILE *nihfile, bool big_endian,
			   float *value)
{
  return (seek_position(nihfile, 4 * position, 0) &&
	  read_f32(value, nihfile, big_endian));
}

// ----------------------------------------------------------------------------
//
static bool read_nih_floats(int position, int num, FILE *nihfile,
			    bool big_endian, float *value)
{
  return (seek_position(nihfile, 4 * position, 0) &&
	  read_f32_array(value, num, nihfile, big_endian));
}
