// ----------------------------------------------------------------------------
// NMR data file reading and writing.
//

#include <iostream.h>		// use ostream
#include <math.h>		// use fabs()

#include "blockfile.h"		// Use Block_File, block_region()
#include "list.h"		// Use List
#include "memalloc.h"		// use new()
#include "nmrdata.h"		// Use NMR_Data
#include "spoint.h"		// Use SPoint, IPoint

static List permuted_label_list(NMR_Data *nmr_data, const IPoint &new_order);
static SPoint region_spectrum_width(NMR_Data *nmr_data, const IRegion &region);
static SPoint region_origin_ppm(NMR_Data *nmr_data, const IRegion &region);
static IPoint reduced_size(const IPoint &size, const IPoint &bsize);
static SPoint corrected_origin_ppm(NMR_Data *nmrdata, const IPoint &bsize);
static List remove_list_component(const List &list, int pos);
static Stringy header_line(const Stringy &name, const IPoint &value);
static Stringy header_line(const Stringy &name, const SPoint &value);
static Stringy header_line(const Stringy &name, const List &strings);

// ----------------------------------------------------------------------------
//
NMR_Data::NMR_Data(const Stringy &path,
		   const IPoint &size,
		   const IPoint &block_size,
		   const SPoint &spectrometer_freq,
		   const SPoint &spectrum_width,
		   const SPoint &origin_ppm,
		   const List &axis_labels)
{
  this->data_path = path;
  this->data_size = size;
  this->bsize = block_size;
  this->sfreq = spectrometer_freq;
  this->swidth = spectrum_width;
  this->oppm = origin_ppm;
  this->axis_labels = duplicate_string_list(axis_labels);
}

// ----------------------------------------------------------------------------
//
NMR_Data::~NMR_Data()
{
  free_string_list_entries(axis_labels);
}

// ----------------------------------------------------------------------------
//
Stringy NMR_Data::path() const
  { return data_path; }
int NMR_Data::dimension() const
  { return data_size.dimension(); }
IPoint NMR_Data::size() const
  { return data_size; }
IPoint NMR_Data::block_size() const
  { return bsize; }
SPoint NMR_Data::spectrometer_frequency() const
  { return sfreq; }
SPoint NMR_Data::spectrum_width() const
  { return swidth; }
SPoint NMR_Data::origin_ppm() const
  { return oppm; }
Stringy NMR_Data::axis_label(int axis) const
  { return *(Stringy *) axis_labels[axis]; }

// ----------------------------------------------------------------------------
//
const List &NMR_Data::axis_label_list() const
{
  return axis_labels;
}

// ----------------------------------------------------------------------------
//
IRegion NMR_Data::data_region() const
{
  IRegion r(dimension());
  for (int a = 0 ; a < dimension() ; ++a)
    {
      r.min[a] = 0;
      r.max[a] = size()[a] - 1;
    }
  return r;
}

// ----------------------------------------------------------------------------
//
class Permuted_NMR_Data : public NMR_Data
{
public:
  Permuted_NMR_Data(NMR_Data *nmr_data, const IPoint &new_order);
  virtual ~Permuted_NMR_Data();
  virtual bool read_value(const IPoint &index, float *value);
  virtual bool read_values(const IRegion &index_region, float *values);
private:
  NMR_Data *nmr_data;
  IPoint new_order;
};

// ----------------------------------------------------------------------------
//
Permuted_NMR_Data::Permuted_NMR_Data(NMR_Data *nmr_data,
				     const IPoint &new_order)
 : NMR_Data(nmr_data->path(),
	    nmr_data->size().permute(new_order),
	    nmr_data->block_size().permute(new_order),
	    nmr_data->spectrometer_frequency().permute(new_order),
	    nmr_data->spectrum_width().permute(new_order),
	    nmr_data->origin_ppm().permute(new_order),
	    permuted_label_list(nmr_data, new_order))
{
  this->nmr_data = nmr_data;
  this->new_order = new_order;
}

// ----------------------------------------------------------------------------
//
Permuted_NMR_Data::~Permuted_NMR_Data()
{
  delete nmr_data;
  nmr_data = NULL;
}

// ----------------------------------------------------------------------------
//
bool Permuted_NMR_Data::read_value(const IPoint &index, float *value)
  { return nmr_data->read_value(index.unpermute(new_order), value); }

// ----------------------------------------------------------------------------
//
bool Permuted_NMR_Data::read_values(const IRegion &index_region, float *values)
{
  IPoint unperm_min = index_region.min.unpermute(new_order);
  IPoint unperm_max = index_region.max.unpermute(new_order);
  IRegion unperm_region = IRegion(unperm_min, unperm_max);

  float *unperm_values = new float [index_region.volume()];
  bool read = nmr_data->read_values(unperm_region, unperm_values);
  if (read)
    {
      // Unpermute array format

      Subarray_Iterator vi(index_region, index_region, true);
      for ( ; !vi.finished() ; vi.next())
	{
	  int k = vi.index();
	  int unperm_k = position_index(vi.position().unpermute(new_order),
					unperm_region, true);
	  values[k] = unperm_values[unperm_k];
	}
    }
  delete [] unperm_values;

  return read;
}

// ----------------------------------------------------------------------------
//
static List permuted_label_list(NMR_Data *nmr_data, const IPoint &new_order)
{
  List labels;
  for (int a = 0 ; a < nmr_data->dimension() ; ++a)
    labels.append(new Stringy(nmr_data->axis_label(new_order[a])));
  return labels;
}

// ----------------------------------------------------------------------------
//
NMR_Data *permute_nmr_axes(NMR_Data *nmr_data, const IPoint &new_order)
  { return new Permuted_NMR_Data(nmr_data, new_order); }

// ----------------------------------------------------------------------------
//
class NMR_Data_Subregion : public NMR_Data
{
public:
  NMR_Data_Subregion(NMR_Data *nmr_data, const IRegion &region);
  virtual ~NMR_Data_Subregion();
  virtual bool read_value(const IPoint &position, float *value);
  virtual bool read_values(const IRegion &region, float *values);
private:
  NMR_Data *nmr_data;
  IRegion region;
};

// ----------------------------------------------------------------------------
//
NMR_Data_Subregion::NMR_Data_Subregion(NMR_Data *nmr_data,
				       const IRegion &region)
 : NMR_Data(nmr_data->path(),
	    region.size(),
	    nmr_data->block_size(),
	    nmr_data->spectrometer_frequency(),
	    region_spectrum_width(nmr_data, region),
	    region_origin_ppm(nmr_data, region),
	    nmr_data->axis_label_list())
{
  this->nmr_data = nmr_data;
  this->region = region;
}

// ----------------------------------------------------------------------------
//
NMR_Data_Subregion::~NMR_Data_Subregion()
{
  delete nmr_data;
  nmr_data = NULL;
}

// ----------------------------------------------------------------------------
//
bool NMR_Data_Subregion::read_value(const IPoint &position, float *value)
{
  IPoint p = position + region.min;
  bool read = nmr_data->read_value(p, value);
  if (read && !region.contains(p))
    *value = 0;
  return read;
}

// ----------------------------------------------------------------------------
//
bool NMR_Data_Subregion::read_values(const IRegion &r, float *values)
{
  IRegion reg = r + region.min;
  bool read = nmr_data->read_values(reg, values);
  if (read && !region.contains(reg))
    {
      Subarray_Iterator ri(reg, reg, true);
      for ( ; !ri.finished() ; ri.next())
	if (!region.contains(ri.position()))
	  values[ri.index()] = 0;
    }
  return read;
}

// ----------------------------------------------------------------------------
//
static SPoint region_spectrum_width(NMR_Data *nmr_data, const IRegion &region)
{
  SPoint swidth = nmr_data->spectrum_width();
  for (int a = 0 ; a < region.dimension() ; ++a)
    swidth[a] *= (double) region.size(a) / nmr_data->size()[a];
  return swidth;
}

// ----------------------------------------------------------------------------
//
static SPoint region_origin_ppm(NMR_Data *nmr_data, const IRegion &region)
{
  SPoint origin = nmr_data->origin_ppm();
  for (int a = 0 ; a < region.dimension() ; ++a)
    {
      double ppm_step = ((nmr_data->spectrum_width()[a] / nmr_data->size()[a])
			 / nmr_data->spectrometer_frequency()[a]);
      origin[a] -= region.min[a] * ppm_step;
    }
  return origin;
}

// ----------------------------------------------------------------------------
//
NMR_Data *nmr_data_region(NMR_Data *nmr_data, const IRegion &region)
  { return new NMR_Data_Subregion(nmr_data, region); }

// ----------------------------------------------------------------------------
//
class Thresholded_NMR_Data : public NMR_Data
{
public:
  Thresholded_NMR_Data(NMR_Data *nmr_data,
		       float neg_threshold, float pos_threshold);
  virtual ~Thresholded_NMR_Data();
  virtual bool read_value(const IPoint &position, float *value);
  virtual bool read_values(const IRegion &region, float *values);
private:
  NMR_Data *nmr_data;
  float neg_threshold, pos_threshold;
};

// ----------------------------------------------------------------------------
//
Thresholded_NMR_Data::Thresholded_NMR_Data(NMR_Data *nmr_data,
					   float neg_threshold,
					   float pos_threshold)
 : NMR_Data(nmr_data->path(),
	    nmr_data->size(),
	    nmr_data->block_size(),
	    nmr_data->spectrometer_frequency(),
	    nmr_data->spectrum_width(),
	    nmr_data->origin_ppm(),
	    nmr_data->axis_label_list())
{
  this->nmr_data = nmr_data;
  this->neg_threshold = neg_threshold;
  this->pos_threshold = pos_threshold;
}

// ----------------------------------------------------------------------------
//
Thresholded_NMR_Data::~Thresholded_NMR_Data()
{
  delete nmr_data;
  nmr_data = NULL;
}

// ----------------------------------------------------------------------------
//
bool Thresholded_NMR_Data::read_value(const IPoint &position, float *value)
{
  bool read = nmr_data->read_value(position, value);
  if (read && *value > neg_threshold && *value < pos_threshold)
    *value = 0;
  return read;
}

// ----------------------------------------------------------------------------
//
bool Thresholded_NMR_Data::read_values(const IRegion &r, float *values)
{
  bool read = nmr_data->read_values(r, values);
  if (read)
    {
      int volume = r.volume();
      for (int k = 0 ; k < volume ; ++k)
	if (values[k] > neg_threshold && values[k] < pos_threshold)
	  values[k] = 0;
    }
  return read;
}

// ----------------------------------------------------------------------------
//
NMR_Data *thresholded_nmr_data(NMR_Data *nmr_data,
			       float neg_threshold, float pos_threshold)
  { return new Thresholded_NMR_Data(nmr_data, neg_threshold, pos_threshold); }

// ----------------------------------------------------------------------------
//
class Squished_NMR_Data : public NMR_Data
{
public:
  Squished_NMR_Data(NMR_Data *nmr_data, const IPoint &cell_size);
  virtual ~Squished_NMR_Data();
  bool read_value(const IPoint &index, float *value);
  bool read_values(const IRegion &index_region, float *values);
private:
  NMR_Data *nmr_data;
  IPoint cell_size;
  float *values;		// temporary storage
};

// ----------------------------------------------------------------------------
//
Squished_NMR_Data::Squished_NMR_Data(NMR_Data *nmr_data,
				     const IPoint &cell_size)
  : NMR_Data(nmr_data->path(),
	     reduced_size(nmr_data->size(), cell_size),
	     nmr_data->block_size(),
	     nmr_data->spectrometer_frequency(),
	     nmr_data->spectrum_width(),
	     corrected_origin_ppm(nmr_data, cell_size),
	     nmr_data->axis_label_list())
{
  this->nmr_data = nmr_data;
  this->cell_size = cell_size;
  this->values = new float [cell_size.product()];
}

// ----------------------------------------------------------------------------
//
Squished_NMR_Data::~Squished_NMR_Data()
{
  delete nmr_data;
  nmr_data = NULL;
  delete [] values;
}

// ----------------------------------------------------------------------------
//
static IPoint reduced_size(const IPoint &size, const IPoint &bsize)
{
  int dim = size.dimension();
  IPoint rsize(dim);
  for (int a = 0 ; a < dim ; ++a)
    rsize[a] = size[a] / bsize[a];
  return rsize;
}

// ----------------------------------------------------------------------------
// The new index 0 ppm value is the ppm value at (bsize-1)/2 ie. at the
// center of the block of points which is reduced to a single point.
//
static SPoint corrected_origin_ppm(NMR_Data *nmrdata, const IPoint &bsize)
{
  SPoint origin = nmrdata->origin_ppm();
  SPoint sw = nmrdata->spectrum_width();
  SPoint sf = nmrdata->spectrometer_frequency();
  IPoint size = nmrdata->size();
  for (int a = 0 ; a < nmrdata->dimension() ; ++a)
    {
      double ppm_per_index = (sw[a] / sf[a]) / size[a];
      origin[a] -= ppm_per_index * (bsize[a] - 1) / 2.0;
    }
  return origin;
}

// ----------------------------------------------------------------------------
//
bool Squished_NMR_Data::read_value(const IPoint &index, float *value)
{
  IRegion iregion = block_region(index, cell_size);
  if (!nmr_data->read_values(iregion, values))
    return false;

  int size = cell_size.product();
  float max_value = 0, min_value = 0;
  for (int k = 0 ; k < size ; ++k)
    {
      if (values[k] > max_value)
	max_value = values[k];
      if (values[k] < min_value)
	min_value = values[k];
    }

  *value = (max_value > -min_value ? max_value : min_value);
  return true;
}

// ----------------------------------------------------------------------------
//
bool Squished_NMR_Data::read_values(const IRegion &index_region, float *values)
{
  IPoint i = index_region.first_point();
  do
    {
      int index = position_index(i, index_region, true);
      if (!read_value(i, values + index))
	return false;
    }
  while (index_region.next_point(&i));
  return true;
}

// ----------------------------------------------------------------------------
//
NMR_Data *squished_nmr_data(NMR_Data *nmr_data, const IPoint &cell_size)
  { return new Squished_NMR_Data(nmr_data, cell_size); }

// ----------------------------------------------------------------------------
//
class Projected_NMR_Data : public NMR_Data
{
public:
  Projected_NMR_Data(NMR_Data *nmr_data, int axis);
  virtual ~Projected_NMR_Data();
  bool read_value(const IPoint &index, float *value);
  bool read_values(const IRegion &index_region, float *values);
private:
  NMR_Data *nmr_data;
  int axis;
};

// ----------------------------------------------------------------------------
//
Projected_NMR_Data::Projected_NMR_Data(NMR_Data *nmr_data, int axis)
  : NMR_Data(nmr_data->path(),
	     nmr_data->size().remove_component(axis),
	     nmr_data->block_size().remove_component(axis),
	     nmr_data->spectrometer_frequency().remove_component(axis),
	     nmr_data->spectrum_width().remove_component(axis),
	     nmr_data->origin_ppm().remove_component(axis),
	     remove_list_component(nmr_data->axis_label_list(), axis))
{
  this->nmr_data = nmr_data;
  this->axis = axis;
}

// ----------------------------------------------------------------------------
//
Projected_NMR_Data::~Projected_NMR_Data()
{
  delete nmr_data;
  nmr_data = NULL;
}

// ----------------------------------------------------------------------------
//
bool Projected_NMR_Data::read_value(const IPoint &index, float *value)
{
  IRegion r(index, index);
  return read_values(r, value);
}

// ----------------------------------------------------------------------------
//
bool Projected_NMR_Data::read_values(const IRegion &index_region,
				     float *values)
{
  int axis_size = nmr_data->size()[axis];
  int axis_block_size = nmr_data->block_size()[axis];
  IRegion base(index_region.min.insert_component(axis, 0),
	       index_region.max.insert_component(axis, 0));

  int volume = index_region.volume();
  for (int k = 0 ; k < volume; ++k)
    values[k] = 0;

  //
  // Read a slab one block size thick at a time to avoid consuming
  // too much memory.
  //

  IRegion slab = base;
  slab.max[axis] = axis_block_size-1;

  float *slab_values = new float[slab.volume()];

  bool read = true;
  while (slab.max[axis] < axis_size &&
	 (read = nmr_data->read_values(slab, slab_values), read))
    {
      Subarray_Iterator bi(base, slab, true);
      Subarray_Iterator vi(index_region, index_region, true);

      int bstep = bi.step_size(axis);
      for ( ; !bi.finished() ; bi.next(), vi.next())
	{
	  float min_h = 0, max_h = 0;
	  float *row = slab_values + bi.index();
	  for (int k = 0 ; k < axis_block_size ; ++k)
	    {
	      double h = row[k * bstep];
	      if (h < min_h) min_h = h;
	      if (h > max_h) max_h = h;
	    }
	  float extreme = (max_h >= -min_h ? max_h : min_h);
	  int vindex = vi.index();
	  float value = values[vindex];
	  if (fabs(extreme) > fabs(value))
	    values[vindex] = extreme;
	}
      slab.translate(axis_block_size, axis);
      base.translate(axis_block_size, axis);
    }

  delete slab_values;

  return read;
}

// ----------------------------------------------------------------------------
//
NMR_Data *projected_nmr_data(NMR_Data *nmr_data, int axis)
  { return new Projected_NMR_Data(nmr_data, axis); }

// ----------------------------------------------------------------------------
//
class NMR_Data_Plane : public NMR_Data
{
public:
  NMR_Data_Plane(NMR_Data *nmr_data, int axis, int plane);
  virtual ~NMR_Data_Plane();
  bool read_value(const IPoint &index, float *value);
  bool read_values(const IRegion &index_region, float *values);
private:
  NMR_Data *nmr_data;
  int axis, plane;
};

// ----------------------------------------------------------------------------
//
NMR_Data_Plane::NMR_Data_Plane(NMR_Data *nmr_data, int axis, int plane)
  : NMR_Data(nmr_data->path(),
	     nmr_data->size().remove_component(axis),
	     nmr_data->block_size().remove_component(axis),
	     nmr_data->spectrometer_frequency().remove_component(axis),
	     nmr_data->spectrum_width().remove_component(axis),
	     nmr_data->origin_ppm().remove_component(axis),
	     remove_list_component(nmr_data->axis_label_list(), axis))
{
  this->nmr_data = nmr_data;
  this->axis = axis;
  this->plane = plane;
}

// ----------------------------------------------------------------------------
//
NMR_Data_Plane::~NMR_Data_Plane()
{
  delete nmr_data;
  nmr_data = NULL;
}

// ----------------------------------------------------------------------------
//
bool NMR_Data_Plane::read_value(const IPoint &index, float *value)
{
  return nmr_data->read_value(index.insert_component(axis, plane), value);
}

// ----------------------------------------------------------------------------
//
bool NMR_Data_Plane::read_values(const IRegion &index_region, float *values)
{
  IRegion expanded_region(index_region.min.insert_component(axis, plane),
			  index_region.max.insert_component(axis, plane));
  return nmr_data->read_values(expanded_region, values);
}

// ----------------------------------------------------------------------------
//
static List remove_list_component(const List &list, int pos)
{
  List copy = list;
  copy.erase(pos);
  return copy;
}

// ----------------------------------------------------------------------------
//
NMR_Data *extract_nmr_data_plane(NMR_Data *nmr_data, int axis, int plane)
  { return new NMR_Data_Plane(nmr_data, axis, plane); }

// ----------------------------------------------------------------------------
//
Stringy standard_nucleus_name(const Stringy &nucleus)
{
  if (nucleus[0] == 'H' ||
      nucleus[0] == 'h' ||
      nucleus == "1H")
    return "1H";
  else if (nucleus[0] == 'C' ||
	   nucleus[0] == 'c' ||
	   nucleus == "13C")
    return "13C";
  else if (nucleus[0] == 'N' ||
	   nucleus[0] == 'n' ||
	   nucleus == "15N")
    return "15N";
  else if (nucleus[0] == 'P' ||
	   nucleus[0] == 'p' ||
	   nucleus == "31P")
    return "31P";

  return "1H";
}

// ----------------------------------------------------------------------------
//
void NMR_Data::print_header(ostream &out)
{
  int dim = dimension();

  Stringy heading = formatted_string("%-20s", "axis");
  for (int a = 0 ; a < dim ; ++a)
    heading = heading + formatted_string("%11s%1d", "w", a+1);
  out << heading << endl;

  SPoint spectrum_width_ppm(dim);
  for (int a = 0 ; a < dim ; ++a)
    {
      double sfreq = spectrometer_frequency()[a];
      spectrum_width_ppm[a] = (sfreq == 0 ? 0 : spectrum_width()[a] / sfreq);
    }

  out << header_line("nucleus", axis_label_list()) << endl;
  out << header_line("matrix size", size()) << endl;
  out << header_line("block size", block_size()) << endl;
  out << header_line("upfield ppm", origin_ppm() - spectrum_width_ppm) << endl;
  out << header_line("downfield ppm", origin_ppm()) << endl;
  out << header_line("spectrum width Hz", spectrum_width()) << endl;
  out << header_line("transmitter MHz", spectrometer_frequency()) << endl;
}

// ----------------------------------------------------------------------------
//
static Stringy header_line(const Stringy &name, const IPoint &value)
{
  Stringy line = formatted_string("%-20s", name.cstring());
  for (int a = 0 ; a < value.dimension() ; ++a)
    line = line + formatted_string("%12d", value[a]);
  return line;
}

// ----------------------------------------------------------------------------
//
static Stringy header_line(const Stringy &name, const SPoint &value)
{
  Stringy line = formatted_string("%-20s", name.cstring());
  for (int a = 0 ; a < value.dimension() ; ++a)
    line = line + formatted_string("%12.3f", value[a]);
  return line;
}

// ----------------------------------------------------------------------------
//
static Stringy header_line(const Stringy &name, const List &strings)
{
  Stringy line = formatted_string("%-20s", name.cstring());
  for (int a = 0 ; a < strings.size() ; ++a)
    line = line + formatted_string("%12s", ((Stringy *)strings[a])->cstring());
  return line;
}

// ----------------------------------------------------------------------------
// Data values are all zero.
//
class Dataless_NMR_Data : public NMR_Data
{
public:
  Dataless_NMR_Data(const Stringy &path,
		    const IPoint &size,
		    const IPoint &block_size,
		    const SPoint &spectrometer_freq,
		    const SPoint &spectrum_width,
		    const SPoint &origin_ppm,
		    const List &axis_labels);

  virtual bool read_value(const IPoint &position, float *value);
  virtual bool read_values(const IRegion &region, float *values);
};

// ----------------------------------------------------------------------------
//
Dataless_NMR_Data::Dataless_NMR_Data(const Stringy &path,
				     const IPoint &size,
				     const IPoint &block_size,
				     const SPoint &spectrometer_freq,
				     const SPoint &spectrum_width,
				     const SPoint &origin_ppm,
				     const List &axis_labels) :
  NMR_Data(path, size, block_size, spectrometer_freq,
	   spectrum_width, origin_ppm, axis_labels)
{
}

// ----------------------------------------------------------------------------
//
bool Dataless_NMR_Data::read_value(const IPoint &, float *value)
  { *value = 0; return 1; }
bool Dataless_NMR_Data::read_values(const IRegion &region, float *values)
{
  int size = region.volume();
  for (int i = 0 ; i < size ; ++i)
    values[i] = 0;
  return 1;
}

// ----------------------------------------------------------------------------
//
NMR_Data *dataless_nmr_data(const Stringy &path,
			    const IPoint &size,
			    const IPoint &block_size,
			    const SPoint &spectrometer_freq,
			    const SPoint &spectrum_width,
			    const SPoint &origin_ppm,
			    const List &axis_labels)
{
  return new Dataless_NMR_Data(path, size, block_size, spectrometer_freq,
			       spectrum_width, origin_ppm, axis_labels);
}

// ----------------------------------------------------------------------------
//
class Block_File_NMR_Data : public NMR_Data
{
public:
  Block_File_NMR_Data(Block_File *bf,
		      const SPoint &spectrometer_freq,
		      const SPoint &spectrum_width,
		      const SPoint &origin_ppm, const List &axis_labels);
  virtual ~Block_File_NMR_Data();

  virtual bool read_value(const IPoint &position, float *value);
  virtual bool read_values(const IRegion &region, float *values);

private:
  Block_File *bf;
};

// ----------------------------------------------------------------------------
//
Block_File_NMR_Data::Block_File_NMR_Data(Block_File *bf,
					 const SPoint &spectrometer_freq,
					 const SPoint &spectrum_width,
					 const SPoint &origin_ppm,
					 const List &axis_labels) :
  NMR_Data(bf->path(), bf->size(), bf->block_size(), spectrometer_freq,
	   spectrum_width, origin_ppm, axis_labels)
{
  this->bf = bf;
}

// ----------------------------------------------------------------------------
//
Block_File_NMR_Data::~Block_File_NMR_Data()
{
  delete bf;
  bf = NULL;
}

// ----------------------------------------------------------------------------
//
bool Block_File_NMR_Data::read_value(const IPoint &position, float *value)
  { return bf->read_value(position, value); }
bool Block_File_NMR_Data::read_values(const IRegion &region, float *values)
  { return bf->read_values(region, values); }

// ----------------------------------------------------------------------------
//
NMR_Data *block_file_nmr_data(Block_File *bf,
			      const SPoint &spectrometer_freq,
			      const SPoint &spectrum_width,
			      const SPoint &origin_ppm,
			      const List &axis_labels)
{
  return new Block_File_NMR_Data(bf, spectrometer_freq,
				 spectrum_width, origin_ppm, axis_labels);
}
