// ----------------------------------------------------------------------------
//
#include "list.h"		// Use List
#include "memalloc.h"		// use new()
#include "notifier.h"		// Use Notice_Type, Notice_Callback
#include "utility.h"		// Use fatal_error()

// ----------------------------------------------------------------------------
//
class Notification
{
public:
  Notification(Notice_Type notice_type, Notice_Callback cb, void *cb_data);

  bool equals(Notice_Type notice_type, Notice_Callback cb, void *cb_data);
  void notify(void *object);

private:
  Notice_Type notice_type;
  Notice_Callback cb;
  void *cb_data;
};

// ----------------------------------------------------------------------------
//
Notification::Notification(Notice_Type notice_type,
			   Notice_Callback cb, void *cb_data)
{
  this->notice_type = notice_type;
  this->cb = cb;
  this->cb_data = cb_data;
}

// ----------------------------------------------------------------------------
//
bool Notification::equals(Notice_Type notice_type,
			  Notice_Callback cb, void *cb_data)
{
  return (this->notice_type == notice_type &&
	  this->cb == cb &&
	  this->cb_data == cb_data);
}

// ----------------------------------------------------------------------------
//
void Notification::notify(void *object)
{
  cb(cb_data, object);
}

// ----------------------------------------------------------------------------
//
Notifier::Notifier()
{
  for (int k = 0 ; k < nt_end ; ++k)
    notice_lists.append(new List());
}

// ----------------------------------------------------------------------------
//
Notifier::~Notifier()
{
  for (int nt = 0 ; nt < nt_end ; ++nt)
    {
      const List &notices = notice_list((Notice_Type) nt);
      for (int ni = 0 ; ni < notices.size() ; ++ni)
	delete (Notification *) notices[ni];
      delete (List *) notice_lists[nt];
    }
}

// ----------------------------------------------------------------------------
//
void Notifier::notify_me(Notice_Type notice_type,
			 Notice_Callback cb, void *cb_data)
{
  if (find(notice_type, cb, cb_data))
    return;

  Notification *n = new Notification(notice_type, cb, cb_data);
  notice_list(notice_type).append(n);
}

// ----------------------------------------------------------------------------
//
bool Notifier::dont_notify_me(Notice_Type notice_type,
			      Notice_Callback cb, void *cb_data)
{
  Notification *n = find(notice_type, cb, cb_data);
  if (n)
    {
      notice_list(notice_type).erase(n);

      //
      // Prevent already scheduled dispatching of this notice.
      //
      for (int ni = 0 ; ni < dispatch_lists.size() ; ++ni)
	((List *) dispatch_lists[ni])->replace(n, NULL);

      delete n;
    }

  return n != NULL;
}

// ----------------------------------------------------------------------------
//
Notification *Notifier::find(Notice_Type notice_type,
			     Notice_Callback cb, void *cb_data)
{
  const List &nlist = notice_list(notice_type);
  for (int ni = 0 ; ni < nlist.size() ; ++ni)
    {
      Notification *n = (Notification *) nlist[ni];
      if (n->equals(notice_type, cb, cb_data))
	return n;
    }
  return NULL;
}

// ----------------------------------------------------------------------------
// Notify everyone interested in the specified notice type.
// Notification requests can be deleted by a callback procedure.
// I want to guarantee that once a notification request is deleted
// the callback will never be called.
//
void Notifier::send_notice(Notice_Type notice_type, void *object)
{

  List nlist = notice_list(notice_type);
  int size = nlist.size();
  if (size == 0)
    return;		// Optimization

  dispatch_lists.append(&nlist);
  for (int ni = 0 ; ni < size ; ++ni)
    {
      Notification *n = (Notification *) nlist[ni];
      if (n != NULL)
	n->notify(object);
    }
  dispatch_lists.erase(&nlist);
}

// ----------------------------------------------------------------------------
//
List &Notifier::notice_list(Notice_Type notice_type)
{
  if (notice_type < 0 || notice_type >= nt_end)
    fatal_error("Notifier::notice_list(): Bad notice type %d\n", notice_type);

  return *(List *)notice_lists[notice_type];
}
