// ----------------------------------------------------------------------------
//  Miscellaneous numerical routines.
//

#include <math.h>	// Use floor(), sqrt(), fabs()
#include <stdlib.h>	// use rand(), qsort()

static int compare_floats(const void *v1, const void *v2);

// ----------------------------------------------------------------------------
//
int min(int a, int b)
  { return a < b ? a : b; }
unsigned int min(unsigned int a, unsigned int b)
  { return a < b ? a : b; }
double min(double a, double b)
  { return a < b ? a : b; }

// ----------------------------------------------------------------------------
//
int max(int a, int b)
  { return a > b ? a : b; }
unsigned int max(unsigned int a, unsigned int b)
  { return a > b ? a : b; }
double max(double a, double b)
  { return a > b ? a : b; }

// ----------------------------------------------------------------------------
//
long abs(long a)
  { return a >= 0 ? a : -a; }
double abs(double a)
  { return a >= 0 ? a : -a; }

// ----------------------------------------------------------------------------
//
double floor(double x, double step)
  { return step * floor(x / step); }
double floor(double x, double a, double b)
  { return a + floor(x-a, b-a); }
double ceil(double x, double step)
  { return step * ceil(x / step); }

// ----------------------------------------------------------------------------
//
double interval_mod(double x, double a, double b)
{
  return (x >= a ? a + fmod(x - a, b - a) : b - fmod(a - x, b - a));
}

// ----------------------------------------------------------------------------
//
int round(double x)
  { return (int) (x >= 0 ? x + .5 : x - .5); }

// ----------------------------------------------------------------------------
// Round to the nearest multiple of unit.
//
double round_multiple(double x, double unit)
{
  return round(x / unit) * unit;
}

// ----------------------------------------------------------------------------
//
double bounded(double lo, double x, double hi)
{
  if (x < lo) x = lo;
  if (x > hi) x = hi;

  return x;
}

// ----------------------------------------------------------------------------
//
int bounded(int lo, int x, int hi)
{
  if (x < lo) x = lo;
  if (x > hi) x = hi;

  return x;
}

// ----------------------------------------------------------------------------
//
double length(double x, double y)
{
  return sqrt(x*x + y*y);
}

// ----------------------------------------------------------------------------
// Return the distance in the plane between segment [x1,x2] and point (x,y).
//
double segment_distance(double x1, double x2, double x, double y)
{
  double d;

  if (x < x1)
    d = length(x - x1, y);
  else if (x > x2)
    d = length(x - x2, y);
  else
    d = fabs(y);

  return d;
}

// ----------------------------------------------------------------------------
// x = mant * 10 ^ exp
//   with 0 <= mant < 1.
//
// decade = 10 ^ exp.
//
void scientific_form(double x, double *mant, long *exp, double *decade)
{
  double e, m, d;

  if (x == 0)
    {
      e = 0;
      m = 0;
      d = 0;
    }
  else
    {
      e = ceil(log10(fabs(x)));
      d = pow(10.0, e);
      m = x / d;
    }

  *exp = (long) e;
  *mant = m;
  *decade = d;
}

// ----------------------------------------------------------------------------
//
void swap(double *a, double *b)
{
  double c = *a;

  *a = *b;
  *b = c;
}

// ----------------------------------------------------------------------------
//
bool is_between(double x, double a, double b)
{
  return (b > a ? (x >= a && x <= b) : (x >= b && x <= a));
}

// ----------------------------------------------------------------------------
//
bool is_bit_set(unsigned int word, int bit)
  { return (word & (0x1 << bit)); }
unsigned int set_bit(unsigned int word, int bit, bool state)
  { return (state ? (word | (0x1 << bit)) : (word & ~(0x1 << bit))); }

// ----------------------------------------------------------------------------
//
int random_int(int min, int max)
{
  return min + rand() % (max - min + 1);
}

// ----------------------------------------------------------------------------
//
void sort_floats(float *values, int n)
{
  qsort(values, n, sizeof(float), compare_floats);
}

// ----------------------------------------------------------------------------
//
static int compare_floats(const void *v1, const void *v2)
{
  float f1 = *(float *) v1;
  float f2 = *(float *) v2;
  if (f1 < f2) return -1;
  else if (f1 > f2) return 1;
  return 0;
}
