/*
 * Peak.C:		Implementation of the Peak class.
 *
 * A Peak object is a single extremum in an NMR spectrum.
 *
 */

#include <stdlib.h>	// Use atof(), atoi()
#include <math.h>	// Use exp()

#include "atom.h"		// use Atom
#include "axismap.h"		// Use Axis_Map
#include "color.h"
#include "group.h"		// Use Group
#include "label.h"
#include "mathconst.h"		// use MC_LN2
#include "memalloc.h"		// use new()
#include "notifier.h"		// use Notifier
#include "num.h"
#include "ornament.h"
#include "peak.h"
#include "peakgp.h"
#include "print.h"		// use color_index()
#include "resonance.h"
#include "spectrum.h"
#include "spectrumdata.h"	// Use symmetric_half_height_width()
#include "utility.h"

// ----------------------------------------------------------------------------
// Construct a Peak in Spectrum <sp>
//
Peak::Peak(Spectrum *sp, const SPoint &pos) : CrossPeak(sp, pos)
{
	int dim = sp->dimension();

	mPeakGroup		= NULL;			// no PeakGp

	//
	// Integration an curve fitting parameters
	//
	mFitHeight		= 0;
	mDataHeightValid	= false;
	mLinewidth		= SPoint(dim);		// no linewidth
	mVolume   		= 0.0;			// no volume
	mIntegrateMethod	= INTEGRATE_NONE;	// no integration meth
	mLinewidthMethod	= "";			// no linewidth method
	mFitResidual		= 0;

	sp->add(*this);

	sp->notifier().send_notice(nt_created_ornament, (Ornament *) this);
}

//
// Destruct a Peak.
//
Peak::~Peak()
{
  Notifier &n = spectrum()->notifier();
  n.send_notice(nt_will_delete_ornament, (Ornament *) this);

  PeakGp *gp = peakgp();
  if (gp)
    gp->remove(*this);	// Remove from peakgroup parent.
  
  spectrum()->remove(*this);
}

// ----------------------------------------------------------------------------
//
Ornament *Peak::copy(Spectrum *to, const Axis_Map &axismap)
{
  SPoint pos = axismap.map(position());
  Peak *pk = new Peak(to, pos);
  pk->set_alias(alias());
  pk->lock(IsLocked());
  pk->SetNote(GetNote());
  pk->SetColor(GetColor());

  for (int a = 0 ; a < dimension() ; ++a)
    if (resonance(a))
      {
	int to_axis = axismap.map(a);
	Resonance *r = resonance(a);
	pk->ChangeAssignment(to_axis,
			     r->atom()->name().cstring(),
			     r->group()->name().cstring());
      }

  //
  // Don't copy integration since it is not valid on new spectrum
  //
  pk->SetIntegrateMethod(INTEGRATE_NONE);

  if (label())
    {
      Label *lbl = (Label *) label()->copy(to, axismap);
      lbl->attach(pk);
      lbl->assignment(label()->is_assignment());
    }

  return pk;
}

// ----------------------------------------------------------------------------
//
Ornament_Type Peak::type() const { return ::peak; }
const char *Peak::type_name() const { return "peak"; }

double Peak::SignalToNoise() const
{
	return abs(DataHeight()) / spectrum()->noise_level();
}

//
// Set & return the linewidth
//
double Peak::linewidth(int a) const
{
	return linewidth()[a];
}

// ----------------------------------------------------------------------------
//
SPoint Peak::linewidth() const
{
  return mLinewidth;
}

void Peak::linewidth(const Stringy &t, const SPoint &lw)
{
  mLinewidthMethod = t;
  mLinewidth = lw;
  if (IntegrateByFit() && t != "fit")
    no_volume();
  Notifier &n = spectrum()->notifier();
  n.send_notice(nt_changed_ornament, (Ornament *) this);
}

Stringy Peak::GetLinewidthMethod() const
  { return mLinewidthMethod; }

// ----------------------------------------------------------------------------
// Range of peak fit used for display and data subtraction.
//
double Peak::linewidth_cutoff_range() const
  { return 10; }

// ----------------------------------------------------------------------------
//
bool Peak::volume(double *vol, Stringy *method) const
{
  if (GetIntegrateMethod() == INTEGRATE_NONE)
    return false;

  if (vol)	*vol = mVolume;
  if (method)	*method = index_name(Integration_Method_Names,
				     GetIntegrateMethod());
  return true;
}

// ----------------------------------------------------------------------------
//
void Peak::set_volume(double v, Integration_Method t)
{
	SetIntegrateMethod(t);
	mVolume = v;
	Notifier &n = spectrum()->notifier();
	n.send_notice(nt_changed_ornament, (Ornament *) this);
	set_volume_error(0, "");
}

// ----------------------------------------------------------------------------
//
void Peak::no_volume()
  { SetIntegrateMethod(INTEGRATE_NONE); }

// ----------------------------------------------------------------------------
//
void Peak::manual_volume(double vol)
{
  set_volume(vol, INTEGRATE_MANUAL);
}

//
// Set the integration region box or ellipse.
//
void Peak::box_volume(Integration_Method t, double vol)
{
	set_volume(vol, t);
}

//
// Set the integration region from the contour thresholds.
//
void Peak::fit_volume(Integration_Method t, double vol, double residual)
{
	set_volume(vol, t);
	mFitResidual = residual;
}

void Peak::peakgp(PeakGp *x)
{
	if (mPeakGroup != NULL && x != NULL && mPeakGroup != x)
	 	mPeakGroup->remove(*this);
	mPeakGroup = x;
}

PeakGp *Peak::peakgp(void) const
  { return mPeakGroup; }

// ----------------------------------------------------------------------------
//
Stringy	Peak::assignment_name(const Stringy &format) const
{
  return (peakgp() ?
	  peakgp()->assignment_name(format) :
	  CrossPeak::assignment_name(format));
}

// ----------------------------------------------------------------------------
//
double Peak::FitHeight() const
  { return mFitHeight; }
void Peak::FitHeight(double h)
  { mFitHeight = h; }
double Peak::fit_residual() const
  { return mFitResidual; }

// ----------------------------------------------------------------------------
//
double Peak::DataHeight(void) const
{
  if (!mDataHeightValid)
    ((Peak *) this)->DataHeight(spectrum()->height_at_point(position()));

  return mDataHeight;
}
void Peak::DataHeight(double h)
{
  mDataHeight = h;
  mDataHeightValid = true;
  Notifier &n = spectrum()->notifier();
  n.send_notice(nt_changed_ornament, (Ornament *) this);
}

/*
 * Return true if the peak has a linewidth by any method
 */
bool Peak::HasLinewidth() const
{
  return !GetLinewidthMethod().is_empty();
}

// ----------------------------------------------------------------------------
//
double Peak::fit_height(const SPoint &p)
{
  const double gaussian_linewidth_factor = 1 / (2 * sqrt(MC_LN2));
  const double lorentzian_linewidth_factor = .5;

  if (!IntegrateByFit())
    return 0.0;

  SPoint delta = position() - p;
  SPoint lw = linewidth();

  double f = 1;
  switch (GetIntegrateMethod())
    {
    case INTEGRATE_GAUSSIAN:
      lw *= gaussian_linewidth_factor;
      for (int a = 0 ; a < dimension() ; ++a)
	{
	  double x = delta[a] / lw[a];
	  f *= exp(- x * x);
	}
      break;
    case INTEGRATE_LORENTZIAN:
      lw *= lorentzian_linewidth_factor;
      for (int a = 0 ; a < dimension() ; ++a)
	{
	  double x = delta[a] / lw[a];
	  f *= 1 / (1 + x * x);
	}
      break;
    default:
      fatal_error("fit_height(): Peak not linefit.\n");
    }

  return f * FitHeight();
}

Integration_Method Peak::GetIntegrateMethod(void) const
  { return mIntegrateMethod; }
bool Peak::IntegrateByBox(void) const
  { return (mIntegrateMethod == INTEGRATE_BOX
	    || mIntegrateMethod == INTEGRATE_ELLIPSE); }
bool Peak::IntegrateByFit(void) const
  { return (mIntegrateMethod==INTEGRATE_GAUSSIAN
	    || mIntegrateMethod == INTEGRATE_LORENTZIAN); }

/*
 * Set the integration method type to <t>
 */
void Peak::SetIntegrateMethod(Integration_Method t)
{
  Notifier &n = spectrum()->notifier();
  n.send_notice(nt_will_change_ornament, (Ornament *) this);
  mIntegrateMethod = t;
  n.send_notice(nt_changed_ornament, (Ornament *) this);
}

/*
 * Return the Resonance along axis <a>
 */
Resonance *Peak::resonance(int a) const
{
	if (peakgp())
		return peakgp()->resonance(a);

	return CrossPeak::resonance(a);
}

/*
 * Set the Resonance of axis <a> to <rp>
 */
void Peak::SetResonance(int a, Resonance *rp)
{
	if (peakgp())
		peakgp()->SetResonance(a, rp);
	else
		CrossPeak::SetResonance(a, rp);
}

// ----------------------------------------------------------------------------
//
void Peak::set_position(const SPoint &r)
{
  CrossPeak::set_position(r);
  PeakGp *pg = peakgp();
  if (pg)
    pg->recompute_position();
}

// ----------------------------------------------------------------------------
//
void Peak::set_alias(const SPoint &a)
{
  CrossPeak::set_alias(a);
  PeakGp *pg = peakgp();
  if (pg)
    pg->recompute_position();
}

// ----------------------------------------------------------------------------
// Set the peak position and erase the integration state.
//
void Peak::SetLocation(const SPoint &r)
{
	set_position(r);

	SetIntegrateMethod(INTEGRATE_NONE);
	mLinewidthMethod = "";
	FitHeight(0);
	set_volume(0, INTEGRATE_NONE);
	mDataHeightValid = false;
}

//
// Move the Peak, rereading the data file for the Peak height.
//
void Peak::IncrementLocation(const SPoint &dr)
{
	SetLocation(position() + dr);
}


//
// Move a peak to the center of its data.
//
bool Peak::MoveToMaximum()
{
	if (IsLocked())
	  return false;

	double	ht;
	SPoint	startPos = position();
	SPoint	maxPos = local_maximum(spectrum(), startPos, &ht);

	if (startPos != maxPos)
		SetLocation(maxPos);

	return true;
}

// ----------------------------------------------------------------------------
//
SRegion Peak::erasure_region(ODraw &dr) const
{
  if (dr.subtract_fit_peaks() && HasLinewidth())
    {
      SPoint c = position();
      SRegion r(c, c);
      SPoint range = linewidth() * linewidth_cutoff_range();
      for (int a = 0 ; a < r.dimension() ; ++a)
	{
	  r.min[a] -= range[a];
	  r.max[a] += range[a];
	}
      r.encompass(CrossPeak::erasure_region(dr));
      return r;
    }

  return CrossPeak::erasure_region(dr);
}

// ----------------------------------------------------------------------------
// Send a Postscript command to draw a peak.
//
void Peak::print(ODraw &dr, FILE *fp,
		 double xsc, double ysc, Rectangle r) const
{
	if (!dr.is_visible(this, r))
		return;

	Rectangle sq = marker_square(dr);
	double x = (r.max(X) - sq.max(X)) * xsc;
	double y = (r.max(Y) - sq.max(Y)) * ysc;
	double w = sq.size(X) * xsc;
	double h = sq.size(Y) * ysc;

	fprintf(fp, "%d %d %.0f %.0f %.0f %.0f %d %d PP\n",
		color_index(GetColor()),	// edge color
		color_index(GetColor()),	// fill color
		x, y, w, h, IsSelected(), dr.in_plane(position()));
}

/*
 * Estimate the peak linewidths of an N-D peak by going to the maximum
 * intensity and looking along each axis until the 1/2 max crossing
 * point is found.
 */
void Peak::estimate_linewidth()
{
	SPoint lw = symmetric_half_height_width(spectrum(), position());
	linewidth("1/2", lw);
}
