/*
 * Peak picking
 */
#include <stdio.h>		// Use sprintf()
#include <string.h>		// Use strcpy(), strlen()
#include <math.h>		/* Use fabs(), HUGE. */

#include "list.h"		// Use List
#include "memalloc.h"		// use new()
#include "num.h"		// Use max()
#include "peak.h"		// Use Peak
#include "peakpick.h"
#include "spectrum.h"		// Use Spectrum
#include "spectrumdata.h"	// use half_height_width()
#include "spoint.h"		// Use SPoint, IPoint, IRegion
#include "wait.h"		// use Wait

// ----------------------------------------------------------------------------
//
static bool has_minimum_linewidth(Spectrum *sp, const IPoint &i,
				  const SPoint &min_linewidth);
static bool has_minimum_drop(Spectrum *sp, const IPoint &i, double factor);
static bool drops_in_direction(Spectrum *sp, const IPoint &i, double factor,
			       int axis, int step);
static List do_peak_pick(Spectrum *, const IRegion &,
			 const Peak_Pick_Parameters &, Wait *);
static bool pick_constraints_satisfied(Spectrum *sp, const IPoint &i,
				       const Peak_Pick_Parameters &params);
static bool is_local_maximum(Spectrum *, const IPoint &);
static bool is_local_minimum(Spectrum *, const IPoint &);
static bool has_close_neighbor_peak(Spectrum *sp, const IPoint &p,
				    const List &peak_points);
static int separation(const IPoint &, const IPoint &);
static void free_ipoint_list_entries(const List &ipoints);

// ----------------------------------------------------------------------------
//
Peak_Pick_Parameters::Peak_Pick_Parameters(int dimension)
{
  minPosHt = 0;
  minNegHt = 0;
  minimum_linewidth = SPoint(dimension);
  minimum_drop_factor = 0;
}

// ----------------------------------------------------------------------------
//
Peak_Pick_Parameters::Peak_Pick_Parameters(const Peak_Pick_Parameters &ppp)
{
  minPosHt = ppp.minPosHt;
  minNegHt = ppp.minNegHt;
  minimum_linewidth = ppp.minimum_linewidth;
  minimum_drop_factor = ppp.minimum_drop_factor;
}

//-----------------------------------------------------------------------------
// Check if half height linewidths are greater than required values.
//
static bool has_minimum_linewidth(Spectrum *sp, const IPoint &i,
				  const SPoint &min_linewidth)
{
  SPoint p = sp->map(i, INDEX, PPM);
  SPoint lw = half_height_width(sp, p);
  for (int a = 0 ; a < sp->dimension() ; ++a)
    if (lw[a] < min_linewidth[a])
      return false;
  return true;
}

//-----------------------------------------------------------------------------
// Check if peak drops off by a certain factor before rising again.
// This filters out overlapped peaks.
//
static bool has_minimum_drop(Spectrum *sp, const IPoint &i, double factor)
{
  if (factor <= 0)
    return true;

  int dim = sp->dimension();
  for (int a = 0 ; a < dim ; ++a)
    if (!drops_in_direction(sp, i, factor, a, 1) ||
	!drops_in_direction(sp, i, factor, a, -1))
      return false;
  return true;
}

//-----------------------------------------------------------------------------
//
static bool drops_in_direction(Spectrum *sp, const IPoint &i, double factor,
			       int axis, int step)
{
  IPoint n = i;
  double h = sp->height_at_index(i);
  double h_prev = h;
  double drop = factor * fabs(h);
  int w = sp->index_range().max[axis];
  for (int s = i[axis] + step ; s >= 0 && s <= w ; s += step)
    {
      n[axis] = s;
      double h_this = sp->height_at_index(n);
      if (h > 0)
	{
	  if (h_this > h_prev)
	    return false;				// Not dropping anymore
	  else if (h - h_this >= drop)
	    return true;				// Dropped enough
	}
      else
	{
	  if (h_this < h_prev)
	    return false;				// Not dropping anymore
	  else if (h_this - h >= drop)
	    return true;				// Dropped enough
	}
      h_prev = h_this;
    }
  return true;		// Hit edge of spectrum
}

/*
 * We allow searching along either W1 or Wn, n the maximum axis. In principle,
 * searching along Wn should be fastest, since Wn is the contiguous data axis,
 * but in practice this is not true because the NMR data reading routines read
 * entire slices at a time, so the amount of data read is dependent more on
 * the length of each data axis than on which axis is contiguous on disk.
 *
 * That an N-D hill-climbing routine is used also means that the initial
 * "search" direction is not that important with respect to performance,
 * because all directions will be tried once a peak is as large as the minimum
 * height
 *
 * The solution for performance is to improve the nmr_slice_read() routine so
 * that random access is improved and full-length slices do not need to be
 * read.
 */

//
// Locate peaks in the region r.
//
// The first step is to determine the two thresholds of peak picking:
//
//  1)	a minimum intensity, which is the maximum of the peak minimum (as set
//	in the peak tool) and the lowest contour visible contour level).
//  2)	a minumum volume, which the peak must meet.
//
// This routine works through every point in the N-D set of points and
// determines if the point is above the minimum intensity. If it is,
// the _getLocalMax() routine walks up the gradient until the local maximum
// is located.
//
// Once located, the peak is tested to see if it is near a peak ornament,
// or adjacent to a temporary peak that this routine has already added.
// If so, the putative peak is rejected.
//
// If the peak is not near any others, the _interpolatePosition() routine is
// called to determine a more accurate position for the peak. The peak is
// now added to the temporary peak list, because we don't want to repeat
// the next expensive step if we don't have to.
//
// The _hasVolume() routine walks the points inside the peak contour to
// determine if the peak has more than the minimum volume. If it does,
// it is accepted as a peak. If it doesn't, it is left on the peak list
// so we don't have to recompute the volume.
//
// Finally, the current peak list is pruned of any temporary peaks that
// do not hold sufficient volume.
//

// ----------------------------------------------------------------------------
// Find new peaks in a region.  The peaks should have sufficient height
// and volume and not be too close to any other peaks.
//
static List do_peak_pick(Spectrum *sp, const IRegion &index_region,
			const Peak_Pick_Parameters &params, Wait *wait)
{
  int count = 0;
  int total = index_region.volume();

  /*
   * A few points, report every 5% complete.
   * A lot of points report every 1% done.
   */
  int report_interval = max(1, (total < 100000 ? total / 20 : total / 100));

  List peak_points;

  IPoint i = index_region.first_point();

  do
    {
      if (pick_constraints_satisfied(sp, i, params) &&
	  !has_close_neighbor_peak(sp, i, peak_points))
	peak_points.append(new IPoint(i));

      //
      // Print out a status message.
      //
      count += 1;
      if (count % report_interval == 0 && wait)
	wait->progress_report("Peak pick %.0f%% finished -- %d peaks added",
				  (100.0 * count) / total, peak_points.size());

      if (count % 1000 == 0)
	if (wait && wait->was_stop_requested())
	  break;
    }
  while (index_region.next_point(&i));

  return peak_points;
}

//-----------------------------------------------------------------------------
//
static bool pick_constraints_satisfied(Spectrum *sp, const IPoint &i,
				       const Peak_Pick_Parameters &params)
{
  float h = sp->height_at_index(i);
  bool extremum = ((h >= params.minPosHt && is_local_maximum(sp, i)) ||
		   (h <= params.minNegHt && is_local_minimum(sp, i)));
  return (extremum &&
	  has_minimum_linewidth(sp, i, params.minimum_linewidth) &&
	  has_minimum_drop(sp, i, params.minimum_drop_factor));
}

//-----------------------------------------------------------------------------
//
static bool is_local_maximum(Spectrum *sp, const IPoint &index)
{
  float h = sp->height_at_index(index);
  IPoint n = index;

  for (int a = 0 ; a < index.dimension() ; ++a)
    {
      n[a] = index[a] - 1;
      if (n[a] >= 0 && sp->height_at_index(n) > h)
	return false;

      n[a] = index[a] + 1;
      if (n[a] < sp->index_range().max[a] && sp->height_at_index(n) > h)
	return false;

      n[a] = index[a];
    }

  return true;
}

//-----------------------------------------------------------------------------
//
static bool is_local_minimum(Spectrum *sp, const IPoint &index)
{
  float h = sp->height_at_index(index);
  IPoint n = index;

  for (int a = 0 ; a < index.dimension() ; ++a)
    {
      n[a] = index[a] - 1;
      if (n[a] >= 0 && sp->height_at_index(n) < h)
	return false;

      n[a] = index[a] + 1;
      if (n[a] < sp->index_range().max[a] && sp->height_at_index(n) < h)
	return false;

      n[a] = index[a];
    }

  return true;
}

//-----------------------------------------------------------------------------
// Make sure there is no peak at this location. We say the
// peaks overlap if they are within NEAR_REAL_PEAK data points
// of each other. This should be tuneable.
//
#define NEAR_REAL_PEAK	2
#define NEAR_TMP_PEAK	1

static bool has_close_neighbor_peak(Spectrum *sp, const IPoint &index,
				    const List &peak_points)
{
  /*
   * First check against the new temporary peaks.
   */
  for (int pi = 0 ; pi < peak_points.size() ; ++pi)
    if (separation(*(IPoint *)peak_points[pi], index) <= NEAR_TMP_PEAK)
      return true;

  const List &olist = sp->ornaments();
  for (int oi = 0 ; oi < olist.size() ; ++oi)
    {
      Ornament *op = (Ornament *) olist[oi];
      if (op->type() == peak)
	{
	  Peak	*pk = (Peak *) op;
	  IPoint i = sp->map(pk->position(), PPM, INDEX).rounded();
	  if (separation(i, index) <= NEAR_REAL_PEAK)
	    return true;
	}
    }

  return false;
}

//-----------------------------------------------------------------------------
//
static int separation(const IPoint &p1, const IPoint &p2)
{
  int max = abs(p1[0] - p2[0]);
  
  for (int i = 1; i < p1.dimension(); i++)
    {
      int sep = abs(p1[i] - p2[i]);
      if (sep > max)
	max = sep;
    }

  return max;
}

/*
 * For the list of peak points add an ornament for those that don't as
 * yet have one.
 *
 * If <Spectrum::mPeakPick.assignedPeaksOnly> is true, then only peaks that
 * could possibly be assigned with assignment guesses are kept and the
 * other peaks are thrown away, decrementing <*peakCountp>.
 */
static List _createOrnaments(Spectrum *sp, const List &peak_points)
{
  List peaks;
  for (int pi = 0 ; pi < peak_points.size() ; ++pi)
    {
      IPoint *i = (IPoint *) peak_points[pi];

      double height;
      SPoint freq = local_maximum(sp, sp->map(*i, INDEX, PPM), &height);
      Peak *op = new Peak(sp, freq);
      peaks.append(op);
    }

  return peaks;
}

/*
 * Locate peaks in spectrum within a region.
 */
static List _peakPick(Spectrum *sp, const SRegion &region,
		      const Peak_Pick_Parameters &params, Wait *wait)
{
	SRegion r = region;
	r.clip(sp->ppm_region());
	if (r.no_interior())
	  return List();

	IRegion ireg = sp->map(r, PPM, INDEX).rounded();
	List peak_points = do_peak_pick(sp, ireg, params, wait);

	if (wait && wait->was_stop_requested()) {
	  free_ipoint_list_entries(peak_points);
	  return List();
	}

	List	peaks = _createOrnaments(sp, peak_points);
	free_ipoint_list_entries(peak_points);
	return peaks;
}

// ----------------------------------------------------------------------------
//
static void free_ipoint_list_entries(const List &ipoints)
{
  for (int pi = 0 ; pi < ipoints.size() ; ++pi)
    delete (IPoint *) ipoints[pi];
}

/*
 * Locate peaks in a region.
 */
List peak_pick_region(Spectrum *sp, const SRegion &region,
		      const Peak_Pick_Parameters &params, Wait *wait)
{
  if (wait)
    wait->begin_waiting("Peak picking region.", true);
  List peaks = _peakPick(sp, region, params, wait);
  if (wait)
    wait->end_waiting();
  return peaks;
}
