/*
 * The Project class provides multiple-spectrum support to make it easier
 * to share information between spectra.
 */
#ifndef _Project_h
#define _Project_h

class view_axis;

#include "attacheddata.h"	// use AttachedData
#include "list.h"		// use List
#include "print.h"		// use Print_Options
#include "savefile.h"		// use SaveFile
#include "spoint.h"		// use SRegion
#include "stringc.h"		// use Stringy
#include "table.h"		// use Table
#include "uiregion.h"		// use View_Regions
#include "winsystem.h"		// use CB_Data

class Condition;
class Contour_Cache;
class CrossPeak;
class Memory_Cache;
class Molecule;
class Spectrum;
class Undo;
class View;

// ----------------------------------------------------------------------------
//
class Preferences
{
public:
  Preferences();
  bool prompt_before_overwrite;	// for Save files
  int key_timeout_interval;	// milliseconds (0 means never timeout)
  int auto_backup_interval;	// seconds (0 means no auto backup)
  int memory_cache_size;	// Mbytes
  bool resize_views;		// Enlarge view windows when adding
				//   resonance or slice panels?
  bool contour_graying;		// Gray out when redrawing contours?
};

// ----------------------------------------------------------------------------
//
class Project
{
public:
  Project(Session &);
  ~Project();

  Session &session() const;

  //
  // Spectra and Views
  //

  const List &	spectrum_list() const;
  Spectrum *	find_spectrum(const Stringy &sname) const;
  Stringy	unique_spectrum_name(const Stringy &name);
  void		set_ppm_zero(Spectrum *sp, const SPoint &zero_position);
  Spectrum *	find_savefile_spectrum(const Stringy &savefile) const;
  int		unsaved_spectra();
  Contour_Cache *contour_cache() const;
  Memory_Cache *memory_cache() const;
  void		set_cache_size(int mbytes);

  List		view_list() const;
  List		view_list(Spectrum *sp) const;
  List		view_list(const Condition *c) const;
  List		top_level_views() const;
  View *	find_view(const Stringy &name);
  View *	spectrum_view(Spectrum *sp) const;
  Stringy	unique_view_name(const Stringy &name, bool top_level);

  //
  // Molecules and Conditions
  //

  List		molecule_list() const;
  Molecule *	find_molecule(const Stringy &molecule_name);
  Molecule *	define_molecule(const Stringy &molecule_name);

  List		condition_list() const;
  Condition *	define_condition(const Stringy &molecule_name,
				 const Stringy &condition_name);
  Condition *	find_condition(const Stringy &molecule_name,
			       const Stringy &condition_name);
  Condition *	default_condition();

  //
  // Save, restore and backup
  //

  bool		unload_query();
  void		unload();
  SaveFile &	save_file();
  void		set_auto_backup(int seconds);
  bool		use_backup_file(const Stringy &path);


  //
  // View axis synchronizations.
  //

  void		synchronize_view_axes(const view_axis &va1,
				      const view_axis &va2);
  void		unsynchronize_view_axis(const view_axis &va);
  const List *	synchronized_view_axes(const view_axis &va);
  const List &	synchronized_view_axis_sets() const;
  void		synchronize_match_views(View *v);

  //
  // View overlays
  //

  void		add_overlay(View *from, View *onto);
  void		remove_overlay(View *from, View *onto);
  List		overlaid_views(View *onto);

  //
  // Assignment copy, paste and delete
  //

  bool		assignment_copy(Spectrum *from);
  void		assignment_paste(Spectrum *to, bool label);
  Spectrum *	assignment_copy_from_spectrum();
  void		set_assignment_copy_from_spectrum(Spectrum *sp);
  void		assignment_delete(Spectrum *sp);

  //
  // Ornament copy and paste
  //

  bool		ornament_copy(Spectrum *from);
  void		ornament_paste(Spectrum *to);

  //
  // Undo ornament deletions, moves, integrations.
  //

  bool		DoUndo();
  Undo *		MakeUndoDelete();
  Undo *		MakeUndoIntegrate(const List &peaks);
  Undo *		MakeUndoMove();

  //
  // Selected ornaments and peaks
  //

  List selected_ornaments();		// From all spectra
  void unselect_all_ornaments();
  List selected_crosspeaks();		// From all spectra, no peaklets
  CrossPeak *selected_peak();
  CrossPeak *single_peak_selected();

  //
  // File paths
  //
  Stringy sparky_path(const Stringy &dir, const Stringy &file) const;
  Stringy list_path(const Stringy &file) const;
  Stringy project_path(const Stringy &file) const;
  Stringy save_path(const Stringy &file) const;
  Stringy new_save_file_path(const Stringy &name) const;

  //
  // Extra data attached to project and saved in files
  //
  AttachedData &saved_values();

  Preferences		preferences;
  View_Regions		view_regions;	// named ppm regions
  Print_Options		print_options;

private:
  Session &	ses;
  List		mSpectra;		// Spectra in project
  List		mViews;			// Views in project
  List		mMolecules;
  SaveFile	mSaveFile;		// the project save file
  AttachedData	attached_data;

  bool use_auto_backups;
  bool ignore_auto_backups;
  bool remove_auto_backups;

  unsigned long backup_interval;
  static void auto_backup_cb(CB_Data);

  bool selection_changed;	// For batching selection change notices
  Table changed_ornaments;	// For batching ornament change notices
  Table deleted_ornaments;

  List assignment_copy_list;	// List of crosspeaks
  Spectrum *assignment_copy_from;	// For assignment copy pointer mode
  List ornament_copy_list;	// List of ornaments

  Undo			*mUndo;	// Undo data

  Memory_Cache *mem_cache;
  Contour_Cache	*contours;

  List	sync_lists;	// List of lists of synchronized view axes.
  List		mOverlays;	// View overlays.

  void		mInit();

  static void ornament_selection_cb(void *project, void *ornament);
  static void send_selection_changed_notice(CB_Data project);
  static void ornament_created_cb(void *project, void *ornament);
  static void ornament_changed_cb(void *project, void *ornament);
  static void send_ornaments_changed_notice(CB_Data project);
  static void will_delete_ornament_cb(void *project, void *ornament);
  static void view_created_cb(void *project, void *view);
  static void will_delete_view_cb(void *project, void *view);
  static void deleted_view_cb(void *project, void *spectrum);
  static void spectrum_created_cb(void *project, void *spectrum);
  static void will_delete_spectrum_cb(void *project, void *spectrum);
  static void molecule_created_cb(void *project, void *molecule);
  static void will_delete_molecule_cb(void *project, void *molecule);

  //
  // Undo
  //
  void		mResetUndo(void);
};

// ----------------------------------------------------------------------------
// Use view_axis for view synchronization.
//
class view_axis
{
 public:
  view_axis(View *view, int axis);
  view_axis(const view_axis &);
  View *view() const;
  int axis() const;
  bool operator==(const view_axis &) const;
 private:
  View *v;
  int a;
};

void free_view_axis_list_entries(List &view_axis_list);

#endif
