// ----------------------------------------------------------------------------
// Python interface to Sparky
// --------------------------
//
// C Sparky objects (Spectra, peaks, resonances, ...) are mirrored in Python.
// There is only one Python object corresponding to a C object.  The Python
// objects are class instances.  They are not built-in extension types
// because classes are more friendly for user programming.  For instance,
// you can add attributes to them or inherit from them.  Immutable data in
// the C objects is represented by a Python data member.  Mutable data is
// looked up in the C object via the Python __getattr__() mechanism and some
// mutable data is settable via __setattr__(). The Python classes are defined
// in C and documented in sparky.py.
//
// Id numbers
// ----------
// Every C object used by Python has an id number.
// There are maps C pointer <-> id number <-> Python object.
// The id number is intended to be invisible to users programming in Python.
// The id number intermediate is used because Python may keep a reference
// to a C object beyond the lifetime of the C object.  If a Python call
// is made on a dead C object I raise an appropriate Python exception.
// The C pointer <-> id number -> Python object maps are maintained as
// static tables by C code in this file.  The Python object -> id number
// is recorded as the (undocumented) sparky_id class member.
// 

#include <stdio.h>		// use FILE
#include <fstream.h>		// use ofstream, ifstream

#include <Python.h>		// Use PyObject
#include <tcl.h>		// Use Tcl_Interp

#include "atom.h"		// use Atom
#include "command.h"		// use Command_Dispatcher
#include "condition.h"		// Use Condition
#include "crosspeak.h"		// Use CrossPeak
#include "format.h"		// use open_sparky_file(), open_spectrum()
#include "grid.h"		// Use Grid
#include "group.h"		// Use Group
#include "integrate.h"		// Use integrate_list()
#include "label.h"		// Use Label
#include "line.h"		// Use Line
#include "list.h"		// Use List
#include "memalloc.h"		// use new()
#include "molecule.h"		// Use Molecule
#include "nmrdata.h"		// use dataless_nmr_data()
#include "notifier.h"		// use Notifier
#include "objectid.h"		// Use Object_Table
#include "ornament.h"		// Use Ornament
#include "paths.h"		// Use sparky_directory(), installation_path()
#include "peak.h"		// Use Peak
#include "peakgp.h"		// Use PeakGp
#include "project.h"		// use Project
#include "reporter.h"		// use Reporter
#include "resonance.h"		// Use Resonance
#include "session.h"		// use Session
#include "spectrum.h"		// Use Spectrum
#include "spoint.h"		// Use IPoint, SPoint
#include "stringc.h"		// Use Stringy
#include "system.h"		// use file_path()
#include "table.h"		// use Table
#include "uidialogs.h"		// use show_manual_url()
#include "uipeak.h"		// Use show_peak_list_dialog()
#include "uiplop.h"		// Use default_peak_list_format()
#include "uiwait.h"		// Use wait_callbacks()
#include "uiview.h"		// Use View
#include "winsystem.h"		// use Widget

// ----------------------------------------------------------------------------
//
struct Get_Set_Functions
{
  const char *attribute_name;
  PyObject *(*get_value)(PyObject *self, void *c_object);
  bool (*set_value)(void *c_object, PyObject *value);
  bool can_change;
};

// ----------------------------------------------------------------------------
//
class Attribute_Name
{
public:
  Attribute_Name(PyObject *py_class, PyObject *py_attr);
  static bool equal(const void *attr_name_1, const void *attr_name_2);
  static unsigned long hash(const void *attr_name);
  PyObject *py_class, *py_attr;
};

// ----------------------------------------------------------------------------
//
class Attribute_Method
{
public:
  Attribute_Method(const Stringy &c_object_type,
		   const Get_Set_Functions &getset);
  Attribute_Method(const Stringy &attr, const Stringy &c_object_type,
		   PyMethodDef *mdef);
  PyObject *getattr(PyObject *self, PyObject *attr);
  PyObject *setattr(PyObject *self, PyObject *value);
private:
  Stringy attribute_name;
  Stringy c_object_type;
  PyObject *(*get_value)(PyObject *self, void *c_object);
  bool (*set_value)(void *c_object, PyObject *value);
  bool can_change;
  PyMethodDef *mdef;
};

// ----------------------------------------------------------------------------
//
class C_Python_Map
{
public:
  void listen_for_deleted_objects(Session &);
  
  int object_id(void *c_object, const Stringy &type);
  void *id_to_object(int id, Stringy *type);
  PyObject *id_to_object(int id);
  void set_python_object_id(PyObject *py_object, int id);

private:
  Object_Table c_object_ids;
  Table id_to_python_object;

  static void will_delete_object_cb(void *cpmap, void *object);
  static void will_delete_id_cb(void *cpmap, void *deleted_id);
};

// ----------------------------------------------------------------------------
// Combines a session and Python callback function.
//
class Python_Callback
{
public:
  Python_Callback(Session &, PyObject *py_func);
  virtual ~Python_Callback();

  Session &session();
  void invoke(PyObject *arg);

private:
  Session &ses;
  PyObject *py_func;
  bool *signal_delete;

  static void session_deleted_cb(void *, void *);
};

// ----------------------------------------------------------------------------
// Callback to Python invoked by C++ notifier.
//
class Notifier_Callback : public Python_Callback
{
public:
  Notifier_Callback(Session &, Notice_Type, Notice_Callback, PyObject *func);
  ~Notifier_Callback();

private:
  Notice_Type ntype;
  Notice_Callback cb;
};

// ----------------------------------------------------------------------------
// Call Python when file is readable
//
class Python_Input_Callback : public Python_Callback
{
public:
  Python_Input_Callback(Session &, FILE *, PyObject *func);
  ~Python_Input_Callback();

private:
  FILE *fp;

  static void call_input_callback(CB_Data icallback);
};

// ----------------------------------------------------------------------------
//
static PyObject *python_session(Session &s);
static PyObject *python_project(Project *proj);
static PyObject *python_spectrum(Spectrum *sp);
static PyObject *python_peak(CrossPeak *xp);
static PyObject *python_ornament(Ornament *op);
static PyObject *python_resonance(Resonance *r);
static PyObject *python_label(Label *label);
static PyObject *python_line(Line *line);
static PyObject *python_grid(Grid *grid);
static PyObject *python_molecule(Molecule *m);
static PyObject *python_group(Group *g);
static PyObject *python_atom(Atom *a);
static PyObject *python_condition(Condition *c);
static PyObject *python_view(View *v);
static PyObject *python_contour_levels(const Contour_Parameters &cp);
static PyObject *python_callback(Python_Callback *pycb);

static PyObject *python_spectrum_e(void *sp);
static PyObject *python_peak_e(void *pk);
static PyObject *python_crosspeak_e(void *xp);
static PyObject *python_ornament_e(void *op);
static PyObject *python_resonance_e(void *r);
static PyObject *python_label_e(void *label);
static PyObject *python_line_e(void *line);
static PyObject *python_grid_e(void *grid);
static PyObject *python_group_e(void *g);
static PyObject *python_atom_e(void *a);
static PyObject *python_molecule_e(void *m);
static PyObject *python_condition_e(void *c);
static PyObject *python_view_e(void *v);

static bool set_ornament_color(Ornament *op, PyObject *value);
static bool set_ornament_selected(Ornament *op, PyObject *value);
static bool set_view_switch(void *view, bool View_Settings::*member,
			    PyObject *value);
static void set_contour_level_attributes(PyObject *levels,
					 const Contour_Parameters &cp);

static PyObject *python_list(const List &c_objects,
			     PyObject *(*py_object)(void *));
static bool python_get_attribute(PyObject *inst, const Stringy &attr,
				 PyObject **value);
static bool python_get_attribute(PyObject *inst, const Stringy &attr,
				 int *value);
static bool python_get_attribute(PyObject *inst, const Stringy &attr,
				 double *value);
static bool python_get_attribute(PyObject *inst, const Stringy &attr,
				 Stringy *value);
static void python_set_attribute(PyObject *instance, const Stringy &attribute,
				 const Stringy &value);
static void python_set_attribute(PyObject *instance, const Stringy &attribute,
				 int value);
static void python_set_attribute(PyObject *instance, const Stringy &attribute,
				 double value);
static void python_set_attribute(PyObject *instance, const Stringy &attribute,
				 PyObject *value);
static PyObject *instance_dictionary_get(PyObject *inst, const Stringy &attr);
static void instance_dictionary_set(PyObject *instance, PyObject *attr,
				    PyObject *value);
static PyObject *python_none();
static PyObject *python_boolean(bool value);
static PyObject *python_int(int value);
static PyObject *python_float(double value);
static PyObject *python_string(const Stringy &string);
static PyObject *python_list(const List &py_objects);
static PyObject *python_tuple(const IPoint &value);
static PyObject *python_tuple(const SPoint &value);
static PyObject *python_tuple(const SRegion &value);
static PyObject *python_tuple(const List &py_objects);

static Notifier_Callback *notifier_callback(Session &, const Stringy &name,
					    PyObject *callback);
static void call_void_callback(void *py_func, void *);
static void call_sregion_callback(void *py_func, void *sregion);
static void call_peak_callback(void *py_func, void *peak);
static void call_spectrum_callback(void *py_func, void *spectrum);

static bool python_menu_callback(Session &, const Stringy &, void *cb);
static bool check_dimension(const SPoint &p, int dim, const Stringy &error);
static bool check_dimension(const SPoint &p, Spectrum *sp);
static bool check_dimension(const SRegion &r, Spectrum *sp);
static double python_number_as_double(PyObject *num);

extern "C" int bool_arg(PyObject *value, void *b);
extern "C" int string_list_arg(PyObject *list, void *slist);
extern "C" int callable_arg(PyObject *func, void *func_p);
extern "C" int file_arg(PyObject *file, void *fp);
extern "C" int ipoint_arg(PyObject *seq, void *ipoint);
static bool permutation_argument(const Stringy &name, int dim,
				 PyObject *arg, IPoint *p);
extern "C" int spoint_arg(PyObject *seq, void *spoint);
static bool spoint_argument(const Stringy &name, int dim,
			    PyObject *arg, SPoint *p);
extern "C" int sregion_arg(PyObject *seq, void *sregion);
extern "C" int spectrum_arg(PyObject *py_spectrum, void *spectrum);
extern "C" int view_arg(PyObject *py_view, void *view);
extern "C" int levels_arg(PyObject *py_levels, void *levels);
extern "C" int peak_arg(PyObject *py_peak, void *crosspeak);
extern "C" int peak_list_arg(PyObject *list, void *plist);
extern "C" int atom_arg(PyObject *py_atom, void *atom);
extern "C" int atom_list_arg(PyObject *py_atom, void *alist);
extern "C" int resonance_arg(PyObject *py_resonance, void *resonance);
extern "C" int resonance_list_arg(PyObject *list, void *rlist);
static int list_arg(PyObject *py_list,
		    int (*list_arg)(PyObject *, void *),
		    void *list);
static PyObject *python_sequence_item(PyObject *seq, int k);
extern "C" int python_callback_arg(PyObject *py_callback, void *pycb);
static int sparky_id(PyObject *inst);
static void *c_object(PyObject *py_object, const Stringy &type);
static void object_type_error(int id, const Stringy &expected_type,
			      const Stringy &received_type);
static void set_python_error(const Stringy &message);
extern "C" void initspy();
extern "C" PyObject *add_input_callback(PyObject *, PyObject *args);
extern "C" PyObject *remove_input_callback(PyObject *, PyObject *args);
extern "C" int widget_tcl_interpretter_arg(PyObject *w, void *ti);
static Tcl_Interp *widget_tcl_interp(PyObject *w);
static bool tk_widget_arg(WinSys &, PyObject *w, void *tkwin);
static bool call_python_callback(Session &, PyObject *cb, PyObject *arg);
static void print_error_message(Session &);
static PyObject *eval_string(const Stringy &command, PyObject *module);
static void define_sparky_classes();
static PyObject *python_class(const Stringy &classname,
			      PyMethodDef *init_method = NULL);
static void set_sparky_id(PyObject *self, int id);
static PyObject *python_class_instance(const Stringy &classname);
static PyObject *new_mirror_object(const Stringy &classname,
				   void *c_object, const Stringy &type);
static PyObject *python_mirror_object(const Stringy &classname, void *cobj,
				      const Stringy &c_object_type);
static PyObject *get_module_value(const Stringy &name, const Stringy &attr);
static PyObject *get_module_value(PyObject *module, const Stringy &attr);
static void set_module_value(const Stringy &name, const Stringy &attr, 
			     PyObject *value);
static void set_module_value(PyObject *module, const Stringy &attr, 
			     PyObject *value);
extern "C" PyObject *sparky_getattr(PyObject *, PyObject *args);
extern "C" PyObject *sparky_setattr(PyObject *, PyObject *args);
static void register_attributes(const Stringy &classname,
				const Stringy &c_object_type,
				Get_Set_Functions *getset,
				PyMethodDef *methods);

// ----------------------------------------------------------------------------
//
static PyObject *this_module = NULL;		// state variable

// ----------------------------------------------------------------------------
// Table mapping Python class attributes to get and set methods.
//
static Table *attribute_table = NULL;		// state variable

// ----------------------------------------------------------------------------
// Map between C++ and Python objects using ID numbers.
//
static C_Python_Map *c_python_map = NULL;	// state variable

// ----------------------------------------------------------------------------
//
static PyObject *python_session(Session &s)
{ 
  return python_mirror_object("Session", &s, "session");
}

// ----------------------------------------------------------------------------
//
extern "C" PyObject *session_init(PyObject *, PyObject *args)
{
  PyObject *self;
  if (PyArg_ParseTuple(args, "O!", &PyInstance_Type, &self))
    return python_none();
  PyErr_Clear();

  //
  // Maybe __init__(self, tk)
  //
  Tcl_Interp *tcl_interp;
  if (PyArg_ParseTuple(args, "O!O&",
		       &PyInstance_Type, &self,
		       widget_tcl_interpretter_arg, &tcl_interp))
    {
      Session *s = new Session((void *) tcl_interp);
      c_python_map->listen_for_deleted_objects(*s);
      set_sparky_id(self, c_python_map->object_id(s, "session"));
      return python_none();
    }

  return NULL;
}

// ----------------------------------------------------------------------------
//
static PyObject *session_project(PyObject *, void *ses)
  { return python_project(&((Session *) ses)->project()); }

// ----------------------------------------------------------------------------
//
extern "C" PyObject *open_file(PyObject *self, PyObject *args)
{
  char *path;
  if (!PyArg_ParseTuple(args, "s", &path))
    return NULL;

  Session *s = (Session *) c_object(self, "session");
  if (s == NULL)
    return NULL;

  Stringy error_msg;
  bool success = open_sparky_file(*s, path, &error_msg);
  if (!success && !error_msg.is_empty())
    s->reporter().warning(error_msg.cstring());

  return python_boolean(success);
}

// ----------------------------------------------------------------------------
//
extern "C" PyObject *open_spectrum(PyObject *self, PyObject *args)
{
  char *path;
  if (!PyArg_ParseTuple(args, "s", &path))
    return NULL;

  Session *s = (Session *) c_object(self, "session");
  if (s == NULL)
    return NULL;

  Stringy error_msg;
  Spectrum *sp = open_spectrum(*s, path, &error_msg);
  if (sp == NULL && !error_msg.is_empty())
    s->reporter().warning(error_msg.cstring());

  return (sp ? python_spectrum(sp) : python_none());
}

// ----------------------------------------------------------------------------
//
extern "C" PyObject *command_characters(PyObject *self, PyObject *args)
{
  char *string;
  if (!PyArg_ParseTuple(args, "s", &string))
    return NULL;

  Session *s = (Session *) c_object(self, "session");
  if (s == NULL)
    return NULL;

  for (char *c = string ; *c != '\0' ; ++c)
    s->command_dispatcher().parse_key(*c);
  
  return python_none();
}

// ----------------------------------------------------------------------------
//
extern "C" PyObject *create_view(PyObject *self, PyObject *args)
{
  Spectrum *sp;
  PyObject *py_parent;
  if (!PyArg_ParseTuple(args, "OO&", &py_parent, spectrum_arg, &sp))
    return NULL;

  Session *s = (Session *) c_object(self, "session");
  if (s == NULL)
    return NULL;

  Widget parent = NULL;
  if (py_parent != Py_None &&
      !tk_widget_arg(s->window_system(), py_parent, &parent))
    return NULL;

  View_Settings settings(sp);
  settings.show_scales = false;
  settings.show_scrollbars = false;
  View *v = new View(*s, parent, sp, settings);

  return python_view(v);
}

// ----------------------------------------------------------------------------
//
extern "C" PyObject *selected_view(PyObject *self, PyObject *args)
{
  if (!PyArg_ParseTuple(args, ""))
    return NULL;

  Session *s = (Session *) c_object(self, "session");
  if (s == NULL)
    return NULL;

  View *v = selected_view(*s);
  return (v ? python_view(v) : python_none());
}

// ----------------------------------------------------------------------------
//
extern "C" PyObject *selected_spectrum(PyObject *self, PyObject *args)
{
  if (!PyArg_ParseTuple(args, ""))
    return NULL;

  Session *s = (Session *) c_object(self, "session");
  if (s == NULL)
    return NULL;

  View *v = selected_view(*s);
  return (v ? python_spectrum(v->spectrum()) : python_none());
}

// ----------------------------------------------------------------------------
//
extern "C" PyObject *unselect_all_ornaments(PyObject *self, PyObject *args)
{
  if (!PyArg_ParseTuple(args, ""))
    return NULL;

  Session *s = (Session *) c_object(self, "session");
  if (s == NULL)
    return NULL;

  s->project().unselect_all_ornaments();

  return python_none();
}

// ----------------------------------------------------------------------------
//
extern "C" PyObject *selected_peaks(PyObject *self, PyObject *args)
{
  if (!PyArg_ParseTuple(args, ""))
    return NULL;

  Session *s = (Session *) c_object(self, "session");
  if (s == NULL)
    return NULL;

  return python_list(s->project().selected_crosspeaks(), python_crosspeak_e);
}

// ----------------------------------------------------------------------------
//
extern "C" PyObject *selected_ornaments(PyObject *self, PyObject *args)
{
  if (!PyArg_ParseTuple(args, ""))
    return NULL;

  Session *s = (Session *) c_object(self, "session");
  if (s == NULL)
    return NULL;

  return python_list(s->project().selected_ornaments(), python_ornament_e);
}

// ----------------------------------------------------------------------------
// Make a new sparky peak list.
//
extern "C" PyObject *show_peak_list(PyObject *self, PyObject *args)
{
  List peaks;
  if (!PyArg_ParseTuple(args, "O&", peak_list_arg, &peaks))
    return NULL;

  Session *s = (Session *) c_object(self, "session");
  if (s == NULL)
    return NULL;

  show_peak_list_dialog(*s, peaks, default_peak_list_format());

  return python_none();
}

// ----------------------------------------------------------------------------
// Make a resonance peak list.
//
extern "C" PyObject *show_resonance_peak_list(PyObject *self, PyObject *args)
{
  Resonance *r;
  if (!PyArg_ParseTuple(args, "O&", resonance_arg, &r))
    return NULL;

  Session *s = (Session *) c_object(self, "session");
  if (s == NULL)
    return NULL;

  show_peak_list_dialog(*s, r);

  return python_none();
}

// ----------------------------------------------------------------------------
// Python interface to Sparky notifier.
//
extern "C" PyObject *notify_python(PyObject *self, PyObject *args)
{
  char *notice_type;
  PyObject *callback;
  if (!PyArg_ParseTuple(args, "sO&", &notice_type, callable_arg, &callback))
    return NULL;

  Session *s = (Session *) c_object(self, "session");
  if (s == NULL)
    return NULL;

  Notifier_Callback *ncb = notifier_callback(*s, notice_type, callback);
  if (ncb == NULL)
    {
      set_python_error(Stringy("Unknown notice type: ") + notice_type);
      return NULL;
    }

  return python_callback(ncb);
}

// ----------------------------------------------------------------------------
// Python interface to Sparky notifier.
//
extern "C" PyObject *dont_notify_python(PyObject *, PyObject *args)
{
  Python_Callback *pcb;
  if (!PyArg_ParseTuple(args, "O&", python_callback_arg, &pcb))
    return NULL;

  delete pcb;

  return python_none();
}

// ----------------------------------------------------------------------------
// Combines a session and Python callback function.
//
Python_Callback::Python_Callback(Session &s, PyObject *py_func) : ses(s)
{
  this->signal_delete = NULL;
  Py_XINCREF(py_func);
  this->py_func = py_func;
  s.notifier().notify_me(nt_will_delete_session, session_deleted_cb, this);
}

// ----------------------------------------------------------------------------
//
Python_Callback::~Python_Callback()
{
  Notifier &nt = session().notifier();
  nt.send_notice(nt_will_delete_py_callback, this);
  nt.dont_notify_me(nt_will_delete_session, session_deleted_cb, this);

  if (signal_delete)
    *signal_delete = true;	// Tells invoke() method not to delete me twice

  Py_XDECREF(py_func);
  py_func = NULL;
}

// ----------------------------------------------------------------------------
//
void Python_Callback::session_deleted_cb(void *py_callback, void *session)
{
  Python_Callback *pcb = (Python_Callback *) py_callback;
  if ((Session *) session == &pcb->session())
    delete pcb;
}

// ----------------------------------------------------------------------------
//
Session &Python_Callback::session()
  { return ses; }

// ----------------------------------------------------------------------------
// If the python callback raises an exception then it is deleted.
// This is a bit tricky because the callback may have already deleted
// this object so there is a danger of deleting it twice.  To avoid
// this the Python_Callback destructor signals it has been deleted
// by setting a local bool variable via a member pointer.  Dicey.
//
void Python_Callback::invoke(PyObject *arg)
{
  bool callback_deleted = false;
  this->signal_delete = &callback_deleted;
  if (!call_python_callback(session(), py_func, arg) && !callback_deleted)
    delete this;
  else
    this->signal_delete = NULL;
}

// ----------------------------------------------------------------------------
//
Notifier_Callback::Notifier_Callback(Session &s, Notice_Type ntype,
				     Notice_Callback cb, PyObject *py_func)
  : Python_Callback(s, py_func)
{
  this->ntype = ntype;
  this->cb = cb;

  s.notifier().notify_me(ntype, cb, this);
}

// ----------------------------------------------------------------------------
//
Notifier_Callback::~Notifier_Callback()
{
  session().notifier().dont_notify_me(ntype, cb, this);
}

// ----------------------------------------------------------------------------
//
static Notifier_Callback *notifier_callback(Session &s, const Stringy &name,
					    PyObject *callback)
{
  struct Python_Notice 
  {
    char *name;
    Notice_Type type;
    void (*call_callback)(void *cb, void *object);
  }
  notices[] = {
    {"selection changed",
     nt_selected_ornaments_changed,	call_void_callback},
    {"drag region",		nt_drag_sregion,	call_sregion_callback},
    {"will drag peak",		nt_will_drag_peak,	call_peak_callback},
    {"dragged peak",		nt_dragged_peak,	call_peak_callback},
    {"removed spectrum from project",
     nt_removed_spectrum_from_project,	call_spectrum_callback},
    {"added spectrum to project",
     nt_added_spectrum_to_project,	call_spectrum_callback},
    {NULL}
  };

  for (int ni = 0 ; notices[ni].name ; ++ni)
    if (name == notices[ni].name)
      return new Notifier_Callback(s, notices[ni].type,
				   notices[ni].call_callback, callback);

  return NULL;
}

// ----------------------------------------------------------------------------
//
static void call_void_callback(void *ncallback, void *)
{
  ((Notifier_Callback *) ncallback)->invoke(NULL);
}

// ----------------------------------------------------------------------------
//
static void call_sregion_callback(void *ncallback, void *sregion)
{
  PyObject *arg = python_tuple(*(SRegion *) sregion);
  ((Notifier_Callback *) ncallback)->invoke(arg);
  Py_XDECREF(arg);
}

// ----------------------------------------------------------------------------
//
static void call_peak_callback(void *ncallback, void *peak)
{
  PyObject *arg = python_peak((Peak *) peak);
  ((Notifier_Callback *) ncallback)->invoke(arg);
  Py_XDECREF(arg);
}

// ----------------------------------------------------------------------------
//
static void call_spectrum_callback(void *ncallback, void *spectrum)
{
  PyObject *arg = python_spectrum((Spectrum *) spectrum);
  ((Notifier_Callback *) ncallback)->invoke(arg);
  Py_XDECREF(arg);
}

// ----------------------------------------------------------------------------
//
extern "C" PyObject *add_menu_command(PyObject *self, PyObject *args)
{
  char *accel, *menu_text;
  PyObject *callback;
  if (!PyArg_ParseTuple(args, "ssO&", &accel, &menu_text,
			callable_arg, &callback))
    return NULL;

  Session *s = (Session *) c_object(self, "session");
  if (s == NULL)
    return NULL;

  Py_XINCREF(callback);
  Command_Dispatcher &cd = s->command_dispatcher();
  cd.add_accelerator(accel, menu_text, python_menu_callback, callback);

  return python_none();
}

// ----------------------------------------------------------------------------
//
static bool python_menu_callback(Session &s, const Stringy &,
				 void *py_callback)
{
  PyObject *py_cb = (PyObject *) py_callback;
  bool result = call_python_callback(s, py_cb, NULL);

  return result;
}

// ----------------------------------------------------------------------------
// Add a Python callback invoked when input on file descriptor is available.
//
extern "C" PyObject *add_input_callback(PyObject *self, PyObject *args)
{
  FILE *fp;
  PyObject *py_callback;
  if (!PyArg_ParseTuple(args, "O&O&",
			file_arg, &fp, callable_arg, &py_callback))
    return NULL;

  Session *s = (Session *) c_object(self, "session");
  if (s == NULL)
    return NULL;

  Python_Input_Callback *icb = new Python_Input_Callback(*s, fp, py_callback);

  return python_callback(icb);
}

// ----------------------------------------------------------------------------
// Add a Python callback invoked when input on file descriptor is available.
//
extern "C" PyObject *remove_input_callback(PyObject *, PyObject *args)
{
  Python_Callback *pycb;
  if (!PyArg_ParseTuple(args, "O&", python_callback_arg, &pycb))
    return NULL;

  delete pycb;

  return python_none();
}

// ----------------------------------------------------------------------------
//
Python_Input_Callback::Python_Input_Callback(Session &s, FILE *fp,
					     PyObject *func)
  : Python_Callback(s, func)
{
  this->fp = fp;
  s.window_system().add_input_callback(fp, call_input_callback, this);
}

// ----------------------------------------------------------------------------
//
Python_Input_Callback::~Python_Input_Callback()
{
  WinSys &ws = session().window_system();
  ws.remove_input_callback(fp, call_input_callback, this);
  fp = NULL;
}

// ----------------------------------------------------------------------------
//
void Python_Input_Callback::call_input_callback(CB_Data icallback)
  { ((Python_Input_Callback *) icallback)->invoke(NULL); }

// ----------------------------------------------------------------------------
//
extern "C" PyObject *show_manual(PyObject *self, PyObject *args)
{
  char *url;
  if (!PyArg_ParseTuple(args, "s", &url))
    return NULL;

  Session *s = (Session *) c_object(self, "session");
  if (s == NULL)
    return NULL;

  show_manual_url(*s, url);

  return python_none();
}

// ----------------------------------------------------------------------------
//
static Get_Set_Functions session_getset[] = {
  {"project",		session_project,		0,	true},
  {NULL},
};
static PyMethodDef session_methods[] = {
  {"open_file",			open_file,			METH_VARARGS},
  {"open_spectrum",		open_spectrum,			METH_VARARGS},
  {"add_input_callback",	add_input_callback,		METH_VARARGS},
  {"remove_input_callback",	remove_input_callback,		METH_VARARGS},
  {"command_characters",	command_characters,		METH_VARARGS},
  {"create_view",		create_view,			METH_VARARGS},
  {"selected_view",		selected_view,			METH_VARARGS},
  {"selected_spectrum",		selected_spectrum,		METH_VARARGS},
  {"unselect_all_ornaments",	unselect_all_ornaments,		METH_VARARGS},
  {"selected_peaks",		selected_peaks,			METH_VARARGS},
  {"selected_ornaments",	selected_ornaments,		METH_VARARGS},
  {"show_peak_list",		show_peak_list,			METH_VARARGS},
  {"show_resonance_peak_list",	show_resonance_peak_list,	METH_VARARGS},
  {"add_command",		add_menu_command,		METH_VARARGS},
  {"show_manual_url",		show_manual,			METH_VARARGS},
  {"notify_me",			notify_python,			METH_VARARGS},
  {"dont_notify_me",		dont_notify_python,		METH_VARARGS},
  {NULL,		NULL},
};

// ----------------------------------------------------------------------------
//
static PyObject *python_project(Project *proj)
{ 
  return python_mirror_object("Project", proj, "project");
}

// ----------------------------------------------------------------------------
//
static PyObject *project_save_path(PyObject *, void *proj)
  { return python_string(((Project *) proj)->save_file().path()); }
static PyObject *project_sparky_directory(PyObject *, void *proj)
  { return python_string(((Project *) proj)->sparky_path("", "")); }

// ----------------------------------------------------------------------------
//
extern "C" PyObject *project_spectra(PyObject *self, PyObject *args)
{
  if (!PyArg_ParseTuple(args, ""))
    return NULL;

  Project *p = (Project *) c_object(self, "project");
  if (p == NULL)
    return NULL;

  return python_list(p->spectrum_list(), python_spectrum_e);
}

// ----------------------------------------------------------------------------
//
extern "C" PyObject *project_views(PyObject *self, PyObject *args)
{
  if (!PyArg_ParseTuple(args, ""))
    return NULL;

  Project *p = (Project *) c_object(self, "project");
  if (p == NULL)
    return NULL;

  return python_list(p->view_list(), python_view_e);
}

// ----------------------------------------------------------------------------
//
extern "C" PyObject *project_molecules(PyObject *self, PyObject *args)
{
  if (!PyArg_ParseTuple(args, ""))
    return NULL;

  Project *p = (Project *) c_object(self, "project");
  if (p == NULL)
    return NULL;

  return python_list(p->molecule_list(), python_molecule_e);
}

// ----------------------------------------------------------------------------
//
extern "C" PyObject *project_conditions(PyObject *self, PyObject *args)
{
  if (!PyArg_ParseTuple(args, ""))
    return NULL;

  Project *p = (Project *) c_object(self, "project");
  if (p == NULL)
    return NULL;

  return python_list(p->condition_list(), python_condition_e);
}

// ----------------------------------------------------------------------------
//
extern "C" PyObject *project_dataless_spectrum(PyObject *self, PyObject *args)
{
  char *name;
  IPoint size, bsize;
  SPoint sfreq, swidth, origin_ppm;
  List axis_labels;
  if (!PyArg_ParseTuple(args, "sO&O&O&O&O&O&", &name, ipoint_arg, &size,
			ipoint_arg, &bsize, spoint_arg, &sfreq,
			spoint_arg, &swidth, spoint_arg, &origin_ppm,
			string_list_arg, &axis_labels))
    return NULL;

  int d = size.dimension();
  if (d < 2)
    {
      set_python_error("spectrum dimension must be >= 2");
      return NULL;
    }

  if (bsize.dimension() != d || sfreq.dimension() != d ||
      swidth.dimension() != d || origin_ppm.dimension() != d ||
      axis_labels.size() != d)
    {
      set_python_error("dimensions of arguments do not match");
      return NULL;
    }

  Project *p = (Project *) c_object(self, "project");
  if (p == NULL)
    return NULL;

  Stringy save_path = "";
  Stringy nmr_path = "";
  NMR_Data *nmr_data = dataless_nmr_data(nmr_path, size, bsize,
					 sfreq, swidth, origin_ppm,
					 axis_labels);
  free_string_list_entries(axis_labels);

  Spectrum *s = new Spectrum(p->session(), name, p->default_condition(),
			     save_path, nmr_data);

  return python_spectrum(s);
}

// ----------------------------------------------------------------------------
//
extern "C" PyObject *project_save_value(PyObject *self, PyObject *args)
{
  char *key, *value;
  if (!PyArg_ParseTuple(args, "ss", &key, &value))
    return NULL;

  Project *proj = (Project *) c_object(self, "project");
  if (proj == NULL)
    return NULL;

  return python_boolean(proj->saved_values().save_value(key, value));
}

// ----------------------------------------------------------------------------
//
extern "C" PyObject *project_saved_value(PyObject *self, PyObject *args)
{
  Project *proj = (Project *) c_object(self, "project");
  if (proj == NULL)
    return NULL;

  char *key;
  if (!PyArg_ParseTuple(args, "s", &key))
    return NULL;

  Stringy value;
  bool got_it = proj->saved_values().saved_value(key, &value);
  return (got_it ? python_string(value) : python_none());
}

// ----------------------------------------------------------------------------
//
extern "C" PyObject *project_close(PyObject *self, PyObject *args)
{
  bool ask;
  if (!PyArg_ParseTuple(args, "O&", bool_arg, &ask))
    return NULL;

  Project *p = (Project *) c_object(self, "project");
  if (p == NULL)
    return NULL;

  bool unload = (!ask || p->unload_query());
  if (unload)
    p->unload();

  return python_boolean(unload);
}

// ----------------------------------------------------------------------------
//
static Get_Set_Functions proj_getset[] = {
  {"save_path",		project_save_path,		0,	true},
  {"sparky_directory",	project_sparky_directory,	0,	true},
  {NULL},
};
static PyMethodDef proj_methods[] = {
  {"spectrum_list",	project_spectra,		METH_VARARGS},
  {"view_list",		project_views,			METH_VARARGS},
  {"molecule_list",	project_molecules,		METH_VARARGS},
  {"condition_list",	project_conditions,		METH_VARARGS},
  {"create_dataless_spectrum",	project_dataless_spectrum,	METH_VARARGS},
  {"save_value",	project_save_value,		METH_VARARGS},
  {"saved_value",	project_saved_value,		METH_VARARGS},
  {"close",		project_close,			METH_VARARGS},
  {NULL,		NULL},
};

// ----------------------------------------------------------------------------
//
static PyObject *python_spectrum(Spectrum *sp)
{
  return python_mirror_object("Spectrum", sp, "spectrum");
}

// ----------------------------------------------------------------------------
//
static PyObject *spectrum_dimension(PyObject *, void *sp)
  { return python_int(((Spectrum *) sp)->dimension()); }
static PyObject *spectrum_name(PyObject *, void *sp)
  { return python_string(((Spectrum *) sp)->name()); }
static PyObject *spectrum_session(PyObject *, void *sp)
  { return python_session(((Spectrum *) sp)->session()); }
static PyObject *spectrum_data_size(PyObject *, void *sp)
  { return python_tuple(((Spectrum *) sp)->index_range().size()); }
static PyObject *spectrum_region(PyObject *, void *sp)
  { return python_tuple(((Spectrum *) sp)->ppm_region()); }
static PyObject *spectrum_nuclei(PyObject *, void *sp)
  {
    Spectrum *spectrum = (Spectrum *) sp;
    int dim = spectrum->dimension();
    List nuclei;
    for (int a = 0 ; a < dim ; ++a)
      nuclei.append(python_string(spectrum->nucleus_type(a)));
    return python_tuple(nuclei);
  }
static PyObject *spectrum_hz_per_ppm(PyObject *, void *sp)
  {
    Spectrum *spectrum = (Spectrum *) sp;
    int dim = spectrum->dimension();
    SPoint hz_per_ppm(dim);
    for (int a = 0 ; a < dim ; ++a)
      hz_per_ppm[a] = spectrum->scale(1.0, a, PPM, HZ);
    return python_tuple(hz_per_ppm);
  }
static PyObject *spectrum_spectrum_width(PyObject *, void *sp)
  { return python_tuple(((Spectrum *) sp)->ppm_spectrum_width()); }
static PyObject *spectrum_sweep_width(PyObject *, void *sp)
  { return python_tuple(((Spectrum *) sp)->ppm_sweep_width()); }
static PyObject *spectrum_data_path(PyObject *, void *sp)
  { return python_string(((Spectrum *) sp)->data_path()); }
static PyObject *spectrum_save_path(PyObject *, void *sp)
  { return python_string(((Spectrum *) sp)->save_file().path()); }
static PyObject *spectrum_condition(PyObject *, void *sp)
  { return python_condition(((Spectrum *) sp)->condition()); }
static PyObject *spectrum_molecule(PyObject *, void *sp)
  { return python_molecule(((Spectrum *) sp)->molecule()); }
static PyObject *spectrum_noise(PyObject *, void *sp)
  { return python_float(((Spectrum *) sp)->noise_level()); }
static PyObject *spectrum_scale_offset(PyObject *, void *sp)
  { return python_tuple(((Spectrum *) sp)->ppm_shift()); }
static PyObject *spectrum_pick_minimum_linewidth(PyObject *, void *sp)
  { return python_tuple(((Spectrum *) sp)->mPeakPick.minimum_linewidth); }
static PyObject *spectrum_pick_minimum_drop_factor(PyObject *, void *sp)
  { return python_float(((Spectrum *) sp)->mPeakPick.minimum_drop_factor); }

// ----------------------------------------------------------------------------
//
static bool set_scale_offset(void *spectrum, PyObject *value)
{
  Spectrum *sp = (Spectrum *) spectrum;

  SPoint offset;
  if (!spoint_argument("scale offset", sp->dimension(), value, &offset))
    return false;

  SPoint zero = sp->ppm_shift() - offset;
  sp->set_ppm_zero(zero);

  return true;
}

// ----------------------------------------------------------------------------
//
extern "C" PyObject *spectrum_peaks(PyObject *self, PyObject *args)
{
  if (!PyArg_ParseTuple(args, "")) return NULL;
  Spectrum *sp = (Spectrum *) c_object(self, "spectrum");
  return (sp ? python_list(sp->crosspeaks(), python_crosspeak_e) : NULL);
}

// ----------------------------------------------------------------------------
//
extern "C" PyObject *spectrum_selected_peaks(PyObject *self, PyObject *args)
{
  if (!PyArg_ParseTuple(args, "")) return NULL;
  Spectrum *sp = (Spectrum *) c_object(self, "spectrum");
  return (sp ? python_list(sp->selected_crosspeaks(), python_crosspeak_e) : NULL);
}

// ----------------------------------------------------------------------------
//
extern "C" PyObject *spectrum_labels(PyObject *self, PyObject *args)
{
  if (!PyArg_ParseTuple(args, "")) return NULL;
  Spectrum *sp = (Spectrum *) c_object(self, "spectrum");
  return (sp ? python_list(sp->ornaments(label), python_label_e) : NULL);
}

// ----------------------------------------------------------------------------
//
extern "C" PyObject *spectrum_lines(PyObject *self, PyObject *args)
{
  if (!PyArg_ParseTuple(args, "")) return NULL;
  Spectrum *sp = (Spectrum *) c_object(self, "spectrum");
  return (sp ? python_list(sp->ornaments(line), python_line_e) : NULL);
}

// ----------------------------------------------------------------------------
//
extern "C" PyObject *spectrum_grids(PyObject *self, PyObject *args)
{
  if (!PyArg_ParseTuple(args, "")) return NULL;
  Spectrum *sp = (Spectrum *) c_object(self, "spectrum");
  return (sp ? python_list(sp->ornaments(grid), python_grid_e) : NULL);
}

// ----------------------------------------------------------------------------
// Make a new peak ornament at a point in a spectrum
//
extern "C" PyObject *data_height(PyObject *self, PyObject *args)
{
  SPoint point;
  if (!PyArg_ParseTuple(args, "O&", spoint_arg, &point))
    return NULL;

  Spectrum *sp = (Spectrum *) c_object(self, "spectrum");
  if (sp == NULL)
    return NULL;

  if (!check_dimension(point, sp))
      return NULL;

  return python_float(sp->height_at_point(point));
}

// ----------------------------------------------------------------------------
// Find a peak in a given spectrum with given assignment
//
extern "C" PyObject *find_peak(PyObject *self, PyObject *args)
{
  Spectrum *sp = (Spectrum *) c_object(self, "spectrum");
  if (sp == NULL)
    return NULL;

  char *assignment;
  List resonances;
  CrossPeak *xp = NULL;
  if (PyArg_ParseTuple(args, "s", &assignment))
    xp = sp->find_assigned_peak(assignment);
  else if ((PyErr_Clear(),
	    PyArg_ParseTuple(args, "O&", resonance_list_arg, &resonances)))
    if (resonances.size() != sp->dimension())
      {
	set_python_error("assignment dimension " +
			 Stringy(resonances.size()) +
			 " doesn't match spectrum dimension " +
			 Stringy(sp->dimension()));
	return NULL;
      }
    else
      xp = sp->find_assigned_peak(resonances);
  else
    return NULL;

  return (xp == NULL ? python_none() : python_peak(xp));
}

// ----------------------------------------------------------------------------
// Pick peaks in a specified region
//
extern "C" PyObject *pick_peaks(PyObject *self, PyObject *args)
{
  Spectrum *sp = (Spectrum *) c_object(self, "spectrum");
  if (sp == NULL)
    return NULL;

  SRegion region;
  SPoint height_thresholds, min_linewidths;
  double min_dropoff;
  if (!PyArg_ParseTuple(args, "O&O&O&d",
			sregion_arg, &region,
			spoint_arg, &height_thresholds,
			spoint_arg, &min_linewidths,
			&min_dropoff))
    return NULL;

  if (!check_dimension(region, sp) ||
      !check_dimension(height_thresholds, 2,
		       "need 2 peak picking height thresholds") ||
      !check_dimension(min_linewidths, sp))
    return NULL;

  Peak_Pick_Parameters params(sp->dimension());
  params.minNegHt = height_thresholds[0];
  params.minPosHt = height_thresholds[1];
  params.minimum_linewidth = min_linewidths;
  params.minimum_drop_factor = min_dropoff;

  List peaks = peak_pick_region(sp, region, params, NULL);

  return python_list(peaks, python_peak_e);
}

// ----------------------------------------------------------------------------
// Make a new peak ornament at a point in a spectrum
//
// This is not a Peak object constructor because having such a constructor
// makes Python peak creation many times slower.  When a spectrum peak list
// is first requested from Python Sparky can freeze for tens of seconds
// while creating Python objects for a few thousand peaks when a Peak
// constructor is used with an init wrapper as done in my python_class()
// routine.
//
extern "C" PyObject *place_peak(PyObject *self, PyObject *args)
{
  SPoint point;
  if (!PyArg_ParseTuple(args, "O&", spoint_arg, &point))
    return NULL;

  Spectrum *sp = (Spectrum *) c_object(self, "spectrum");
  if (sp == NULL)
    return NULL;

  if (!check_dimension(point, sp))
      return NULL;

  CrossPeak *xp = new Peak(sp, point);

  return python_peak(xp);
}

// ----------------------------------------------------------------------------
//
extern "C" PyObject *spectrum_save_value(PyObject *self, PyObject *args)
{
  char *key, *value;
  if (!PyArg_ParseTuple(args, "ss", &key, &value))
    return NULL;
  Spectrum *sp = (Spectrum *) c_object(self, "spectrum");
  if (sp == NULL)
    return NULL;
  return python_boolean(sp->saved_values().save_value(key, value));
}

// ----------------------------------------------------------------------------
//
extern "C" PyObject *spectrum_saved_value(PyObject *self, PyObject *args)
{
  Spectrum *sp = (Spectrum *) c_object(self, "spectrum");
  if (sp == NULL)
    return NULL;

  char *key;
  if (!PyArg_ParseTuple(args, "s", &key))
    return NULL;

  Stringy value;
  bool got_it = sp->saved_values().saved_value(key, &value);
  return (got_it ? python_string(value) : python_none());
}

// ----------------------------------------------------------------------------
//
static Get_Set_Functions spect_getset[] = {
  {"dimension",		spectrum_dimension,	0,	false},
  {"name",		spectrum_name,		0,	true},
  {"session",		spectrum_session,	0,	false},
  {"data_size",		spectrum_data_size,	0,	false},
  {"region",		spectrum_region,	0,	true},
  {"nuclei",		spectrum_nuclei,	0,	false},
  {"hz_per_ppm",	spectrum_hz_per_ppm,	0,	false},
  {"spectrum_width",	spectrum_spectrum_width,0,	false},
  {"sweep_width",	spectrum_sweep_width,	0,	true},
  {"data_path",		spectrum_data_path,	0,	true},
  {"save_path",		spectrum_save_path,	0,	true},
  {"condition",		spectrum_condition,	0,	true},
  {"molecule",		spectrum_molecule,	0,	true},
  {"noise",		spectrum_noise,		0,	true},
  {"scale_offset",	spectrum_scale_offset,	set_scale_offset,	true},
  {"pick_minimum_linewidth",spectrum_pick_minimum_linewidth, 0,	true},
  {"pick_minimum_drop_factor",spectrum_pick_minimum_drop_factor, 0,	true},
};
static PyMethodDef spect_methods[] = {
  {"selected_peaks",	spectrum_selected_peaks,	METH_VARARGS},
  {"peak_list",		spectrum_peaks,			METH_VARARGS},
  {"label_list",	spectrum_labels,		METH_VARARGS},
  {"line_list",		spectrum_lines,			METH_VARARGS},
  {"grid_list",		spectrum_grids,			METH_VARARGS},
  {"data_height",	data_height,			METH_VARARGS},
  {"find_peak",		find_peak,			METH_VARARGS},
  {"pick_peaks",	pick_peaks,			METH_VARARGS},
  {"place_peak",	place_peak,			METH_VARARGS},
  {"save_value",	spectrum_save_value,		METH_VARARGS},
  {"saved_value",	spectrum_saved_value,		METH_VARARGS},
  {NULL,		NULL},
};

// ----------------------------------------------------------------------------
//
static PyObject *python_peak(CrossPeak *xp)
{
  if (xp == NULL)
    return python_none();

  return python_mirror_object("Peak", xp, "crosspeak");
}

// ----------------------------------------------------------------------------
//
static PyObject *peak_spectrum(PyObject *, void *xp)
  { return python_spectrum(((CrossPeak *) xp)->spectrum()); }
static PyObject *peak_color(PyObject *, void *xp)
  { return python_string(((CrossPeak *) xp)->GetColor().name()); }
static PyObject *peak_selected(PyObject *, void *xp)
  { return python_boolean(((CrossPeak *) xp)->IsSelected()); }
static PyObject *peak_position(PyObject *, void *xp)
  { return python_tuple(((CrossPeak *) xp)->position()); }
static PyObject *peak_alias(PyObject *, void *xp)
  { return python_tuple(((CrossPeak *) xp)->alias()); }
static PyObject *peak_frequency(PyObject *, void *xp)
  { return python_tuple(((CrossPeak *) xp)->frequency()); }
static PyObject *peak_is_assigned(PyObject *, void *xp)
  { return python_boolean(((CrossPeak *) xp)->is_fully_assigned()); }
static PyObject *peak_assignment(PyObject *, void *xp)
  { return python_string(((CrossPeak *) xp)->assignment_name()); }
static PyObject *peak_data_height(PyObject *, void *xp)
{
  Peak *pk = (((CrossPeak *) xp)->type() == peak ? (Peak *) xp : NULL);
  return (pk ? python_float(pk->DataHeight()) : python_none());
}
static PyObject *peak_volume(PyObject *, void *xp)
{
  double volume;
  return (((CrossPeak *) xp)->volume(&volume) ?
	  python_float(volume) : python_none());
}
static bool set_peak_volume(void *crosspeak, PyObject *value)
{
  CrossPeak *xp = (CrossPeak *) crosspeak;
  if (xp->type() != peak)
    { set_python_error("can't set peakgroup volume"); return false; }

  if (!PyNumber_Check(value))
    { set_python_error("volume must be a number"); return false; }
  double vol = python_number_as_double(value);

  Peak *pk = (Peak *) xp;
  if (pk->IntegrateByBox())
    pk->box_volume(pk->GetIntegrateMethod(), vol);
  else if (pk->IntegrateByFit())
    pk->fit_volume(pk->GetIntegrateMethod(), vol, pk->fit_residual());
  else
    pk->manual_volume(vol);

  return true;
}
static PyObject *peak_volume_method(PyObject *, void *xp)
{
  Stringy method;
  return python_string(((CrossPeak *) xp)->volume(NULL, &method) ?
		       method : Stringy(""));
}
static bool set_peak_volume_method(void *crosspeak, PyObject *value)
{
  CrossPeak *xp = (CrossPeak *) crosspeak;
  if (xp->type() != peak)
    { set_python_error("can't set peakgroup volume method"); return false; }

  if (!PyString_Check(value))
    { set_python_error("volume method must be a string"); return false; }
  char *method = PyString_AsString(value);
  int mi = name_index(Integration_Method_Names, method);
  if (mi == -1)
    { set_python_error("invalid volume method name"); return false; }
  Integration_Method m = (Integration_Method) mi;

  Peak *pk = (Peak *) xp;

  double vol = 0;
  pk->volume(&vol);

  if (m == INTEGRATE_BOX || m == INTEGRATE_ELLIPSE)
    pk->box_volume(m, vol);
  else if (m == INTEGRATE_GAUSSIAN || m == INTEGRATE_LORENTZIAN)
    pk->fit_volume(m, vol, pk->fit_residual());
  else if (m == INTEGRATE_MANUAL)
    pk->manual_volume(vol);
  else
    pk->no_volume();

  return true;
}
static PyObject *peak_volume_error(PyObject *, void *xp)
{
  double verror;
  return (((CrossPeak *) xp)->volume_error(&verror) ?
	  python_float(verror) : python_none());
}
static PyObject *peak_volume_error_method(PyObject *, void *xp)
{
  Stringy method;
  return python_string(((CrossPeak *) xp)->volume_error(NULL, &method) ?
		       method : Stringy(""));
}
static PyObject *peak_fit_height(PyObject *, void *xp)
{
  Peak *pk = (((CrossPeak *) xp)->type() == peak ? (Peak *) xp : NULL);
  return (pk && pk->IntegrateByFit() ?
	  python_float(pk->FitHeight()) : python_none());
}
static bool set_peak_fit_height(void *crosspeak, PyObject *value)
{
  CrossPeak *xp = (CrossPeak *) crosspeak;
  if (xp->type() != peak)
    { set_python_error("can't set peakgroup fit height"); return false; }

  if (!PyNumber_Check(value))
    { set_python_error("fit height must be a number"); return false; }
  double h = python_number_as_double(value);

  Peak *pk = (Peak *) xp;
  pk->FitHeight(h);

  return true;
}
static PyObject *peak_line_width(PyObject *, void *xp)
{
  Peak *pk = (((CrossPeak *) xp)->type() == peak ? (Peak *) xp : NULL);
  return (pk && pk->HasLinewidth() ?
	  python_tuple(pk->linewidth()) : python_none());
}
static PyObject *peak_line_width_method(PyObject *, void *xp)
{
  Peak *pk = (((CrossPeak *) xp)->type() == peak ? (Peak *) xp : NULL);
  return python_string(pk ? pk->GetLinewidthMethod() : Stringy(""));
}
static PyObject *peak_note(PyObject *, void *xp)
  { return python_string(((CrossPeak *) xp)->GetNote()); }

// ----------------------------------------------------------------------------
//
static bool set_peak_alias(void *crosspeak, PyObject *value)
{
  CrossPeak *xp = (CrossPeak *) crosspeak;

  SPoint alias;
  if (!spoint_argument("peak alias", xp->dimension(), value, &alias))
    return false;

  xp->set_alias(alias);
  return true;
}

// ----------------------------------------------------------------------------
//
static PyObject *peak_label(PyObject *, void *crosspeak)
{
  Label *label = ((CrossPeak *) crosspeak)->label();
  return (label ? python_label(label) : python_none());
}

// ----------------------------------------------------------------------------
//
static bool set_peak_color(void *crosspeak, PyObject *value)
  { return set_ornament_color((CrossPeak *) crosspeak, value); }

// ----------------------------------------------------------------------------
//
static bool set_peak_frequency(void *crosspeak, PyObject *value)
{
  CrossPeak *xp = (CrossPeak *) crosspeak;
  if (xp->type() != peak)
    { set_python_error("can't set peakgroup frequency"); return false; }

  SPoint freq;
  if (!spoint_argument("peak frequency", xp->dimension(), value, &freq))
    return false;

  xp->set_position(freq - xp->alias());
  return true;
}

// ----------------------------------------------------------------------------
//
static bool set_peak_linewidth(void *crosspeak, PyObject *value)
{
  CrossPeak *xp = (CrossPeak *) crosspeak;
  if (xp->type() != peak)
    { set_python_error("can't set peakgroup linewidth"); return false; }

  SPoint lw;
  if (!spoint_argument("peak linewidth", xp->dimension(), value, &lw))
    return false;

  Peak *pk = (Peak *) xp;
  pk->linewidth(pk->GetLinewidthMethod(), lw);
  return true;
}

// ----------------------------------------------------------------------------
//
static bool set_peak_linewidth_method(void *crosspeak, PyObject *value)
{
  CrossPeak *xp = (CrossPeak *) crosspeak;
  if (xp->type() != peak)
    { set_python_error("can't set peakgroup linewidth method"); return false; }

  if (!PyString_Check(value))
    { set_python_error("linewidth method must be a string"); return false; }

  Peak *pk = (Peak *) xp;
  pk->linewidth(PyString_AsString(value), pk->linewidth());
  return true;
}

// ----------------------------------------------------------------------------
//
static bool set_peak_note(void *crosspeak, PyObject *value)
{
  CrossPeak *xp = (CrossPeak *) crosspeak;

  if (!PyString_Check(value))
    { set_python_error("note must be a string"); return false; }

  xp->SetNote(PyString_AsString(value));
  return true;
}

// ----------------------------------------------------------------------------
//
static bool set_peak_position(void *crosspeak, PyObject *value)
{
  CrossPeak *xp = (CrossPeak *) crosspeak;
  if (xp->type() != peak)
    { set_python_error("can't set peakgroup position"); return false; }

  SPoint pos;
  if (!spoint_argument("peak position", xp->dimension(), value, &pos))
    return false;

  xp->set_position(pos);
  return true;
}

// ----------------------------------------------------------------------------
//
static bool set_peak_selected(void *crosspeak, PyObject *value)
  { return set_ornament_selected((CrossPeak *) crosspeak, value); }

// ----------------------------------------------------------------------------
//
static bool set_peak_volume_error(void *crosspeak, PyObject *value)
{
  CrossPeak *xp = (CrossPeak *) crosspeak;

  if (!PyNumber_Check(value))
    { set_python_error("volume error must be a number"); return false; }

  Stringy method;
  xp->volume_error(NULL, &method);
  xp->set_volume_error(python_number_as_double(value), method);
  return true;
}

// ----------------------------------------------------------------------------
//
static bool set_peak_volume_error_method(void *crosspeak, PyObject *value)
{
  CrossPeak *xp = (CrossPeak *) crosspeak;

  if (!PyString_Check(value))
    { set_python_error("volume error method must be a string"); return false; }

  double verror = 0;
  xp->volume_error(&verror, NULL);
  xp->set_volume_error(verror, PyString_AsString(value));
  return true;
}

// ----------------------------------------------------------------------------
//
static PyObject *python_assignment(CrossPeak *xp)
{
  List rlist;
  for (int a = 0 ; a < xp->dimension() ; ++a)
    rlist.append(python_resonance(xp->resonance(a)));
  return python_tuple(rlist);
}

// ----------------------------------------------------------------------------
//
extern "C" PyObject *peak_resonances(PyObject *self, PyObject *args)
{
  if (!PyArg_ParseTuple(args, "")) return NULL;
  CrossPeak *xp = (CrossPeak *) c_object(self, "crosspeak");
  return (xp ? python_assignment(xp) : NULL);
}

// ----------------------------------------------------------------------------
//
extern "C" PyObject *peak_peaklets(PyObject *self, PyObject *args)
{
  CrossPeak *xp = (CrossPeak *) c_object(self, "crosspeak");
  if (xp == NULL || !PyArg_ParseTuple(args, ""))
    return NULL;

  PeakGp *pg = (xp->type() == peakgroup ? (PeakGp *) xp : NULL);
  return python_list((pg ? pg->peaklets() : List()), python_peak_e);
}

// ----------------------------------------------------------------------------
//
extern "C" PyObject *assign_peak(PyObject *self, PyObject *args)
{
  int axis;
  char *gname, *aname;
  if (!PyArg_ParseTuple(args, "iss", &axis, &gname, &aname))
    return NULL;

  CrossPeak *xp = (CrossPeak *) c_object(self, "crosspeak");
  if (xp == NULL)
    return NULL;

  if (axis < 0 || axis >= xp->dimension())
    {
      PyErr_SetString(PyExc_ValueError, "out of range axis value");
      return NULL;
    }

  return python_boolean(xp->ChangeAssignment(axis, aname, gname) != NULL);
}

// ----------------------------------------------------------------------------
//
extern "C" PyObject *show_assignment_label(PyObject *self, PyObject *args)
{
  if (!PyArg_ParseTuple(args, ""))
    return NULL;

  CrossPeak *xp = (CrossPeak *) c_object(self, "crosspeak");
  if (xp == NULL)
    return NULL;

  Label *label = xp->label();
  if (label == NULL)
    label = new Label(xp);
  label->assignment(true);

  return python_label(label);
}

// ----------------------------------------------------------------------------
//
extern "C" PyObject *show_label(PyObject *self, PyObject *args)
{
  char *text;
  if (!PyArg_ParseTuple(args, "s", &text))
    return NULL;

  CrossPeak *xp = (CrossPeak *) c_object(self, "crosspeak");
  if (xp == NULL)
    return NULL;

  Label *label = xp->label();
  if (label == NULL)
    label = new Label(xp);
  label->assignment(false);
  label->SetString(text);

  return python_label(label);
}

// ----------------------------------------------------------------------------
//
extern "C" PyObject *fit_peak(PyObject *self, PyObject *args)
{
  PyObject *py_view;
  if (!PyArg_ParseTuple(args, "O", &py_view))
    return NULL;

  CrossPeak *xp = (CrossPeak *) c_object(self, "crosspeak");
  View *v = (View *) c_object(py_view, "view");
  if (xp == NULL || v == NULL)
    return NULL;

  if (xp->type() != peak)
    {
      PyErr_SetString(PyExc_TypeError, "argument is a peakgroup");
      return NULL;
    }

  List plist;
  plist.append((Peak *) xp);
  Session &s = v->session();
  s.project().MakeUndoIntegrate(plist);
  double pos, neg;
  v->lowest_contour_levels(&pos, &neg);
  integrate_list(v->spectrum(), plist, SRegion(), pos, neg,
		 wait_callbacks(s), s.reporter());

  return (xp->volume() ? python_peak(xp) : python_none());
}

// ----------------------------------------------------------------------------
//
extern "C" PyObject *center_peak(PyObject *self, PyObject *args)
{
  if (!PyArg_ParseTuple(args, ""))
    return NULL;

  CrossPeak *xp = (CrossPeak *) c_object(self, "crosspeak");
  if (xp == NULL)
    return NULL;

  if (xp->type() != peak)
    {
      PyErr_SetString(PyExc_TypeError, "argument is a peakgroup");
      return NULL;
    }

  bool moved = ((Peak *) xp)->MoveToMaximum();

  return python_boolean(moved);
}

// ----------------------------------------------------------------------------
//
static Get_Set_Functions peak_getset[] = {
  {"spectrum",	peak_spectrum,	0,	false},
  {"color",	peak_color,	set_peak_color,	true},
  {"selected",	peak_selected,	set_peak_selected,	true},
  {"position", peak_position,	set_peak_position,	true},
  {"alias", peak_alias,	set_peak_alias,	true},
  {"frequency", peak_frequency,	set_peak_frequency,	true},
  {"is_assigned", peak_is_assigned,	0,	true},
  {"assignment", peak_assignment,	0,	true},
  {"data_height", peak_data_height,	0,	true},
  {"volume", peak_volume,	set_peak_volume,	true},
  {"volume_method", peak_volume_method,	set_peak_volume_method,	true},
  {"volume_error", peak_volume_error,	set_peak_volume_error,	true},
  {"volume_error_method", peak_volume_error_method,	set_peak_volume_error_method,	true},
  {"fit_height", peak_fit_height,	set_peak_fit_height,	true},
  {"line_width", peak_line_width,	set_peak_linewidth,	true},
  {"line_width_method", peak_line_width_method,	set_peak_linewidth_method,	true},
  {"note", peak_note,	set_peak_note,	true},
  {"label", peak_label,			0,	true},
  {NULL},
};

static PyMethodDef peak_methods[] = {
    {"resonances",		peak_resonances,		METH_VARARGS},
    {"peaklets",		peak_peaklets,			METH_VARARGS},
    {"assign",			assign_peak,			METH_VARARGS},
    {"center",			center_peak,			METH_VARARGS},
    {"fit",			fit_peak,			METH_VARARGS},
    {"show_assignment_label",	show_assignment_label,		METH_VARARGS},
    {"show_label",		show_label,			METH_VARARGS},
    {NULL,			NULL}
};

// ----------------------------------------------------------------------------
//
static PyObject *python_ornament(Ornament *op)
{
  if (op == NULL)
    return python_none();

  PyObject *py_orn;
  switch (op->type())
    {
    case peak:
    case peakgroup:
      py_orn = python_peak((CrossPeak *)op); break;
    case label:
      py_orn = python_label((Label *)op); break;
    case grid:
      py_orn = python_grid((Grid *)op); break;
    case line:
      py_orn = python_line((Line *)op); break;
    default:
      py_orn = python_none(); break;
    }

  return py_orn;
}

// ----------------------------------------------------------------------------
//
static bool set_ornament_color(Ornament *op, PyObject *value)
{
  if (!PyString_Check(value))
    { set_python_error("ornament color must be a string"); return false; }

  op->SetColor(PyString_AsString(value));
  return true;
}

// ----------------------------------------------------------------------------
//
static bool set_ornament_selected(Ornament *op, PyObject *value)
{
  if (!PyInt_Check(value))
    { set_python_error("selection state must be an integer"); return false; }

  op->select(PyInt_AsLong(value));
  return true;
}

// ----------------------------------------------------------------------------
//
static PyObject *python_resonance(Resonance *r)
{
  if (r == NULL)
    return python_none();

  return python_mirror_object("Resonance", r, "resonance");
}

// ----------------------------------------------------------------------------
//
static PyObject *resonance_frequency(PyObject *, void *res)
  { return python_float(((Resonance *) res)->frequency()); }
static PyObject *resonance_atom(PyObject *, void *res)
  { return python_atom(((Resonance *) res)->atom()); }
static PyObject *resonance_group(PyObject *, void *res)
  { return python_group(((Resonance *) res)->group()); }
static PyObject *resonance_condition(PyObject *, void *res)
  { return python_condition(((Resonance *) res)->condition()); }
static PyObject *resonance_name(PyObject *, void *res)
  { return python_string(((Resonance *) res)->name()); }
static PyObject *resonance_deviation(PyObject *, void *res)
  { return python_float(((Resonance *) res)->frequencySigma()); }
static PyObject *resonance_peak_count(PyObject *, void *res)
  { return python_int(((Resonance *) res)->crosspeaks().size()); }

// ----------------------------------------------------------------------------
//
extern "C" PyObject *resonance_peaks(PyObject *self, PyObject *args)
{
  if (!PyArg_ParseTuple(args, "")) return NULL;
  Resonance *r = (Resonance *) c_object(self, "resonance");
  return (r ? python_list(r->crosspeaks(), python_crosspeak_e) : NULL);
}

// ----------------------------------------------------------------------------
//
static Get_Set_Functions res_getset[] = {
  {"frequency", resonance_frequency,	0,	true},
  {"atom", resonance_atom,	0,	false},
  {"group", resonance_group,	0,	false},
  {"condition", resonance_condition,	0,	false},
  {"name", resonance_name,	0,	false},
  {"deviation", resonance_deviation,	0,	true},
  {"peak_count", resonance_peak_count,	0,	true},
  {NULL},
};

static PyMethodDef res_methods[] = {
  {"peak_list",			resonance_peaks,		METH_VARARGS},
  {NULL,			NULL}
};

// ----------------------------------------------------------------------------
//
static PyObject *python_label(Label *label)
{
  if (label == NULL)
    return python_none();

  return python_mirror_object("Label", label, "label");
}

// ----------------------------------------------------------------------------
//
static PyObject *label_spectrum(PyObject *, void *label)
  { return python_spectrum(((Label *) label)->spectrum()); }
static PyObject *label_color(PyObject *, void *label)
  { return python_string(((Label *) label)->GetColor().name()); }
static PyObject *label_selected(PyObject *, void *label)
  { return python_boolean(((Label *) label)->IsSelected()); }
static PyObject *label_text(PyObject *, void *label)
  { return python_string(((Label *) label)->GetString()); }
static PyObject *label_shows_assignment(PyObject *, void *label)
  { return python_boolean(((Label *) label)->is_assignment()); }
static PyObject *label_peak(PyObject *, void *label)
  { return python_peak(((Label *) label)->attached()); }

// ----------------------------------------------------------------------------
//
static bool set_label_color(void *label, PyObject *value)
  { return set_ornament_color((Label *) label, value); }
static bool set_label_selected(void *label, PyObject *value)
  { return set_ornament_selected((Label *) label, value); }

// ----------------------------------------------------------------------------
//
static Get_Set_Functions label_getset[] = {
  {"spectrum",	label_spectrum,	0,	false},
  {"color",	label_color,	set_label_color,	true},
  {"selected",	label_selected,	set_label_selected,	true},
  {"text",	label_text,	0,	true},
  {"shows_assignment",	label_shows_assignment,	0,	true},
  {"peak",	label_peak,	0,	false},
  {NULL},
};

// ----------------------------------------------------------------------------
//
static PyObject *python_line(Line *line)
{
  return python_mirror_object("Line", line, "line");
}

// ----------------------------------------------------------------------------
// Make a new line ornament between two points in a spectrum
//
extern "C" PyObject *line_init(PyObject *, PyObject *args)
{
  PyObject *self;
  if (PyArg_ParseTuple(args, "O!", &PyInstance_Type, &self))
    return python_none();
  PyErr_Clear();

  //
  // Maybe __init__(self, spectrum, start, end)
  //
  SPoint p1_ppm, p2_ppm;
  Spectrum *sp;
  if (PyArg_ParseTuple(args, "O!O&O&O&",
		       &PyInstance_Type, &self,
		       spectrum_arg, &sp,
		       spoint_arg, &p1_ppm,
		       spoint_arg, &p2_ppm))
    {
      if (!check_dimension(p1_ppm, sp) || !check_dimension(p2_ppm, sp))
	return NULL;

      Line *line = new Line(sp, p1_ppm, p2_ppm);
      set_sparky_id(self, c_python_map->object_id(line, "line"));
      line->select(true);
      return python_none();
    }

  return NULL;
}

// ----------------------------------------------------------------------------
//
static PyObject *line_spectrum(PyObject *, void *line)
  { return python_spectrum(((Line *) line)->spectrum()); }
static PyObject *line_color(PyObject *, void *line)
  { return python_string(((Line *) line)->GetColor().name()); }
static PyObject *line_selected(PyObject *, void *line)
  { return python_boolean(((Line *) line)->IsSelected()); }
static PyObject *line_start(PyObject *, void *line)
  { return python_tuple(((Line *) line)->bl); }
static PyObject *line_end(PyObject *, void *line)
  { return python_tuple(((Line *) line)->tr); }

// ----------------------------------------------------------------------------
//
static bool set_line_color(void *line, PyObject *value)
  { return set_ornament_color((Line *) line, value); }
static bool set_line_selected(void *line, PyObject *value)
  { return set_ornament_selected((Line *) line, value); }

// ----------------------------------------------------------------------------
//
static Get_Set_Functions line_getset[] = {
  {"spectrum",	line_spectrum,	0,	false},
  {"color",	line_color,	set_line_color,	true},
  {"selected",	line_selected,	set_line_selected,	true},
  {"start",	line_start,	0,	true},
  {"end",	line_end,	0,	true},
  {NULL},
};

// ----------------------------------------------------------------------------
//
static PyObject *python_grid(Grid *grid)
{
  return python_mirror_object("Grid", grid, "grid");
}

// ----------------------------------------------------------------------------
//
static PyObject *grid_spectrum(PyObject *, void *grid)
  { return python_spectrum(((Grid *) grid)->spectrum()); }
static PyObject *grid_color(PyObject *, void *grid)
  { return python_string(((Grid *) grid)->GetColor().name()); }
static PyObject *grid_selected(PyObject *, void *grid)
  { return python_boolean(((Grid *) grid)->IsSelected()); }
static PyObject *grid_position(PyObject *, void *grid)
  { return python_tuple(((Grid *) grid)->location); }
static PyObject *grid_axis(PyObject *, void *grid)
  { return python_int(((Grid *) grid)->axis); }

// ----------------------------------------------------------------------------
//
static bool set_grid_color(void *grid, PyObject *value)
  { return set_ornament_color((Grid *) grid, value); }
static bool set_grid_selected(void *grid, PyObject *value)
  { return set_ornament_selected((Grid *) grid, value); }

// ----------------------------------------------------------------------------
//
static Get_Set_Functions grid_getset[] = {
  {"spectrum",	grid_spectrum,	0,	false},
  {"color",	grid_color,	set_grid_color,	true},
  {"selected",	grid_selected,	set_grid_selected,	true},
  {"position",	grid_position,	0,	true},
  {"axis",	grid_axis,	0,	true},
  {NULL},
};

// ----------------------------------------------------------------------------
//
static PyObject *python_molecule(Molecule *m)
{
  return python_mirror_object("Molecule", m, "molecule");
}

// ----------------------------------------------------------------------------
//
static PyObject *molecule_name(PyObject *, void *mol)
  { return python_string(((Molecule *) mol)->name()); }
static PyObject *molecule_session(PyObject *, void *mol)
  { return python_session(((Molecule *) mol)->session()); }

// ----------------------------------------------------------------------------
//
extern "C" PyObject *molecule_groups(PyObject *self, PyObject *args)
{
  if (!PyArg_ParseTuple(args, "")) return NULL;
  Molecule *m = (Molecule *) c_object(self, "molecule");
  return (m ? python_list(m->GroupList(), python_group_e) : NULL);
}

// ----------------------------------------------------------------------------
//
extern "C" PyObject *molecule_conditions(PyObject *self, PyObject *args)
{
  if (!PyArg_ParseTuple(args, "")) return NULL;
  Molecule *m = (Molecule *) c_object(self, "molecule");
  return (m ? python_list(m->condition_list(), python_condition_e) : NULL);
}

// ----------------------------------------------------------------------------
//
extern "C" PyObject *molecule_atoms(PyObject *self, PyObject *args)
{
  Molecule *m = (Molecule *) c_object(self, "molecule");
  if (m == NULL)
    return NULL;

  if (!PyArg_ParseTuple(args, "")) return NULL;

  return python_list(m->AtomList(), python_atom_e);
}

// ----------------------------------------------------------------------------
//
extern "C" PyObject *molecule_atom(PyObject *self, PyObject *args)
{
  Molecule *m = (Molecule *) c_object(self, "molecule");
  if (m == NULL)
    return NULL;

  char *group_name, *atom_name;
  if (!PyArg_ParseTuple(args, "ss", &group_name, &atom_name))
    return NULL;

  Atom *a = m->find_atom(group_name, atom_name);
  return (a ? python_atom(a) : python_none());
}

// ----------------------------------------------------------------------------
//
extern "C" PyObject *molecule_save_value(PyObject *self, PyObject *args)
{
  char *key, *value;
  if (!PyArg_ParseTuple(args, "ss", &key, &value))
    return NULL;
  Molecule *m = (Molecule *) c_object(self, "molecule");
  if (m == NULL)
    return NULL;
  return python_boolean(m->saved_values().save_value(key, value));
}

// ----------------------------------------------------------------------------
//
extern "C" PyObject *molecule_saved_value(PyObject *self, PyObject *args)
{
  Molecule *m = (Molecule *) c_object(self, "molecule");
  if (m == NULL)
    return NULL;

  char *key;
  if (!PyArg_ParseTuple(args, "s", &key))
    return NULL;

  Stringy value;
  bool got_it = m->saved_values().saved_value(key, &value);
  return (got_it ? python_string(value) : python_none());
}

// ----------------------------------------------------------------------------
//
static Get_Set_Functions mol_getset[] = {
  {"name",	molecule_name,		0,	false},
  {"session",	molecule_session,	0,	false},
  {NULL},
};

static PyMethodDef mol_methods[] = {
  {"group_list",		molecule_groups,		METH_VARARGS},
  {"atom_list",			molecule_atoms,			METH_VARARGS},
  {"atom",			molecule_atom,			METH_VARARGS},
  {"condition_list",		molecule_conditions,		METH_VARARGS},
  {"save_value",		molecule_save_value,		METH_VARARGS},
  {"saved_value",		molecule_saved_value,		METH_VARARGS},
  {NULL,			NULL}
};

// ----------------------------------------------------------------------------
//
static PyObject *python_group(Group *g)
{
  return python_mirror_object("Group", g, "group");
}

// ----------------------------------------------------------------------------
//
static PyObject *group_name(PyObject *, void *group)
  { return python_string(((Group *) group)->name()); }
static PyObject *group_symbol(PyObject *, void *group)
  { return python_string(((Group *) group)->symbol()); }
static PyObject *group_number(PyObject *, void *group)
{
  Group *g = (Group *) group;
  return (g->number() == 0 ? python_none() : python_int(g->number()));
}
static PyObject *group_suffix(PyObject *, void *group)
  { return python_string(((Group *) group)->suffix()); }

// ----------------------------------------------------------------------------
//
static Get_Set_Functions group_getset[] = {
  {"name", group_name,	0,	false},
  {"symbol", group_symbol,	0,	false},
  {"number", group_number,	0,	false},
  {"suffix", group_suffix,	0,	false},
  {NULL},
};

// ----------------------------------------------------------------------------
//
static PyObject *python_atom(Atom *a)
{
  return python_mirror_object("Atom", a, "atom");
}

// ----------------------------------------------------------------------------
//
static PyObject *atom_name(PyObject *, void *atom)
  { return python_string(((Atom *) atom)->name()); }
static PyObject *atom_nucleus(PyObject *, void *atom)
  { return python_string(((Atom *) atom)->nucleus()); }
static PyObject *atom_group(PyObject *, void *atom)
  { return python_group(((Atom *) atom)->group()); }

// ----------------------------------------------------------------------------
//
static Get_Set_Functions atom_getset[] = {
  {"name", atom_name,	0,	false},
  {"nucleus", atom_nucleus,	0,	false},
  {"group", atom_group,	0,	false},
  {NULL},
};

// ----------------------------------------------------------------------------
//
static PyObject *python_condition(Condition *c)
{
  return python_mirror_object("Condition", c, "condition");
}

// ----------------------------------------------------------------------------
//
static PyObject *condition_name(PyObject *, void *cond)
  { return python_string(((Condition *) cond)->name()); }
static PyObject *condition_molecule(PyObject *, void *cond)
  { return python_molecule(((Condition *) cond)->molecule()); }

// ----------------------------------------------------------------------------
//
extern "C" PyObject *condition_resonances(PyObject *self, PyObject *args)
{
  Condition *c = (Condition *) c_object(self, "condition");
  if (c && PyArg_ParseTuple(args, ""))
    {
      List rlist = c->resonance_list();
      rlist.sort(compare_resonance_frequencies);
      return python_list(rlist, python_resonance_e);
    }
  return NULL;
}

// ----------------------------------------------------------------------------
//
extern "C" PyObject *interval_resonances(PyObject *self, PyObject *args)
{
  Condition *c = (Condition *) c_object(self, "condition");
  double ppm_min, ppm_max, alias_min, alias_max;
  char *nucleus;
  if (c && PyArg_ParseTuple(args, "ddsdd",
			    &ppm_min, &ppm_max, &nucleus,
			    &alias_min, &alias_max))
    {
      List rlist = c->interval_resonances(ppm_min, ppm_max, nucleus,
					  alias_min, alias_max);
      return python_list(rlist, python_resonance_e);
    }
  return NULL;
}

// ----------------------------------------------------------------------------
//
extern "C" PyObject *find_resonance(PyObject *self, PyObject *args)
{
  Condition *c = (Condition *) c_object(self, "condition");
  if (c == NULL)
    return NULL;

  char *group_name, *atom_name;
  Atom *a;
  Resonance *r;
  if (PyArg_ParseTuple(args, "ss", &group_name, &atom_name))
    r = c->find_resonance(group_name, atom_name);
  else if ((PyErr_Clear(), PyArg_ParseTuple(args, "O&", atom_arg, &a)))
    r = c->find_resonance(*a);
  else
    return NULL;

  return (r ? python_resonance(r) : python_none());
}

// ----------------------------------------------------------------------------
//
static Get_Set_Functions cond_getset[] = {
  {"name", condition_name,	0,	false},
  {"molecule", condition_molecule,	0,	false},
  {NULL},
};

static PyMethodDef cond_methods[] = {
  {"resonance_list",		condition_resonances,		METH_VARARGS},
  {"interval_resonances",	interval_resonances,		METH_VARARGS},
  {"find_resonance",		find_resonance,			METH_VARARGS},
  {NULL,			NULL}
};

// ----------------------------------------------------------------------------
//
static PyObject *python_view(View *v)
{
  return python_mirror_object("View", v, "view");
}

// ----------------------------------------------------------------------------
//
static PyObject *view_spectrum(PyObject *, void *view)
  { return python_spectrum(((View *) view)->spectrum()); }
static PyObject *view_session(PyObject *, void *view)
  { return python_session(((View *) view)->session()); }
static PyObject *view_name(PyObject *, void *view)
  { return python_string(((View *) view)->name()); }
static PyObject *view_frame(PyObject *, void *view)
{
  View *v = (View *) view;
  Stringy name = v->session().window_system().widget_name(v->frame());
  return python_string(name);
}
static PyObject *view_is_shown(PyObject *, void *view)
  { return python_boolean(((View *) view)->settings().show_view); }
static PyObject *view_is_top_level_window(PyObject *, void *view)
  { return python_boolean(((View *) view)->is_top_level_window()); }
static PyObject *view_center(PyObject *, void *view)
  { return python_tuple(((View *) view)->settings().center); }
static PyObject *view_pixel_size(PyObject *, void *view)
  { return python_tuple(((View *) view)->settings().pixel_size); }
static PyObject *view_region(PyObject *, void *view)
  { return python_tuple(((View *) view)->view_region()); }
static PyObject *view_visible_depth(PyObject *, void *view)
  { return python_tuple(((View *) view)->settings().visible_depth); }
static PyObject *view_axis_order(PyObject *, void *view)
  { return python_tuple(((View *) view)->settings().axis_order); }
static PyObject *view_positive_levels(PyObject *, void *view)
  { return python_contour_levels(((View *) view)->positive_contours()); }
static PyObject *view_negative_levels(PyObject *, void *view)
  { return python_contour_levels(((View *) view)->negative_contours()); }
static PyObject *view_scale_units(PyObject *, void *view)
{
  int units = ((View *) view)->settings().scale_units;
  return python_string(index_name(Unit_Names, units));
}
static PyObject *view_show_ornaments(PyObject *, void *view)
  { return python_boolean(((View *) view)->settings().show_ornaments); }
static PyObject *view_show_peaks(PyObject *, void *view)
  { return python_boolean(((View *) view)->settings().show_peaks); }
static PyObject *view_show_peakgroups(PyObject *, void *view)
  { return python_boolean(((View *) view)->settings().show_peakgroups); }
static PyObject *view_show_labels(PyObject *, void *view)
  { return python_boolean(((View *) view)->settings().show_labels); }
static PyObject *view_show_lines(PyObject *, void *view)
  { return python_boolean(((View *) view)->settings().show_lines); }
static PyObject *view_show_grids(PyObject *, void *view)
  { return python_boolean(((View *) view)->settings().show_grids); }
static PyObject *view_subtract_fit_peaks(PyObject *, void *view)
  { return python_boolean(((View *) view)->settings().subtract_fit_peaks); }
static PyObject *view_show_scales(PyObject *, void *view)
  { return python_boolean(((View *) view)->settings().show_scales); }
static PyObject *view_show_nucleus_names(PyObject *, void *view)
  { return python_boolean(((View *) view)->settings().show_nucleus_names); }
static PyObject *view_show_scrollbars(PyObject *, void *view)
  { return python_boolean(((View *) view)->settings().show_scrollbars); }
static PyObject *view_show_resonance_panels(PyObject *, void *view)
  { return python_boolean(((View *) view)->settings().show_resonance_panels); }
static PyObject *view_filter_resonances(PyObject *, void *view)
  { return python_boolean(((View *) view)->settings().filter_resonances); }
static PyObject *view_show_slices(PyObject *, void *view)
  { return python_boolean(((View *) view)->settings().show_slices); }
static PyObject *view_slice_auto_scale(PyObject *, void *view)
  { return python_boolean(((View *) view)->settings().slice_auto_scale); }
static PyObject *view_slice_subtract_peaks(PyObject *, void *view)
  { return python_boolean(((View *) view)->settings().slice_subtract_peaks); }
static PyObject *view_show_peak_info(PyObject *, void *view)
  { return python_boolean(((View *) view)->settings().show_peak_info); }
static PyObject *view_show_contour_scale(PyObject *, void *view)
  { return python_boolean(((View *) view)->settings().show_contour_scale); }
static PyObject *view_show_crosshair(PyObject *, void *view)
  { return python_boolean(((View *) view)->settings().show_crosshair); }
static PyObject *view_show_crosshair_from_other_views(PyObject *, void *view)
  { return python_boolean(((View *) view)->settings().show_crosshair_from_other_views); }
static PyObject *view_show_transpose_crosshair(PyObject *, void *view)
  { return python_boolean(((View *) view)->settings().show_transpose_crosshair); }
static PyObject *view_pointer_position(PyObject *, void *view)
{
  SPoint p;
  return (((View *) view)->pointer_position(&p) ?
	  python_tuple(p) : python_none());
}

// ----------------------------------------------------------------------------
//
static bool set_view_axis_order(void *view, PyObject *value)
{
  View *v = (View *) view;
  Spectrum *sp = v->spectrum();

  IPoint axis_order;
  if (!permutation_argument("axis order", sp->dimension(), value, &axis_order))
    return false;

  v->set_axis_order(axis_order);
  return true;
}

// ----------------------------------------------------------------------------
//
static bool set_view_center(void *view, PyObject *value)
{
  View *v = (View *) view;
  Spectrum *sp = v->spectrum();

  SPoint center;
  if (!spoint_argument("view center", sp->dimension(), value, &center))
    return false;

  v->View_Window::set_view(center, v->pixel_size());
  return true;
}

// ----------------------------------------------------------------------------
//
static bool set_view_pixel_size(void *view, PyObject *value)
{
  View *v = (View *) view;
  Spectrum *sp = v->spectrum();

  SPoint pixel_size;
  if (!spoint_argument("view pixel size", sp->dimension(), value, &pixel_size))
    return false;

  v->View_Window::set_view(v->center(), pixel_size);
  return true;
}

// ----------------------------------------------------------------------------
//
static bool set_view_visible_depth(void *view, PyObject *value)
{
  View *v = (View *) view;
  Spectrum *sp = v->spectrum();

  SPoint depth;
  if (!spoint_argument("view visible depth", sp->dimension(), value, &depth))
    return false;

  v->set_visible_depth(depth);
  return true;
}

// ----------------------------------------------------------------------------
//
static bool set_view_contour_levels(void *view, PyObject *value)
{
  View *v = (View *) view;

  Contour_Parameters cp;
  if (!levels_arg(value, &cp))
    return false;

  v->set_contour_parameters(cp);
  return true;
}

// ----------------------------------------------------------------------------
//
static bool set_view_is_shown(void *view, PyObject *value)
  { return set_view_switch(view, &View_Settings::show_view, value); }
static bool set_view_show_ornaments(void *view, PyObject *value)
  { return set_view_switch(view, &View_Settings::show_ornaments, value); }
static bool set_view_show_peaks(void *view, PyObject *value)
  { return set_view_switch(view, &View_Settings::show_peaks, value); }
static bool set_view_show_peakgroups(void *view, PyObject *value)
  { return set_view_switch(view, &View_Settings::show_peakgroups, value); }
static bool set_view_show_labels(void *view, PyObject *value)
  { return set_view_switch(view, &View_Settings::show_labels, value); }
static bool set_view_show_lines(void *view, PyObject *value)
  { return set_view_switch(view, &View_Settings::show_lines, value); }
static bool set_view_show_grids(void *view, PyObject *value)
  { return set_view_switch(view, &View_Settings::show_grids, value); }
static bool set_view_subtract_fit_peaks(void *view, PyObject *value)
  { return set_view_switch(view, &View_Settings::subtract_fit_peaks, value); }
static bool set_view_show_scales(void *view, PyObject *value)
  { return set_view_switch(view, &View_Settings::show_scales, value); }
static bool set_view_show_nucleus_names(void *view, PyObject *value)
  { return set_view_switch(view, &View_Settings::show_nucleus_names, value); }
static bool set_view_show_scrollbars(void *view, PyObject *value)
  { return set_view_switch(view, &View_Settings::show_scrollbars, value); }
static bool set_view_show_resonance_panels(void *view, PyObject *value)
  { return set_view_switch(view, &View_Settings::show_resonance_panels, value); }
static bool set_view_filter_resonances(void *view, PyObject *value)
  { return set_view_switch(view, &View_Settings::filter_resonances, value); }
static bool set_view_show_slices(void *view, PyObject *value)
  { return set_view_switch(view, &View_Settings::show_slices, value); }
static bool set_view_slice_auto_scale(void *view, PyObject *value)
  { return set_view_switch(view, &View_Settings::slice_auto_scale, value); }
static bool set_view_slice_subtract_peaks(void *view, PyObject *value)
  { return set_view_switch(view, &View_Settings::slice_subtract_peaks, value); }
static bool set_view_show_peak_info(void *view, PyObject *value)
  { return set_view_switch(view, &View_Settings::show_peak_info, value); }
static bool set_view_show_contour_scale(void *view, PyObject *value)
  { return set_view_switch(view, &View_Settings::show_contour_scale, value); }
static bool set_view_show_crosshair(void *view, PyObject *value)
  { return set_view_switch(view, &View_Settings::show_crosshair, value); }
static bool set_view_show_crosshair_from_other_views(void *view,
						     PyObject *value)
  { return set_view_switch(view,
			   &View_Settings::show_crosshair_from_other_views,
			   value); }
static bool set_view_show_transpose_crosshair(void *view, PyObject *value)
  { return set_view_switch(view, &View_Settings::show_transpose_crosshair,
			   value); }

// ----------------------------------------------------------------------------
//
static bool set_view_switch(void *view, bool View_Settings::*member,
			    PyObject *value)
{
  View *v = (View *) view;

  if (!PyInt_Check(value))
    { set_python_error("view switch value must be an integer"); return false; }

  bool onoff = PyInt_AsLong(value);
  View_Settings s = v->settings();
  s.*member = onoff;
  v->configure(s);
  return true;
}

// ----------------------------------------------------------------------------
//
extern "C" PyObject *destroy_view(PyObject *self, PyObject *args)
{
  View *v = (View *) c_object(self, "view");
  if (v == NULL)
    return NULL;

  if (!PyArg_ParseTuple(args, ""))
    return NULL;

  delete v;

  return python_none();
}

// ----------------------------------------------------------------------------
//
extern "C" PyObject *set_crosshair_position(PyObject *self, PyObject *args)
{
  View *v = (View *) c_object(self, "view");
  if (v == NULL)
    return NULL;

  SPoint pos;
  if (!PyArg_ParseTuple(args, "O&", spoint_arg, &pos))
    return NULL;

  if (!check_dimension(pos, v->spectrum()))
    return NULL;

  v->set_crosshair_position(pos);

  return python_none();
}

// ----------------------------------------------------------------------------
//
static Get_Set_Functions view_getset[] = {
  {"spectrum", view_spectrum,	0,	false},
  {"session", view_session,	0,	false},
  {"name", view_name,	0,	true},
  {"frame", view_frame,	0,	false},
  {"is_shown", view_is_shown,	set_view_is_shown,	true},
  {"is_top_level_window", view_is_top_level_window,	0,	false},
  {"center", view_center,	set_view_center,	true},
  {"pixel_size", view_pixel_size,	set_view_pixel_size,	true},
  {"region", view_region,	0,	true},
  {"visible_depth", view_visible_depth,	set_view_visible_depth,	true},
  {"axis_order", view_axis_order,	set_view_axis_order,	true},
  {"positive_levels", view_positive_levels,	set_view_contour_levels,true},
  {"negative_levels", view_negative_levels,	set_view_contour_levels,true},
  {"scale_units", view_scale_units,	0,	true},
  {"show_ornaments", view_show_ornaments,	set_view_show_ornaments,true},
  {"show_peaks", view_show_peaks,	set_view_show_peaks,	true},
  {"show_peakgroups", view_show_peakgroups, set_view_show_peakgroups, true},
  {"show_labels", view_show_labels,	set_view_show_labels,	true},
  {"show_lines", view_show_lines,	set_view_show_lines,	true},
  {"show_grids", view_show_grids,	set_view_show_grids,	true},
  {"subtract_fit_peaks", view_subtract_fit_peaks,	set_view_subtract_fit_peaks,	true},
  {"show_scales", view_show_scales,	set_view_show_scales,	true},
  {"show_nucleus_names", view_show_nucleus_names,	set_view_show_nucleus_names,	true},
  {"show_scrollbars", view_show_scrollbars, set_view_show_scrollbars, true},
  {"show_resonance_panels", view_show_resonance_panels,	set_view_show_resonance_panels,	true},
  {"filter_resonances", view_filter_resonances,	set_view_filter_resonances,	true},
  {"show_slices", view_show_slices,	set_view_show_slices,	true},
  {"slice_auto_scale", view_slice_auto_scale,	set_view_slice_auto_scale,	true},
  {"slice_subtract_peaks", view_slice_subtract_peaks,	set_view_slice_subtract_peaks,	true},
  {"show_peak_info", view_show_peak_info, set_view_show_peak_info, true},
  {"show_contour_scale", view_show_contour_scale,	set_view_show_contour_scale,	true},
  {"show_crosshair", view_show_crosshair, set_view_show_crosshair, true},
  {"show_crosshair_from_other_views", view_show_crosshair_from_other_views,	set_view_show_crosshair_from_other_views,	true},
  {"show_transpose_crosshair", view_show_transpose_crosshair,	set_view_show_transpose_crosshair,	true},
  {"pointer_position", view_pointer_position,	0,	true},
  {NULL},
};

static PyMethodDef view_methods[] = {
  {"destroy",			destroy_view,			METH_VARARGS},
  {"set_crosshair_position",	set_crosshair_position,		METH_VARARGS},
  {NULL,			NULL}
};

// ----------------------------------------------------------------------------
//
static PyObject *python_contour_levels(const Contour_Parameters &cp)
{
  PyObject *py_levels = python_class_instance("Contour_Levels");
  set_contour_level_attributes(py_levels, cp);
  return py_levels;
}

// ----------------------------------------------------------------------------
//
extern "C" PyObject *levels_init(PyObject *, PyObject *args)
{
  PyObject *self;
  if (!PyArg_ParseTuple(args, "O!", &PyInstance_Type, &self))
    return NULL;

  set_contour_level_attributes(self, Contour_Parameters(true));

  return python_none();
}

// ----------------------------------------------------------------------------
//
static void set_contour_level_attributes(PyObject *levels,
					 const Contour_Parameters &cp)
{
  python_set_attribute(levels, "lowest", cp.levels.lowest);
  python_set_attribute(levels, "factor", cp.levels.factor);
  python_set_attribute(levels, "levels", cp.levels.levels);
  python_set_attribute(levels, "color", cp.color_scheme());
}

// ----------------------------------------------------------------------------
//
static PyObject *python_callback(Python_Callback *pycb)
{ 
  return python_mirror_object("Python_Callback", pycb, "python_callback");
}

// ----------------------------------------------------------------------------
//
static PyObject *python_spectrum_e(void *sp)
  { return python_spectrum((Spectrum *)sp); }
static PyObject *python_peak_e(void *pk)
  { return python_peak((Peak *)pk); }
static PyObject *python_crosspeak_e(void *xp)
  { return python_peak((CrossPeak *)xp); }
static PyObject *python_ornament_e(void *op)
  { return python_ornament((Ornament *)op); }
static PyObject *python_resonance_e(void *r)
  { return python_resonance((Resonance *)r); }
static PyObject *python_label_e(void *label)
  { return python_label((Label *)label); }
static PyObject *python_line_e(void *line)
  { return python_line((Line *)line); }
static PyObject *python_grid_e(void *grid)
  { return python_grid((Grid *)grid); }
static PyObject *python_group_e(void *g)
  { return python_group((Group *)g); }
static PyObject *python_atom_e(void *a)
  { return python_atom((Atom *)a); }
static PyObject *python_molecule_e(void *m)
  { return python_molecule((Molecule *)m); }
static PyObject *python_condition_e(void *c)
  { return python_condition((Condition *)c); }
static PyObject *python_view_e(void *v)
  { return python_view((View *)v); }

// ----------------------------------------------------------------------------
//
static PyObject *python_list(const List &c_objects,
			     PyObject *(*py_object)(void *))
{
  List py_objects;

  for (int oi = 0 ; oi < c_objects.size() ; ++oi)
    py_objects.append(py_object(c_objects[oi]));

  return python_list(py_objects);
}

// ----------------------------------------------------------------------------
//
static bool python_get_attribute(PyObject *inst, const Stringy &attr,
				 PyObject **value)
{
  char *cattr = (char *) attr.cstring();
  if (!PyInstance_Check(inst) || !PyObject_HasAttrString(inst, cattr))
    return false;

  *value = PyObject_GetAttrString(inst, cattr);
  return true;
}

// ----------------------------------------------------------------------------
//
static bool python_get_attribute(PyObject *inst, const Stringy &attr,
				 int *value)
{
  PyObject *py_value;
  if (!python_get_attribute(inst, attr, &py_value))
    return false;

  bool valid = PyInt_Check(py_value);
  if (valid)
    *value = (int) PyInt_AsLong(py_value);
  Py_XDECREF(py_value);

  return valid;
}

// ----------------------------------------------------------------------------
//
static bool python_get_attribute(PyObject *inst, const Stringy &attr,
				 double *value)
{
  PyObject *py_value;
  if (!python_get_attribute(inst, attr, &py_value))
    return false;

  bool valid = PyNumber_Check(py_value);
  if (valid)
    *value = python_number_as_double(py_value);
  Py_XDECREF(py_value);

  return valid;
}

// ----------------------------------------------------------------------------
//
static bool python_get_attribute(PyObject *inst, const Stringy &attr,
				 Stringy *value)
{
  PyObject *py_value;
  if (!python_get_attribute(inst, attr, &py_value))
    return false;

  bool valid = PyString_Check(py_value);
  if (valid)
    *value = PyString_AsString(py_value);
  Py_XDECREF(py_value);

  return valid;
}

// ----------------------------------------------------------------------------
//
static void python_set_attribute(PyObject *instance, const Stringy &attribute,
				 const Stringy &value)
  { python_set_attribute(instance, attribute, python_string(value)); }
static void python_set_attribute(PyObject *instance, const Stringy &attribute,
				 int value)
  { python_set_attribute(instance, attribute, python_int(value)); }
static void python_set_attribute(PyObject *instance, const Stringy &attribute,
				 double value)
  { python_set_attribute(instance, attribute, python_float(value)); }

// ----------------------------------------------------------------------------
// Caller passes on ownership of value.
//
static void python_set_attribute(PyObject *instance, const Stringy &attribute,
				 PyObject *value)
{
  PyObject_SetAttrString(instance, (char *) attribute.cstring(), value);

  Py_XDECREF(value);	// PyObject_SetAttribute() increments reference count.
}

// ----------------------------------------------------------------------------
//
static PyObject *instance_dictionary_get(PyObject *inst, const Stringy &attr)
{
  PyObject *dict = PyObject_GetAttrString(inst, "__dict__");
  PyObject *attrib = PyString_FromString((char *) attr.cstring());

  PyObject *value = PyDict_GetItem(dict, attrib);
  Py_XINCREF(value);   // PyDict_GetItem() doesn't increment reference count.

  Py_XDECREF(dict);
  Py_XDECREF(attrib);	

  return value;
}

// ----------------------------------------------------------------------------
// Reference count of value is incremented.
//
static void instance_dictionary_set(PyObject *instance, PyObject *attr,
				    PyObject *value)
{
  PyObject *dict = PyObject_GetAttrString(instance, "__dict__");
  PyDict_SetItem(dict, attr, value);
  Py_XDECREF(dict);
}

// ----------------------------------------------------------------------------
// Return a python value passing ownership to caller.
//
static PyObject *python_none()
  { Py_XINCREF(Py_None);
    return Py_None; }
static PyObject *python_boolean(bool value)
  { return PyInt_FromLong((value ? 1 : 0)); }
static PyObject *python_int(int value)
  { return PyInt_FromLong(value); }
static PyObject *python_float(double value)
  { return PyFloat_FromDouble(value); }
static PyObject *python_string(const Stringy &string)
  { return PyString_FromString((char *) string.cstring()); }

// ----------------------------------------------------------------------------
// Caller passes ownership of the python objects.
//
static PyObject *python_list(const List &py_objects)
{
  PyObject *py_list = PyList_New(py_objects.size());

  for (int k = 0 ; k < py_objects.size() ; ++k)
    PyList_SetItem(py_list, k, (PyObject *) py_objects[k]);

  return py_list;
}

// ----------------------------------------------------------------------------
//
static PyObject *python_tuple(const IPoint &value)
{
  PyObject *py_value = PyTuple_New(value.dimension());

  for (int a = 0 ; a < value.dimension() ; ++a)
    PyTuple_SetItem(py_value, a, python_int(value[a]));

  return py_value;
}

// ----------------------------------------------------------------------------
//
static PyObject *python_tuple(const SPoint &value)
{
  PyObject *py_tuple = PyTuple_New(value.dimension());

  for (int a = 0 ; a < value.dimension() ; ++a)
    PyTuple_SetItem(py_tuple, a, python_float(value[a]));

  return py_tuple;
}

// ----------------------------------------------------------------------------
//
static PyObject *python_tuple(const SRegion &value)
{
  PyObject *py_tuple = PyTuple_New(2);

  PyTuple_SetItem(py_tuple, 0, python_tuple(value.min));
  PyTuple_SetItem(py_tuple, 1, python_tuple(value.max));

  return py_tuple;
}

// ----------------------------------------------------------------------------
// Caller passes ownership of list elements.
//
static PyObject *python_tuple(const List &py_objects)
{
  PyObject *py_tuple = PyTuple_New(py_objects.size());

  for (int a = 0 ; a < py_objects.size() ; ++a)
    PyTuple_SetItem(py_tuple, a, (PyObject *) py_objects[a]);

  return py_tuple;
}

// ----------------------------------------------------------------------------
// Converter function for use with PyArg_ParseTuple() format O&
//
extern "C" int bool_arg(PyObject *value, void *b)
{
  bool t = PyObject_IsTrue(value);

  *(bool *)b = t;

  return 1;
}

// ----------------------------------------------------------------------------
// Converter function for use with PyArg_ParseTuple() format O&
//
extern "C" int string_list_arg(PyObject *list, void *slist)
{
  if (!PySequence_Check(list))
    {
      PyErr_SetString(PyExc_TypeError, "argument is not a sequence");
      return 0;
    }

  List strings;
  int size = PySequence_Length(list);
  for (int k = 0 ; k < size ; ++k)
    {
      PyObject *string = python_sequence_item(list, k);
      if (!PyString_Check(string))
	{
	  PyErr_SetString(PyExc_TypeError, "list component is not a string");
	  return 0;
	}
      strings.append(new Stringy(PyString_AsString(string)));
    }

  *(List *)slist = strings;

  return 1;
}

// ----------------------------------------------------------------------------
// Converter function for use with PyArg_ParseTuple() format O&
//
extern "C" int callable_arg(PyObject *func, void *func_p)
{
  if (!PyCallable_Check(func))
    {
      PyErr_SetString(PyExc_TypeError, "argument is not a function");
      return 0;
    }

  *(PyObject **) func_p = func;
  return 1;
}

// --------------------------------------------------------------------------
// Gross hack below -- parental guidance suggested.
// Python provides public access to file descriptors but not FILE pointers.
// So I ripped this typedef from Python distribution fileobject.c.
//
struct PythonFileStruct
{
  PyObject_HEAD
  FILE *fp;
};

// ----------------------------------------------------------------------------
// Converter function for use with PyArg_ParseTuple() format O&
//
extern "C" int file_arg(PyObject *file, void *fp)
{
  if (!PyFile_Check(file))
    {
      PyErr_SetString(PyExc_TypeError, "argument is not a file");
      return 0;
    }

  *(FILE **) fp = ((PythonFileStruct *) file)->fp;
  return 1;
}

// ----------------------------------------------------------------------------
// Converter function for use with PyArg_ParseTuple() format O&
//
extern "C" int ipoint_arg(PyObject *seq, void *ipoint)
{
  if (!PySequence_Check(seq))
    {
      PyErr_SetString(PyExc_TypeError, "argument is not a point");
      return 0;
    }

  int size = PySequence_Length(seq);
  if (size > DIM)
    {
      Stringy msg = formatted_string("point dimension > %d", DIM);
      PyErr_SetString(PyExc_ValueError, (char *) msg.cstring());
      return 0;
    }

  IPoint p(size);
  for (int a = 0 ; a < size ; ++a)
    {
      PyObject *c = python_sequence_item(seq, a);
      if (PyInt_Check(c))
	p[a] = PyInt_AsLong(c);
      else
	{
	  PyErr_SetString(PyExc_TypeError, "point component is not an int");
	  return 0;
	}
    }

  *(IPoint *)ipoint = p;

  return 1;
}

// ----------------------------------------------------------------------------
//
static bool permutation_argument(const Stringy &name, int dim,
				 PyObject *arg, IPoint *p)
{
  if (ipoint_arg(arg, p) && p->dimension() == dim && p->is_permutation())
    return true;
  set_python_error("bad " + name);
  return false;
}

// ----------------------------------------------------------------------------
// Converter function for use with PyArg_ParseTuple() format O&
//
extern "C" int spoint_arg(PyObject *seq, void *spoint)
{
  if (!PySequence_Check(seq))
    {
      PyErr_SetString(PyExc_TypeError, "argument is not a point");
      return 0;
    }

  int size = PySequence_Length(seq);
  if (size > DIM)
    {
      Stringy msg = formatted_string("point dimension > %d", DIM);
      PyErr_SetString(PyExc_ValueError, (char *) msg.cstring());
      return 0;
    }

  SPoint p(size);
  for (int a = 0 ; a < size ; ++a)
    {
      PyObject *c = python_sequence_item(seq, a);
      if (PyNumber_Check(c))
	p[a] = python_number_as_double(c);
      else
	{
	  PyErr_SetString(PyExc_TypeError,
			  "point component is not float or int");
	  return 0;
	}
    }

  *(SPoint *)spoint = p;

  return 1;
}

// ----------------------------------------------------------------------------
//
static bool spoint_argument(const Stringy &name, int dim,
			    PyObject *arg, SPoint *p)
{
  if (spoint_arg(arg, p) && p->dimension() == dim)
    return true;
  set_python_error("bad " + name);
  return false;
}

// ----------------------------------------------------------------------------
//
static bool check_dimension(const SPoint &p, int dim, const Stringy &error)
{
  bool same_dimension = (p.dimension() == dim);
  if (!same_dimension)
    PyErr_SetString(PyExc_ValueError, (char *) error.cstring());
  return same_dimension;
}

// ----------------------------------------------------------------------------
//
static bool check_dimension(const SPoint &p, Spectrum *sp)
{
  bool same_dimension = (p.dimension() == sp->dimension());
  if (!same_dimension)
    PyErr_SetString(PyExc_ValueError,
		    "spectrum dimension doesn't match region dimension");
  return same_dimension;
}

// ----------------------------------------------------------------------------
//
static bool check_dimension(const SRegion &r, Spectrum *sp)
{
  bool same_dimension = (r.dimension() == sp->dimension());
  if (!same_dimension)
    PyErr_SetString(PyExc_ValueError,
		    "spectrum dimension doesn't match region dimension");
  return same_dimension;
}

// ----------------------------------------------------------------------------
//
static double python_number_as_double(PyObject *num)
{
  if (PyFloat_Check(num))
    return PyFloat_AsDouble(num);
  else if (PyInt_Check(num))
    return (double) PyInt_AsLong(num);
  else if (PyLong_Check(num))
    return PyLong_AsDouble(num);
  return 0;
}

// ----------------------------------------------------------------------------
// Converter function for use with PyArg_ParseTuple() format O&
//
extern "C" int sregion_arg(PyObject *seq, void *sregion)
{
  SRegion r;
  if (!PySequence_Check(seq) || PySequence_Length(seq) != 2 ||
      !spoint_arg(python_sequence_item(seq, 0), &r.min) ||
      !spoint_arg(python_sequence_item(seq, 1), &r.max) ||
      r.min.dimension() != r.max.dimension())
    {
      PyErr_SetString(PyExc_TypeError, "illegal region argument");
      return 0;
    }

  *(SRegion *)sregion = r;

  return 1;
}

// ----------------------------------------------------------------------------
// Converter function for use with PyArg_ParseTuple() format O&
//
extern "C" int spectrum_arg(PyObject *py_spectrum, void *spectrum)
{
  Spectrum *sp = (Spectrum *) c_object(py_spectrum, "spectrum");
  if (sp == NULL)
    return 0;

  *(Spectrum **) spectrum = sp;

  return 1;
}

// ----------------------------------------------------------------------------
// Converter function for use with PyArg_ParseTuple() format O&
//
extern "C" int view_arg(PyObject *py_view, void *view)
{
  View *v = (View *) c_object(py_view, "view");
  if (v == NULL)
    return 0;

  *(View **) view = v;

  return 1;
}

// ----------------------------------------------------------------------------
// Converter function for use with PyArg_ParseTuple() format O&
//
extern "C" int levels_arg(PyObject *py_levels, void *levels)
{
  Contour_Parameters cp;
  Stringy color_name;

  if (!PyInstance_Check(py_levels) ||
      !python_get_attribute(py_levels, "lowest", &cp.levels.lowest) ||
      !python_get_attribute(py_levels, "factor", &cp.levels.factor) ||
      !python_get_attribute(py_levels, "levels", &cp.levels.levels) ||
      !python_get_attribute(py_levels, "color", &color_name))
    {
      PyErr_SetString(PyExc_TypeError, "bad Contour_Levels instance");
      return 0;
    }

  cp.set_color_scheme(color_name);

  *(Contour_Parameters *) levels = cp;

  return 1;
}

// ----------------------------------------------------------------------------
// Converter function for use with PyArg_ParseTuple() format O&
//
extern "C" int peak_arg(PyObject *py_peak, void *crosspeak)
{
  CrossPeak *xp = (CrossPeak *) c_object(py_peak, "crosspeak");
  if (xp == NULL)
    return 0;

  *(CrossPeak **) crosspeak = xp;

  return 1;
}

// ----------------------------------------------------------------------------
// Converter function for use with PyArg_ParseTuple() format O&
//
extern "C" int peak_list_arg(PyObject *list, void *plist)
{
  return list_arg(list, peak_arg, plist);
}

// ----------------------------------------------------------------------------
// Converter function for use with PyArg_ParseTuple() format O&
//
extern "C" int atom_arg(PyObject *py_atom, void *atom)
{
  Atom *a = (Atom *) c_object(py_atom, "atom");
  if (a == NULL)
    return 0;

  *(Atom **) atom = a;

  return 1;
}

// ----------------------------------------------------------------------------
// Converter function for use with PyArg_ParseTuple() format O&
//
extern "C" int atom_list_arg(PyObject *list, void *alist)
{
  return list_arg(list, atom_arg, alist);
}

// ----------------------------------------------------------------------------
// Converter function for use with PyArg_ParseTuple() format O&
//
extern "C" int resonance_arg(PyObject *py_resonance, void *resonance)
{
  Resonance *r = (Resonance *) c_object(py_resonance, "resonance");
  if (r == NULL)
    return 0;

  *(Resonance **) resonance = r;

  return 1;
}

// ----------------------------------------------------------------------------
// Converter function for use with PyArg_ParseTuple() format O&
//
extern "C" int resonance_list_arg(PyObject *list, void *rlist)
{
  return list_arg(list, resonance_arg, rlist);
}

// ----------------------------------------------------------------------------
//
static int list_arg(PyObject *py_list,
		    int (*list_arg)(PyObject *, void *),
		    void *list)
{
  if (!PySequence_Check(py_list))
    {
      PyErr_SetString(PyExc_TypeError, "argument is not a sequence");
      return 0;
    }

  List elist;
  int size = PySequence_Length(py_list);
  for (int k = 0 ; k < size ; ++k)
    {
      PyObject *py_element = python_sequence_item(py_list, k);
      void *e;
      if (!list_arg(py_element, &e))
	return 0;
      elist.append(e);
    }

  *(List *)list = elist;

  return 1;
}

// ----------------------------------------------------------------------------
// Return a sequence item with reference count not incremented.
//
static PyObject *python_sequence_item(PyObject *seq, int k)
{
  PyObject *item = PySequence_GetItem(seq, k);
  Py_XDECREF(item);
  return item;
}

// ----------------------------------------------------------------------------
// Converter function for use with PyArg_ParseTuple() format O&
//
extern "C" int python_callback_arg(PyObject *py_callback, void *pycb)
{
  Python_Callback *pcb = (Python_Callback *) c_object(py_callback,
						      "python_callback");
  if (pcb == NULL)
    return 0;

  *(Python_Callback **) pycb = pcb;

  return 1;
}

// ----------------------------------------------------------------------------
//
static int sparky_id(PyObject *inst)
{
  PyObject *py_id = NULL;
  if (!PyInstance_Check(inst) ||
      (py_id = instance_dictionary_get(inst, "sparky_id"), py_id == NULL) ||
      !PyInt_Check(py_id))
    PyErr_SetString(PyExc_TypeError, "not a sparky class instance");

  int id = -1;
  if (py_id)
    id = PyInt_AsLong(py_id);

  Py_XDECREF(py_id);

  return id;
}

// ----------------------------------------------------------------------------
// Need to remove table entry mapping between C object pointer and Python id
// when a C object is deleted.  If the Python object is referenced after
// the C object is deleted the table lookup based on the id number fails and
// a Python exception is raised.
//
void C_Python_Map::listen_for_deleted_objects(Session &s)
{
  //
  // When C object is deleted, the entry in the id table must be deleted.
  //
  Notifier &n = s.notifier();
  n.notify_me(nt_will_delete_atom, will_delete_object_cb, this);
  n.notify_me(nt_will_delete_condition, will_delete_object_cb, this);
  n.notify_me(nt_will_delete_crosspeak, will_delete_object_cb, this);
  n.notify_me(nt_will_delete_grid, will_delete_object_cb, this);
  n.notify_me(nt_will_delete_group, will_delete_object_cb, this);
  n.notify_me(nt_will_delete_label, will_delete_object_cb, this);
  n.notify_me(nt_will_delete_line, will_delete_object_cb, this);
  n.notify_me(nt_will_delete_molecule, will_delete_object_cb, this);
  n.notify_me(nt_will_delete_project, will_delete_object_cb, this);
  n.notify_me(nt_will_delete_resonance, will_delete_object_cb, this);
  n.notify_me(nt_will_delete_session, will_delete_object_cb, this);
  n.notify_me(nt_will_delete_spectrum, will_delete_object_cb, this);
  n.notify_me(nt_will_delete_view, will_delete_object_cb, this);
  n.notify_me(nt_will_delete_py_callback, will_delete_object_cb, this);

  //
  // Normally all notify_me() calls have a corresponding dont_notify_me().
  // The above notify_me() calls have no corresponding dont_notify_me().
  // I just let deleting the notifier clean up the above notify_me() calls.
  //
  // The dont_notify_me() calls should be done when the session is deleted
  // and all objects of the above types are destroyed.  Doing it on a
  // will_delete_session notice is too soon because that notice is sent
  // before all objects have been deleted.  That notice is issued at the
  // start of the session destructor.  If the above notifier callbacks are
  // removed before all objects are deleted then the objects will not be
  // removed from the C_Python_Map and references to them could be made
  // from Python after they have been deleted.  There is no later notice in
  // the session destructor indicating when all the above objects are cleaned
  // up.  They will be all deleted before the notifier is deleted.
  //
}

// ----------------------------------------------------------------------------
//
void C_Python_Map::will_delete_object_cb(void *cpmap, void *object)
{
  C_Python_Map *cpm = (C_Python_Map *) cpmap;
  List idlist = cpm->c_object_ids.delete_ids(object);
  for (int k = 0 ; k < idlist.size() ; ++k)
    {
      TableKey deleted_id = (TableKey) idlist[k];
      TableData py_object;
      if (cpm->id_to_python_object.find(deleted_id, &py_object))
	{
	  cpm->id_to_python_object.remove(deleted_id);
	  Py_XDECREF((PyObject *) py_object);
	}
    }
}

// ----------------------------------------------------------------------------
//
int C_Python_Map::object_id(void *pointer, const Stringy &type)
  { return c_object_ids.id(pointer, type); }

// ----------------------------------------------------------------------------
//
void *C_Python_Map::id_to_object(int id, Stringy *type)
  { return c_object_ids.object(id, type); }

// ----------------------------------------------------------------------------
//
PyObject *C_Python_Map::id_to_object(int id)
{
  TableData pyobj;
  if (id_to_python_object.find((TableKey) id, &pyobj))
    return (PyObject *) pyobj;
  return NULL;
}

// ----------------------------------------------------------------------------
//
void C_Python_Map::set_python_object_id(PyObject *py_object, int id)
{
  Py_XINCREF(py_object);
  id_to_python_object.insert((TableKey) id, (TableData) py_object);
}

// ----------------------------------------------------------------------------
//
static void *c_object(PyObject *py_object, const Stringy &type)
{
  int id = sparky_id(py_object);
  if (id == -1)
    return NULL;

  Stringy id_type;
  void *object = c_python_map->id_to_object(id, &id_type);
  if (object == NULL)
    {
      Stringy error_msg = formatted_string("%s %d not found",
					   type.cstring(), id);
      set_python_error(error_msg);
      return NULL;
    }
  else if (id_type != type)
    { object_type_error(id, type, id_type); return NULL; }

  return object;
}

// ----------------------------------------------------------------------------
//
static void object_type_error(int id, const Stringy &expected_type,
			      const Stringy &received_type)
{
  Stringy error_msg = formatted_string("expected a %s "
				       "but object %d is a %s",
				       expected_type.cstring(), id,
				       received_type.cstring());
  set_python_error(error_msg);
}

// ----------------------------------------------------------------------------
//
static void set_python_error(const Stringy &message)
{
  PyObject *error_object = get_module_value(this_module, "error");
  PyErr_SetString(error_object, (char *) message.cstring());
}

// ----------------------------------------------------------------------------
//
extern "C" PyObject *install_path(PyObject *, PyObject *args)
{
  if (!PyArg_ParseTuple(args, ""))
    return NULL;

  return python_string(installation_path(""));
}

// ----------------------------------------------------------------------------
//
extern "C" PyObject *object_exists(PyObject *, PyObject *args)
{
  PyObject *sparky_object;
  if (!PyArg_ParseTuple(args, "O", &sparky_object))
    return NULL;

  int id = sparky_id(sparky_object);
  if (id == -1)
    return NULL;

  bool deleted = (c_python_map->id_to_object(id) == NULL);
  return python_boolean(!deleted);
}

// ----------------------------------------------------------------------------
//
static struct PyMethodDef sparky_methods[] =
{
  // name, address, '1' = tuple arg-lists
 {"installation_path",		install_path,			METH_VARARGS},
 {"object_exists",		object_exists,			METH_VARARGS},
 {NULL,		        NULL}
};

// ----------------------------------------------------------------------------
// Initialization routine called by python when module is dynamically loaded.
//
extern "C" void initspy()
{
  this_module = Py_InitModule("spy", sparky_methods);
  Py_XINCREF(this_module);

  set_module_value(this_module, "error", python_string("SparkyError"));
  set_module_value(this_module, "user_sparky_directory",
		   python_string(home_directory()));

  define_sparky_classes();

  c_python_map = new C_Python_Map();
}

// --------------------------------------------------------------------------
// The Python Tkinter interface to Tk doesn't provide public access
// to its C types.  So I ripped this typedef from _tkinter.c.
// I need it to get the Python/Tkinter Tcl_Interp for use by the
// C++ Tk interface.
//
struct TkappObject
{
  PyObject_HEAD
  Tcl_Interp *interp;
};

// ----------------------------------------------------------------------------
// Converter used with PyArg_ParseTuple() O& format
//
extern "C" int widget_tcl_interpretter_arg(PyObject *w, void *ti)
{
  Tcl_Interp *tcl_interp = widget_tcl_interp(w);

  if (tcl_interp)
    *(Tcl_Interp **)ti = tcl_interp;
  else
    PyErr_SetString(PyExc_TypeError, "not a Tk widget");

  return (tcl_interp ? 1 : 0);
}

// ----------------------------------------------------------------------------
//
static Tcl_Interp *widget_tcl_interp(PyObject *w)
{
  Tcl_Interp *tcl_interp = NULL;

  if (PyInstance_Check(w) && PyObject_HasAttrString(w, "tk"))
    {
      PyObject *py_tkapp = PyObject_GetAttrString(w, "tk");
      if (py_tkapp != NULL)
	{
	  tcl_interp = ((struct TkappObject *) py_tkapp)->interp;
	  Py_XDECREF(py_tkapp);
	}
    }

  return tcl_interp;
}

// ----------------------------------------------------------------------------
// Converter used with PyArg_ParseTuple() O& format
//
static bool tk_widget_arg(WinSys &ws, PyObject *w, void *widget)
{
  Stringy name;
  Widget wt = (python_get_attribute(w, "_w", &name) ?
	       ws.named_widget(name) : NULL);
  if (wt)
    *(Widget *)widget = wt;
  else
    PyErr_SetString(PyExc_TypeError, "not a Tk widget");

  return (wt ? true : false);
}

// ----------------------------------------------------------------------------
//
static bool call_python_callback(Session &s, PyObject *py_callback,
				 PyObject *arg)
{
  List args;
  if (arg)
    {
      args.append(arg);
      Py_XINCREF(arg);
    }
  PyObject *py_args = python_tuple(args);
  PyObject *result = PyEval_CallObject((PyObject *) py_callback, py_args);
  if (result == NULL)
    print_error_message(s);

  Py_XDECREF(result);
  Py_XDECREF(py_args);
  return result != NULL;
}

// ----------------------------------------------------------------------------
// Use the session.stderr to print the error.  This goes to the Sparky shell.
// If session.stderr doesn't exist use sys.stderr.
//
static void print_error_message(Session &s)
{
  PyObject *session = python_session(s);
  PyObject *session_stderr;
  if (!python_get_attribute(session, "stderr", &session_stderr))
    PyErr_Print();
  else
    {
      PyObject *sys_stderr = get_module_value("sys", "stderr");
      set_module_value("sys", "stderr", session_stderr);
      PyErr_Print();
      set_module_value("sys", "stderr", sys_stderr);
    }
  Py_XDECREF(session);
}

// ----------------------------------------------------------------------------
//
static PyObject *eval_string(const Stringy &s, PyObject *module)
{
  Stringy command = s;
  if (command[command.length()-1] != '\n')
    command << "\n";

  PyObject *main_module = PyImport_AddModule("__main__");
  PyObject *globals = PyModule_GetDict(main_module);
  PyObject *locals = PyModule_GetDict(module);
  char *cmd = (char *) command.cstring();
  PyObject *result = PyRun_String(cmd, Py_single_input, globals, locals);

  return result;
}

// ----------------------------------------------------------------------------
//  
static void define_sparky_classes()
{
  //
  // The following PyMethodDef arrays are static because PyMethod_New()
  // takes a pointer to a PyMethodDef but does not copy the object.
  //
  static PyMethodDef
    session_init_method = {"__init__", session_init, METH_VARARGS},
    line_init_method = {"__init__", line_init, METH_VARARGS},
    contour_level_init_method = {"__init__", levels_init, METH_VARARGS};

  python_class("Session", &session_init_method);
  python_class("Project");
  python_class("Spectrum");
  python_class("Peak");
  python_class("Resonance");
  python_class("Label");
  python_class("Line", &line_init_method);
  python_class("Grid");
  python_class("Molecule");
  python_class("Atom");
  python_class("Group");
  python_class("Condition");
  python_class("View");
  python_class("Contour_Levels", &contour_level_init_method);
  python_class("Python_Callback");

  register_attributes("Session", "session", session_getset, session_methods);
  register_attributes("Project", "project", proj_getset, proj_methods);
  register_attributes("Spectrum", "spectrum", spect_getset, spect_methods);
  register_attributes("Peak", "crosspeak", peak_getset, peak_methods);
  register_attributes("Resonance", "resonance", res_getset, res_methods);
  register_attributes("Label", "label", label_getset, NULL);
  register_attributes("Line", "line", line_getset, NULL);
  register_attributes("Grid", "grid", grid_getset, NULL);
  register_attributes("Molecule", "molecule", mol_getset, mol_methods);
  register_attributes("Group", "group", group_getset, NULL);
  register_attributes("Atom", "atom", atom_getset, NULL);
  register_attributes("Condition", "condition", cond_getset, cond_methods);
  register_attributes("View", "view", view_getset, view_methods);
}

// ----------------------------------------------------------------------------
//
static PyObject *python_class(const Stringy &classname,
			      PyMethodDef *init_method)
{
  static PyMethodDef getset_methods[] = {
    {"__getattr__",	sparky_getattr,		METH_VARARGS},
    {"__setattr__",	sparky_setattr,		METH_VARARGS},
    {NULL,		NULL}};

  //
  // Create the class dictionary
  //
  PyObject *dict = PyDict_New();
  for (int m = 0 ; getset_methods[m].ml_name != NULL ; ++m)
    PyDict_SetItemString(dict, getset_methods[m].ml_name,
			 PyCFunction_New(&getset_methods[m], NULL));

  //
  // This hack deals with a Python 1.5 problem that class members which
  // are builtin functions don't get the self argument when invoked with
  // and expression like instance.method(args).
  // I replace __init__ with a Python code wrapper that calls the
  // C method which is renamed to __builtin_init__.
  // Perhaps this will be unnecessary in a future version of python.
  //
  if (init_method)
    {
      static bool first_call = true;		// state variable
      if (first_call)
	{
	  first_call = false;
	  eval_string("def __init_wrapper__(*args):\n"
		      "  apply(args[0].__builtin_init__, args)",
		      this_module);
	}
      PyObject *init = PyCFunction_New(init_method, NULL);
      PyDict_SetItemString(dict, "__builtin_init__", init);
      PyObject *init_wrapper = get_module_value(this_module,
						"__init_wrapper__");
      PyDict_SetItemString(dict, "__init__", init_wrapper);
    }

  //
  // Create the class
  //
  PyObject *name = python_string(classname);
  PyObject *bases = NULL;
  PyObject *c = PyClass_New(bases, dict, name);
  Py_XDECREF(name);

  //
  // Add this class to the module
  //
  set_module_value(this_module, classname, c);

  return c;
}

// ----------------------------------------------------------------------------
//
static PyObject *python_class_instance(const Stringy &classname)
{
  PyObject *pclass = get_module_value(this_module, classname);
  PyObject *args = NULL;
  PyObject *kw = NULL;
  return PyInstance_New(pclass, args, kw);
}

// ----------------------------------------------------------------------------
// Ref count incremented
//
static PyObject *python_mirror_object(const Stringy &classname,
				      void *cobj, const Stringy &c_object_type)
{
  int id = c_python_map->object_id(cobj, c_object_type);
  PyObject *py_object = c_python_map->id_to_object(id);
  if (py_object)
    {
      Py_XINCREF(py_object);
      return py_object;
    }
  
  return new_mirror_object(classname, cobj, c_object_type);
}

// ----------------------------------------------------------------------------
//
static PyObject *new_mirror_object(const Stringy &classname,
				   void *c_object, const Stringy &type)
{
  PyObject *pclass = get_module_value(this_module, classname);
  PyObject *args = NULL;
  PyObject *kw = NULL;
  PyObject *instance = PyInstance_New(pclass, args, kw);

  set_sparky_id(instance, c_python_map->object_id(c_object, type));

  return instance;
}

// ----------------------------------------------------------------------------
//
static void set_sparky_id(PyObject *self, int id)
{
  python_set_attribute(self, "sparky_id", id);
  c_python_map->set_python_object_id(self, id);
}

// ----------------------------------------------------------------------------
// Return value ref count is not incremented.
//
static PyObject *get_module_value(const Stringy &name, const Stringy &attr)
{
  PyObject *module = PyImport_AddModule((char *) name.cstring());
  return get_module_value(module, attr);
}
static PyObject *get_module_value(PyObject *module, const Stringy &attr)
{
  PyObject *dict = PyModule_GetDict(module);
  PyObject *value = PyDict_GetItemString(dict, (char *) attr.cstring());

  return value;
}

// ----------------------------------------------------------------------------
//
static void set_module_value(const Stringy &name, const Stringy &attr, 
			     PyObject *value)
{
  PyObject *module = PyImport_AddModule((char *) name.cstring());
  set_module_value(module, attr, value);
}
static void set_module_value(PyObject *module, const Stringy &attr, 
			     PyObject *value)
{
  PyObject *dict = PyModule_GetDict(module);
  PyDict_SetItemString(dict, (char *) attr.cstring(), value);
}

// ----------------------------------------------------------------------------
// Put Python class attribute get and set and method functions in a hash table
// for later lookup by sparky_getattr() and sparky_setattr().
//
static void register_attributes(const Stringy &classname,
				const Stringy &c_object_type,
				Get_Set_Functions *getset,
				PyMethodDef *methods)
{
  if (attribute_table == NULL)
    attribute_table = new Table(Attribute_Name::equal, Attribute_Name::hash);

  PyObject *py_class = get_module_value(this_module, classname);

  if (getset)
    for (int a = 0 ; getset[a].attribute_name ; ++a)
      {
	PyObject *py_attr = python_string(getset[a].attribute_name);
	Attribute_Name *an = new Attribute_Name(py_class, py_attr);
	Attribute_Method *meth = new Attribute_Method(c_object_type, getset[a]);
	attribute_table->insert((TableKey) an, (TableData) meth);
      }

  if (methods)
    for (int m = 0 ; methods[m].ml_name ; ++m)
      {
	PyObject *py_attr = python_string(methods[m].ml_name);
	Attribute_Name *an = new Attribute_Name(py_class, py_attr);
	Stringy attr = PyString_AsString(py_attr);
	Attribute_Method *meth = new Attribute_Method(attr, c_object_type,
						      &methods[m]);
	attribute_table->insert((TableKey) an, (TableData) meth);
      }
}
// ----------------------------------------------------------------------------
//
static Attribute_Method *lookup_attribute(PyObject *py_class,
					  PyObject *py_attr)
{
  Attribute_Name name(py_class, py_attr);

  TableData meth;
  if (attribute_table && attribute_table->find((TableKey) &name, &meth))
    return (Attribute_Method *) meth;

  return NULL;
}

// ----------------------------------------------------------------------------
//
extern "C" PyObject *sparky_getattr(PyObject *, PyObject *args)
{
  PyObject *self, *attr;
  if (!PyArg_ParseTuple(args, "O!O!", &PyInstance_Type, &self,
			&PyString_Type, &attr))
    return NULL;

  
  PyObject *py_class = (PyObject *)((PyInstanceObject *) self)->in_class;
  Attribute_Method *m = lookup_attribute(py_class, attr);
  if (m)
    return m->getattr(self, attr);

  PyErr_SetString(PyExc_AttributeError, PyString_AsString(attr));
  return NULL;
}

// ----------------------------------------------------------------------------
//
extern "C" PyObject *sparky_setattr(PyObject *, PyObject *args)
{
  PyObject *self, *attr, *value;
  if (!PyArg_ParseTuple(args, "O!O!O", &PyInstance_Type, &self,
			&PyString_Type, &attr, &value))
    return NULL;

  
  PyObject *py_class = (PyObject *)((PyInstanceObject *) self)->in_class;
  Attribute_Method *m = lookup_attribute(py_class, attr);
  if (m)
    return m->setattr(self, value);

  instance_dictionary_set(self, attr, value);

  return python_none();
}

// ----------------------------------------------------------------------------
//
Attribute_Name::Attribute_Name(PyObject *py_class, PyObject *py_attr)
{
  this->py_class = py_class;
  this->py_attr = py_attr;
}

// ----------------------------------------------------------------------------
//
bool Attribute_Name::equal(const void *attr_name_1, const void *attr_name_2)
{
  Attribute_Name *an1 = (Attribute_Name *) attr_name_1;
  Attribute_Name *an2 = (Attribute_Name *) attr_name_2;
  return (PyObject_Compare(an1->py_class, an2->py_class) == 0 &&
	  PyObject_Compare(an1->py_attr, an2->py_attr) == 0);
}

// ----------------------------------------------------------------------------
//
unsigned long Attribute_Name::hash(const void *attr_name)
{
  Attribute_Name *an = (Attribute_Name *) attr_name;
  return (unsigned long) (PyObject_Hash(an->py_class) |
			  PyObject_Hash(an->py_attr));
}

// ----------------------------------------------------------------------------
//
Attribute_Method::Attribute_Method(const Stringy &c_object_type,
				   const Get_Set_Functions &getset)
{
  this->attribute_name = getset.attribute_name;
  this->c_object_type = c_object_type;
  this->get_value = getset.get_value;
  this->set_value = getset.set_value;
  this->can_change = getset.can_change;
  this->mdef = NULL;
}

// ----------------------------------------------------------------------------
//
Attribute_Method::Attribute_Method(const Stringy &attr,
				   const Stringy &c_object_type,
				   PyMethodDef *mdef)

{
  this->attribute_name = attr;
  this->c_object_type = c_object_type;
  this->get_value = NULL;
  this->set_value = NULL;
  this->can_change = false;
  this->mdef = mdef;
}

// ----------------------------------------------------------------------------
//
PyObject *Attribute_Method::getattr(PyObject *self, PyObject *attr)
{
  void *cobj = c_object(self, c_object_type);
  if (cobj == NULL)
    return NULL;

  if (get_value)
    {
      PyObject *value = get_value(self, cobj);
      if (!can_change && value)
	instance_dictionary_set(self, attr, value);
      return value;
    }
  else if (mdef)
    {
      PyObject *value = PyCFunction_New(mdef, self);
      instance_dictionary_set(self, attr, value);
      return value;
    }

  return python_none();
}

// ----------------------------------------------------------------------------
//
PyObject *Attribute_Method::setattr(PyObject *self, PyObject *value)
{
  void *cobj = c_object(self, c_object_type);
  if (cobj == NULL)
    return NULL;

  if (set_value)
    if (set_value(cobj, value))
      return python_none();
    else
      return NULL;

  set_python_error(c_object_type + " " + attribute_name + " not settable");
  return NULL;
}
