// ----------------------------------------------------------------------------
//

#include "memalloc.h"		// use new()
#include "nmrdata.h"		// Use NMR_Data
#include "num.h"		// Use is_between(), interval_mod()
#include "spectrumdata.h"
#include "utility.h"		// Use fatal_error()

// ----------------------------------------------------------------------------
//
static bool retry_read(NMR_Data *nmr_data);
static void half_max_range(SpectrumData *sp, const SPoint &pos, int a,
			   double *x1, double *x2);
static bool half_max_position(SpectrumData *sp, const SPoint &pos,
			      int a, int step, double *x);
static bool neighbor_values(SpectrumData *sp, const IPoint &p, int axis,
			    float data[3]);

// ----------------------------------------------------------------------------
//
const char *Unit_Names[] = {"index", "hz", "ppm", "none", NULL};

// ----------------------------------------------------------------------------
//
SpectrumData::SpectrumData(NMR_Data *nmr_data)
{
	mNMRData = nmr_data;
	mDimension = nmr_data->dimension();
	mHaveNoise = false;

	for (int a = 0 ; a < mDimension ; ++a)
	  {
	    Stringy nucleus = nucleus_type(a);
	    double f = 1;
	    if (nucleus == "1H") f = 1;
	    else if (nucleus == "13C") f = 4;
	    else if (nucleus == "15N") f = 10;
	    nucleus_factor[a] = f;
	  }

	axis_shift = SPoint(mDimension);
	sweep_width = ppm_spectrum_width();
}

// ----------------------------------------------------------------------------
//
SpectrumData::~SpectrumData()
{
  delete mNMRData;
  mNMRData = NULL;
}

// ----------------------------------------------------------------------------
//
IRegion SpectrumData::index_range() const
{
  IRegion r(dimension());

  IPoint size = mNMRData->size();
  for (int a = 0 ; a < dimension() ; ++a)
    {
      r.min[a] = 0;
      r.max[a] = size[a] - 1;
    }

  return r;
}

// ----------------------------------------------------------------------------
//
SRegion SpectrumData::ppm_region() const
  { return map(index_range(), INDEX, PPM); }

// ----------------------------------------------------------------------------
//
SPoint SpectrumData::ppm_spectrum_width() const
{
  SPoint sfreq_mhz = mNMRData->spectrometer_frequency();
  SPoint swidth_hz = mNMRData->spectrum_width();
  SPoint swidth(dimension());
  for (int a = 0 ; a < dimension() ; ++a)
    swidth[a] = swidth_hz[a] / sfreq_mhz[a];
  return swidth;
}

// ----------------------------------------------------------------------------
//
SPoint SpectrumData::ppm_sweep_width() const
  { return sweep_width; }
void SpectrumData::set_ppm_sweep_width(const SPoint &sweepwidth)
  { sweep_width = sweepwidth; }

// ----------------------------------------------------------------------------
//
SRegion SpectrumData::alias_region() const
{
  SPoint zero(dimension());
  SPoint rmax = map(zero, INDEX, PPM);
  SPoint rmin = rmax - ppm_sweep_width();
  SRegion r(rmin, rmax);
  return r;
}

// ----------------------------------------------------------------------------
//
SPoint SpectrumData::ppm_shift() const
{
  return axis_shift;
}

// ----------------------------------------------------------------------------
//
void SpectrumData::set_ppm_shift(const SPoint &ppm_shift)
{
  axis_shift = ppm_shift;
}

// ----------------------------------------------------------------------------
//
double SpectrumData::nucleus_ppm_factor(int axis) const
  { return nucleus_factor[axis]; }

/*
 * Return the name of NMR nucleus for Axis <a>.
 */
Stringy SpectrumData::nucleus_type(int a) const
{
  return (a < dimension() ? mNMRData->axis_label(a) : Stringy(""));
}

// ----------------------------------------------------------------------------
//
const List &SpectrumData::nucleus_types() const
{
  return mNMRData->axis_label_list();
}

/*
 * Return true if the nucleus label along axis <x> is unique amoung all
 * other axes.
 */
bool SpectrumData::is_nucleus_unique(const Stringy &nucleus, int *axis)
{
  int a = 0;	// Suppress compiler warning about uninitialized var
  int count = 0;

  for (int i = 0; i < dimension(); i++)
    if (nucleus_type(i) == nucleus)
      { a = i; count += 1; }

  if (count == 1 && axis)
    *axis = a;

  return count == 1;
}

// ----------------------------------------------------------------------------
//
Stringy SpectrumData::data_path() const
{
  return mNMRData->path();
}

// ----------------------------------------------------------------------------
// Returns 0 if point out of range.
//
float SpectrumData::height_at_index(const IPoint &index) const
{
  bool success;
  float value;

  do
    success = mNMRData->read_value(index, &value);
  while (!success && retry_read(mNMRData));

  return (success ? value : 0);
}

// ----------------------------------------------------------------------------
//
float SpectrumData::height_at_point(const SPoint &p) const
  { return height_at_index(map(p, PPM, INDEX).rounded()); }

// ----------------------------------------------------------------------------
//
bool SpectrumData::heights_for_region(const IRegion &region, float *data) const
{
  bool success;

  do
    success = mNMRData->read_values(region, data);
  while (!success && retry_read(mNMRData));

  if (!success)
    {
      int size = region.volume();
      for (int k = 0 ; k < size ; ++k)
	data[k] = 0;
    }

  return true;
}

// ----------------------------------------------------------------------------
//
static bool retry_read(NMR_Data *nmr_data)
{
  static bool dont_warn = false;	// state variable

  if (dont_warn)
    return false;

  Stringy msg = formatted_string("Error reading data file %s\n"
				 "Will use zeros.",
				 nmr_data->path().cstring());
  warn(msg.cstring());
  dont_warn = true;

  return false;
}

int SpectrumData::dimension() const
  { return mDimension; }

// ----------------------------------------------------------------------------
bool SpectrumData::is_homonuclear(int x, int y) const
{
  /*
   * The spectrometer frequency is characteristic of the
   * atom type. If the frequency is nearly the same
   * along both axes, assume it is a HOMONUCLEAR pair of
   * axis.
   *
   * Why not use the atom name? Historically, old UCSF
   * NMR format had no atom name.
   */
  SPoint sfreq_mhz = mNMRData->spectrometer_frequency();
  return is_between(sfreq_mhz[x] / sfreq_mhz[y], 0.95, 1.05);
}

// ----------------------------------------------------------------------------
// Scale a length from one coordinate system to another.
//
double SpectrumData::scale(double x, int a, Units from, Units to) const
{
  double sfreq_mhz = mNMRData->spectrometer_frequency()[a];
  double spectrum_width_ppm = mNMRData->spectrum_width()[a] / sfreq_mhz;
  double ppm_step = spectrum_width_ppm / mNMRData->size()[a];

  if (from == INDEX)
    switch (to)
      {
      case PPM:
	x *= ppm_step;
	break;
      case HZ:
	x *= ppm_step * sfreq_mhz;
	break;
      case INDEX:
	break;
      case NOUNITS:
	fatal_error("SpectrumData::scale(): Bad unit type.\n");
      }
  else if (to == INDEX)
    switch (from)
      {
      case PPM:
	x /= ppm_step;
	break;
      case HZ:
	x /= ppm_step * sfreq_mhz;
	break;
      case INDEX:
	break;
      case NOUNITS:
	fatal_error("SpectrumData::scale(): Bad unit type.\n");
      }
  else
    {
      x = scale(x, a, from, INDEX);
      x = scale(x, a, INDEX, to);
    }

  return x;
}

// ----------------------------------------------------------------------------
//
SPoint SpectrumData::scale(const SPoint &point, Units from, Units to) const
{
  SPoint topoint = point;

  for (int a = 0 ; a < point.dimension() ; ++a)
    topoint[a] = scale(point[a], a, from, to);

  return topoint;
}

// ----------------------------------------------------------------------------
//
SRegion SpectrumData::scale(const SRegion &region, Units from, Units to) const
{
  int dim = region.dimension();
  SPoint min(dim), max(dim);
  for (int a = 0 ; a < dim ; ++a)
    {
      bool swap = (scale(1.0, a, from, to) < 0);
      min[a] = scale((swap ? region.max[a] : region.min[a]), a, from, to);
      max[a] = scale((swap ? region.min[a] : region.max[a]), a, from, to);
    }
  return SRegion(min, max);
}

// ----------------------------------------------------------------------------
// Transform from one coordinate system to another.
//
double SpectrumData::map(double x, int a, Units from, Units to) const
{
  double tx;

  double sfreq_mhz = mNMRData->spectrometer_frequency()[a];
  double spectrum_width_ppm = mNMRData->spectrum_width()[a] / sfreq_mhz;
  double ppm_step = spectrum_width_ppm / mNMRData->size()[a];
  double ppm_offset = mNMRData->origin_ppm()[a] + axis_shift[a];

  if (from == INDEX)
    switch (to)
      {
      case PPM:
	tx = ppm_offset - ppm_step * x;
	break;
      case HZ:
	tx = map(x, a, INDEX, PPM) * sfreq_mhz;
	break;
      case INDEX:
	tx = x;
	break;
      case NOUNITS:
      default:
	tx = 0;		// Suppress compiler warning about uninitialized var
	fatal_error("SpectrumData::map(): Bad unit type.\n");
	break;
      }
  else if (to == INDEX)
    switch (from)
      {
      case PPM:
	tx = (ppm_offset - x) / ppm_step;
	break;
      case HZ:
	tx = map(x / sfreq_mhz, a, PPM, INDEX);
	break;
      case INDEX:
	tx = x;
	break;
      case NOUNITS:
      default:
	tx = 0;		// Suppress compiler warning about uninitialized var
	fatal_error("SpectrumData::map(): Bad unit type.\n");
	break;
      }
  else
    {
      tx = map(x, a, from, INDEX);
      tx = map(tx, a, INDEX, to);
    }

  return tx;
}

// ----------------------------------------------------------------------------
//
SPoint SpectrumData::map(const SPoint &point, Units from, Units to) const
{
  SPoint topoint = point;

  for (int a = 0 ; a < dimension() ; ++a)
    topoint[a] = map(point[a], a, from, to);

  return topoint;
}

// ----------------------------------------------------------------------------
//
SRegion SpectrumData::map(const SRegion &region, Units from, Units to) const
{
  int dim = region.dimension();
  SPoint min(dim), max(dim);
  for (int a = 0 ; a < dim ; ++a)
    {
      bool swap = (map(0.0, a, from, to) > map(1.0, a, from, to));
      min[a] = map((swap ? region.max[a] : region.min[a]), a, from, to);
      max[a] = map((swap ? region.min[a] : region.max[a]), a, from, to);
    }
  return SRegion(min, max);
}

// ----------------------------------------------------------------------------
//
SPoint half_height_width(SpectrumData *sp, const SPoint &p)
{
  SPoint lw(sp->dimension());

  for (int a = 0; a < sp->dimension(); ++a)
    {
      double x1, x2;
      half_max_range(sp, p, a, &x1, &x2);
      lw[a] = x2 - x1;
    }

  return lw;
}

// ----------------------------------------------------------------------------
//
SPoint symmetric_half_height_width(SpectrumData *sp, const SPoint &p)
{
  SPoint lw(sp->dimension());

  for (int a = 0; a < sp->dimension(); ++a)
    {
      double x1, x2;
      half_max_range(sp, p, a, &x1, &x2);
      double x = p[a];
      lw[a] = 2 * min(x2 - x, x - x1);
    }

  return lw;
}

//
// Return the full-width half max of the data as a position away from
// <pos>[<a>] along the <dir> direction. <dir> is +1 or -1.
//
static void half_max_range(SpectrumData *sp, const SPoint &pos, int a,
			   double *x1, double *x2)
{
  half_max_position(sp, pos, a, -1, x1);
  half_max_position(sp, pos, a, +1, x2);
}

// ----------------------------------------------------------------------------
//
static bool half_max_position(SpectrumData *sp, const SPoint &p,
			      int a, int step, double *x)
{
  IPoint indx = sp->map(p, PPM, INDEX).rounded();
  int w = sp->index_range().max[a];
  double z0 = sp->height_at_index(indx);
  double zhalf = z0 / 2;
  double zlast = z0;
  IPoint n = indx;
  step = -step;		// ppm direction to index direction
  for (int i = indx[a] + step ; i >= 0 && i <= w ; i += step)
    {
      n[a] = i;
      double z = sp->height_at_index(n);
      if (z0 > 0 ? z < zhalf : z > zhalf)
	{
	  // Interpolate
	  *x = sp->map(i - step * (zhalf - z) / (zlast - z), a, INDEX, PPM);
	  return true;
	}
      zlast = z;
    }

  *x = sp->map((step > 0 ? w : 0), a, INDEX, PPM);

  return false;
}

//
// Return the data local maximum starting at startPos.
// startPos must be in data units.
//
static IPoint grid_extremum(SpectrumData *sp, const IPoint &index)
{
	float		dat[3];		// this point plus the two around

	//
	// Set the initial position
	//
	IPoint		i = index;
	double		z0 = sp->height_at_index(i);

	//
	// Read the data in a slice along one of the directions, if
	// we can move along the direction, do so, increment the position
	// and try again. If we try all directions and cannot move,
	// we are finished
	//
	int	movedAnyDir = true;
	int	movedThisDir;

	while (movedAnyDir) {

		movedAnyDir = false;

		for (int dir = sp->dimension() - 1; dir >= 0; dir--) {

			//
			// Move in this direction until you can't move any more
			//
			for (;;) {

				movedThisDir = false;

				//
				// If either data limit has been reached, stop
				// working on this direction.
				//
				if (!neighbor_values(sp, i, dir, dat))
					break;

				//
				// If positive, move ever more positive in
				// the steepest direction.
				//
				if (z0 > 0) {
					if (dat[0] > dat[1] && dat[0]>dat[2]) {
						movedThisDir = true;
						i[dir] -= 1;
					}
					else if (dat[2] > dat[1]) {
						movedThisDir = true;
						i[dir] += 1;
					}
				}

				//
				// If negative, move ever more negative in
				// the steepest direction.
				//
				else {
					if (dat[0] < dat[1] && dat[0]<dat[2]) {
						movedThisDir = true;
						i[dir] -= 1;
					}
					else if (dat[2] < dat[1]) {
						movedThisDir = true;
						i[dir] += 1;
					}
				}

				//
				// If a local maximum, break out
				//
				if (movedThisDir == false)
					break;

				//
				// Otherwise, say we at least moved.
				//
				movedAnyDir = true;
			}
		}
	}

	return i;
}

// ----------------------------------------------------------------------------
//
static bool neighbor_values(SpectrumData *sp, const IPoint &p, int axis,
			    float data[3])
{
  const IPoint &i1 = p;

  if (i1[axis] <= sp->index_range().min[axis] ||
      i1[axis] >= sp->index_range().max[axis])
    return false;

  IPoint i0 = i1;  i0[axis] -= 1;
  IPoint i2 = i1;  i2[axis] += 1;

  data[0] = sp->height_at_index(i0);
  data[1] = sp->height_at_index(i1);
  data[2] = sp->height_at_index(i2);

  return true;
}

//
// Do a 3-point interpolation of the data. This is done by fitting
// the three points to a parabola, and determining the apex. The
// points are implicitly numbered 0, 1, 2, so the middle of the
// points is '1'.
//
static double _threePointInterpolate(float dat[], double *heightp)
{
	//
	// We are going to solve 3 equations of the form:
	//
	//		a0 + a1 * x(i) + a2 * x(i)^2 = b(i)
	//
	// For x,y pairs of 0,dat[0], 1,dat[1], 2,dat[2].
	// This is the set of equations:
	//
	//		a0	+ 0	+ 0	= dat[0]
	//		a0	+ a1	+ a2	= dat[1]
	//		a0	+ 2 a1	+ 4 a2	= dat[2]
	//
	// In matrix form this is:
	//
	//		(1 0 0)   (a0)   (dat[0])
	//		(1 1 1) . (a1) = (dat[1])
	//		(1 2 4)   (a2)   (dat[2])
	//
	//
  double a0 = dat[0];
  double a1 = -1.5 * dat[0] + 2 * dat[1] - .5 * dat[2];
  double a2 = .5 * dat[0] - dat[1] + .5 * dat[2];

  //
  // Find the maximum of a2 x^2 + a1 x + a0 which occurs when:
  //
  //		2 * a2 * x + a1 = 0	 -ie-
  //		x = - a1 / (2 * a2)
  //
  if (a2 == 0)
    {
      *heightp = dat[1];	// return the middle point
      return 1;
    }

  double x = -0.5 * a1 / a2;
  if (heightp)
    *heightp = a2 * x * x + a1 * x + a0;
  return x;
}

//
// Interpolate the position in N-dimensions.
//
static SPoint quadratic_max(SpectrumData *sp, const IPoint &i, double *heightp)
{
	SPoint		r = i;
	float		dat[3];		// this point plus the two around

	for (int dir = i.dimension() - 1; dir >= 0; dir--) {

		//
		// If the position is near an edge we can't interpolate.
		//
		if (!neighbor_values(sp, i, dir, dat))
			continue;

		//
		// Do an interpolation to find the maximum
		//
		double	delta = _threePointInterpolate(dat, heightp);

		if (0 <= delta && delta <= 2)
			r[dir] = i[dir] + delta - 1;
	}

	return r;
}

//
// Return the data interpolated maximum starting at startPos.
// startPos must be in data units.
//
SPoint local_maximum(SpectrumData *sp, const SPoint &p, double *h)
{
  IPoint i = sp->map(p, PPM, INDEX).rounded();
  IPoint imax = grid_extremum(sp, i);
  return sp->map(quadratic_max(sp, imax, h), INDEX, PPM);
}

/*
 * Return the Spectrum noise, which is measured by computing the random
 * deviations from the baseline in peakless regions of the spectrum
 */
#define NOISE_POINTS 30
double SpectrumData::noise_level()
{
	if (!mHaveNoise) {
		mNoise = sample_spectrum_noise(this, NOISE_POINTS);
		if (mNoise == 0)
		  mNoise = 1;
		mHaveNoise = true;
	}
	return mNoise;
}
bool SpectrumData::have_noise_level() const
  { return mHaveNoise; }
void SpectrumData::set_noise_level(double noise)
  { mNoise = noise; mHaveNoise = true; }

// ----------------------------------------------------------------------------
// Take the median value of randomly sampled data absolute values.
// For this to work well most of the data points in the spectrum should
// be noise with no contribution from a peak.
//
double sample_spectrum_noise(SpectrumData *sp, int sample_count)
{
  float *values = new float[sample_count];
  IRegion r = sp->index_range();
  for (int k = 0 ; k < sample_count ; ++k)
    values[k] = abs(sp->height_at_index(r.random_point()));
  sort_floats(values, sample_count);
  double median = values[sample_count/2];
  delete values;
  return median;
}

// ----------------------------------------------------------------------------
//
SPoint alias_onto_spectrum(const SPoint &freq, SpectrumData *sp)
{
  SPoint rmax = sp->ppm_region().max;
  SPoint swidth = sp->ppm_sweep_width();
  SPoint f(sp->dimension());
  for (int a = 0 ; a < sp->dimension() ; ++a)
    {
      double fmax = rmax[a];
      double fmin = fmax - swidth[a];
      f[a] = interval_mod(freq[a], fmin, fmax);
    }
  return f;
}
