// ----------------------------------------------------------------------------
//
#include <iostream.h>	// Use ostream, istream
#include <stdarg.h>	// use va_start()
#include <stdlib.h>	// use atoi()
#include <stdio.h>	// Use sprintf()
#include <string.h>	// Use strcmp()
#include <ctype.h>	// Use isspace()

#include "list.h"	// Use List
#include "memalloc.h"		// use new()
#include "stringc.h"

#define MAX_LINE_LENGTH		4096

// ----------------------------------------------------------------------------
// Optimization to avoid allocating empty strings.
//
static char *empty_string = "";

// ----------------------------------------------------------------------------
//
Stringy::Stringy()
{
  this->s = empty_string;
}

// ----------------------------------------------------------------------------
//
Stringy::Stringy(const char *s)
{
  this->s = ((s == NULL || s[0] == '\0') ?
	     empty_string : allocate_character_array(s));
}

// ----------------------------------------------------------------------------
//
Stringy::Stringy(const Stringy &s)
{
  this->s = (s.s == empty_string ?
	     empty_string : allocate_character_array(s.s));
}

// ----------------------------------------------------------------------------
//
Stringy::Stringy(char c)
{
  char buffer[2] = {c, '\0'};
  this->s = allocate_character_array(buffer);
}

// ----------------------------------------------------------------------------
//
Stringy::Stringy(int i)
{
  char buffer[64];
  sprintf(buffer, "%d", i);
  this->s = allocate_character_array(buffer);
}

// ----------------------------------------------------------------------------
//
Stringy::~Stringy()
{
  if (s != empty_string)
    delete [] s;
  s = NULL;	// Safety measure.
}

// ----------------------------------------------------------------------------
//
Stringy &Stringy::operator=(const Stringy &s)
{
  *this = s.s;
  return *this;
}

// ----------------------------------------------------------------------------
//
Stringy &Stringy::operator=(const char *s)
{
  char *p = this->s;
  this->s = ((s == NULL || s[0] == '\0') ?
	     empty_string : allocate_character_array(s));
  if (p != empty_string)
    delete [] p;
  return *this;
}

// ----------------------------------------------------------------------------
//
bool Stringy::operator==(const Stringy &s) const
  { return strcmp(this->s, s.s) == 0; }
bool Stringy::operator==(const char *s) const
  { return strcmp(this->s, s) == 0; }

// ----------------------------------------------------------------------------
//
bool Stringy::operator!=(const Stringy &s) const
 { return !(*this == s); }
bool Stringy::operator!=(const char *s) const
 { return !(*this == s); }

// ----------------------------------------------------------------------------
//
static Stringy concatenate_strings(const char *s1, const char *s2)
{
  size_t length = strlen(s1) + strlen(s2) + 1;
  char *buf = new char [length];

  strcpy(buf, s1);
  strcat(buf, s2);

  Stringy s(buf);

  delete [] buf;

  return s;
}

// ----------------------------------------------------------------------------
//
Stringy &Stringy::operator<<(const char *s)
{
  *this = concatenate_strings(cstring(), s);

  return *this;
}

// ----------------------------------------------------------------------------
//
Stringy &Stringy::operator<<(const Stringy &s)
{
  return *this << s.cstring();
}

// ----------------------------------------------------------------------------
//
Stringy &Stringy::operator<<(char c)
{
  char value[2] = {c, '\0'};
  return *this << value;
}

// ----------------------------------------------------------------------------
//
Stringy &Stringy::operator<<(int k)
{
  char value[64];

  sprintf(value, "%d", k);

  return *this << value;
}

// ----------------------------------------------------------------------------
//
Stringy Stringy::operator+(const Stringy &s) const
{
  return concatenate_strings(cstring(), s.cstring());
}

// ----------------------------------------------------------------------------
//
char Stringy::operator[](int k) const
{
  return this->s[k];
}

// ----------------------------------------------------------------------------
//
int Stringy::length() const
{
  return strlen(s);
}

// ----------------------------------------------------------------------------
//
bool Stringy::is_empty() const
{
  return s[0] == '\0';
}

// ----------------------------------------------------------------------------
//
bool Stringy::is_white() const
{
  for (int k = 0 ; k < length() ; ++k)
    if (!isspace(s[k]))
      return false;

  return true;
}

// ----------------------------------------------------------------------------
//
Stringy Stringy::substring(int start, int end) const
  { return ::substring(cstring() + start, cstring() + end); }
Stringy Stringy::substring(int start) const
  { return ::substring(cstring() + start, cstring() + length()); }

// ----------------------------------------------------------------------------
//
bool has_prefix(const Stringy &s, const Stringy &prefix)
{
  return strncmp(s.cstring(), prefix.cstring(), prefix.length()) == 0;
}

// ----------------------------------------------------------------------------
//
Stringy remove_suffix(const Stringy &s, const Stringy &suffix)
{
  if (s.length() >= suffix.length() &&
      s.substring(s.length()-suffix.length()) == suffix)
    return s.substring(0, s.length() - suffix.length());
  return s;
}

// ----------------------------------------------------------------------------
//
Stringy replace_character(const Stringy &s, char from, char to)
{
  char f[2] = {from, '\0'};
  char t[2] = {to, '\0'};
  int pos;
  if (find_character(s, f, &pos))
    return (s.substring(0, pos) +
	    t +
	    replace_character(s.substring(pos+1), from, to));
  return s;
}

// ----------------------------------------------------------------------------
//
const char *Stringy::cstring() const
{
  return s;
}

// ----------------------------------------------------------------------------
//
Stringy operator+(const char* s1, const Stringy& s2)
{
  return concatenate_strings(s1, s2.cstring());
}

// ----------------------------------------------------------------------------
//
ostream &operator<<(ostream &strm, const Stringy &s)
{
  return strm << s.cstring();
}

// ----------------------------------------------------------------------------
//
istream &operator>>(istream &strm, Stringy &s)
{
  char buf[MAX_LINE_LENGTH];
  strm >> buf;
  s = buf;

  return strm;
}

// ----------------------------------------------------------------------------
//
bool stream_line(istream &strm, Stringy *s)
{
  char buf[MAX_LINE_LENGTH];
  strm.getline(buf, sizeof buf);
  *s = buf;

  return !strm.fail() && !strm.eof();
}

// ----------------------------------------------------------------------------
//
int compare_strings(const Stringy &s1, const Stringy &s2)
{
  return strcmp(s1.cstring(), s2.cstring());
}

// ----------------------------------------------------------------------------
//
Stringy formatted_string(const char *format, ...)
{
  va_list args;
  va_start(args, format);
  Stringy msg = va_formatted_string(format, args);
  va_end(args);

  return msg;
}

// ----------------------------------------------------------------------------
//
Stringy va_formatted_string(const char *format, va_list args)
{
  char buf[MAX_LINE_LENGTH];
  vsprintf(buf, format, args);
  return buf;
}

// ----------------------------------------------------------------------------
// Precede specified characters by a backslash.
//
Stringy backslashify_characters(const Stringy &s, const Stringy &chars)
{
  int pos;
  if (find_character(s, chars, &pos))
    return (s.substring(0, pos) +
	    "\\" + s.substring(pos, pos+1) +
	    backslashify_characters(s.substring(pos+1), chars));
  return s;
}

// ----------------------------------------------------------------------------
// Find first occurence of one of a set of characters.
//
bool find_character(const Stringy &s, const Stringy &chars, int *pos)
{
  for (int k = 0 ; k < s.length() ; ++k)
    if (strchr(chars.cstring(), s[k]))
      {
	if (pos)
	  *pos = k;
	return true;
      }

  return false;
}

// ----------------------------------------------------------------------------
//
bool contains_whitespace(const Stringy &s)
{
  for (int k = 0 ; k < s.length() ; ++k)
  if (isspace(s[k]))
    return true;
  return false;
}

// ----------------------------------------------------------------------------
//
Stringy spaces(int num)
{
  Stringy format = formatted_string("%%%ds", num);
  return formatted_string(format.cstring(), "");
}

// ----------------------------------------------------------------------------
//
Stringy substring(const char *start, const char *end)
{
  int length = end - start;

  char *buf = new char [length+1];
  strncpy(buf, start, length);
  buf[length] = '\0';
  Stringy s = buf;
  delete [] buf;

  return s;
}

// ----------------------------------------------------------------------------
//
int string_compare(const void *string1, const void *string2)
{
  return compare_strings(*((const Stringy *) string1),
			 *((const Stringy *) string2));
}

// ----------------------------------------------------------------------------
//
bool equal_strings(const void *string1, const void *string2)
  { return string_compare(string1, string2) == 0; }

// ----------------------------------------------------------------------------
//
unsigned long hash_string(const void *string)
{
  unsigned long hash = 0;
  unsigned char *b = (unsigned char *) &hash;

  Stringy *s = (Stringy *) string;
  for (int c = 0 ; c < s->length() ; ++c)
    b[c % sizeof(unsigned long)] ^= (*s)[c];

  return hash;
}

// ----------------------------------------------------------------------------
//
Stringy first_word(const Stringy &s, const Stringy &separators, Stringy *rest)
{
  int pos;
  if (find_character(s, separators, &pos))
    {
      if (rest)
	*rest = s.substring(pos + 1);
      return s.substring(0, pos);
    }
  if (rest)
    *rest = "";
  return s;
}

// ----------------------------------------------------------------------------
//
Stringy first_token(const Stringy &s, Stringy *rest)
{
  const char *start = skip_white(s.cstring());
  const char *end = skip_nonwhite(start);

  Stringy tok = substring(start, end);
  if (rest)
    *rest = end;

  return tok;
}

// ----------------------------------------------------------------------------
// Remove leading and trailing white space.
//
Stringy trim_white(const Stringy &s)
{
  const char *first = skip_white(s.cstring());
  const char *end = first + strlen(first);
  while (end > first && isspace(*(end - 1)))
    end -= 1;
  return substring(first, end);
}

// ----------------------------------------------------------------------------
//
const char *skip_white(const char *string)
{
  while (*string && isspace(*string)) ++string;

  return string;
}

// ----------------------------------------------------------------------------
//
char *skip_white(char *string)
{
  while (*string && isspace(*string)) ++string;

  return string;
}

// ----------------------------------------------------------------------------
//
const char *skip_nonwhite(const char *string)
{
  while (*string && !isspace(*string)) ++string;

  return string;
}

// ----------------------------------------------------------------------------
//
char *skip_nonwhite(char *string)
{
  while (*string && !isspace(*string)) ++string;

  return string;
}

// ----------------------------------------------------------------------------
//
bool number_suffix(const Stringy &s, char suffix_separator,
		   Stringy *basename, int *suffix_number)
{
  const char *cs = s.cstring();
  const char *sep = strrchr(cs, suffix_separator);
  *basename = (sep ? substring(cs, sep) : s);
  *suffix_number = (sep ? atoi(sep + 1) : 0);
  return (sep != NULL);
}

// ----------------------------------------------------------------------------
//
void free_string_list_entries(List &strings)
{
  for (int si = 0 ; si < strings.size() ; ++si)
    delete (Stringy *) strings[si];
  strings.erase();
}

// ----------------------------------------------------------------------------
//
List duplicate_string_list(const List &strings)
{
  List scopy;
  for (int si = 0 ; si < strings.size() ; ++si)
    scopy.append(new Stringy(*(Stringy *) strings[si]));
  return scopy;
}

// ----------------------------------------------------------------------------
//
int name_index(const char *names[], const Stringy &name)
{
  for (int k = 0 ; names[k] ; ++k)
    if (name == names[k])
      return k;
  return -1;
}

// ----------------------------------------------------------------------------
//
Stringy index_name(const char *names[], int index)
{
  int k;
  for (k = 0 ; names[k] ; ++k)
    ;
  if (index < 0 || index > k)
    return "";
  return names[index];
}

// ----------------------------------------------------------------------------
//
const char *matching_string(const char *names[], const Stringy &name)
{
  for (int k = 0 ; names[k] ; ++k)
    if (name == names[k])
      return names[k];
  return NULL;
}

// ----------------------------------------------------------------------------
// Duplicate a string into newly allocated memory block.
//
char *allocate_character_array(const char *s)
{
  if (s == NULL)
    return NULL;

  char *copy = new char [strlen(s) + 1];
  strcpy(copy, s);

  return copy;
}
