// ----------------------------------------------------------------------------
//
#include <fcntl.h>		// Use fcntl()
#include <pwd.h>		// Use getpwuid()
#include <sys/stat.h>		// Use mkdir(), stat()
#include <sys/types.h>		// Use struct stat, struct passwd
#include <sys/wait.h>		// Use waitpid()
#include <unistd.h>		// Use unlink(), rename(), getuid(), chdir()

#include "system.h"
#include "utility.h"		// use warn()

// ----------------------------------------------------------------------------
//
static void try_waiting();
static Stringy resolve_links(const Stringy &path, int *max_resolutions);

// ----------------------------------------------------------------------------
// Include system routines portable to all platforms.
//
#include "system-all.cc"

// ----------------------------------------------------------------------------
//
Stringy path_separator()
  { return "/"; }

// ----------------------------------------------------------------------------
//
#define MAX_LINK_RESOLUTIONS 256
Stringy resolve_links(const Stringy &path)
{
  int max_res = MAX_LINK_RESOLUTIONS;
  Stringy p = resolve_links(path, &max_res);
  if (max_res == 0)
    warn("resolve_links(): Circular link %s", path.cstring());
  return p;
}

// ----------------------------------------------------------------------------
//
static Stringy resolve_links(const Stringy &path, int *max_resolutions)
{
  if (path.is_empty())
    return path;

  Stringy p = path;
  Stringy directory, file;
  if (split_path(p, &directory, &file))
    {
      Stringy d = resolve_links(directory, max_resolutions);
      p = d + "/" + file;
    }

  char link_path[4096];
  for ( ; *max_resolutions > 0 ; --*max_resolutions)
    {
      int count = readlink(p.cstring(), link_path, sizeof(link_path));
      if (count == -1)
	break;
      else
	{
	  link_path[count] = '\0';
	  p = link_path;
	}
    }

  return (p == path ? p : resolve_links(p, max_resolutions));
}

// ----------------------------------------------------------------------------
//
bool make_directory(const Stringy &path)
{
  return mkdir(path.cstring(), 0777) == 0;
}

// ----------------------------------------------------------------------------
//
bool same_file(const Stringy &path1, const Stringy &path2)
{
  if (path1 == path2)
    return true;

  struct stat s1, s2;
  return (stat(path1.cstring(), &s1) == 0 && stat(path2.cstring(), &s2) == 0 &&
	  s1.st_dev == s2.st_dev && s1.st_ino == s2.st_ino);
}

// ----------------------------------------------------------------------------
//
Stringy default_sparky_install_path()
{
  return file_path(file_path(file_path("", "usr"), "local"), "sparky");
}

// ----------------------------------------------------------------------------
//
bool remove_file(const Stringy &path)
{
  return unlink(path.cstring()) == 0;
}

// ----------------------------------------------------------------------------
//
bool move_file(const Stringy &from, const Stringy &to)
{
  return rename(from.cstring(), to.cstring()) == 0;
}

// ----------------------------------------------------------------------------
//
bool user_owns_file(const Stringy &path)
{
  struct stat	sbuf;
  uid_t uid = getuid();
  return (stat(path.cstring(), &sbuf) == 0 && sbuf.st_uid == uid);
}

// ----------------------------------------------------------------------------
//
Stringy get_username()
{
  uid_t uid = getuid();
  struct passwd *pwd = getpwuid(uid);
  if (pwd)
    return pwd->pw_name;

  const char *u = getenv("USERNAME");
  if (u)
    return u;

  u = getenv("USER");
  if (u)
    return u;

  return "";
}

// ----------------------------------------------------------------------------
//
static Stringy user_home_directory()
{
  struct passwd *pwent = getpwuid(getuid());
  if (pwent)
    return pwent->pw_dir;

  return getenv("HOME");
}

// ----------------------------------------------------------------------------
//
static Stringy user_home_directory(const Stringy &username)
{
  struct passwd	*pwent;
  pwent = getpwnam(username.cstring());
  if (pwent)
    return pwent->pw_dir;

  return "";
}

// ----------------------------------------------------------------------------
//
bool catch_signal(int sig, void (*handler)(int))
{
  struct sigaction action;

  *(void **) &action.sa_handler = (void *) handler;
  sigemptyset(&action.sa_mask);
  action.sa_flags = 0;
  
  return sigaction(sig, &action, NULL) == 0;

}

// ----------------------------------------------------------------------------
//
bool ignore_signal(int sig)
{
  struct sigaction action;

  *(void **) &action.sa_handler = (void *) SIG_IGN;
  sigemptyset(&action.sa_mask);
  action.sa_flags = 0;
  
  return sigaction(sig, &action, NULL) == 0;
}

// ----------------------------------------------------------------------------
//
Stringy file_owner(FILE *fp)
{
  struct stat	f_stat;
  if (fstat(fileno(fp), &f_stat) != EOF)
    {
      struct passwd *pwe = getpwuid(f_stat.st_uid);
      if (pwe)
	return pwe->pw_name;
    }

  return "";
}

// ----------------------------------------------------------------------------
//
bool is_data_available(FILE *fp)
{
  int fd = fileno(fp);
  int flags = fcntl(fd, F_GETFL, 0);
  fcntl(fd, F_SETFL, flags | O_NONBLOCK);
  bool avail = (read(fd, NULL, 0) == 0);
  fcntl(fd, F_SETFL, flags);
  return avail;
}

// ----------------------------------------------------------------------------
//
void ignore_broken_pipes()
  { ignore_signal(SIGPIPE); }

// ----------------------------------------------------------------------------
//
void catch_fatal_errors(void (*handler)(int))
{
  catch_signal(SIGQUIT, handler);
  catch_signal(SIGSEGV, handler);
  catch_signal(SIGBUS, handler);
  catch_signal(SIGFPE, handler);
  catch_signal(SIGTERM, handler);
}

// ----------------------------------------------------------------------------
//
void catch_user_signal(void (*handler)(int))
{
  catch_signal(SIGUSR1, handler);
}

// ----------------------------------------------------------------------------
//
void core_dump()
{
  char *core_directory = getenv("CORE_DIRECTORY");
  if (core_directory)
    chdir(core_directory);
  abort();
}

// ----------------------------------------------------------------------------
//
static List wait_pids;		// state variable
static void try_waiting()
{
  int status;
  List pids = wait_pids;
  for (int k = 0 ; k < pids.size() ; ++k)
    {
      int pid = (int) (long) pids[k];
      if (waitpid(pid, &status, WNOHANG) == pid)
	wait_pids.erase(pids[k]);
    }
}

// ----------------------------------------------------------------------------
// Start a sub process with bidirectional I/O.
// The client should close the streams when they are done.
// If the streams are not needed NULL arguments can be provided
// and stdin and stdout of the subprocess will be left connected
// to stdin and stdout for this process.
//
bool start_process(const Stringy &command,
		   FILE **to_process, FILE **from_process)
{
  const int READ = 0;
  const int WRITE = 1;

  int to_desc[2], from_desc[2];

  if (to_process && pipe(to_desc) != 0)
    return false;
  if (from_process && pipe(from_desc) != 0)
    {
      if (to_process)
	{ close(to_desc[READ]); close(to_desc[WRITE]); }
      return false;
    }

  pid_t pid = fork();

  if (pid == -1)
    {
      if (to_process)
	{ close(to_desc[READ]); close(to_desc[WRITE]); }
      if (from_process)
	{ close(from_desc[READ]); close(from_desc[WRITE]); }
      return false;
    }
  
  if (pid == 0)
    {
      if (to_process)
	{
	  close(to_desc[WRITE]);
	  dup2(to_desc[READ], fileno(stdin));
	}
      if (from_process)
	{
	  close(from_desc[READ]);
	  dup2(from_desc[WRITE], fileno(stdout));
	}
      const char *argv[] = {"/bin/csh", "-c", command.cstring(), NULL};
      execv(argv[0], (char * const *) argv);
      perror("start_process()");
      fprintf(stderr, "start_process(): Couldn't start %s\n",
	      command.cstring());
      exit(3);					// exec() failed.
    }

  if (to_process)
    {
      close(to_desc[READ]);
      *to_process = fdopen(to_desc[WRITE], "w");
    }

  if (from_process)
    {
      close(from_desc[WRITE]);
      *from_process = fdopen(from_desc[READ], "r");
    }

  wait_pids.append((void *) pid);
  try_waiting();

  return true;
}

// ----------------------------------------------------------------------------
//
Stringy default_print_command()
{
  Stringy cmd = "lpr";
  const char *pname = getenv("PRINTER");
  if (pname)
    cmd << formatted_string(" -P%s", pname);

  return cmd;
}

// ----------------------------------------------------------------------------
//
bool start_netscape(const Stringy &url)
{
  Stringy command = formatted_string("netscape %s", url.cstring());
  command = backslashify_characters(command, "#");
  return start_process(command);
}
