// ----------------------------------------------------------------------------
// UCSF NMR data file format
//
// This file format was developped for a processing program call Striker
// written by Mark Day at UCSF, 1990-93.  There are 3 versions of the file
// format.  Most of the file header parameters describe how the data was
// processed and are not used by Sparky.  This file contains alot of
// superfluous junk pertaining to those unused header parameters.
//
#include <string.h>		// Use memset()

#include "binaryIO.h"		// Use read_f32(), ...
#include "blockfile.h"		// Use Block_File
#include "memalloc.h"		// use new()
#include "nmrdata.h"		// Use NMR_Data
#include "num.h"		// Use round(), min(), max()
#include "spoint.h"		// Use IPoint, IRegion
#include "system.h"		// Use file_owner(), get_username()
#include "ucsffile.h"		// Use is_ucsf_nmr_file()
#include "utility.h"		// Use fatal_error()

// ----------------------------------------------------------------------------
// File versions.
//
#define NMR_UNKNOWN	-1
#define NMR_CLASSIC	0
#define NMR_NEW		1
#define NMR_NEWER	2

// ----------------------------------------------------------------------------
// The version 2 file format has a 180 byte header followed by one 128 byte
// header for each axis, followed by the data matrix in block format.
//
#define NMR_HEADER_IO_SIZE ((size_t) 180)
#define NMR_AXIS_IO_SIZE ((size_t) 128)
#define NMR_HEADER_V0_IO_SIZE ((size_t) 2048)

#define REC_SIZE	512
#define MAXBLOCK	32768

/*
 * return codes for functions that calculate the window premultiplication
 * factors.
 */
#define APO_SINE	1
#define APO_SQUARE 	2
#define APO_EXPONENTIAL	3
#define APO_GAUSS	4
#define APO_NONE	5
#define APO_CONVOLUTION	6
#define APO_CUSTOM	7
#define APO_EXTERNAL	10

// ----------------------------------------------------------------------------
// Version 2 data structures.
//

/*
 * data structure for solvent removal via convolution of fid.
 */
typedef struct {
	short		width;		/* half width of window */
	short		extrapolation;	/* npoints at ends to extrapolate */
} NMR_CONVOLUTION;

/*
 * parameters for apodization.
 */
typedef struct {
	union {
		float	shift;
		float	line_broad;
	}	p1;
	union {
		float	gaussian;
	}	p2;
} NMR_APO_PARAMS;

/*
 * parameters for polynomial fitting of transformed baseline.
 */
typedef struct {
	unsigned	solvent_start;
	short		poly_order;
	short		solvent_width;
} NMR_BASE_FIT;

/*
 * region for calculating offest of transformed baseline.
 */
typedef struct {
	unsigned	start, end;
} NMR_BASE_OFFSET;

/*
 * Processing flags / small bitfields
 */
typedef struct {
	unsigned	fid_convolution : 2;
	unsigned	dc_offset	: 1;
	unsigned	forward_extend	: 1;
	unsigned	backwards_extend: 1;
	unsigned	replace		: 2;
	unsigned	apodization	: 4;
	unsigned	ft_code		: 2;
	unsigned	nfills		: 4;
	unsigned	absolute_value	: 1;
	unsigned	spectrum_reverse: 1;
	unsigned	baseline_offset	: 1;
	unsigned	baseline_fit	: 2;
	unsigned	reserved	: 10;
} NMR_FLAG;

/*
 * results flag.
 */
typedef struct {
	unsigned	transformed	: 1;
	unsigned	base_corrected	: 1;
	unsigned	reserved	: 14;
} NMR_PROCESSED;

#define LP_REPLACE_NONE		0
#define LP_REPLACE_BEFORE	1
#define LP_REPLACE_AFTER	2

typedef struct {
	unsigned	start;		/* 1st pt. in coeffs. calculation */
	unsigned short	poly_order;	/* polynomial order */
	unsigned short	npredicted;	/* # of additional points */
	unsigned short	npoints;	/* # of point in coeffs region */
} LP_EXTEND;

typedef struct {
	unsigned	before_start;	/* 1st pt. in downfield coeffs. calc. */
	unsigned	after_start;	/* 1st pt. in upfield coeffs calc. */
	unsigned	first;		/* first pt to be replaced */
	unsigned short	npredicted;	/* length of region replacement. */
	unsigned short	poly_order;	/* polynomial order */
	unsigned short	before_npoints;	/* # of point in coeffs region */
	unsigned short	after_npoints;	/* # of point in coeffs region */
} LP_REPLACE;

/*
 * Data structures for version 2 files.
 */
typedef struct {
	char		nucleus[6];
	short		spectral_shift;		/* to left or right shift */
	unsigned	npoints;		/* # of active data points */
	unsigned	size;			/* total size of axis */
	unsigned	bsize;			// # of points per cache block
	float		spectrometer_freq;	// MHz
	float		spectral_width;		// Hz
	float		xmtr_freq;		/* transmitter offset (ppm) */
	float		zero_order;		/* phase corrections */
	float		first_order;
	float		first_pt_scale;		/* scaling for first point */
	NMR_PROCESSED	status;			/* completion flags */
	NMR_FLAG	flags;			/* processing options */
	NMR_CONVOLUTION	conv;			/* FID convolution parameters */
	NMR_APO_PARAMS	apo;			/* apodization parameters */
	LP_EXTEND	forward;		/* FID extension */
	LP_EXTEND	backwards;		/* FID beginning correction */
	LP_REPLACE	replace;		/* FID replacement */
	NMR_BASE_OFFSET	base_offset;		/* baseline offset correction */
	NMR_BASE_FIT	base_fit;
	void		*unused;
} NMR_AXIS;

typedef struct {
	char		ident[10];
	char		naxis;
	char		ncomponents;
	char		encoding;
	char		version;
	char		owner[9];
	char		date[26];
	char		comment[80];
	long		seek_pos;
	char		scratch[40];
	NMR_AXIS	*axis;
} NMR_HEADER;

typedef struct {
	char		*fname;
	NMR_HEADER	*hdr;
} NMR_FILE;

// ----------------------------------------------------------------------------
// Version 0 data structures.
//
typedef struct {
	float spect_width;	/* spectral width (sw)*/
	float carrier_ref;	/* carrier reference (xr)*/
	float window_code;	/* code for window function (iwndw)*/
	float shift;		/* shift value for sine bell (sh)*/
	float lb;		/* line broadening (in Hz.) for exp. mult. */
	float gn;		/* gn value for gm window? (gn)*/
	float n_zero_fills;	/* number of times zero filled (nzf)*/
	float nterms;		/* number of terms in polynomial for baseline */
	float zero_order;	/* zero order phase correction (pa)*/
	float first_order;	/* first order phase correction (pb)*/
	float size;		/* unused */
	float offset_correct;	/* code for offset correction (0 or 1) (ibc)*/
	float ft_code;		/* code for transform; 2 complex, 1 real */
	float spectrum_reverse; /* code for spectrum reverse (0 or 1) (isr)*/
	float start_point;	/* start data point for operation (ifx)*/
	float end_point;	/* end data point for current operation (ilx)*/
	float abs_val_code;	/* code for absolute value calculation (iav)*/
	float car_index;	/* carrier postion in data points (icar)*/
	float ridge_subtract;	/* code for t1 ridge subtraction (irs)*/
	float baseline_correct;	/* code for baseline correction (0 or 1) (iba)*/
} NMR_AXIS_V0;

typedef struct {
	float left_shift[2];	/* Amount to left shift FID */
	float right_shift[2];	/* Amount to right shift FID */
} EXTRA_PARAMS_V0;

typedef struct {
	float nxpoints;		/* number of data points in x dimension  (nxh)*/
	float nypoints;		/* number of data points in y dimension  (nyh)*/ 
	float t1t2; 		/* current domain (t1 or t2) (it1t2)*/
	float sfreq;		/* spectrometer frequency (sf) */
	float carrier_pos;	/* carrier position (car)*/
	float processed[2];	/* true if axis already processed. */
	float junk1[3];		/* unused */
	NMR_AXIS_V0 w1;		/* w1 dimension parameters */
	NMR_AXIS_V0 w2;		/* w2 dimension parameters */
	EXTRA_PARAMS_V0 xp;	/* additional parameters for axis */
	float junk2[74];	/* unused */
	char  misc[1536];	/* miscellaneous additional info */
} NMR_HEADER_V0;


// ----------------------------------------------------------------------------
// Version 1 data structures.
//

/*
 * data structure for solvent removal via convolution of fid.
 */
typedef struct {
	unsigned	type;
	short		width;		/* half width of window */
	short		extrapolation;	/* npoints at ends to extrapolate */
} NMR_CONV_V1;

/*
 * Data structures for new format files.
 */
typedef struct {
	char		nucleus[6];
	unsigned	npoints;		/* # of active data points */
	unsigned	bsize;			/* # of points per cache block*/
	float		spectrometer_freq;
	float		spectral_width;
	float		xmtr_freq;
	unsigned	transformed;		/* true if axis transformed */
	unsigned	nfills;			/* zero fills */
	unsigned	apodization;		/* code for apodization fn. */
	float		shift;			/* sine (squared) shift */
	float		line_broad;
	float		gn;			/* gaussian multiplier */
	float		zero_order;		/* phas corrections */
	float		first_order;
	unsigned	abs_value;		/* true for abs val mode */
	unsigned	baseline_correct;
	unsigned	offset_correct;
	unsigned	ft_code;		/* 1 real, 2 if complex */
	unsigned	ridge_subtract;
	unsigned	spectrum_reverse;
	int		spectral_shift;		/* to left or right shift */
	unsigned	size;			/* total size of axis */
	void		*unused;
	NMR_CONV_V1	conv;
	char		scratch[24];		/* for future use */
} NMR_AXIS_V1;

// ----------------------------------------------------------------------------
//
static IPoint nmr_data_size(NMR_FILE *nmrfp);
static IPoint nmr_block_size(NMR_FILE *nmrfp);
static SPoint nmr_spectrometer_frequency(NMR_FILE *nmrfp);
static SPoint nmr_spectral_width(NMR_FILE *nmrfp);
static IPoint nmr_data_ipoint(NMR_FILE *nmrfp,
			      int (*fn)(const NMR_FILE *, int a));
static SPoint nmr_data_spoint(NMR_FILE *nmrfp,
			      float (*fn)(const NMR_FILE *, int a));
static SPoint nmr_origin_ppm(NMR_FILE *nmrfp);
static List nmr_axis_labels(NMR_FILE *nmrfp);
static int nmr_data_start(NMR_FILE *nmrfp);

static int read_version(FILE *fp);
static bool nmr_file_dimension(const Stringy &path, int *dim, int dimarray[]);
static NMR_FILE *nmr_open(const Stringy &path);
static bool nmr_header_read(NMR_FILE *fp, FILE *);
static long nmr_seekpos(NMR_HEADER *hdr);
static int nmr_file_getaxis(NMR_HEADER *hdr);
static NMR_HEADER *nmr_file_header(const NMR_FILE *fp);
static int nmr_file_version(const NMR_FILE *fp);
static const char *nmr_file_path(NMR_FILE *fp);
static int nmr_file_naxes(const NMR_FILE *fp);
static NMR_AXIS *nmr_file_axis(const NMR_FILE *fp, int axis);
static int nmr_axis_size(const NMR_FILE *fp, int axis);
static const char *nmr_axis_nucleus(const NMR_FILE *fp, int axis);
static float nmr_axis_spectrometer_frequency(const NMR_FILE *fp, int axis);
static float nmr_axis_transmitter_frequency(const NMR_FILE *fp, int axis);
static float nmr_axis_spectral_width(const NMR_FILE *fp, int axis);
static int nmr_block_volume(NMR_FILE *fp);
static int nmr_block_size(const NMR_FILE *fp, int axis);
static int nmr_file_ncomponents(NMR_FILE *fp);
static void nmr_fclose(NMR_FILE *fp);
static NMR_FILE *nmr_fcreate(const char *fname, const NMR_HEADER *in_hdr);
static void nmr_header_copy(const NMR_HEADER *in, NMR_HEADER *out);
static void nmr_file_resize(NMR_FILE *nmrfp);
static bool nmr_header_write(NMR_FILE *nmrfp, NMR_HEADER *hdr, FILE *);
static void nmr_header_print(NMR_FILE *f, FILE *out);
static void initialize_ucsf_header(NMR_Data *nmr_data, NMR_HEADER *hucsf);
static bool write_nmr_data(NMR_Data *nmr_data, Block_File *bf,
			   Stringy *err_msg);
static bool change_ucsf_header_value(const Stringy &path, int axis,
				     int offset, float value);

// ----------------------------------------------------------------------------
//  Routines for reading version 2 nmr data files.
//
static int read_nmr_header(NMR_HEADER *hdr, FILE *fp);
static int read_nmr_axis(NMR_AXIS *axis, FILE *fp);
static int read_nmr_axes(NMR_AXIS *axes, size_t num, FILE *fp);
static int read_nmr_processed(NMR_PROCESSED *p, FILE *fp);
static int read_nmr_flag(NMR_FLAG *flags, FILE *fp);
static int read_nmr_convolution(NMR_CONVOLUTION *conv, FILE *fp);
static int read_nmr_apo_params(NMR_APO_PARAMS *apo, FILE *fp);
static int read_lp_extend(LP_EXTEND *lpe, FILE *fp);
static int read_lp_replace(LP_REPLACE *lpr, FILE *fp);
static int read_nmr_base_offset(NMR_BASE_OFFSET *bo, FILE *fp);
static int read_nmr_base_fit(NMR_BASE_FIT *bf, FILE *fp);

// ----------------------------------------------------------------------------
//  Routines for reading older version 1 nmr files.
//
static int read_nmr_axis_v1(NMR_AXIS_V1 *axis, FILE *fp);
static int read_nmr_axes_v1(NMR_AXIS_V1 *axes, size_t num, FILE *fp);
static void convert_axes(NMR_AXIS_V1 *in, NMR_AXIS *out, int naxes);
static int read_nmr_conv_v1(NMR_CONV_V1 *conv, FILE *fp);

// ----------------------------------------------------------------------------
//  Routines for reading older version 0 nmr files.
//
static int read_nmr_header_v0(NMR_HEADER_V0 *hv0, FILE *fp);
static int read_nmr_axis_v0(NMR_AXIS_V0 *aparams, FILE *fp);
static int read_extra_params_v0(EXTRA_PARAMS_V0 *eparams, FILE *fp);

/* ----------------------------------------------------------------------------
*  Routines for writing version 2 nmr data files.
*/
static int write_nmr_axis(NMR_AXIS *axis, FILE *fp);
static int write_nmr_processed(NMR_PROCESSED *p, FILE *fp);
static int write_nmr_flag(NMR_FLAG *flags, FILE *fp);
static int write_nmr_convolution(NMR_CONVOLUTION *conv, FILE *fp);
static int write_nmr_apo_params(NMR_APO_PARAMS *apo, FILE *fp);
static int write_lp_extend(LP_EXTEND *lpe, FILE *fp);
static int write_lp_replace(LP_REPLACE *lpr, FILE *fp);
static int write_nmr_base_offset(NMR_BASE_OFFSET *bo, FILE *fp);
static int write_nmr_base_fit(NMR_BASE_FIT *bf, FILE *fp);

static int write_nmr_header(NMR_HEADER *hdr, FILE *fp);
static int write_nmr_axes(NMR_AXIS *axes, size_t num, FILE *fp);

// ----------------------------------------------------------------------------
//
NMR_Data *ucsf_nmr_data(const Stringy &path, Memory_Cache *mcache)
{
  if (!is_ucsf_nmr_data(path))
    return NULL;

  NMR_FILE *nmrfp = nmr_open(path);
  if (!nmrfp || nmr_file_ncomponents(nmrfp) != 1)
    return NULL;

  IPoint size = nmr_data_size(nmrfp);
  IPoint block_size = nmr_block_size(nmrfp);
  SPoint spectrometer_freq = nmr_spectrometer_frequency(nmrfp);
  SPoint spectral_width = nmr_spectral_width(nmrfp);
  SPoint origin_ppm = nmr_origin_ppm(nmrfp);
  List axis_labels = nmr_axis_labels(nmrfp);
  int data_offset = nmr_data_start(nmrfp);

  Block_File *bf = packed_block_file(path, "rb", size, block_size,
				     data_offset, true, mcache);
  NMR_Data *nmrdata = NULL;
  if (bf)
    nmrdata = block_file_nmr_data(bf, spectrometer_freq, spectral_width,
				  origin_ppm, axis_labels);

  free_string_list_entries(axis_labels);

  nmr_fclose(nmrfp);

  return nmrdata;
}

// ----------------------------------------------------------------------------
//
bool write_ucsf_nmr_data(NMR_Data *nmr_data, const Stringy &path,
			 Stringy *err_msg, Memory_Cache *mcache)
{
  NMR_HEADER hucsf;
  initialize_ucsf_header(nmr_data, &hucsf);

  NMR_FILE *nmrfp = nmr_fcreate(path.cstring(), &hucsf);
  if (!nmrfp)
    { *err_msg = "Couldn't create UCSF file."; return false; }

  //
  // Mode is rb+ allowing reading and writing because cache mechanism
  // reads a block before modifying it so partial writes can be done.
  // Mode wb+ won't work because current file will be truncated to zero
  // length losing the header.  Mode ab+ won't work because fseek() to
  // position 0 then puts you at the end of file.
  //
  Block_File *bf = packed_block_file(path, "rb+", nmr_data_size(nmrfp),
				     nmr_block_size(nmrfp),
				     nmr_data_start(nmrfp), true,
				     mcache);
  if (bf == NULL)
    { *err_msg = "Failed opening UCSF file."; return false; }

  bf->write_zeros();

  bool written = write_nmr_data(nmr_data, bf, err_msg);

  nmr_fclose(nmrfp);
  delete bf;

  return written;
}

// ----------------------------------------------------------------------------
// Below is the old UCSF file format code.
//

#define T1	0
#define T2	1
#define MAX_DIMEN	6

// ----------------------------------------------------------------------------
//
static IPoint nmr_data_size(NMR_FILE *nmrfp)
  { return nmr_data_ipoint(nmrfp, nmr_axis_size); }
static IPoint nmr_block_size(NMR_FILE *nmrfp)
  { return nmr_data_ipoint(nmrfp, nmr_block_size); }
static SPoint nmr_spectrometer_frequency(NMR_FILE *nmrfp)
  { return nmr_data_spoint(nmrfp, nmr_axis_spectrometer_frequency); }
static SPoint nmr_spectral_width(NMR_FILE *nmrfp)
  { return nmr_data_spoint(nmrfp, nmr_axis_spectral_width); }

// ----------------------------------------------------------------------------
//
static IPoint nmr_data_ipoint(NMR_FILE *nmrfp,
			      int (*fn)(const NMR_FILE *, int axis))
{
  IPoint value(nmr_file_naxes(nmrfp));
  for (int a = 0 ; a < value.dimension() ; ++a)
    value[a] = fn(nmrfp, a);
  return value;
}

// ----------------------------------------------------------------------------
//
static SPoint nmr_data_spoint(NMR_FILE *nmrfp,
			      float (*fn)(const NMR_FILE *, int a))
{
  SPoint value(nmr_file_naxes(nmrfp));
  for (int a = 0 ; a < value.dimension() ; ++a)
    value[a] = fn(nmrfp, a);
  return value;
}

// ----------------------------------------------------------------------------
//
static SPoint nmr_origin_ppm(NMR_FILE *nmrfp)
{
  SPoint carrier_ppm = nmr_data_spoint(nmrfp, nmr_axis_transmitter_frequency);
  SPoint sfreq = nmr_spectrometer_frequency(nmrfp);
  SPoint swidth = nmr_spectral_width(nmrfp);
  SPoint origin_ppm(sfreq.dimension());
  for (int a = 0 ; a < origin_ppm.dimension() ; ++a)
    origin_ppm[a] = carrier_ppm[a] + .5 * swidth[a] / sfreq[a];
  return origin_ppm;
}

// ----------------------------------------------------------------------------
//
static List nmr_axis_labels(NMR_FILE *nmrfp)
{
  List axis_labels;
  for (int a = 0 ; a < nmr_file_naxes(nmrfp) ; ++a)
    axis_labels.append(new Stringy(nmr_axis_nucleus(nmrfp, a)));
  return axis_labels;
}

// ----------------------------------------------------------------------------
//
static int nmr_data_start(NMR_FILE *nmrfp)
{
  return NMR_HEADER_IO_SIZE + NMR_AXIS_IO_SIZE * nmr_file_naxes(nmrfp);
}

/*
 * Returns the version of the NMR file fp.
 */
static int read_version(FILE *fp)
{
	NMR_HEADER	hdr;
	NMR_HEADER_V0	hv0;

	/*
	 * first check to see if new format file.
	 */
	if (beginning_of_file(fp) &&
	    read_nmr_header(&hdr, fp) &&
	    hdr.ident[8] == '\0') {
		if (strcmp(hdr.ident, "UCSF NMR") == 0) {
			return hdr.version;
		}
	}

	/*
	 * Now check if old style by examining t1t2 value.
	 * Check that the IEEE float value is valid so that
	 * a floating point exception does not occur.
	 */
	float one = 1, two = 2;
	if (beginning_of_file(fp) &&
	    read_nmr_header_v0(&hv0, fp) &&
	    (memcmp(&hv0.t1t2, &one, sizeof(float)) == 0 ||
	     memcmp(&hv0.t1t2, &two, sizeof(float)) == 0)) {
	  return NMR_CLASSIC;
	}

	return NMR_UNKNOWN;
}

/*
 * Return the dimensions of the nmr file
 */
static bool nmr_file_dimension(const Stringy &path, int *dim, int dimarray[])
{
	FILE *fp = fopen(path.cstring(), "rb");		// NMR data file
	if (fp == NULL)
	  return false;

	int		version = read_version(fp);
	NMR_HEADER_V0	hv0;
	NMR_HEADER	hdr;
	NMR_AXIS	axis;
	int		i, naxes;
	bool		success = false;

	/*
	 * NMR_CLASSIC files are 2D.
	 */
	if (version == NMR_CLASSIC) {
		if (beginning_of_file(fp) && read_nmr_header_v0(&hv0, fp)) {
		  dimarray[0] = (int) hv0.nxpoints;
		  dimarray[1] = (int) hv0.nypoints;
		  *dim = 2;
		  success = true;
		}
	}

	/*
	 * NMR_NEW files are variable length, with a header followed by
	 * a number of NMR_AXIS.
	 */
	else if (version == NMR_NEW || version == NMR_NEWER) {
	  
		if (beginning_of_file(fp) && read_nmr_header(&hdr, fp)) {
			naxes = hdr.naxis;
			for (i = 0; i < naxes; i++) {
			    if (read_nmr_axes(&axis, 1, fp)) {
					dimarray[i] = axis.npoints;
			    }
			    else
				dimarray[i] = 0;
			}
			*dim = naxes;
			success = true;
		}
	}

	fclose(fp);

	return success;
}

// ----------------------------------------------------------------------------
//
bool is_ucsf_nmr_data(const Stringy &path)
{
  int dim, sizes[DIM];
  return nmr_file_dimension(path, &dim, sizes);
}

/*
 * Open an existing UCSF nmr data file.
 * Returns NULL on failure.
 */
static NMR_FILE *nmr_open(const Stringy &path)
{
	/*
	 * open file.
	 */
	FILE *filep = fopen(path.cstring(), "rb");	// NMR data file
	if (filep == NULL)
		return NULL;

	NMR_FILE *fp = new NMR_FILE();
	fp->fname = allocate_character_array(path.cstring());

	/*
	 * allocate and read in header.
	 */
	fp->hdr = new NMR_HEADER;
	memset(fp->hdr, 0, sizeof(NMR_HEADER));
	bool header_ok = nmr_header_read(fp, filep);
	fclose(filep);

	if (!header_ok)
	  {
	    delete fp->hdr;
	    delete fp->fname;
	    delete fp;
	    return NULL;
	  }

	return fp;
}

/*
 * reads header information from data file.
 */
static bool nmr_header_read(NMR_FILE *fp, FILE *filep)
{
	int		i, naxes, version;
	char		*date;
	NMR_HEADER_V0	p;
	NMR_AXIS_V0	*a;
	NMR_AXIS	*b;
	NMR_AXIS_V1	*old_axis;
	NMR_HEADER	*hdr;
	const char	*fname;

	hdr = nmr_file_header(fp);
	fname = nmr_file_path(fp);
	version = read_version(filep);
	if (version == NMR_UNKNOWN) {
		warn("Unrecognizable file header in file %s\n", fname);
		return false;
	}
	hdr->version = version;

	if (!beginning_of_file(filep))
	  return false;

	switch (nmr_file_version(fp)) {

	/*
	 * old style file.  Must read in and convert the parameters.
	 */
	case NMR_CLASSIC:
	  {
		if (!read_nmr_header_v0(&p, filep)) {
			warn("Error reading: %s\n", fname);
			return false;
		}

		/*
		 * Convert old style parameters to new style header.
		 * bsize is appropriated to hold current axis orientation
		 * of "Classic" file.
		 */
		hdr->naxis = 2;
		hdr->ncomponents = 1;
		(void) strcpy(hdr->ident, "UCSF NMR");
		(void) strcpy(hdr->owner, file_owner(filep).cstring());
		time_t time = file_modification_time(fname);
		date = ctime(&time);
		strncpy(hdr->date, date, 24);
		hdr->date[24] = '\0';
		hdr->axis = new NMR_AXIS [2];
		memset(hdr->axis, 0, 2 * sizeof(NMR_AXIS));
		hdr->axis[T1].bsize = hdr->axis[T2].bsize = round(p.t1t2) - 1;
		if (p.t1t2 == 2.0) {
			hdr->axis[T1].npoints = round(p.nypoints);
			hdr->axis[T2].npoints = round(p.nxpoints);
		}
		else {
			hdr->axis[T1].npoints = round(p.nxpoints);
			hdr->axis[T2].npoints = round(p.nypoints);
		}

		/*
		 * fill in original axis sizes. May not be originally defined.
		 */
		if ((hdr->axis[T1].size = round(p.w1.size)) == 0.0)
			hdr->axis[T1].size = hdr->axis[T1].npoints;
		if ((hdr->axis[T2].size = round(p.w2.size)) == 0.0)
			hdr->axis[T2].size = hdr->axis[T2].npoints;


		/*
		 * fill in axis parameters.
		 */
		for (i = 0; i < 2; i++) {
			a = &p.w1 + i;
			b = &hdr->axis[i];
			(void) strcpy(b->nucleus, "H1");
			b->spectrometer_freq = p.sfreq;
			b->spectral_width = a->spect_width;
			b->xmtr_freq = a->carrier_ref;
			b->flags.apodization = (unsigned) a->window_code;
			b->apo.p1.shift = a->shift;
			b->apo.p1.line_broad = a->lb;
			b->apo.p2.gaussian = a->gn;
			b->flags.nfills = round(a->n_zero_fills);
			b->zero_order = a->zero_order;
			b->first_order = a->first_order;
			b->flags.dc_offset = round(a->offset_correct);
			b->flags.ft_code = round(a->ft_code);
			b->flags.spectrum_reverse = round(a->spectrum_reverse);
			b->flags.absolute_value = round(a->abs_val_code);
			b->flags.baseline_fit = round(a->baseline_correct);
			b->flags.baseline_offset = round(a->ridge_subtract);
			b->status.transformed = (unsigned) p.processed[i];
			b->spectral_shift = (short) (p.xp.right_shift[i] -
						     p.xp.left_shift[i]);
		}
		break;
	  }

	case NMR_NEWER:

		if (!read_nmr_header(hdr, filep)) {
			warn("Error reading: %s\n", fname);
			return false;
		}

		/*
		 * allocate memory and read in axis data.
		 */
		naxes = min(hdr->naxis, MAX_DIMEN);
		hdr->axis = new NMR_AXIS [naxes];
		if (!read_nmr_axes(hdr->axis, naxes, filep)) {
			warn("Error reading: %s\n", fname);
			delete [] hdr->axis;
			return false;
		}

		break;

	case NMR_NEW:

		if (!read_nmr_header(hdr, filep)) {
			warn("Error reading: %s\n", fname);
			return false;
		}

		/*
		 * allocate memory and read in axis data.
		 */
		naxes = min(hdr->naxis, MAX_DIMEN);
		hdr->axis = new NMR_AXIS [naxes];
		memset(hdr->axis, 0, naxes * sizeof(NMR_AXIS));
		old_axis = new NMR_AXIS_V1 [naxes];
		if (!read_nmr_axes_v1(old_axis, naxes, filep)) {
			warn("Error reading: %s\n", fname);
			delete [] hdr->axis;
			delete [] old_axis;
			return false;
		}

		/*
		 * convert between axis formats.
		 */
		convert_axes(old_axis, hdr->axis, naxes);
		hdr->version = NMR_NEWER;
		delete [] old_axis;

		/*
		 * Check to make sure older file.have "size" set correctly.
		 */
		for (i = 0; i < hdr->naxis; i++) {
			if (hdr->axis[i].size == 0)
				hdr->axis[i].size = hdr->axis[i].npoints;
		}
		break;
	}

	/*
	 * insure that older file.have seek_pos set properly
	 */
	if (hdr->seek_pos == 0)
		hdr->seek_pos = nmr_seekpos(hdr);

	/*
	 * make sure first point scaling factor is set.
	 */
	for (i = 0; i < hdr->naxis; i++) {
		if (hdr->axis[i].first_pt_scale == 0.)
			hdr->axis[i].first_pt_scale = 0.5;
	}

	return true;
}

/*
 * convert_axes(): Converts from version 1 to version 2 of NMR_AXIS structure.
 */
static void convert_axes(NMR_AXIS_V1 *in, NMR_AXIS *out, int naxes)
{
	int	i;

	for (i = 0; i < naxes; i++) {

		/*
		 * assign processing flags.
		 */
		out->flags.fid_convolution = in->conv.type;
		out->flags.dc_offset = in->offset_correct;
		out->flags.apodization = in->apodization;
		out->flags.ft_code = in->ft_code;
		out->flags.nfills = in->nfills;
		out->flags.absolute_value = in->abs_value;
		out->flags.spectrum_reverse = in->spectrum_reverse;
		out->flags.baseline_offset = in->ridge_subtract;
		out->flags.baseline_fit = in->baseline_correct;

		out->status.transformed = in->transformed;

		(void) strcpy(out->nucleus, in->nucleus);
		out->npoints = in->npoints;
		out->size = in->size;
		out->bsize = in->bsize;
		out->spectrometer_freq = in->spectrometer_freq;
		out->spectral_width = in->spectral_width;
		out->xmtr_freq = in->xmtr_freq;
		out->zero_order = in->zero_order;
		out->first_order = in->first_order;
		out->first_pt_scale = 0.5;
		out->spectral_shift = in->spectral_shift;
		out->conv.width = in->conv.width;
		out->conv.extrapolation = in->conv.extrapolation;

		switch (in->apodization) {
			case APO_SINE:
			case APO_SQUARE:
				out->apo.p1.shift = in->shift;
				break;

			case APO_EXPONENTIAL:
			case APO_CONVOLUTION:
				out->apo.p1.line_broad = in->line_broad;
				break;

			case APO_GAUSS:
				out->apo.p1.line_broad = in->line_broad;
				out->apo.p2.gaussian = in->gn;
				break;
		}
		in++;
		out++;
	}
}

static long nmr_seekpos(NMR_HEADER *hdr)
{
	NMR_AXIS	*a;
	int	nx, ny, nblocks;
	long		seekpos;

	if (hdr->version == NMR_CLASSIC) {

		if (nmr_file_getaxis(hdr) == T2) {
			nx = hdr->axis[T2].size;
			ny = hdr->axis[T1].size;
		}
		else {
			nx = hdr->axis[T1].size;
			ny = hdr->axis[T2].size;
		}
		seekpos = nx / REC_SIZE * REC_SIZE;
		if (nx % REC_SIZE)
			seekpos += REC_SIZE;
		seekpos *= ny * sizeof(float);
		seekpos += NMR_HEADER_V0_IO_SIZE;
	}

	else {
		seekpos = (int) sizeof(float) * hdr->ncomponents;

		/*
		 * Multiply by blocksize and number of blocks.
		 */
		for (int i = 0; i < hdr->naxis; i++) {
			a = &hdr->axis[i];
			seekpos *= a->bsize;
			nblocks = a->size / a->bsize;
			if (a->size % a->bsize)
				nblocks++;
			seekpos *= nblocks;
		}
		seekpos += NMR_HEADER_IO_SIZE + NMR_AXIS_IO_SIZE * hdr->naxis;
	}

	return seekpos;
}

/*
 * gets the orientation for file i/o.  This is only really appropriate for
 * old format files, but it will return naxis - 1 (which should be the
 * axis for collection in the time domain) for new format files.
 */
static int nmr_file_getaxis(NMR_HEADER *hdr)
{
	if (hdr->version != NMR_CLASSIC)
		return hdr->naxis - 1;

	return hdr->axis[0].bsize;
}

/*
 * access function to return pointer to header of NMR data file.
 */
static NMR_HEADER *nmr_file_header(const NMR_FILE *fp)
{
	return fp->hdr;
}

// ----------------------------------------------------------------------------
//
static int nmr_file_version(const NMR_FILE *fp)
{
  return nmr_file_header(fp)->version;
}

/*
 * access function to return name of nmr data file.
 */
static const char *nmr_file_path(NMR_FILE *fp)
{
	return fp->fname;
}

static int nmr_file_naxes(const NMR_FILE *fp)
{
	return fp->hdr->naxis;
}

/*
 * acess function to return pointer to header data for particular axis in
 * NMR data file.
 */
static NMR_AXIS *nmr_file_axis(const NMR_FILE *fp, int axis)
{
	return &fp->hdr->axis[axis];
}

/*
 * nmr_axis_size(): Returns the number of data points along axis.
 */
static int nmr_axis_size(const NMR_FILE *fp, int axis)
{
	return nmr_file_axis(fp, axis)->npoints;
}

/*
 * nmr_axis_nucleus(): Returns nucleus observed for axis.
 */
static const char *nmr_axis_nucleus(const NMR_FILE *fp, int axis)
{
  return (nmr_file_version(fp) != NMR_UNKNOWN &&
	  nmr_file_version(fp) != NMR_CLASSIC ?
	  nmr_file_axis(fp, axis)->nucleus :
	  "");
}

/*
 * nmr_axis_spectromter_frequency(): Returns spectrometer frequency of axis.
 */
static float nmr_axis_spectrometer_frequency(const NMR_FILE *fp, int axis)
{
	return nmr_file_axis(fp, axis)->spectrometer_freq;
}

/*
 * nmr_axis_transfmitter_frequency(): Returns transmitter frequency of 
 * center point of spectrum.
 */
static float nmr_axis_transmitter_frequency(const NMR_FILE *fp, int axis)
{
	return nmr_file_axis(fp, axis)->xmtr_freq;
}

/*
 * nmr_axis_spectral_width(): Returns spectral_width of axis.
 */
static float nmr_axis_spectral_width(const NMR_FILE *fp, int axis)
{
	return nmr_file_axis(fp, axis)->spectral_width;
}

/* ----------------------------------------------------------------------------
*/
static int nmr_block_volume(NMR_FILE *fp)
{
  int size = 1;
  for (int axis = 0; axis < nmr_file_naxes(fp) ; axis++)
    size *= nmr_block_size(fp, axis);

  return size;
}

/* ----------------------------------------------------------------------------
*/
static int nmr_block_size(const NMR_FILE *fp, int axis)
{
  return fp->hdr->axis[axis].bsize;
}

// ----------------------------------------------------------------------------
//
static int nmr_file_ncomponents(NMR_FILE *fp)
{
  return fp->hdr->ncomponents;
}


// ----------------------------------------------------------------------------
// Closes file and deallocates memory used for block cache.
//
static void nmr_fclose(NMR_FILE *fp)
{
	if (fp == NULL)
	  return;

	/*
	 * close files and  free memory used by both formats.
	 */
	delete fp->fname;		fp->fname = NULL;
	delete [] fp->hdr->axis;	fp->hdr->axis = NULL;
	delete fp->hdr;			fp->hdr = NULL;
	delete fp;
}

/*
 * nmr_fcreate() : Creates a n-dimensional spectra file.
 * Arguments passed:
 *	char *fname : name of file to create.
 * 	NMR_HEADER *in_hdr : header for file (THIS IS NOT MODIFIED)
 */
static NMR_FILE *nmr_fcreate(const char *fname, const NMR_HEADER *in_hdr)
{
	NMR_FILE	*fp;
	NMR_HEADER	*hdr;
	time_t		tm;


	FILE *filep = fopen(fname, "wb+");		// NMR data file
	if (filep == NULL) {
		fatal_error("Error creating %s", fname);
		return NULL;
	}
	fp = new NMR_FILE();
	fp->fname = allocate_character_array(fname);

	/*
	 * Allocate memory and copy header information.
	 */
	hdr = fp->hdr = new NMR_HEADER;
	hdr->axis = new NMR_AXIS [in_hdr->naxis];
	nmr_header_copy(in_hdr, hdr);
	strcpy(hdr->ident, "UCSF NMR");

	/*
	 * initialize owner of file.
	 */
	if (!strcmp(hdr->owner, ""))
		strcpy(fp->hdr->owner, get_username().cstring());

	/*
	 * initialize date if missing.
	 */
	if (!strcmp(hdr->date, "")) {
		tm = time((time_t *) NULL);
		strcpy(hdr->date, ctime(&tm));
		hdr->date[24] = '\0';
	}

	/*
	 * initialize nucleus if missing, and set original size to
	 * npoints in axis.
	 */
	for (int i = 0; i < hdr->naxis; i++) {
		if (!strcmp(hdr->axis[i].nucleus, ""))
			strcpy(hdr->axis[i].nucleus, "1H");
		hdr->axis[i].size = hdr->axis[i].npoints;
	}

	/*
	 * For old style file, don't need block caching structures.
	 */
	if (fp->hdr->version == NMR_CLASSIC) {
		return fp;
	}

	/*
	 * allocate memory for, and initialize block caching structure.
	 */
	nmr_file_resize(fp);

	nmr_header_write(fp, hdr, filep);
	fclose(filep);

	return fp;
}


/*
 * copys the header information from one file to another.
 */
static void nmr_header_copy(const NMR_HEADER *in, NMR_HEADER *out)
{
	NMR_AXIS	*tmp;

	tmp = out->axis;
	memcpy((char *) out, (const char *) in, sizeof *out);
	out->axis = tmp;
	memcpy((char *) out->axis, (const char *) in->axis,
	       sizeof(NMR_AXIS) * out->naxis);
}


/*
 * Resizes the caching blocks used for reading and
 * writing of slices to new format file.  This is necessary if file is
 * originally created without information on its eventual size.  This function
 * sets the blocks size properly after the header information has been filled
 * in.
 */
static void nmr_file_resize(NMR_FILE *fp)
{
	NMR_HEADER	*hdr;

	if (nmr_file_version(fp) == NMR_CLASSIC)
		return;

	/*
	 * calculate block size for new file.
	 */
	hdr = fp->hdr;
	for (int i = 0; i < hdr->naxis; i++)
		hdr->axis[i].bsize = max(hdr->axis[i].npoints, 1u);

	while (nmr_block_volume(fp) * sizeof(float) > MAXBLOCK)
		for (int i = 0; i < hdr->naxis; i++)
			hdr->axis[i].bsize = max(hdr->axis[i].bsize/2, 1u);
}


/*
 * write header to NMR_FILE
 */
static bool nmr_header_write(NMR_FILE *fp, NMR_HEADER *hdr, FILE *filep)
{
	int		i, naxes;
	const char	*fname;


	hdr->seek_pos = nmr_seekpos(hdr);
	fname = nmr_file_path(fp);
	naxes = (int) hdr->naxis;

	/*
	 * Check to make sure older file.have "size" set correctly.
	 */
	for (i = 0; i < naxes; i++) {
		if (hdr->axis[i].size == 0)
			hdr->axis[i].size = hdr->axis[i].npoints;
	}

	if (!beginning_of_file(filep))
	  return false;

	switch (hdr->version) {

	/*
	 * old style file.  Must convert the parameters.
	 */
	case NMR_CLASSIC:
		fatal_error("nmr_header_write(): "
			    "Old style UCSF format not supported.\n");
	case NMR_NEWER:
		if (!write_nmr_header(hdr, filep)) {
			fatal_error("Error writing %s\n", fname);
			return false;
		}

		/*
		 * write to disk
		 */
		if (!write_nmr_axes(hdr->axis, naxes, filep)) {
			fatal_error("Error writing %s\n", fname);
			return false;
		}

		break;

	case NMR_NEW:
		fatal_error("nmr_header_write(): "
			    "New style UCSF format not supported.\n");
	}
	return true;
}



/*
 * Creates formatted print out of header contents.
 */
#define NAXIS 3
#define CONV_NONE 0

static char *code_str[] =	{"Off", "On"};
static char *ft_str[]   =	{"None", "Real", "Complex", "Inverse"};
static char *status_str[]   =	{"No", "Yes"};
static char *domain_str[]   =	{"Time\t", "Frequency"};
static char *conv_str[]   =	{"None", "Sine Bell", "Gaussian"};
static char *replace_str[]   =	{"None", "One sided", "One Sided", "Two Sided"};
static char *apo_str[]  =	{ "None", "Sine", "Sine Square", "Exponential",
				  "Gaussian", "None", "Convolution", "",
				  "", "","Custom"};

static void nmr_header_print(NMR_FILE *f, FILE *out)
{
	int		i, j, nloops, naxis;
	int		shift, print;

	NMR_HEADER	*hdr = nmr_file_header(f);
	fprintf(out, "\nParameters for file:\t%s\n", nmr_file_path(f));
	fprintf(out, "Number of axes\t\t= %d\n", hdr->naxis);
	fprintf(out, "Number of components\t= %d\n", hdr->ncomponents);
	if (strcmp(hdr->owner, ""))
		fprintf(out, "Owner\t\t\t= %s\n", hdr->owner);
	if (strcmp(hdr->date, ""))
		fprintf(out, "Creation Date\t\t= %s\n", hdr->date);
	if (strcmp(hdr->comment, ""))
		fprintf(out, "Comment\t\t= %s\n", hdr->comment);

	nloops = hdr->naxis / NAXIS;
	if (hdr->naxis % NAXIS)
		nloops++;

	for (i = 0; i < nloops; i++) {
		naxis = min(hdr->naxis, i * NAXIS + NAXIS);
		fprintf(out, "\nAxis:\t\t");
		for (j = i * NAXIS; j < naxis; j++)
			fprintf(out, "\t\tT%d", j+1);
		fprintf(out, "\n");
		fprintf(out, "Nucleus:\t");
		for (j = i * NAXIS; j < naxis; j++)
			fprintf(out, "\t\t%s", hdr->axis[j].nucleus);
		fprintf(out, "\n");
		fprintf(out, "Number of Data Points:");
		for (j = i * NAXIS; j < naxis; j++)
			fprintf(out, "\t\t%d", hdr->axis[j].npoints);
		fprintf(out, "\n");
		fprintf(out, "Axis Domain:\t\t");
		for (j = i * NAXIS; j < naxis; j++) {
			fprintf(out, "\t%s",
			  domain_str[hdr->axis[j].status.transformed]);
		}
		fprintf(out, "\n");
		fprintf(out, "Axis Baseline Corrected:");
		for (j = i * NAXIS; j < naxis; j++) {
			fprintf(out, "\t%s\t",
			  status_str[hdr->axis[j].status.base_corrected]);
		}
		fprintf(out, "\n");
		fprintf(out, "Spectrometer Frequency:\t");
		for (j = i * NAXIS; j < naxis; j++)
			fprintf(out, "\t%-8.3f",
				hdr->axis[j].spectrometer_freq);
		fprintf(out, "\n");
		fprintf(out, "Spectral Width:\t\t");
		for (j = i * NAXIS; j < naxis; j++)
			fprintf(out, "\t%-8.3f", hdr->axis[j].spectral_width);
		fprintf(out, "\n");
		fprintf(out, "Transmitter Frequency:\t");
		for (j = i * NAXIS; j < naxis; j++)
			fprintf(out, "\t%-8.3f", hdr->axis[j].xmtr_freq);
		fprintf(out, "\n");
		fprintf(out, "\nTIME DOMAIN CALCULATIONS\n");
		shift = 0;
		for (j = i * NAXIS; j < naxis; j++)
			shift |= max(0, hdr->axis[j].spectral_shift);
		if (shift) {
			fprintf(out, "Right Shift:\t");
			for (j = i * NAXIS; j < naxis; j++) {
				shift = max(0, hdr->axis[j].spectral_shift);
				fprintf(out, "\t\t%4d", shift);
			}
			fprintf(out, "\n");
		}
		shift = 0;
		for (j = i * NAXIS; j < naxis; j++)
			shift |= min(0, hdr->axis[j].spectral_shift);
		if (shift) {
			fprintf(out, "Left Shift:\t");
			for (j = i * NAXIS; j < naxis; j++) {
				shift = min(0, hdr->axis[j].spectral_shift);
				shift *= -1;
				fprintf(out, "\t\t%4d", shift);
			}
			fprintf(out, "\n");
		}
		fprintf(out, "Time Domain Convolution:");
		for (j = i * NAXIS; j < naxis; j++)
			fprintf(out, "\t%-11s",
			  conv_str[hdr->axis[j].flags.fid_convolution]);
		fprintf(out, "\n");
		print = false;
		for (j = i * NAXIS; j < naxis; j++) {
			print |= (hdr->axis[j].flags.fid_convolution
			  != CONV_NONE);
		}
		if (print) {
			fprintf(out, "Convolution 1/2 Width:");
			for (j = i * NAXIS; j < naxis; j++)
				fprintf(out, "\t\t%4d",
					hdr->axis[j].conv.width);
			fprintf(out, "\n");
			fprintf(out, "Convolution Extrapolation:");
			for (j = i * NAXIS; j < naxis; j++)
				fprintf(out, "\t%4d\t",
					hdr->axis[j].conv.extrapolation);
			fprintf(out, "\n");
		}
		fprintf(out, "FID DC Offset Correction:");
		for (j = i * NAXIS; j < naxis; j++)
			fprintf(out, "\t%s\t",
			  code_str[hdr->axis[j].flags.dc_offset]);
		fprintf(out, "\n");

		fprintf(out, "Backwards Extension:");
		for (j = i * NAXIS; j < naxis; j++)
			fprintf(out, "\t\t%s",
			  code_str[hdr->axis[j].flags.backwards_extend]);
		fprintf(out, "\n");
		print = false;
		for (j = i * NAXIS; j < naxis; j++) {
			print |= (hdr->axis[j].flags.backwards_extend);
		}
		if (print) {
			fprintf(out, "  Frequency Components:");
			for (j = i * NAXIS; j < naxis; j++)
				fprintf(out, "\t\t%d",
					hdr->axis[j].backwards.poly_order);
			fprintf(out, "\n");
			fprintf(out, "  Additional Points:");
			for (j = i * NAXIS; j < naxis; j++)
				fprintf(out, "\t\t%d",
					hdr->axis[j].backwards.npredicted);
			fprintf(out, "\n");
			fprintf(out, "  Coefficient Region Start:");
			for (j = i * NAXIS; j < naxis; j++)
				fprintf(out, "\t%d\t",
					hdr->axis[j].backwards.start + 1);
			fprintf(out, "\n");
			fprintf(out, "  Coefficient Region Length:");
			for (j = i * NAXIS; j < naxis; j++)
				fprintf(out, "\t%d\t",
					hdr->axis[j].backwards.npoints);
			fprintf(out, "\n");
		}

		fprintf(out, "Forwards Extension:");
		for (j = i * NAXIS; j < naxis; j++)
			fprintf(out, "\t\t%s",
			  code_str[hdr->axis[j].flags.forward_extend]);
		fprintf(out, "\n");
		print = false;
		for (j = i * NAXIS; j < naxis; j++) {
			print |= (hdr->axis[j].flags.forward_extend);
		}
		if (print) {
			fprintf(out, "  Frequency Components:");
			for (j = i * NAXIS; j < naxis; j++)
				fprintf(out, "\t\t%d",
					hdr->axis[j].forward.poly_order);
			fprintf(out, "\n");
			fprintf(out, "  Additional Points:");
			for (j = i * NAXIS; j < naxis; j++)
				fprintf(out, "\t\t%d",
					hdr->axis[j].forward.npredicted);
			fprintf(out, "\n");
			fprintf(out, "  Coefficient Region Start:");
			for (j = i * NAXIS; j < naxis; j++)
				fprintf(out, "\t%d\t",
					hdr->axis[j].forward.start + 1);
			fprintf(out, "\n");
			fprintf(out, "  Coefficient Region Length:");
			for (j = i * NAXIS; j < naxis; j++)
				fprintf(out, "\t%d\t",
					hdr->axis[j].forward.npoints);
			fprintf(out, "\n");
		}

		fprintf(out, "Replacement:\t");
		for (j = i * NAXIS; j < naxis; j++)
			fprintf(out, "\t\t%s",
				replace_str[hdr->axis[j].flags.replace]);
		fprintf(out, "\n");
		print = false;
		for (j = i * NAXIS; j < naxis; j++) {
			print |= (hdr->axis[j].flags.replace);
		}
		if (print) {
			fprintf(out, "  Frequency Components:");
			for (j = i * NAXIS; j < naxis; j++)
				fprintf(out, "\t\t%d",
					hdr->axis[j].replace.poly_order);
			fprintf(out, "\n");
			fprintf(out, "  Before Region Start:");
			for (j = i * NAXIS; j < naxis; j++)
				fprintf(out, "\t\t%d",
				  hdr->axis[j].replace.before_start + 1);
			fprintf(out, "\n");
			fprintf(out, "  After Region Start:");
			for (j = i * NAXIS; j < naxis; j++)
				fprintf(out, "\t\t%d",
				  hdr->axis[j].replace.after_start + 1);
			fprintf(out, "\n");
			fprintf(out, "  Coefficient Region Length:");
			for (j = i * NAXIS; j < naxis; j++)
				fprintf(out, "\t%d\t",
				  hdr->axis[j].replace.before_npoints);
			fprintf(out, "\n");
		}

		fprintf(out, "Apodization Function:\t");
		for (j = i * NAXIS; j < naxis; j++)
			fprintf(out, "\t%-11s",
				apo_str[hdr->axis[j].flags.apodization]);
		fprintf(out, "\n");
		print = false;
		for (j = i * NAXIS; j < naxis; j++) {
			print |= (hdr->axis[j].flags.apodization == APO_SINE ||
			  hdr->axis[j].flags.apodization == APO_SQUARE);
		}
		if (print) {
			fprintf(out, "  Sine Shift:\t\t");
			for (j = i * NAXIS; j < naxis; j++)
				fprintf(out, "\t%-8.3f",
					hdr->axis[j].apo.p1.shift);
			fprintf(out, "\n");
		}
		print = false;
		for (j = i * NAXIS; j < naxis; j++) {
			print |= (hdr->axis[j].flags.apodization == APO_GAUSS ||
			  hdr->axis[j].flags.apodization == APO_EXPONENTIAL);
		}
		if (print) {
			fprintf(out, "  Line Broadening:\t");
			for (j = i * NAXIS; j < naxis; j++)
				fprintf(out, "\t%-8.3f",
					hdr->axis[j].apo.p1.line_broad);
			fprintf(out, "\n");
		}
		print = false;
		for (j = i * NAXIS; j < naxis; j++) {
			print |= (hdr->axis[j].flags.apodization == APO_GAUSS);
		}
		if (print) {
			fprintf(out, "  Gaussian Multiplier:\t");
			for (j = i * NAXIS; j < naxis; j++)
				fprintf(out, "\t%-8.3f",
					hdr->axis[j].apo.p2.gaussian);
			fprintf(out, "\n");
		}
		fprintf(out, "Fourier Transform Type:");
		for (j = i * NAXIS; j < naxis; j++)
			fprintf(out, "\t\t%s",
				ft_str[hdr->axis[j].flags.ft_code]);
		fprintf(out, "\n");
		print = false;
		for (j = i * NAXIS; j < naxis; j++) {
			print |= hdr->axis[j].flags.ft_code;
		}
		if (print) {
			fprintf(out, "  Number of Zero Fills:");
			for (j = i * NAXIS; j < naxis; j++)
				fprintf(out, "\t\t%d",
					hdr->axis[j].flags.nfills);
			fprintf(out, "\n");
		}
		fprintf(out, "\nFREQUENCY DOMAIN CALCULATIONS\n");
		fprintf(out, "Zero Order Phase Correction:");
		for (j = i * NAXIS; j < naxis; j++)
			fprintf(out, "\t%-8.3f", hdr->axis[j].zero_order);
		fprintf(out, "\n");
		fprintf(out, "First Order Phase Correction:");
		for (j = i * NAXIS; j < naxis; j++)
			fprintf(out, "\t%-8.3f", hdr->axis[j].first_order);
		fprintf(out, "\n");
		fprintf(out, "Absolute Value Mode:");
		for (j = i * NAXIS; j < naxis; j++)
			fprintf(out, "\t\t%s",
			  code_str[hdr->axis[j].flags.absolute_value]);
		fprintf(out, "\n");
		fprintf(out, "Spectrum Reverse:");
		for (j = i * NAXIS; j < naxis; j++)
			fprintf(out, "\t\t%s",
			  code_str[hdr->axis[j].flags.spectrum_reverse]);
		fprintf(out, "\n");
		fprintf(out, "Baseline Offset Correction:");
		for (j = i * NAXIS; j < naxis; j++)
			fprintf(out, "\t%s\t",
			  code_str[hdr->axis[j].flags.baseline_offset]);
		fprintf(out, "\n");
		fprintf(out, "Baseline Polynomial Fit:");
		for (j = i * NAXIS; j < naxis; j++)
			fprintf(out, "\t%s\t",
			  code_str[hdr->axis[j].flags.baseline_fit]);
		fprintf(out, "\n");
		print = false;
		for (j = i * NAXIS; j < naxis; j++) {
			print |= hdr->axis[j].flags.baseline_fit;
		}
		if (print) {
			fprintf(out, "  Polynomial Order:");
			for (j = i * NAXIS; j < naxis; j++)
				fprintf(out, "\t\t%d",
					hdr->axis[j].base_fit.poly_order);
			fprintf(out, "\n");
			fprintf(out, "  First Solvent Data Point:");
			for (j = i * NAXIS; j < naxis; j++) {
				if (hdr->axis[j].base_fit.solvent_width)
					fprintf(out, "\t%d\t",
					 hdr->axis[j].base_fit.solvent_start+1);
				else
					fprintf(out, "\tSingle Region");
			}
			fprintf(out, "\n");
			fprintf(out, "  Last Solvent Data Point:");
			for (j = i * NAXIS; j < naxis; j++) {
				if (hdr->axis[j].base_fit.solvent_width)
					fprintf(out, "\t%d\t",
					  hdr->axis[j].base_fit.solvent_start +
				          hdr->axis[j].base_fit.solvent_width);
				else
					fprintf(out, "\tSingle Region");
			}
			fprintf(out, "\n");
		}
	}
	fprintf(out, "\n");
}

// ----------------------------------------------------------------------------
//
static void initialize_ucsf_header(NMR_Data *nmr_data, NMR_HEADER *hucsf)
{
  memset(hucsf, 0, sizeof(NMR_HEADER));

  hucsf->naxis = (char) nmr_data->dimension();
	
  hucsf->axis = new NMR_AXIS [hucsf->naxis];
  memset(hucsf->axis, 0, hucsf->naxis * sizeof(NMR_AXIS));
	
  hucsf->encoding = false;
	
  hucsf->version = (char) NMR_NEWER;

  hucsf->ncomponents = 1;		// Take only real component.

  time_t time = file_modification_time(nmr_data->path());
  strncpy(hucsf->date, ctime(&time), 24);
  hucsf->date[24] = '\0';

  /* Copy sizes, sweepwidths, xmtr offsets, nuclei,
   * spectrometer frequencies from nih file header
   */
  for (int a = 0 ; a < nmr_data->dimension() ; ++a)
    {
      int size = nmr_data->size()[a];

      hucsf->axis[a].size = hucsf->axis[a].npoints = size;
      hucsf->axis[a].spectral_width = nmr_data->spectrum_width()[a];

      // The xmtr frequency, also called carrier frequency, is in ppm.
      // Sparky assumes it is the ppm position of the center of the data.

      float sfreq = nmr_data->spectrometer_frequency()[a];
      float origin_ppm = nmr_data->origin_ppm()[a];
      float ppm_per_data_point = (nmr_data->spectrum_width()[a]/size) / sfreq;
      float ppm_middle = origin_ppm - ppm_per_data_point * size / 2.0;

      hucsf->axis[a].xmtr_freq = ppm_middle;
      hucsf->axis[a].spectrometer_freq = sfreq;
      strncpy(hucsf->axis[a].nucleus, nmr_data->axis_label(a).cstring(), 5);
      hucsf->axis[a].nucleus[5] = '\0';
      hucsf->axis[a].bsize = 1;
      hucsf->axis[a].status.transformed = 1;
    }
}

// ----------------------------------------------------------------------------
//
static bool write_nmr_data(NMR_Data *nmr_data, Block_File *bf,
			   Stringy *err_msg)
{
  int dim = nmr_data->dimension();
  IPoint block_size(dim);
  for (int a = 0 ; a < dim ; ++a)
    block_size[a] = max(nmr_data->block_size()[a], bf->block_size()[a]);

  int volume = block_size.product();
  float *values = new float [volume];

  IRegion bspace = block_space(bf->size(), block_size);
  Subarray_Iterator bi(bspace, bspace, true);
  for ( ; !bi.finished() ; bi.next())
    {
      IRegion r = block_region(bi.position(), block_size);
      if (!nmr_data->read_values(r, values))
	{ *err_msg = "Failed reading data."; return false; }
      if (!bf->write_values(r, values))
	{ *err_msg = "Failed writing UCSF data."; return false; }
    }

  delete [] values;

  return true;
}

// ----------------------------------------------------------------------------
//
bool print_ucsf_nmr_header(const Stringy &path, FILE *outfp)
{
  NMR_FILE *nmrfp = nmr_open(path.cstring());

  if (nmrfp)
    {
      nmr_header_print(nmrfp, outfp);
      nmr_fclose(nmrfp);
    }

  return nmrfp != NULL;
}

/* ----------------------------------------------------------------------------
*/
static int read_nmr_header(NMR_HEADER *hdr, FILE *fp)
{
  return (read_u8_array(hdr->ident, 10, fp)
	  && read_u8(&hdr->naxis, fp)
	  && read_u8(&hdr->ncomponents, fp)
	  && read_u8(&hdr->encoding, fp)
	  && read_u8(&hdr->version, fp)
	  && read_u8_array(hdr->owner, 9, fp)
	  && read_u8_array(hdr->date, 26, fp)
	  && read_u8_array(hdr->comment, 80, fp)
	  && read_padding(3, fp)		/* Structure padding. */
	  && read_s32(&hdr->seek_pos, fp)
	  && read_u8_array(hdr->scratch, 40, fp)
	  && (hdr->axis = (NMR_AXIS *) 0, read_padding(4, fp))
	  );
}

/* ----------------------------------------------------------------------------
*/
static int read_nmr_header_v0(NMR_HEADER_V0 *hv0, FILE *fp)
{
  return (read_f32(&hv0->nxpoints, fp)
	  && read_f32(&hv0->nypoints, fp)
	  && read_f32(&hv0->t1t2, fp)
	  && read_f32(&hv0->sfreq, fp)
	  && read_f32(&hv0->carrier_pos, fp)
	  && read_f32_array(hv0->processed, 2, fp)
	  && read_f32_array(hv0->junk1, 3, fp)
	  && read_nmr_axis_v0(&hv0->w1, fp)
	  && read_nmr_axis_v0(&hv0->w2, fp)
	  && read_extra_params_v0(&hv0->xp, fp)
	  && read_f32_array(hv0->junk2, 74, fp)
	  && read_u8_array(hv0->misc, 1536, fp)
	  );
}

/* ----------------------------------------------------------------------------
*/
static int read_nmr_axis_v0(NMR_AXIS_V0 *aparams, FILE *fp)
{
  return (read_f32(&aparams->spect_width, fp)
	  && read_f32(&aparams->carrier_ref, fp)
	  && read_f32(&aparams->window_code, fp)
	  && read_f32(&aparams->shift, fp)
	  && read_f32(&aparams->lb, fp)
	  && read_f32(&aparams->gn, fp)
	  && read_f32(&aparams->n_zero_fills, fp)
	  && read_f32(&aparams->nterms, fp)
	  && read_f32(&aparams->zero_order, fp)
	  && read_f32(&aparams->first_order, fp)
	  && read_f32(&aparams->size, fp)
	  && read_f32(&aparams->offset_correct, fp)
	  && read_f32(&aparams->ft_code, fp)
	  && read_f32(&aparams->spectrum_reverse, fp)
	  && read_f32(&aparams->start_point, fp)
	  && read_f32(&aparams->end_point, fp)
	  && read_f32(&aparams->abs_val_code, fp)
	  && read_f32(&aparams->car_index, fp)
	  && read_f32(&aparams->ridge_subtract, fp)
	  && read_f32(&aparams->baseline_correct, fp)
	  );
}

/* ----------------------------------------------------------------------------
*/
static int read_extra_params_v0(EXTRA_PARAMS_V0 *eparams, FILE *fp)
{
  return (read_f32_array(eparams->left_shift, 2, fp)
	  && read_f32_array(eparams->right_shift, 2, fp)
	  );
}

/* ----------------------------------------------------------------------------
*/
static int read_nmr_axes(NMR_AXIS *axes, size_t num, FILE *fp)
{
  int success;
  size_t k;

  success = 1;
  for (k = 0 ; k < num && success ; ++k)
    success = read_nmr_axis(&axes[k], fp);

  return success;
}

/* ----------------------------------------------------------------------------
*/
static int read_nmr_axis(NMR_AXIS *axis, FILE *fp)
{
  return (read_u8_array(axis->nucleus, 6, fp)
	  && read_s16(&axis->spectral_shift, fp)
	  && read_u32(&axis->npoints, fp)
	  && read_u32(&axis->size, fp)
	  && read_u32(&axis->bsize, fp)
	  && read_f32(&axis->spectrometer_freq, fp)
	  && read_f32(&axis->spectral_width, fp)
	  && read_f32(&axis->xmtr_freq, fp)
	  && read_f32(&axis->zero_order, fp)
	  && read_f32(&axis->first_order, fp)
	  && read_f32(&axis->first_pt_scale, fp)
	  && read_nmr_processed(&axis->status, fp)
	  && read_nmr_flag(&axis->flags, fp)
	  && read_nmr_convolution(&axis->conv, fp)
	  && read_nmr_apo_params(&axis->apo, fp)
	  && read_lp_extend(&axis->forward, fp)
	  && read_lp_extend(&axis->backwards, fp)
	  && read_lp_replace(&axis->replace, fp)
	  && read_nmr_base_offset(&axis->base_offset, fp)
	  && read_nmr_base_fit(&axis->base_fit, fp)
	  && (axis->unused = (void *) 0, read_padding(4, fp))
	  );
}

/* ----------------------------------------------------------------------------
*/
static int read_nmr_processed(NMR_PROCESSED *p, FILE *fp)
{
  bit_field bf;

  if (read_bit_field(bf, fp))
    {
      p->transformed = bit_field_bits(bf, 0, 0);
      p->base_corrected = bit_field_bits(bf, 1, 1);
      p->reserved = bit_field_bits(bf, 2, 15);

      return 1;
    }

  return 0;
}

/* ----------------------------------------------------------------------------
*/
static int read_nmr_flag(NMR_FLAG *flags, FILE *fp)
{
  bit_field bf;

  if (read_bit_field(bf, fp))
    {
      flags->fid_convolution = bit_field_bits(bf, 0, 1);
      flags->dc_offset = bit_field_bits(bf, 2, 2);
      flags->forward_extend = bit_field_bits(bf, 3, 3);
      flags->backwards_extend = bit_field_bits(bf, 4, 4);
      flags->replace = bit_field_bits(bf, 5, 6);
      flags->apodization = bit_field_bits(bf, 7, 10);
      flags->ft_code = bit_field_bits(bf, 11, 12);
      flags->nfills = bit_field_bits(bf, 13, 16);
      flags->absolute_value = bit_field_bits(bf, 17, 17);
      flags->spectrum_reverse = bit_field_bits(bf, 18, 18);
      flags->baseline_offset = bit_field_bits(bf, 19, 19);
      flags->baseline_fit = bit_field_bits(bf, 20, 21);
      flags->reserved = bit_field_bits(bf, 22, 31);

      return 1;
    }

  return 0;
}

/* ----------------------------------------------------------------------------
*/
static int read_nmr_convolution(NMR_CONVOLUTION *conv, FILE *fp)
{
  return (read_s16(&conv->width, fp)
	  && read_s16(&conv->extrapolation, fp)
	  );
}

/* ----------------------------------------------------------------------------
*/
static int read_nmr_apo_params(NMR_APO_PARAMS *apo, FILE *fp)
{
  return (read_f32(&apo->p1.shift, fp)
	  && read_f32(&apo->p2.gaussian, fp)
	  );
}

/* ----------------------------------------------------------------------------
*/
static int read_lp_extend(LP_EXTEND *lpe, FILE *fp)
{
  return (read_u32(&lpe->start, fp)
	  && read_u16(&lpe->poly_order, fp)
	  && read_u16(&lpe->npredicted, fp)
	  && read_u16(&lpe->npoints, fp)
	  && read_padding(2, fp)		/* Structure padding. */
	  );
}

/* ----------------------------------------------------------------------------
*/
static int read_lp_replace(LP_REPLACE *lpr, FILE *fp)
{
  return (read_u32(&lpr->before_start, fp)
	  && read_u32(&lpr->after_start, fp)
	  && read_u32(&lpr->first, fp)
	  && read_u16(&lpr->npredicted, fp)
	  && read_u16(&lpr->poly_order, fp)
	  && read_u16(&lpr->before_npoints, fp)
	  && read_u16(&lpr->after_npoints, fp)
	  );
}

/* ----------------------------------------------------------------------------
*/
static int read_nmr_base_offset(NMR_BASE_OFFSET *bo, FILE *fp)
{
  return (read_u32(&bo->start, fp)
	  && read_u32(&bo->end, fp)
	  );
}

/* ----------------------------------------------------------------------------
*/
static int read_nmr_base_fit(NMR_BASE_FIT *bf, FILE *fp)
{
  return (read_u32(&bf->solvent_start, fp)
	  && read_s16(&bf->poly_order, fp)
	  && read_s16(&bf->solvent_width, fp)
	  );
}

/* ----------------------------------------------------------------------------
*/
static int read_nmr_axes_v1(NMR_AXIS_V1 *axes, size_t num, FILE *fp)
{
  int success;
  size_t k;

  success = 1;
  for (k = 0 ; k < num && success ; ++k)
    success = read_nmr_axis_v1(&axes[k], fp);

  return success;
}

/* ----------------------------------------------------------------------------
*/
static int read_nmr_axis_v1(NMR_AXIS_V1 *axis, FILE *fp)
{
  return (read_u8_array(axis->nucleus, 6, fp)
	  && read_padding(2, fp)		/* Structure padding. */
	  && read_u32(&axis->npoints, fp)
	  && read_u32(&axis->bsize, fp)
	  && read_f32(&axis->spectrometer_freq, fp)
	  && read_f32(&axis->spectral_width, fp)
	  && read_f32(&axis->xmtr_freq, fp)
	  && read_u32(&axis->transformed, fp)
	  && read_u32(&axis->nfills, fp)
	  && read_u32(&axis->apodization, fp)
	  && read_f32(&axis->shift, fp)
	  && read_f32(&axis->line_broad, fp)
	  && read_f32(&axis->gn, fp)
	  && read_f32(&axis->zero_order, fp)
	  && read_f32(&axis->first_order, fp)
	  && read_u32(&axis->abs_value, fp)
	  && read_u32(&axis->baseline_correct, fp)
	  && read_u32(&axis->offset_correct, fp)
	  && read_u32(&axis->ft_code, fp)
	  && read_u32(&axis->ridge_subtract, fp)
	  && read_u32(&axis->spectrum_reverse, fp)
	  && read_s32(&axis->spectral_shift, fp)
	  && read_u32(&axis->size, fp)
	  && (axis->unused = (void *) 0, read_padding(4, fp))
	  && read_nmr_conv_v1(&axis->conv, fp)
	  && read_u8_array(axis->scratch, 24, fp)
	  );
}

/* ----------------------------------------------------------------------------
*/
static int read_nmr_conv_v1(NMR_CONV_V1 *conv, FILE *fp)
{
  return (read_u32(&conv->type, fp)
	  && read_s16(&conv->width, fp)
	  && read_s16(&conv->extrapolation, fp)
	  );
}

/* ----------------------------------------------------------------------------
*/
static int write_nmr_header(NMR_HEADER *hdr, FILE *fp)
{
  return (write_u8_array(hdr->ident, 10, fp)
	  && write_u8(hdr->naxis, fp)
	  && write_u8(hdr->ncomponents, fp)
	  && write_u8(hdr->encoding, fp)
	  && write_u8(hdr->version, fp)
	  && write_u8_array(hdr->owner, 9, fp)
	  && write_u8_array(hdr->date, 26, fp)
	  && write_u8_array(hdr->comment, 80, fp)
	  && write_padding(3, fp)		/* Structure padding. */
	  && write_s32(hdr->seek_pos, fp)
	  && write_u8_array(hdr->scratch, 40, fp)
	  && write_padding(4, fp)		/* NMR_AXIS *hdr->axis */
	  );
}

/* ----------------------------------------------------------------------------
*/
static int write_nmr_axes(NMR_AXIS *axes, size_t num, FILE *fp)
{
  int success;
  size_t k;

  success = 1;
  for (k = 0 ; k < num && success ; ++k)
    success = write_nmr_axis(&axes[k], fp);

  return success;
}

/* ----------------------------------------------------------------------------
*/
static int write_nmr_axis(NMR_AXIS *axis, FILE *fp)
{
  return (write_u8_array(axis->nucleus, 6, fp)
	  && write_s16(axis->spectral_shift, fp)
	  && write_u32(axis->npoints, fp)
	  && write_u32(axis->size, fp)
	  && write_u32(axis->bsize, fp)
	  && write_f32(axis->spectrometer_freq, fp)
	  && write_f32(axis->spectral_width, fp)
	  && write_f32(axis->xmtr_freq, fp)
	  && write_f32(axis->zero_order, fp)
	  && write_f32(axis->first_order, fp)
	  && write_f32(axis->first_pt_scale, fp)
	  && write_nmr_processed(&axis->status, fp)
	  && write_nmr_flag(&axis->flags, fp)
	  && write_nmr_convolution(&axis->conv, fp)
	  && write_nmr_apo_params(&axis->apo, fp)
	  && write_lp_extend(&axis->forward, fp)
	  && write_lp_extend(&axis->backwards, fp)
	  && write_lp_replace(&axis->replace, fp)
	  && write_nmr_base_offset(&axis->base_offset, fp)
	  && write_nmr_base_fit(&axis->base_fit, fp)
	  && write_padding(4, fp)		/* void *axis->unused */
	  );
}

/* ----------------------------------------------------------------------------
*/
static int write_nmr_processed(NMR_PROCESSED *p, FILE *fp)
{
  bit_field bf;

  zero_bit_field(bf);

  set_bit_field_bits(bf, 0, 0, p->transformed);
  set_bit_field_bits(bf, 1, 1, p->base_corrected);
  set_bit_field_bits(bf, 2, 15, p->reserved);

  return write_bit_field(bf, fp);
}

/* ----------------------------------------------------------------------------
*/
static int write_nmr_flag(NMR_FLAG *flags, FILE *fp)
{
  bit_field bf;

  zero_bit_field(bf);

  set_bit_field_bits(bf, 0, 1, flags->fid_convolution);
  set_bit_field_bits(bf, 2, 2, flags->dc_offset);
  set_bit_field_bits(bf, 3, 3, flags->forward_extend);
  set_bit_field_bits(bf, 4, 4, flags->backwards_extend);
  set_bit_field_bits(bf, 5, 6, flags->replace);
  set_bit_field_bits(bf, 7, 10, flags->apodization);
  set_bit_field_bits(bf, 11, 12, flags->ft_code);
  set_bit_field_bits(bf, 13, 16, flags->nfills);
  set_bit_field_bits(bf, 17, 17, flags->absolute_value);
  set_bit_field_bits(bf, 18, 18, flags->spectrum_reverse);
  set_bit_field_bits(bf, 19, 19, flags->baseline_offset);
  set_bit_field_bits(bf, 20, 21, flags->baseline_fit);
  set_bit_field_bits(bf, 22, 31, flags->reserved);

  return write_bit_field(bf, fp);
}

/* ----------------------------------------------------------------------------
*/
static int write_nmr_convolution(NMR_CONVOLUTION *conv, FILE *fp)
{
  return (write_s16(conv->width, fp)
	  && write_s16(conv->extrapolation, fp)
	  );
}

/* ----------------------------------------------------------------------------
*/
static int write_nmr_apo_params(NMR_APO_PARAMS *apo, FILE *fp)
{
  return (write_f32(apo->p1.shift, fp)
	  && write_f32(apo->p2.gaussian, fp)
	  );
}

/* ----------------------------------------------------------------------------
*/
static int write_lp_extend(LP_EXTEND *lpe, FILE *fp)
{
  return (write_u32(lpe->start, fp)
	  && write_u16(lpe->poly_order, fp)
	  && write_u16(lpe->npredicted, fp)
	  && write_u16(lpe->npoints, fp)
	  && write_padding(2, fp)		/* Structure padding. */
	  );
}

/* ----------------------------------------------------------------------------
*/
static int write_lp_replace(LP_REPLACE *lpr, FILE *fp)
{
  return (write_u32(lpr->before_start, fp)
	  && write_u32(lpr->after_start, fp)
	  && write_u32(lpr->first, fp)
	  && write_u16(lpr->npredicted, fp)
	  && write_u16(lpr->poly_order, fp)
	  && write_u16(lpr->before_npoints, fp)
	  && write_u16(lpr->after_npoints, fp)
	  );
}

/* ----------------------------------------------------------------------------
*/
static int write_nmr_base_offset(NMR_BASE_OFFSET *bo, FILE *fp)
{
  return (write_u32(bo->start, fp)
	  && write_u32(bo->end, fp)
	  );
}

/* ----------------------------------------------------------------------------
*/
static int write_nmr_base_fit(NMR_BASE_FIT *bf, FILE *fp)
{
  return (write_u32(bf->solvent_start, fp)
	  && write_s16(bf->poly_order, fp)
	  && write_s16(bf->solvent_width, fp)
	  );
}

// ----------------------------------------------------------------------------
// Change an axis label in a UCSF nmr data file.
// Only works for version 2 file format.
//
bool change_ucsf_axis_label(const Stringy &path, int axis,
			    const Stringy &label)
{
  NMR_FILE *nmr_file = nmr_open(path);
  if (nmr_file == NULL)
    return false;

  int version = nmr_file_version(nmr_file);
  int dim = nmr_file_naxes(nmr_file);
  nmr_fclose(nmr_file);

  if (version != NMR_NEWER || axis >= dim)
    return false;

  char nucleus_name[6];
  strncpy(nucleus_name, label.cstring(), 5);
  nucleus_name[5] = '\0';

  FILE *fp = fopen(path.cstring(), "rb+");		// NMR data file
  if (fp == NULL)
    return false;

  long offset = (long) (NMR_HEADER_IO_SIZE + axis * NMR_AXIS_IO_SIZE);
  bool success = (seek_position(fp, offset, 0) &&
		  write_u8_array(nucleus_name, 6, fp));
  fclose(fp);

  return success;
}

// ----------------------------------------------------------------------------
// Change an axis ppm origin in a UCSF nmr data file.
// Only works for version 2 file format.
//
bool change_ucsf_origin(const Stringy &path, int axis, float origin_ppm)
{
  NMR_FILE *nmr_file = nmr_open(path);
  if (nmr_file == NULL)
    return false;

  int version = nmr_file_version(nmr_file);
  int dim = nmr_file_naxes(nmr_file);
  float width_ppm = (nmr_axis_spectral_width(nmr_file, axis) /
		     nmr_axis_spectrometer_frequency(nmr_file, axis));
  nmr_fclose(nmr_file);

  if (version != NMR_NEWER || axis >= dim)
    return false;

  float xmtr_freq = origin_ppm - width_ppm / 2.0;

  FILE *fp = fopen(path.cstring(), "rb+");		// NMR data file
  if (fp == NULL)
    return false;

  long offset = (long) (NMR_HEADER_IO_SIZE + axis * NMR_AXIS_IO_SIZE + 28);
  bool success = (seek_position(fp, offset, 0) && write_f32(xmtr_freq, fp));
  fclose(fp);

  return success;
}

// ----------------------------------------------------------------------------
// Change an axis spectral width in a UCSF nmr data file.
// Only works for version 2 file format.
//
bool change_ucsf_spectral_width(const Stringy &path, int axis, float sw_hz)
{
  return change_ucsf_header_value(path, axis, 24, sw_hz);
}

// ----------------------------------------------------------------------------
// Change an axis spectrometer frequency in a UCSF nmr data file.
// Only works for version 2 file format.
//
bool change_ucsf_spectrometer_frequency(const Stringy &path, int axis,
					float sfreq_mhz)
{
  return change_ucsf_header_value(path, axis, 20, sfreq_mhz);
}

// ----------------------------------------------------------------------------
// Change an axis header floating point value in a UCSF nmr data file.
// Only works for version 2 file format.
//
static bool change_ucsf_header_value(const Stringy &path, int axis,
				     int offset, float value)
{
  NMR_FILE *nmr_file = nmr_open(path);
  if (nmr_file == NULL)
    return false;

  int version = nmr_file_version(nmr_file);
  int dim = nmr_file_naxes(nmr_file);
  nmr_fclose(nmr_file);

  if (version != NMR_NEWER || axis >= dim)
    return false;

  FILE *fp = fopen(path.cstring(), "rb+");		// NMR data file
  if (fp == NULL)
    return false;

  long file_offset = (long) (NMR_HEADER_IO_SIZE + axis * NMR_AXIS_IO_SIZE
			     + offset);
  bool success = (seek_position(fp, file_offset, 0) && write_f32(value, fp));
  fclose(fp);

  return success;
}
