// ----------------------------------------------------------------------------
//
#include <math.h>

#include "color.h"		// Use Color
#include "list.h"
#include "memalloc.h"		// use new()
#include "num.h"
#include "rectangle.h"
#include "uidrawing.h"
#include "utility.h"
#include "winsystem.h"		// use Drawing_Window

static Rectangle first_tile(Rectangle rect, Rectangle tile);
static void containing_interval(double x, double t1, double t2,
				double *s1, double *s2);

// ----------------------------------------------------------------------------
//
Drawing::Drawing(WinSys &winsys, Widget parent,
		 int w, int h, double x_center, double y_center,
		 double x_pixel_size, double y_pixel_size, Rectangle drawing,
		 Rectangle repaint_tile, double repaint_priority,
		 bool contour_graying)
  : ws(winsys)
{
  this->vw = w;
  this->vh = h;
  this->x_center = x_center;
  this->y_center = y_center;
  this->x_pixel_size = x_pixel_size;
  this->y_pixel_size = y_pixel_size;
  this->drawing_rect = drawing;
  this->redraw_tile = repaint_tile;
  this->redraw_priority = repaint_priority;
  this->pointer_in_view = false;
  this->contour_graying = contour_graying;

  this->c = new Drawing_Window(ws, parent, "drawing");
  c->request_backing_store();

  Widget dw = c->widget();
  ws.set_widget_size(dw, w, h);
  ws.add_event_callback(expose_event, dw, expose_cb, this);
  ws.add_event_callback(resize_event, dw, resize_cb, this);
  ws.add_event_callback(enter_window_event, dw, enter_window_cb, this);
  ws.add_event_callback(leave_window_event, dw, leave_window_cb, this);
  ws.add_event_callback(button_1_press_event, dw, button_press_cb, this);
  ws.add_event_callback(button_1_release_event, dw, button_release_cb, this);
  ws.add_event_callback(pointer_move_event, dw, pointer_move_cb, this);
  ws.add_event_callback(pointer_drag_event, dw, pointer_drag_cb, this);
  ws.add_event_callback(got_focus_event, dw, got_focus_cb, this);
  ws.add_event_callback(key_press_event, dw, key_press_cb, this);

  this->font_height = 0;
}

// ----------------------------------------------------------------------------
// Drawing canvas widget is deleted by top level dialog.
//
Drawing::~Drawing()
{
  Widget dw = c->widget();
  ws.remove_event_callback(expose_event, dw, expose_cb, this);
  ws.remove_event_callback(resize_event, dw, resize_cb, this);
  ws.remove_event_callback(enter_window_event, dw, enter_window_cb, this);
  ws.remove_event_callback(leave_window_event, dw, leave_window_cb, this);
  ws.remove_event_callback(button_1_press_event, dw, button_press_cb, this);
  ws.remove_event_callback(button_1_release_event, dw, button_release_cb, this);
  ws.remove_event_callback(pointer_move_event, dw, pointer_move_cb, this);
  ws.remove_event_callback(pointer_drag_event, dw, pointer_drag_cb, this);
  ws.remove_event_callback(got_focus_event, dw, got_focus_cb, this);
  ws.remove_event_callback(key_press_event, dw, key_press_cb, this);

  ws.remove_work_procedure(repaint_tile, this);

  delete c;

  c = 0;			// Catch errors quickly.
}

// ----------------------------------------------------------------------------
//
void Drawing::expose_cb(Widget, CB_Data drawing, CB_Data event)
{
  Drawing *d = (Drawing *) drawing;

  int x, y, width, height;
  if (d->ws.exposure_region(event, &x, &y, &width, &height))
    d->repaint(x, y, width, height);
}

// ----------------------------------------------------------------------------
//
void Drawing::resize_cb(Widget w, CB_Data drawing, CB_Data)
{
  Drawing *d = (Drawing *) drawing;

  d->resize(d->ws.widget_width(w), d->ws.widget_height(w));
}

// ----------------------------------------------------------------------------
//
void Drawing::enter_window_cb(Widget, CB_Data drawing, CB_Data event)
{
  Drawing *d = (Drawing *) drawing;
  double x, y;
  if (d->event_position(event, &x, &y))
    d->enter_window(x, y);
}

// ----------------------------------------------------------------------------
//
void Drawing::leave_window_cb(Widget, CB_Data drawing, CB_Data event)
{
  Drawing *d = (Drawing *) drawing;
  double x, y;
  if (d->event_position(event, &x, &y))
    {
      d->pointer_in_view = false;
      d->exit_window(x, y);
    }
}

// ----------------------------------------------------------------------------
//
void Drawing::button_press_cb(Widget, CB_Data drawing, CB_Data event)
{
  Drawing *d = (Drawing *) drawing;
  double x, y;
  if (d->event_position(event, &x, &y))
    d->buttondown(x, y, d->ws.shift_modifier(event));
}

// ----------------------------------------------------------------------------
//
void Drawing::button_release_cb(Widget, CB_Data drawing, CB_Data event)
{
  Drawing *d = (Drawing *) drawing;
  double x, y;
  if (d->event_position(event, &x, &y))
    d->buttonup(x, y);
}

// ----------------------------------------------------------------------------
//
void Drawing::pointer_move_cb(Widget, CB_Data drawing, CB_Data event)
{
  Drawing *d = (Drawing *) drawing;
  double x, y;
  if (d->event_position(event, &x, &y))
    d->move(x, y);
}

// ----------------------------------------------------------------------------
//
void Drawing::pointer_drag_cb(Widget, CB_Data drawing, CB_Data event)
{
  Drawing *d = (Drawing *) drawing;
  double x, y;
  if (d->event_position(event, &x, &y))
    d->drag(x, y);
}

// ----------------------------------------------------------------------------
//
void Drawing::got_focus_cb(Widget, CB_Data drawing, CB_Data)
{
  Drawing *d = (Drawing *) drawing;
  d->got_focus();
}

// ----------------------------------------------------------------------------
//
void Drawing::key_press_cb(Widget, CB_Data drawing, CB_Data event)
{
  Drawing *d = (Drawing *) drawing;

  bool shift;
  char c;
  int f;
  if (d->ws.key_pressed(event, &c))
    d->key_pressed(c);
  else if (d->ws.function_key_pressed(event, &f, &shift))
    d->function_key_pressed(f, shift);
}

// ----------------------------------------------------------------------------
//
bool Drawing::event_position(CB_Data event, double *x, double *y)
{
  if (ws.event_xy(event, &pointer_x, &pointer_y))
    {
      pointer_in_view = true;
      drawing_coordinates(pointer_x, pointer_y, x, y);
      return true;
    }
  return false;
}

// ----------------------------------------------------------------------------
//
bool Drawing::pointer_position(double *x, double *y)
{
  if (pointer_in_view)
    {
      drawing_coordinates(pointer_x, pointer_y, x, y);
      return true;
    }

  return false;
}

// ----------------------------------------------------------------------------
//
void Drawing::resize(int width, int height)
{
  vw = width;
  vh = height;

  drawing_resized();

  // Wouldn't need this if window bit gravity was set to ForgetGravity.
  erase();
}

// ----------------------------------------------------------------------------
//
void Drawing::drawing_resized() {}

// ----------------------------------------------------------------------------
//
void Drawing::set_contour_graying(bool onoff)
{
  this->contour_graying = onoff;
}

// ----------------------------------------------------------------------------
//
#define UNREFRESHED_COLOR "gray30"
void Drawing::repaint(int x, int y, int w, int h)
{
  double x1, y1, x2, y2;

  if (contour_graying)
    {
      set_drawing_color(UNREFRESHED_COLOR);
      c->fill_rectangle(x, y, w, h);
    }

  drawing_coordinates(x, y, &x1, &y1);
  drawing_coordinates(x + w, y + h, &x2, &y2);
  Rectangle rect = Rectangle(x1, y1, x2, y2);

  rect.clip(drawing_rectangle());
  rect.clip(view_rectangle());
  if (!rect.no_interior())
    {
      if (repaint_region().is_empty())
	ws.add_work_procedure(repaint_tile, this, redraw_priority);
      repaint_reg.add(rect);
    }
}

// ----------------------------------------------------------------------------
//
void Drawing::repaint_tile(CB_Data dr)
{
  Drawing *d = (Drawing *) dr;

  if (d->repaint_region().is_empty())
    return;

  Rectangle rect = d->repaint_region().first_rectangle();
  Rectangle tile = first_tile(rect, d->redraw_tile);
  d->repaint_reg.subtract(tile);
  d->redraw_rectangle(tile);

  if (!d->repaint_region().is_empty())
    d->ws.add_work_procedure(repaint_tile, d, d->redraw_priority);
}

// ----------------------------------------------------------------------------
//
static Rectangle first_tile(Rectangle rect, Rectangle tile)
{
  double tx1, tx2, ty1, ty2;
  containing_interval(rect.min(X), tile.min(X), tile.max(X), &tx1, &tx2);
  containing_interval(rect.min(Y), tile.min(Y), tile.max(Y), &ty1, &ty2);

  return Rectangle(tx1, ty1, tx2, ty2);
}

// ----------------------------------------------------------------------------
// Tile the real line with points t = t1 + n * (t2 - t1).
// Find the interval [s1,s2) containing x.
//
static void containing_interval(double x, double t1, double t2,
				double *s1, double *s2)
{
  double step = t2 - t1;
  int n1 = (int) floor((x - t1) / step);
  double r1 = t1 + step * n1;
  double r2 = t1 + step * (n1 + 1);

  //
  // The above can produce an interval to the left or right of the correct one
  // because of inexact arithmetic.  Fix these cases.
  //

  //
  // The following line does nothing but is needed to work around a compiler
  // bug in gcc 2.95.2 on PC linux.  Without it the x >= r2 test will
  // sometimes evaluate false even though the 2 numbers are equal.  At the
  // end of the if else clause, with neither body being executed, x >= r2
  // can be printed and comes out true.  This bug makes Sparky ceasely
  // redraw the same contour tile.  Compiling without optimization gets
  // rid of the problem.  Also compiling with -ffloat-store which prevents
  // floating point variables from being placed in registers fixes the
  // problem.
  //
  // Reading gcc news groups on -ffloat-store indicates that the problem is
  // that the x86 FPU registers are 80 bits long.  So below I can end up
  // comparing x which comes from a double in memory (64 bits) to an 80 bit
  // register value r2 calculated in exactly the same way as x but not
  // truncated to 64 bits.  The below call to is_between() in a different
  // module forces all FPU values to spill to 64 bit locations.
  //
  is_between(x, r1, r2);

  if (x >= r2)
    {
      r1 = t1 + step * (n1 + 1);
      r2 = t1 + step * (n1 + 2);
    }
  else if (x < r1)
    {
      r1 = t1 + step * (n1 - 1);
      r2 = t1 + step * n1;
    }

  *s1 = r1;
  *s2 = r2;
}

// ----------------------------------------------------------------------------
//
void Drawing::redraw_rectangle(Rectangle) {}

// ----------------------------------------------------------------------------
//
void Drawing::set_redraw_priority(double priority)
{
  redraw_priority = priority;
  ws.set_work_procedure_priority(repaint_tile, this, redraw_priority);
}

// ----------------------------------------------------------------------------
//
void Drawing::erase_rectangle(const Rectangle &r, bool exposures)
{
  if (r.intersects(view_rectangle()))
    {
      int px1, py1, px2, py2, w, h;

      paint_coordinates(r.min(X), r.min(Y), &px2, &py1);
      paint_coordinates(r.max(X), r.max(Y), &px1, &py2);
      w = px2 - px1 + 1;
      h = py2 - py1 + 1;

      if (exposures)
	c->clear_area(px1, py1, w, h);
      else
	c->draw_background(px1, py1, w, h);
    }
}

// ----------------------------------------------------------------------------
//
void Drawing::erase()
{
  erase_rectangle(view_rectangle());
}

// ----------------------------------------------------------------------------
//
const Region2 &Drawing::repaint_region()
{
  return repaint_reg;
}

// ----------------------------------------------------------------------------
//
void Drawing::update_font_height()
{
  int pixel_height = bounded(1, round(font_height / pixel_size(Y)), 100);
  c->set_font_height(pixel_height);
}

// ----------------------------------------------------------------------------
//
void Drawing::set_font_height(double height)
{
  font_height = height;
}

// ----------------------------------------------------------------------------
//
Widget Drawing::drawing_widget()
  { return c->widget(); }

// ----------------------------------------------------------------------------
// Pixel size in data units.
//
double Drawing::pixel_size(Axis a) const
{
  return (a == X ? x_pixel_size : y_pixel_size);
}

// ----------------------------------------------------------------------------
// Display aspect of a ppm unit square.
//
double Drawing::aspect() const
{
  return pixel_size(Y) / pixel_size(X);
}

// ----------------------------------------------------------------------------
//
void Drawing::screen_size(int *xsize, int *ysize)
{
  *xsize = vw;
  *ysize = vh;
}

// ----------------------------------------------------------------------------
//
void Drawing::set_display_size(int width, int height)
{
  ws.set_widget_size(drawing_widget(), width, height);
}

// ----------------------------------------------------------------------------
//
Rectangle Drawing::view_rectangle() const
{
  double w = vw * x_pixel_size;
  double h = vh * y_pixel_size;
  return Rectangle(x_center - w/2, y_center - h/2,
		   x_center + w/2, y_center + h/2);
}

// ----------------------------------------------------------------------------
//
Rectangle Drawing::drawing_rectangle() const
  { return drawing_rect; }

// ----------------------------------------------------------------------------
//
void Drawing::paint_coordinates(double x, double y, int *px, int *py)
{
  *px = round(.5 * vw - (x - x_center) / x_pixel_size);
  *py = round(.5 * vh + (y - y_center) / y_pixel_size);
}

// ----------------------------------------------------------------------------
//
void Drawing::drawing_coordinates(int px, int py, double *x, double *y)
{
  *x = x_center - (px - .5 * vw) * x_pixel_size;
  *y = y_center + (py - .5 * vh) * y_pixel_size;
}

// ----------------------------------------------------------------------------
//
int Drawing::paint_coordinate(double d, Axis a)
{
  return (a == X ?
	  round(.5 * vw - (d - x_center) / x_pixel_size) :
	  round(.5 * vh + (d - y_center) / y_pixel_size));
}

// ----------------------------------------------------------------------------
//
void Drawing::translate_view(double delta_x, double delta_y)
{
  int dx = round(delta_x / pixel_size(X));
  int dy = round(delta_y / pixel_size(Y));

// This is used before changing the mapping between X window coordinates
// and the image coordinates in translate_view().  If pretranslation exposure
// events are processed after the coordinate translation is done, the wrong
// region is repainted.  This leaves the correct region unpainted.  This
// routine is called to minimize the problem.

  ws.process_exposure_events(drawing_widget());
  c->translate_contents(dx, -dy);

  this->x_center += dx * pixel_size(X);
  this->y_center += dy * pixel_size(Y);
  view_rectangle_changed();
  repaint_reg.intersect(view_rectangle());
}

// ----------------------------------------------------------------------------
//
void Drawing::set_center(double x, double y)
  { translate_view(x - x_center, y - y_center); }

// ----------------------------------------------------------------------------
//
void Drawing::set_pixel_size(double xsize, double ysize)
{
  this->x_pixel_size = xsize;
  this->y_pixel_size = ysize;
  view_rectangle_changed();
  erase();
}

// ----------------------------------------------------------------------------
//
void Drawing::view_rectangle_changed()
{
  //
  // Since pointer move events are reported in drawing coordinates
  // when I alter coordinates for window I need to say pointer moved.
  //
  double x, y;
  if (pointer_position(&x, &y))
    move(x, y);
}

// ----------------------------------------------------------------------------
//
void Drawing::set_drawing(Rectangle r)
  { drawing_rect = r; }
void Drawing::set_redraw_tile(Rectangle r)
  { redraw_tile = r; }

// ----------------------------------------------------------------------------
//
void Drawing::set_drawing_color(const Color &color)
{
  c->set_foreground(color);
}

// ----------------------------------------------------------------------------
//
void Drawing::draw_line(double x1, double y1, double x2, double y2,
			bool xor_mode)
{
  int px1, py1, px2, py2;

  paint_coordinates(x1, y1, &px1, &py1);
  paint_coordinates(x2, y2, &px2, &py2);

  c->draw_line(px1, py1, px2, py2, xor_mode);
}

// ----------------------------------------------------------------------------
//
void Drawing::draw_point(double x, double y)
{
  int px, py;

  paint_coordinates(x, y, &px, &py);

  c->draw_point(px, py);
}

// ----------------------------------------------------------------------------
//
void Drawing::draw_rectangle(Rectangle r, bool xor_mode)
{
  int x, y, w, h;

  rectangle_paint_coordinates(r, &x, &y, &w, &h);
  c->draw_rectangle(x, y, w, h, xor_mode);
}

// ----------------------------------------------------------------------------
//
#define ARC360 (360 * 64)
void Drawing::draw_ellipse(Rectangle r)
{
  int x, y, w, h;

  rectangle_paint_coordinates(r, &x, &y, &w, &h);
  c->draw_arc(x, y, w, h, 0, ARC360);
}

// ----------------------------------------------------------------------------
//
void Drawing::draw_triangle(double x1, double y1,
			    double x2, double y2,
			    double x3, double y3)
{
  int px1, py1, px2, py2, px3, py3;

  paint_coordinates(x1, y1, &px1, &py1);
  paint_coordinates(x2, y2, &px2, &py2);
  paint_coordinates(x3, y3, &px3, &py3);
  c->fill_triangle(px1, py1, px2, py2, px3, py3);
}

// ----------------------------------------------------------------------------
//
void Drawing::rectangle_paint_coordinates(Rectangle r,
					  int *x, int *y, int *w, int *h)
{
  int px1, py1, px2, py2;

  paint_coordinates(r.min(X), r.min(Y), &px2, &py1);
  paint_coordinates(r.max(X), r.max(Y), &px1, &py2);

  *x = px1;
  *y = py1;
  *w = (px2 > px1 ? px2 - px1 : 1);
  *h = (py2 > py1 ? py2 - py1 : 1);
}

// ----------------------------------------------------------------------------
// Draw a string centered at a point.
//
void Drawing::draw_text(double center_x, double center_y, const Stringy &text)
{
  update_font_height();

  double w, ascent, descent;
  text_size(text, &w, &ascent, &descent);
  double h = ascent + descent;

  double x = center_x + w/2;
  double y = center_y + h/2 - descent;

  int px, py;
  paint_coordinates(x, y, &px, &py);
  c->draw_string(px, py, text);
}

// ----------------------------------------------------------------------------
//
void Drawing::text_size(const Stringy &text, double *width,
			double *ascent, double *descent)
{
  int w, a, d;

  update_font_height();
  c->text_size(text, &w, &a, &d);

  *width = w * pixel_size(X);
  *ascent = a * pixel_size(Y);
  *descent = d * pixel_size(Y);
}

// ----------------------------------------------------------------------------
// Default action on mouse events is to do nothing.
//
void Drawing::buttondown(double, double, bool) {}
void Drawing::buttonup(double, double) {}
void Drawing::move(double, double) {}
void Drawing::enter_window(double, double) {}
void Drawing::exit_window(double, double) {}
void Drawing::drag(double, double) {}
void Drawing::key_pressed(char) {}
void Drawing::function_key_pressed(int, bool) {}
void Drawing::got_focus() {}
