// ----------------------------------------------------------------------------
//
#include <math.h>		// Use fabs()

#include "color.h"		// Use Color
#include "memalloc.h"		// use new()
#include "num.h"		// Use round()
#include "uiepanel.h"
#include "winsystem.h"		// Use create_drawing_area()

// ----------------------------------------------------------------------------
//
edge_panel::edge_panel(WinSys &winsys, double a, double b, int screen_length,
		       bool is_horz, Widget parent, const Stringy &name,
		       bool scroll) : ws(winsys)
{
  new_interval(a, b);
  this->is_horz = is_horz;
  this->len = screen_length;
  this->erase_queued = true;	// window exposed when created
  this->scroll = scroll;

  this->c = new Drawing_Window(ws, parent, name);
  c->use_label_font();
  Widget dw = c->widget();
  if (is_horz)
    ws.set_widget_width(dw, screen_length);
  else
    ws.set_widget_height(dw, screen_length);
  ws.add_event_callback(expose_event, dw, expose_cb, this);
  ws.add_event_callback(resize_event, dw, resize_cb, this);
}

// ----------------------------------------------------------------------------
//
edge_panel::~edge_panel()
{
  Widget dw = c->widget();
  ws.remove_event_callback(expose_event, dw, expose_cb, this);
  ws.remove_event_callback(resize_event, dw, resize_cb, this);

  delete c;

  c = 0;
}

// ----------------------------------------------------------------------------
//
void edge_panel::expose_cb(Widget, CB_Data epanel, CB_Data event)
{
  Edge_Panel ep = (Edge_Panel) epanel;

  int x, y, width, height;
  if (ep->ws.exposure_region(event, &x, &y, &width, &height))
    ep->repaint(x, y, width, height);
}

// ----------------------------------------------------------------------------
//
void edge_panel::resize_cb(Widget w, CB_Data client_data, CB_Data)
{
  Edge_Panel ep = (Edge_Panel) client_data;

  ep->resize(ep->ws.widget_width(w), ep->ws.widget_height(w));
}

// ----------------------------------------------------------------------------
//
void edge_panel::resize(int width, int height)
{
  double length = (is_horizontal() ? width : height) * pixel_size();
  double center = (a + b) / 2;

  len = (is_horizontal() ? width : height);

  new_interval(center - length/2, center + length/2);

  erase();	// Wouldn't need this if window bit gravity
		//  was set to ForgetGravity.
}

// ----------------------------------------------------------------------------
//
bool edge_panel::is_horizontal()
  { return is_horz; }

// ----------------------------------------------------------------------------
//
void edge_panel::set_drawing_color(const Color &color)
  { c->set_foreground(color); }

// ----------------------------------------------------------------------------
//
Widget edge_panel::widget()
  { return c->widget(); }

// ----------------------------------------------------------------------------
//
void edge_panel::range(double *a, double *b)
{
  *a = this->a;
  *b = this->b;
}

// ----------------------------------------------------------------------------
//
void edge_panel::set_range(double a, double b)
{
  if (is_translation(a, b))
    translate(a - this->a);
  else
    {
      new_interval(a, b);
      erase();
    }
}

// ----------------------------------------------------------------------------
//
void edge_panel::view_region_changed() {}

// ----------------------------------------------------------------------------
//
void edge_panel::new_interval(double a, double b)
{
  this->a = a;
  this->b = b;
}

// ----------------------------------------------------------------------------
//
#define MARKER_TRANSLATION_TOLERANCE 1e-6

bool edge_panel::is_translation(double a, double b)
{
  return (fabs((b - a) - (this->b - this->a)) <
	  MARKER_TRANSLATION_TOLERANCE * fabs(this->b - this->a));
}

// ----------------------------------------------------------------------------
//
void edge_panel::translate(double delta)
{
  int d = round(delta / pixel_size());

  if (d != 0)
    {
      double shift = d * pixel_size();

//
// Process all queued exposure events.
// This is used before changing the mapping between X window coordinates
// and the edge_panel coordinates in translate().  If pretranslation exposure
// events are processed after the coordinate translation is done, the wrong
// region is repainted.  This leaves the correct region unpainted.  This
// routine is called to minimize the problem.
//
      if (scroll)
	{
	  ws.process_exposure_events(widget());
	  if (is_horizontal())
	    c->translate_contents(d, 0);
	  else
	    c->translate_contents(0, -d);
	}
      a += shift;
      b += shift;
    }
}

// ----------------------------------------------------------------------------
//
void edge_panel::erase()
{
  if (!erase_queued)
    {
      c->clear_window();
      erase_queued = true;
    }
}

// ----------------------------------------------------------------------------
//
void edge_panel::repaint(int x, int y, int w, int h)
{
  erase_queued = false;

  if (is_horizontal())
    redraw(draw_position(x + w), draw_position(x));
  else
    redraw(draw_position(y), draw_position(y + h));
}

// ----------------------------------------------------------------------------
//
void edge_panel::draw_tick(double x, double xlabel)
{
  int pos = paint_position(x);
  int labelpos = paint_position(xlabel);
  int length = c->font_height() / 2;

  if (is_horizontal())
    c->draw_line(pos, 0, labelpos, length);
  else
    c->draw_line(0, pos, length, labelpos);
}

// ----------------------------------------------------------------------------
//
void edge_panel::draw_label(double xlabel, const char *label, bool parallel)
{
  int w, ascent, descent;

  c->text_size(label, &w, &ascent, &descent);
  int h = ascent + descent;
  int pos = paint_position(xlabel);
  int space = (3 * h) / 4;

  if (is_horizontal())
    if (parallel)
      {
	int x = pos - w / 2;
	int y = space + ascent;
	c->draw_string(x, y, label);
      }
    else
      {
	int x = pos - ascent / 2;
	int y = space;
	c->draw_vertical_string(x, y, label, false);
      }
  else			// Vertical scale.
    if (parallel)
      {
	int x = space + ascent;
	int y = pos + w / 2;
	c->draw_vertical_string(x, y, label, true);
      }
    else
      {
	int x = space;
	int y = pos + ascent / 2;
	c->draw_string(x, y, label);
      }
}

// ----------------------------------------------------------------------------
//
void edge_panel::text_width(const char *text, bool parallel,
			    double *left, double *right)
{
  int w, ascent, descent;

  c->text_size(text, &w, &ascent, &descent);
  if (parallel)
    *left = *right = w * pixel_size() / 2;
  else
    {
      *left = (descent + ascent/2) * pixel_size();
      *right = ascent/2 * pixel_size();
    }
}

// ----------------------------------------------------------------------------
//
double edge_panel::label_spacing()
{
  return c->font_height() * fabs(pixel_size());
}

// ----------------------------------------------------------------------------
//
double edge_panel::thickness()
{
  Widget w = widget();
  int pixels = (is_horizontal() ?
		ws.widget_height(w) - 1 :
		ws.widget_width(w) - 1);
  return pixels * pixel_size();
}

// ----------------------------------------------------------------------------
//
void edge_panel::draw_line(double x1, double y1, double x2, double y2,
			   bool xor_mode)
{
  int px1, py1, px2, py2;
  paint_coordinates(x1, y1, &px1, &py1);
  paint_coordinates(x2, y2, &px2, &py2);
  c->draw_line(px1, py1, px2, py2, xor_mode);
}

// ----------------------------------------------------------------------------
//
void edge_panel::fill_rectangle(double x1, double y1, double x2, double y2)
{
  int px1, py1, px2, py2;
  paint_coordinates(x1, y1, &px1, &py1);
  paint_coordinates(x2, y2, &px2, &py2);
  int px = min(px1, px2);
  int py = min(py1, py2);
  unsigned int pw = max(px1, px2) - px;
  unsigned int ph = max(py1, py2) - py;
  c->fill_rectangle(px, py, pw, ph);
}

// ----------------------------------------------------------------------------
//
void edge_panel::paint_coordinates(double x, double y, int *px, int *py)
{
  if (is_horizontal())
    {
      double h = thickness();
      *px = paint_position(x);
      *py = round((h - y) / pixel_size());
    }
  else
    {
      *px = round(x / pixel_size());
      *py = paint_position(y);
    }
}

// ----------------------------------------------------------------------------
//
int edge_panel::paint_position(double x)
{
  return round((is_horizontal() ? b - x : x - a) / pixel_size());
}

// ----------------------------------------------------------------------------
//
double edge_panel::draw_position(int p)
{
  return (is_horizontal() ? b - p * pixel_size() : a + p * pixel_size());
}

// ----------------------------------------------------------------------------
//
double edge_panel::pixel_size()
{
  return (b - a) / len;
}

// ----------------------------------------------------------------------------
//
void edge_panel::redraw(double, double) {}

// ----------------------------------------------------------------------------
//
void edge_panel::pointer_in(double, double) {}
void edge_panel::pointer_move(double, double) {}
void edge_panel::pointer_out(double, double) {}

// ----------------------------------------------------------------------------
//
bool is_clipped(double c, double d, double a_clip, double b_clip)
{
  return (max(c, d) < min(a_clip, b_clip) ||
	  min(c, d) > max(a_clip, b_clip));
}
