// ----------------------------------------------------------------------------
//
#include <stdarg.h>		// use va_start()
#include <stdio.h>		// Use FILE
#include <stdlib.h>		// use getenv(), atof()
#include <string.h>		// Use strcmp()

#include "atom.h"		// use Atom
#include "color.h"		// Use Color
#include "crosspeak.h"		// Use CrossPeak
#include "group.h"		// Use Group
#include "list.h"		// Use List
#include "memalloc.h"		// use new()
#include "model.h"		// use Model
#include "molecule.h"		// Use Molecule
#include "notifier.h"		// use Notifier
#include "project.h"		// use Project
#include "reporter.h"		// use Reporter
#include "resonance.h"		// Use Resonance
#include "session.h"		// use Session
#include "spectrum.h"		// Use Spectrum
#include "spoint.h"		// Use SPoint
#include "system.h"		// Use start_process()
#include "uicomponents.h"	// Use Spectrum_Menu
#include "utility.h"		// Use warning()
#include "uidialogs.h"		// use help_cb()
#include "uidialog.h"		// use Dialog, Dialog_Table
#include "uiview.h"		// Use View
#include "winsystem.h"		// Use add_input_callback()

#define PDB_MODEL_NUMBER 100
#define ASSIGNMENT_MODEL_NUMBER 200
#define HIGHLIGHT_MODEL_NUMBER 300

#define MAX_LINE_LENGTH		4096

static SPoint highlight_offset(const SPoint &dir);
static int fully_assigned(const List &peaks);
static bool fully_assigned(CrossPeak *xp);
static Stringy skip_line(const Stringy &s);

// ----------------------------------------------------------------------------
//
class midas_connection
{
public:
  static midas_connection *new_connection(WinSys &);
  ~midas_connection();
  bool is_connection_good();
  int show_assignments(const List &peaks, const Stringy &color,
		       int model_number, bool highlight);
  Stringy send_command(const char *format, ...);
private:
  FILE *to_midas;
  FILE *from_midas;
  Stringy object_file_path;
  FILE *object_file;
  bool good_connection;
  WinSys &ws;

  midas_connection(FILE *to_midas, FILE *from_midas, const Stringy &temp_path,
		   WinSys &);
  static void midas_request_cb(CB_Data);

  void set_color(const Stringy &color);
  void move_to(const SPoint &p);
  void draw_to(const SPoint &p);
  void marker(const SPoint &p);
  void send_objects();
  void replace_objects(int model_number);

  bool show_assignment(CrossPeak *xp, bool highlight);
  Stringy model_pdb_path(int model_number);
};

// ----------------------------------------------------------------------------
//
class midas_dialog : public Dialog
{
public:
  midas_dialog(Session &);
  ~midas_dialog();

  static midas_dialog *the(Session &);

  void show(Spectrum *sp);
  void update(Spectrum *sp);

private:
  Widget pdb_file, line_color, sw;
  Spectrum_Menu *spectrum_menu;
  Spectrum *spectrum;
  Stringy pdb_path;
  Stringy color;
  bool show_all_assignments;
  midas_connection *mc;

  static void will_delete_spectrum_cb(void *idialog, void *spectrum);
  static void selections_changed_cb(void *mdialog, void *);
  static void file_dialog_cb(Widget, CB_Data, CB_Data);

  void apply();
  bool start_midas();
  int show_assignments(const List &peaks, bool highlight);
};

// ----------------------------------------------------------------------------
//
void show_midas_dialog(Session &s, Spectrum *sp)
  { midas_dialog::the(s)->show(sp); }

// ----------------------------------------------------------------------------
//
midas_dialog::midas_dialog(Session &s) : Dialog(s, "midasDialog")
{
  this->mc = NULL;
  this->spectrum = NULL;

  spectrum_menu = new Spectrum_Menu(s, dialog, "spectrumMenu");

  const char *colors[] = {"black", "white", "gray", "red", "blue", "green",
			  "cyan", "magenta", NULL};
  line_color = ws.option_menu(dialog, "colors", colors);
			   
  ws.set_option(line_color, "green");

  const char *swnames[] = {"allAssignments", NULL};
  sw = ws.switches(dialog, "switches", swnames);

  pdb_file = ws.edit_field(dialog, "pdbFile");

  Widget separator = ws.create_separator(dialog, "separator");

  Widget controls = ws.button_row(dialog, "controls",
			       "ok", ok_cb, this,
			       "apply", apply_cb, this,
			       "file", file_dialog_cb, this,
			       "close", close_cb, this,
			       "help", help_cb, &s,
			       NULL);

  ws.column_attachments(separator, spectrum_menu->option_menu(),
		     line_color, sw, pdb_file,
		     separator, controls, END_OF_WIDGETS);


  Notifier &n = session.notifier();
  n.notify_me(nt_will_delete_spectrum, will_delete_spectrum_cb, this);
  n.notify_me(nt_selected_ornaments_changed, selections_changed_cb, this);
}

// ----------------------------------------------------------------------------
//
midas_dialog::~midas_dialog()
{
  session.dialog_table().delete_dialog("midas_dialog", this);

  Notifier &n = session.notifier();
  n.dont_notify_me(nt_will_delete_spectrum, will_delete_spectrum_cb, this);
  n.dont_notify_me(nt_selected_ornaments_changed, selections_changed_cb, this);

  delete spectrum_menu;
}

// ----------------------------------------------------------------------------
// The default midas_dialog instance.
//
midas_dialog *midas_dialog::the(Session &s)
{
  Stringy name = "midas_dialog";
  Dialog_Table &dt = s.dialog_table();
  if (dt.get_dialog(name) == NULL)
    dt.set_dialog(name, new midas_dialog(s));
  return (midas_dialog *) dt.get_dialog(name);
}

// ----------------------------------------------------------------------------
//
void midas_dialog::file_dialog_cb(Widget, CB_Data mdialog, CB_Data)
{
  midas_dialog *md = (midas_dialog *) mdialog;

  Stringy defpath = md->session.project().sparky_path("", "");
  Stringy path = md->ws.open_file_dialog(md->dialog, "Select PDB File",
					 defpath, "pdb", true);
  if (! path.is_empty())
    md->ws.set_edit_value(md->pdb_file, path);
}

// ----------------------------------------------------------------------------
//
void midas_dialog::will_delete_spectrum_cb(void *mdialog, void *spectrum)
{
  midas_dialog *md = (midas_dialog *) mdialog;
  Spectrum *sp = (Spectrum *) spectrum;

  if (md->spectrum == sp)
    md->update(NULL);
}

// ----------------------------------------------------------------------------
//
void midas_dialog::selections_changed_cb(void *mdialog, void *)
{
  midas_dialog *md = (midas_dialog *) mdialog;

  if (md->mc && md->spectrum)
    md->show_assignments(md->spectrum->selected_crosspeaks(),
			 md->show_all_assignments);
}

// ----------------------------------------------------------------------------
//
void midas_dialog::show(Spectrum *sp)
{
  update(sp);
  ws.show_dialog(dialog);
  ws.raise_widget(dialog);
}

// ----------------------------------------------------------------------------
//
void midas_dialog::update(Spectrum *sp)
{
  this->spectrum = sp;

  Stringy dialog_title = (spectrum == NULL ? Stringy("Midas") :
			  Stringy("Midas ") + spectrum->name());
  ws.set_dialog_title(dialog, dialog_title);

  spectrum_menu->set_spectrum_choice(spectrum);
}

// ----------------------------------------------------------------------------
//
void midas_dialog::apply()
{
  bool midas_started = start_midas();
  if (mc == NULL)
    return;		// Couldn't start Midas

  Stringy new_pdb_path = ws.edit_value(pdb_file);
  bool pdb_file_changed = (new_pdb_path != pdb_path);
  if (pdb_file_changed)
    {
      Model *model = read_pdb_model(new_pdb_path);
      if (model == NULL)
	{
	  Reporter &rr = session.reporter();
	  rr.warning("Couldn't read pdb file %s\n", new_pdb_path.cstring());
	  return;
	}
      if (spectrum)
	{
	  Molecule *m = spectrum->molecule();
	  m->remove_all_models();
	  m->add_model(model);
	}
    }

  if (pdb_file_changed || midas_started)
    {
      mc->send_command("~open %d\n", PDB_MODEL_NUMBER);
      mc->send_command("open %d %s\n", PDB_MODEL_NUMBER,
		       new_pdb_path.cstring());
      mc->send_command("echo Opened PDB file %s\n", new_pdb_path.cstring());
    }

  Spectrum *new_spectrum = spectrum_menu->selected_spectrum();
  Stringy new_color = ws.option_selected(line_color);
  bool new_all_assign = ws.switch_state(sw, "allAssignments");
  bool redisplay = (pdb_file_changed || midas_started ||
		    (new_spectrum != spectrum && new_spectrum != NULL) ||
		    new_color != color ||
		    new_all_assign != show_all_assignments);
  this->spectrum = new_spectrum;
  this->color = new_color;
  this->show_all_assignments = new_all_assign;

  if (redisplay)
    {
      if (spectrum && show_all_assignments)
	{
	  mc->send_command("echo Showing assignments for spectrum %s\n",
			   spectrum->name().cstring());
	  int shown = show_assignments(spectrum->crosspeaks(), false);
	  mc->send_command("echo Displayed %d of %d peak assignments.\n",
			   shown, fully_assigned(spectrum->crosspeaks()));
	}
      if (spectrum)
	show_assignments(spectrum->selected_crosspeaks(),
			 show_all_assignments);
      if (!show_all_assignments)
	show_assignments(List(), true);		// Erase highlighting
    }
}

// ----------------------------------------------------------------------------
// Start Midas subprocess, or restart it if user exitted Midas.
//
bool midas_dialog::start_midas()
{
  if (mc && !mc->is_connection_good())
    {
      delete mc;
      mc = NULL;
    }

  if (mc == NULL)
    {
      mc = midas_connection::new_connection(ws);
      if (mc == NULL)
	{
	  Reporter &rr = session.reporter();
	  rr.warning("Couldn't start midas\n");
	}
      return mc != NULL;
    }

  return false;		// already started
}

// ----------------------------------------------------------------------------
// Two Midas models, one for showing assignment lines and one for highlighting
// certain lines are used.  This call replaces the current display lines
// for the appropriate model based on the highlight argument.
//
int midas_dialog::show_assignments(const List &peaks, bool highlight)
{
  Stringy color = (highlight ? Stringy("white") : this->color);
  int model_number = (highlight ?
		      HIGHLIGHT_MODEL_NUMBER : ASSIGNMENT_MODEL_NUMBER);
  return mc->show_assignments(peaks, color, model_number, highlight);
}

// ----------------------------------------------------------------------------
//
midas_connection::midas_connection(FILE *to_midas, FILE *from_midas,
				   const Stringy &temp_path, WinSys &winsys)
  : ws(winsys)
{
  this->to_midas = to_midas;
  this->from_midas = from_midas;
  object_file_path = temp_path;
  object_file = fopen(object_file_path.cstring(), "w");
  good_connection = true;
  ws.add_input_callback(from_midas, midas_request_cb, this);
}

// ----------------------------------------------------------------------------
//
midas_connection::~midas_connection()
{
  ws.remove_input_callback(from_midas, midas_request_cb, this);
  fclose(to_midas);
  fclose(from_midas);
  fclose(object_file);
  remove_file(object_file_path);
}

// ----------------------------------------------------------------------------
//
static Stringy midas_command()
{
  return getenv("MIDAS_COMMAND");
}

// ----------------------------------------------------------------------------
//
midas_connection *midas_connection::new_connection(WinSys &ws)
{
  Stringy mc = midas_command();
  if (mc.is_empty())
    return NULL;
  Stringy cmd = mc + " -d sparky";

  FILE *to_midas, *from_midas;
  if (!start_process(cmd, &to_midas, &from_midas))
    return NULL;

  fprintf(to_midas, "SYNC\n");
  fflush(to_midas);

  Stringy temp_path = temporary_file_name();

  return new midas_connection(to_midas, from_midas, temp_path, ws);
}

// ----------------------------------------------------------------------------
//
bool midas_connection::is_connection_good()
{
  if (good_connection)
    {
      good_connection = (ferror(to_midas) == 0 && feof(from_midas) == 0);
      if (! good_connection)
	ws.remove_input_callback(from_midas, midas_request_cb, this);
    }

  return good_connection;
}

// ----------------------------------------------------------------------------
//
void midas_connection::midas_request_cb(CB_Data mconn)
{
  midas_connection *mc = (midas_connection *) mconn;

  char buf[MAX_LINE_LENGTH];
  if (fgets(buf, sizeof buf, mc->from_midas))
    {
      fprintf(stderr, "Unrecognized midas request: %s\n", buf);

      fprintf(mc->to_midas, "SYNC\n");
      fflush(mc->to_midas);
    }
  else
    mc->is_connection_good();	// this removes input handler if Midas exitted
}

// ----------------------------------------------------------------------------
//
void midas_connection::set_color(const Stringy &color)
  { fprintf(object_file, ".color %s\n", color.cstring()); }
void midas_connection::move_to(const SPoint &p)
  { fprintf(object_file, ".m %s\n", point_format(p, "%.3g", false)); }
void midas_connection::draw_to(const SPoint &p)
  { fprintf(object_file, ".d %s\n", point_format(p, "%.3g", false)); }
void midas_connection::marker(const SPoint &p)
  { fprintf(object_file, ".marker %s\n", point_format(p, "%.3g", false)); }

// ----------------------------------------------------------------------------
//
Stringy midas_connection::send_command(const char *format, ...)
{
  Stringy reply;

  if (is_connection_good())
    {
      va_list args;
      va_start(args, format);
      vfprintf(to_midas, format, args);
      va_end(args);
      fflush(to_midas);

      char buf[MAX_LINE_LENGTH];
      while (fgets(buf, sizeof buf, from_midas) &&
	     strcmp(buf, "SYNC\n") != 0)
	reply << buf;
    }

  return reply;
}

// ----------------------------------------------------------------------------
//
int midas_connection::show_assignments(const List &peaks, const Stringy &color,
				       int model_number, bool highlight)
{
  set_color(color);

  int shown = 0;
  for (int pi = 0 ; pi < peaks.size() ; ++pi)
    if (show_assignment((CrossPeak *) peaks[pi], highlight))
      shown += 1;

  replace_objects(model_number);

  return shown;
}

// ----------------------------------------------------------------------------
//
void midas_connection::send_objects()
{
  fclose(object_file);
  send_command("open object %s\n", object_file_path.cstring());
  object_file = fopen(object_file_path.cstring(), "w");
}

// ----------------------------------------------------------------------------
//
void midas_connection::replace_objects(int model_number)
{
  fclose(object_file);
  send_command("~open %d\n", model_number);
  send_command("open object %d %s\n",
	       model_number, object_file_path.cstring());
  object_file = fopen(object_file_path.cstring(), "w");
}

// ----------------------------------------------------------------------------
//
bool midas_connection::show_assignment(CrossPeak *xp, bool highlight)
{
  Molecule *m = xp->spectrum()->molecule();
  const List &models = m->model_list();
  if (models.size() == 0)
    return false;
  Model *model = (Model *) models[0];

  SPoint position[DIM];
  for (int a = 0 ; a < xp->dimension() ; ++a)
    if (xp->resonance(a) == NULL ||
	! model->coordinates(*xp->resonance(a)->atom(), &position[a]))
      return false;

  if (highlight)
    {
      for (int a = 1 ; a < xp->dimension() ; ++a)
	{
	  SPoint offset = highlight_offset(position[a] - position[a-1]);
	  move_to(position[a-1] + offset);
	  draw_to(position[a] + offset);
	  move_to(position[a-1] - offset);
	  draw_to(position[a] - offset);
	}
    }
  else
    {
      move_to(position[0]);
      for (int a = 1 ; a < xp->dimension() ; ++a)
	draw_to(position[a]);
    }

  return true;
}

// ----------------------------------------------------------------------------
//
#define HIGHLIGHT_OFFSET_DISTANCE .2

static SPoint highlight_offset(const SPoint &dir)
{
  return dir.normal() * HIGHLIGHT_OFFSET_DISTANCE;
}

// ----------------------------------------------------------------------------
//
static int fully_assigned(const List &peaks)
{
  int count = 0;
  for (int pi = 0 ; pi < peaks.size() ; ++pi)
    if (fully_assigned((CrossPeak *) peaks[pi]))
      count += 1;

  return count;
}

// ----------------------------------------------------------------------------
//
static bool fully_assigned(CrossPeak *xp)
{
  for (int a = 0 ; a < xp->dimension() ; ++a)
    if (xp->resonance(a) == NULL)
      return false;

  return true;
}

// ----------------------------------------------------------------------------
//
static Stringy skip_line(const Stringy &s)
{
  char *end_line = strchr(s.cstring(), '\n');

  return (end_line ? end_line + 1 : "");
}

// ----------------------------------------------------------------------------
//
Stringy midas_connection::model_pdb_path(int model_number)
{
  Stringy reply = send_command("open");
  Stringy rest = skip_line(reply);	// Skip open command echo.
  while (!rest.is_empty())
    {
      Stringy model = first_token(rest, &rest);
      Stringy path = first_token(rest, &rest);
      if (atof(model.cstring()) == model_number)
	return path;
    }
  return "";
}
