// ----------------------------------------------------------------------------
//
#include "list.h"		// Use List
#include "memalloc.h"		// use new()
#include "notifier.h"		// use Notifier
#include "ornament.h"		// Use Ornament
#include "peak.h"		// Use Peak
#include "project.h"		// Use Project
#include "session.h"		// use Session
#include "spectrum.h"		// Use Spectrum
#include "stringc.h"		// Use Stringy
#include "uicomponents.h"	// use table_axis_label()
#include "uidialogs.h"		// Use show_ornament_color_dialog()
				// Use show_ornament_note_dialog()
				// Use show_ornament_size_dialog()
				// Use show_view_depth_dialog()
#include "uidialog.h"		// use Dialog, Dialog_Table
#include "uiview.h"		// Use selected_view()
#include "winsystem.h"		// Use ws.edit_field() ...

// ----------------------------------------------------------------------------
//
class ornament_dialog : public Dialog
{
public:
  ornament_dialog(Session &);
  ~ornament_dialog();

  static ornament_dialog *the(Session &);

  void show();
  void update();

private:
  Widget selected, ptable, locked;
  bool pos_modified, lw_modified, vol_modified, volerror_modified;

  static void selections_changed_cb(void *odialog, void *);
  static void color_cb(Widget, CB_Data, CB_Data);
  static void note_cb(Widget, CB_Data, CB_Data);
  static void sizes_cb(Widget, CB_Data, CB_Data);
  static void planes_cb(Widget, CB_Data, CB_Data);
  static void position_modified_cb(Widget, CB_Data, CB_Data);
  static void linewidth_modified_cb(Widget, CB_Data, CB_Data);
  static void volume_modified_cb(Widget, CB_Data, CB_Data);
  static void volume_error_modified_cb(Widget, CB_Data, CB_Data);

  void apply();
  void change_position(Peak *pk);
  void change_linewidth(Peak *pk);
  void change_volume(Peak *pk);
  void change_volume_error(CrossPeak *xp);
};

static Stringy ornament_description(const List &ornaments);
static int ornament_counts(const List &ornaments,
			   int *peaks, int *labels, int *lines, int *grids);
static Stringy item_description(int count, const char *name,
				const char **separator);

// ----------------------------------------------------------------------------
//
void show_ornament_dialog(Session &s)
  { ornament_dialog::the(s)->show(); }

// ----------------------------------------------------------------------------
//
ornament_dialog::ornament_dialog(Session &s)
  : Dialog(s, "ornamentDialog")
{
  selected = ws.create_label(dialog, "selected");

  Table_Entry label = TABLE_LABEL;
  Table_Entry value = TABLE_TEXT_FIELD;
  ptable = ws.widget_table(dialog, "peakProps", 5, 5,
			label, label, label, label, label,
			label, value, value, value, value,
			label, value, value, value, value,
			label, value, label, label, label,
			label, value, label, label, label);
  for (int a = 0 ; a < DIM ; ++a)
    {
      ws.add_text_field_changed_callback(ws.table_element(ptable, 1, a+1),
					 position_modified_cb, this);
      ws.add_text_field_changed_callback(ws.table_element(ptable, 2, a+1),
					 linewidth_modified_cb, this);
    }
  ws.add_text_field_changed_callback(ws.table_element(ptable, 3, 1),
				     volume_modified_cb, this);
  ws.add_text_field_changed_callback(ws.table_element(ptable, 4, 1),
				     volume_error_modified_cb, this);

  locked = ws.create_toggle_button(dialog, "lock");

  Widget props = ws.button_row(dialog, "properties",
			    "color", color_cb, this,
			    "note", note_cb, this,
			    "sizes", sizes_cb, this,
			    "planes", planes_cb, this,
			    NULL);

  Widget separator = ws.create_separator(dialog, "separator");

  Widget controls = ws.button_row(dialog, "controls",
			       "ok", ok_cb, this,
			       "apply", apply_cb, this,
			       "close", close_cb, this,
			       "help", help_cb, &s,
			       NULL);

  ws.column_attachments(separator, selected, ptable, locked, props,
		     separator, controls, END_OF_WIDGETS);

  Notifier &n = session.notifier();
  n.notify_me(nt_selected_ornaments_changed, selections_changed_cb, this);
}

// ----------------------------------------------------------------------------
//
ornament_dialog::~ornament_dialog()
{
  session.dialog_table().delete_dialog("ornament_dialog", this);

  Notifier &n = session.notifier();
  n.dont_notify_me(nt_selected_ornaments_changed, selections_changed_cb, this);
}

// ----------------------------------------------------------------------------
// The default ornament_dialog instance.
//
ornament_dialog *ornament_dialog::the(Session &s)
{
  Stringy name = "ornament_dialog";
  Dialog_Table &dt = s.dialog_table();
  if (dt.get_dialog(name) == NULL)
    dt.set_dialog(name, new ornament_dialog(s));
  return (ornament_dialog *) dt.get_dialog(name);
}

// ----------------------------------------------------------------------------
//
void ornament_dialog::selections_changed_cb(void *odialog, void *)
{
  ornament_dialog *od = (ornament_dialog *) odialog;

  if (od->shown())
    od->update();
}

// ----------------------------------------------------------------------------
//
void ornament_dialog::color_cb(Widget, CB_Data odialog, CB_Data)
{
  ornament_dialog *od = (ornament_dialog *) odialog;
  show_ornament_color_dialog(od->session);
}

// ----------------------------------------------------------------------------
//
void ornament_dialog::note_cb(Widget, CB_Data odialog, CB_Data)
{
  ornament_dialog *od = (ornament_dialog *) odialog;
  show_ornament_note_dialog(od->session);
}

// ----------------------------------------------------------------------------
//
void ornament_dialog::sizes_cb(Widget, CB_Data odialog, CB_Data)
{
  ornament_dialog *od = (ornament_dialog *) odialog;
  show_ornament_size_dialog(od->session);
}

// ----------------------------------------------------------------------------
// The view depth button doesn't belong in the ornament dialog.
// It is here only for historical reasons.
//
void ornament_dialog::planes_cb(Widget, CB_Data odialog, CB_Data)
{
  ornament_dialog *od = (ornament_dialog *) odialog;
  show_view_depth_dialog(od->session, selected_view(od->session));
}

// ----------------------------------------------------------------------------
//
void ornament_dialog::position_modified_cb(Widget, CB_Data odialog, CB_Data)
  { ((ornament_dialog *) odialog)->pos_modified = true; }
void ornament_dialog::linewidth_modified_cb(Widget, CB_Data odialog, CB_Data)
  { ((ornament_dialog *) odialog)->lw_modified = true; }
void ornament_dialog::volume_modified_cb(Widget, CB_Data odialog, CB_Data)
  { ((ornament_dialog *) odialog)->vol_modified = true; }
void ornament_dialog::volume_error_modified_cb(Widget, CB_Data odialog, CB_Data)
  { ((ornament_dialog *) odialog)->volerror_modified = true; }

// ----------------------------------------------------------------------------
//
void ornament_dialog::show()
{
  update();
  ws.show_dialog(dialog);
  ws.raise_widget(dialog);
}

// ----------------------------------------------------------------------------
//
void ornament_dialog::update()
{
  List sel = session.project().selected_ornaments();

  Stringy descrip = ornament_description(sel);
  ws.set_label_text(selected, descrip);

  if (sel.size() == 1)
    {
      Ornament *op = (Ornament *) sel[0];
      if (op->type() == peak)
	{
	  Peak *pk = (Peak *) op;
	  Spectrum *sp = op->spectrum();
	  SPoint pos = pk->position();
	  for (int a = 0 ; a < sp->dimension() ; ++a)
	    {
	      ws.set_label_text(ws.table_element(ptable, 0, a+1),
				table_axis_label(sp, a, ""));
	      ws.set_numeric_text_field(ws.table_element(ptable, 1, a+1),
					"%.5g", pos[a]);
	      if (pk->HasLinewidth())
		{
		  double lw = sp->scale(pk->linewidth(a), a, PPM, HZ);
		  ws.set_numeric_text_field(ws.table_element(ptable, 2, a+1),
					 "%.1f", lw);
		}
	      else
		ws.set_text_field(ws.table_element(ptable, 2, a+1), "");
	    }

	  double vol;
	  if (pk->volume(&vol))
	    ws.set_numeric_text_field(ws.table_element(ptable, 3, 1), "%.3g", vol);
	  else
	    ws.set_text_field(ws.table_element(ptable, 3, 1), "");

	  pos_modified = lw_modified = vol_modified = false;
	  ws.manage_table_children(ptable, 4, 1 + sp->dimension());
	}
      else
	ws.manage_table_children(ptable, 0, 0);

      if (is_crosspeak(op))
	{
	  CrossPeak *xp = (CrossPeak *) op;
	  double verr;
	  Stringy method;
	  if (xp->volume_error(&verr, &method))
	    ws.set_numeric_text_field(ws.table_element(ptable, 4, 1), "%.3g",
				      100 * verr);
	  else
	    ws.set_text_field(ws.table_element(ptable, 4, 1), "");
	  volerror_modified = false;
	  ws.manage_table_row(ptable, 4);
	}

      ws.set_toggle_button(locked, op->IsLocked(), true);
    }
  else
    {
      ws.manage_table_children(ptable, 0, 0);
      ws.set_toggle_button(locked, false, true);
    }
}

// ----------------------------------------------------------------------------
//
static Stringy ornament_description(const List &ornaments)
{
  int peaks, labels, lines, grids;
  Stringy descrip = "Description: ";

  ornament_counts(ornaments, &peaks, &labels, &lines, &grids);

  const char *separator = "";
  descrip << item_description(peaks, "peak", &separator);
  descrip << item_description(labels, "label", &separator);
  descrip << item_description(lines, "line", &separator);
  descrip << item_description(grids, "grid", &separator);

  return descrip;
}

// ----------------------------------------------------------------------------
//
static int ornament_counts(const List &ornaments,
			   int *peaks, int *labels, int *lines, int *grids)
{
  *peaks = *labels = *lines = *grids = 0;

  for (int oi = 0 ; oi < ornaments.size() ; ++oi)
    {
      Ornament *op = (Ornament *) ornaments[oi];
      if (is_crosspeak(op))
	*peaks += 1;
      else if (op->type() == label)
	*labels += 1;
      else if (op->type() == line)
	*lines += 1;
      else if (op->type() == grid)
	*grids += 1;
    }

  return *peaks + *labels + *lines + *grids;
}

// ----------------------------------------------------------------------------
//
static Stringy item_description(int count, const char *name,
				const char **separator)
{
  Stringy descrip;

  if (count > 0)
    {
      descrip << *separator << count << " " << name;
      if (count > 1)
	descrip << "s";
      *separator = ", ";
    }

  return descrip;
}

// ----------------------------------------------------------------------------
//
void ornament_dialog::apply()
{
  bool l = ws.toggle_button_state(locked);

  List sel = session.project().selected_ornaments();
  if (sel.size() == 1)
    {
      Ornament *op = (Ornament *) sel[0];

      if (op->type() == peak)
	{
	  Peak *pk = (Peak *) op;
	  change_position(pk);
	  change_linewidth(pk);
	  change_volume(pk);
	}
      if (is_crosspeak(op))
	change_volume_error((CrossPeak *) op);
      op->lock(l);
    }
}

// ----------------------------------------------------------------------------
//
void ornament_dialog::change_position(Peak *pk)
{
  if (pos_modified)
    {
      SPoint new_ppm(pk->dimension());
      for (int a = 0 ; a < pk->dimension() ; ++a)
	new_ppm[a] = ws.numeric_text_field(ws.table_element(ptable, 1, a+1));
      pk->SetLocation(new_ppm);
      pos_modified = false;
    }
}

// ----------------------------------------------------------------------------
//
void ornament_dialog::change_linewidth(Peak *pk)
{
  if (lw_modified)
    {
      SPoint new_hz(pk->dimension());
      for (int a = 0 ; a < pk->dimension() ; ++a)
	new_hz[a] = ws.numeric_text_field(ws.table_element(ptable, 2, a+1));
      SPoint new_linewidth = pk->spectrum()->scale(new_hz, HZ, PPM);
      pk->linewidth("manual", new_linewidth);
      lw_modified = false;
    }
}

// ----------------------------------------------------------------------------
//
void ornament_dialog::change_volume(Peak *pk)
{
  if (vol_modified)
    {
      double volume = ws.numeric_text_field(ws.table_element(ptable, 3, 1));
      pk->manual_volume(volume);
      vol_modified = false;
    }
}

// ----------------------------------------------------------------------------
//
void ornament_dialog::change_volume_error(CrossPeak *xp)
{
  if (volerror_modified)
    {
      double verror =
	ws.numeric_text_field(ws.table_element(ptable, 4, 1)) / 100;
      xp->set_volume_error(verror, "manual");
      volerror_modified = false;
    }
}
