// ----------------------------------------------------------------------------
// Resonance panel display
//
#include <math.h>		// Use fabs()

#include "assignguess.h"	// use Guess_Assignments
#include "condition.h"		// use Condition
#include "crosspeak.h"		// use CrossPeak
#include "label.h"		// use display_string()
#include "memalloc.h"		// use new()
#include "memalloc.h"		// use new()
#include "num.h"		// use interval_mod()
#include "rectangle.h"		// Use Axis
#include "resonance.h"		// Use Resonance
#include "session.h"		// use Session
#include "spectrum.h"		// Use Spectrum
#include "stringc.h"		// Use Stringy
#include "uiepanel.h"
#include "uirpanel.h"
#include "uiview.h"
#include "winsystem.h"		// Use add_work_procedure()

static bool find_resonance(const Stringy &rname, const List &pstrings,
			   int *index);
static Panel_String *closest_panel_string(const List &panel_strings,
					  double freq_ppm);
static List resonance_positioning(const List &reslist, double ppm_spacing,
				  double alias_min, double alias_max);
static void spread_positions(const List &pstrings, double spacing);
static bool join_clumps(const List &pstrings, bool joined[], double spacing);
static void position_clump(int k, const List &pstrings,
			   bool joined[], double spacing);

// ----------------------------------------------------------------------------
//
RPanel::RPanel(View *view, double a, double b, int screen_length, bool is_horz)
   : edge_panel(view->session().window_system(), a, b, screen_length, is_horz,
		view->frame(), (is_horz ? "xResPanel" : "yResPanel"))
{
  this->view = view;
  this->axis = (is_horz ? X : Y);
  this->filter = false;
  this->a_cluster = this->b_cluster = 0;
  this->highlight_color = "";
  this->update_strings = true;
  this->update_in_progress = false;
}

// ----------------------------------------------------------------------------
//
RPanel::~RPanel()
{
  ws.remove_work_procedure(draw_resonances_cb, this);
  free_panel_string_list_entries(panel_strings);
}

// ----------------------------------------------------------------------------
//
bool RPanel::filter_resonances() const
  { return filter; }
void RPanel::set_resonance_filtering(bool onoff)
{
  if (onoff != filter)
    {
      filter = onoff;
      resonance_changed();
    }
}

// ----------------------------------------------------------------------------
//
#define RPANEL_CLUSTER_FACTOR 1

void RPanel::view_region_changed()
{
  if (filter)
    resonance_changed();
}

// ----------------------------------------------------------------------------
//
void RPanel::resonance_changed()
{
  update_strings = true;
  erase();
}

// ----------------------------------------------------------------------------
//
void RPanel::redraw(double a_clip, double b_clip)
{
  double a, b;
  range(&a, &b);

  double csize = (RPANEL_CLUSTER_FACTOR + 1) * (b - a);
  bool new_cluster = (fabs(csize - (b_cluster-a_cluster)) > .01*csize ||
		      !is_between(a, a_cluster, b_cluster) ||
		      !is_between(b, a_cluster, b_cluster));
  if (new_cluster)
    {				// Recluster.
      double space = RPANEL_CLUSTER_FACTOR * (b-a)/2;
      a_cluster = a - space;
      b_cluster = b + space;
      update_strings = true;
      erase();
      return;
    }

  if (update_strings)
    {
      update_strings = false;
      free_panel_string_list_entries(panel_strings);
      panel_strings = resonance_positions(view->view_region(),
					  label_spacing());
    }

  this->a_clip = (update_in_progress ? min(this->a_clip, a_clip) : a_clip);
  this->b_clip = (update_in_progress ? max(this->b_clip, b_clip) : b_clip);
  update_in_progress = true;
  next_to_draw = 0;
  ws.add_work_procedure(draw_resonances_cb, this, -5);
}

// ----------------------------------------------------------------------------
//
void RPanel::draw_resonances_cb(CB_Data rpanel)
{
  RPanel *rp = (RPanel *) rpanel;

  if (rp->next_to_draw < rp->panel_strings.size())
    {
      Panel_String *ps = (Panel_String *) rp->panel_strings[rp->next_to_draw];
      rp->next_to_draw += 1;
      rp->draw_resonance(ps);
    }
  else
    rp->update_in_progress = false;

  if (rp->update_in_progress)
    rp->ws.add_work_procedure(draw_resonances_cb, rp, -5);
}

// ----------------------------------------------------------------------------
//
void RPanel::draw_resonance(Panel_String *ps)
{
  double left, right;
  Stringy dname = display_string(ps->label);
  text_width(dname.cstring(), false, &left, &right);
  bool redraw_tick = !is_clipped(ps->tick_position, ps->label_position,
				 a_clip, b_clip);
  bool redraw_label = !is_clipped(ps->label_position - left,
				  ps->label_position + right,
				  a_clip, b_clip);
  if (redraw_tick || redraw_label)
    set_drawing_color(ps->label == highlight_resonance ?
		      highlight_color : Color("black"));
  if (redraw_tick)
    draw_tick(ps->tick_position, ps->label_position);
  if (redraw_label)
    draw_label(ps->label_position, dname.cstring(), false);
}

// ----------------------------------------------------------------------------
//
List RPanel::resonance_positions(const SRegion &region, double ppm_spacing)
{
  Spectrum *sp = view->spectrum();
  SRegion r = sp->ppm_region();
  int a = view->axis(axis);
  double alias_max = r.max[a];
  double alias_min = alias_max - sp->ppm_sweep_width()[a];
  Condition *c = sp->condition();

  List reslist;
  if (filter)
    {
      const Guess_Assignments &ga = sp->GetAssignGuessOptions();
      reslist = ga.guessable_resonances(sp, region, a);
    }
  else
    {
      bool use_cluster = (region == view->view_region());
      double ppm_min = (use_cluster ? a_cluster : region.min[a]);
      double ppm_max = (use_cluster ? b_cluster : region.max[a]);
      reslist = c->interval_resonances(ppm_min, ppm_max,
				       sp->nucleus_type(a),
				       alias_min, alias_max);
    }

  return resonance_positioning(reslist, ppm_spacing, alias_min, alias_max);
}

// ----------------------------------------------------------------------------
//
void RPanel::new_highlight(const Stringy &rname, const Color &c)
{
  if (rname == highlight_resonance && c == highlight_color)
    return;

  draw_highlight(highlight_resonance, "black");
  draw_highlight(rname, c);
}

// ----------------------------------------------------------------------------
//
void RPanel::draw_highlight(const Stringy &rname, const Color &c)
{
  int index;
  if (find_resonance(rname, panel_strings, &index))
    {
      Panel_String *ps = (Panel_String *) panel_strings[index];
      set_drawing_color(c);
      draw_tick(ps->tick_position, ps->label_position);
      draw_label(ps->label_position,
		 display_string(ps->label).cstring(), false);
      highlight_resonance = rname;
      highlight_color = c;
    }
  else
    {
      highlight_resonance = "";
      highlight_color = "";
    }
}

// ----------------------------------------------------------------------------
//
static bool find_resonance(const Stringy &rname, const List &pstrings,
			   int *index)
{
  for (int k = 0 ; k < pstrings.size() ; ++k)
    if (((Panel_String *) pstrings[k])->label == rname)
      {
	*index = k;
	return true;
      }

  return false;
}

// ----------------------------------------------------------------------------
//
Panel_String::Panel_String(const Stringy &label,
			   double tick_pos, double label_pos)
{
  this->label = label;
  this->tick_position = tick_pos;
  this->label_position = label_pos;
}

// ----------------------------------------------------------------------------
//
List resonance_positioning(const List &reslist, double ppm_spacing,
			   double alias_min, double alias_max)
{
  List pstrings;
  for (int ri = 0 ; ri < reslist.size() ; ++ri)
    {
      Resonance *r = (Resonance *) reslist[ri];
      Stringy name = r->name();
      double pos = interval_mod(r->frequency(), alias_min, alias_max);
      pstrings.append(new Panel_String(name, pos, pos));
    }

  spread_positions(pstrings, ppm_spacing);

  return pstrings;
}

// ----------------------------------------------------------------------------
//
void free_panel_string_list_entries(List &pslist)
{
  for (int psi = 0 ; psi < pslist.size() ; ++psi)
    delete (Panel_String *) pslist[psi];
  pslist.erase();
}

// ----------------------------------------------------------------------------
// Find marker positions to avoid overlaps.
//
static void spread_positions(const List &pstrings, double spacing)
{
  bool *joined = new bool [pstrings.size()];

  for (int k = 0 ; k < pstrings.size() ; ++k)
    {
      Panel_String *ps = (Panel_String *) pstrings[k];
      ps->label_position = ps->tick_position;
      joined[k] = false;
    }

  while (join_clumps(pstrings, joined, spacing)) ;

  delete [] joined;
}

// ----------------------------------------------------------------------------
// Find marker positions to avoid overlaps.
//
static bool join_clumps(const List &pstrings, bool joined[], double spacing)
{
  bool did_merge = false;

  for (int k = 0 ; k < pstrings.size()-1 ; ++k)
    {
      Panel_String *ps0 = (Panel_String *) pstrings[k];
      Panel_String *ps1 = (Panel_String *) pstrings[k+1];
      if (!joined[k] && ps1->label_position - ps0->label_position < spacing)
	{
	  joined[k] = true;
	  position_clump(k, pstrings, joined, spacing);
	  did_merge = true;
	}
    }

  return did_merge;
}

// ----------------------------------------------------------------------------
// Find marker positions to avoid overlaps.
//
static void position_clump(int k, const List &pstrings,
			   bool joined[], double spacing)
{
  int a, b;
  double mean, mid;

  for (a = k ; a > 0 && joined[a-1] ; --a) ;
  for (b = k ; b < pstrings.size()-1 && joined[b] ; ++b) ;

  mean = 0;
  for (int i = a ; i <= b ; ++i)
    mean += ((Panel_String *)pstrings[i])->tick_position;
  mean /= b - a + 1;

  mid = (a+b) / 2.0;
  for (int i = a ; i <= b ; ++i)
    ((Panel_String *)pstrings[i])->label_position = mean + (i - mid) * spacing;
}

// ----------------------------------------------------------------------------
//
void RPanel::pointer_in(double x, double y)
{
  RPanel::pointer_move(x, y);
}

// ----------------------------------------------------------------------------
//
void RPanel::pointer_move(double x, double y)
{
  CrossPeak *xp = view->over_visible_crosspeak(x, y);
  int a = view->axis(axis);
  if (xp && xp->resonance(a))
    new_highlight(xp->resonance(a)->name(), "red");
  else
    {
      double freq = (is_horizontal() ? x : y);
      Panel_String *ps = closest_panel_string(panel_strings, freq);
      if (ps)
	new_highlight(ps->label, "blue");
      else
	new_highlight("", "");
    }
}

// ----------------------------------------------------------------------------
//
static Panel_String *closest_panel_string(const List &panel_strings,
					  double freq_ppm)
{
  Panel_String *closest = NULL;
  double min_sep = 0;	// Suppress compiler warning about uninitialized var

  for (int psi = 0 ; psi < panel_strings.size() ; ++psi)
    {
      Panel_String *ps = (Panel_String *) panel_strings[psi];
      double sep = abs(ps->tick_position - freq_ppm);
      if (closest == NULL || sep < min_sep)
	{
	  min_sep = sep;
	  closest = ps;
	}
    }
  return closest;
}

// ----------------------------------------------------------------------------
//
void RPanel::pointer_out(double, double)
{
  new_highlight("", "");
}
