// ----------------------------------------------------------------------------
//

#include <math.h>	// Use fabs()
#include <stdio.h>	// Use sprintf()
#include <string.h>	// use strcpy()

#include "crosspeak.h"		// Use CrossPeak
#include "num.h"		// Use abs()
#include "session.h"		// use Session
#include "spectrum.h"		// Use Spectrum
#include "stringc.h"		// Use Stringy
#include "uiepanel.h"
#include "uiscale.h"
#include "uiview.h"		// Use View
#include "winsystem.h"		// Use set_label_string()

static double graph_scale(double min);
static void value_string(double x, double precision, char *buffer);

// ----------------------------------------------------------------------------
//
Scale::Scale(View *view, double a, double b, int screen_length, bool is_horz)
  : edge_panel(view->session().window_system(), a, b, screen_length, is_horz,
	       view->frame(), (is_horz ? "xScale" : "yScale"))
	    
{
  this->view = view;
  value = ws.create_label(edge_panel::widget(), "value");
  lbl = ws.create_label(edge_panel::widget(), "label");
  ws.place_widget(lbl, 0, 0);

  update();
}

// ----------------------------------------------------------------------------
//
void Scale::update()
{
  this->axis = (is_horizontal() ? view->axis(X) : view->axis(Y));
  ws.set_label_text(lbl, view->axis_label(axis));
  if (ws.is_viewable(value))
    {
      int pos = (is_horizontal() ? ws.widget_x(value) : ws.widget_y(value));
      int size = (is_horizontal() ?
		  ws.widget_width(value) : ws.widget_height(value));
      show_pointer_value(draw_position(pos + size/2), NULL);
    }
  erase();
}

// ----------------------------------------------------------------------------
//
double Scale::scale_value(double d)
{
  return view->spectrum()->map(d, axis, PPM, view->axis_units());
}

// ----------------------------------------------------------------------------
//
double Scale::ppm_value(double scale_value)
{
  return view->spectrum()->map(scale_value, axis, view->axis_units(), PPM);
}

// ----------------------------------------------------------------------------
//
void Scale::pointer_in(double x, double y)
{
  show_pointer_value((is_horizontal() ? x : y),
		     view->over_visible_crosspeak(x, y));
}

// ----------------------------------------------------------------------------
//
void Scale::pointer_move(double x, double y)
{
  show_pointer_value((is_horizontal() ? x : y),
		     view->over_visible_crosspeak(x, y));
}

// ----------------------------------------------------------------------------
//
void Scale::pointer_out(double, double)
{
  ws.unplace_widget(value);
}

// ----------------------------------------------------------------------------
//
void Scale::show_pointer_value(double p, CrossPeak *xp)
{
  Stringy vstring = formatted_string("%.4g", scale_value(p));
  if (xp && xp->IsAliased(axis))
    vstring = formatted_string("%.4g\naliased",
			       scale_value(xp->frequency(axis)));

  ws.set_label_text(value, vstring);

  int size = (is_horizontal() ?
	      ws.widget_width(value) : ws.widget_height(value));
  int pos = paint_position(p) - size / 2;
  if (is_horizontal())
    ws.place_widget(value, pos, 0);
  else
    ws.place_widget(value, 0, pos);
}

// ----------------------------------------------------------------------------
//
void Scale::redraw(double a_clip, double b_clip)
{
  double a, b, first, last, spacing, x;
  char val_string[64];

  range(&a, &b);
  tick_spacing(a, b, &first, &last, &spacing);
  set_drawing_color("black");
  for (x = first ; x <= last ; x += spacing)
    {
      value_string(x, spacing, val_string);
      double position = ppm_value(x);
      double left, right;
      text_width(val_string, is_horizontal(), &left, &right);
      if (!is_clipped(position - left, position + right, a_clip, b_clip))
	{
	  draw_tick(position, position);
	  draw_label(position, val_string, is_horizontal());
	}
    }
}

// ----------------------------------------------------------------------------
//
void Scale::tick_spacing(double a, double b,
			 double *first, double *last, double *spacing)
{
  double x1 = scale_value(a);
  double x2 = scale_value(b);
  double minimum_spacing =
    4 * abs(scale_value(label_spacing()) - scale_value(0));

  *spacing = graph_scale(minimum_spacing);
  *first = round_multiple(min(x1, x2), *spacing);
  *last = max(x1, x2);
}

// ----------------------------------------------------------------------------
//
static double graph_scale(double min)
{
  long exp;
  double mant, decade, f;

  scientific_form(min, &mant, &exp, &decade);
  if (mant <= .2)
    f = .2;
  else if (mant <= .5)
    f = .5;
  else
    f = 1;

  return f * decade;
}

// ----------------------------------------------------------------------------
//
static void value_string(double x, double precision, char *buffer)
{
  char format[64];
  long e;

  e = (long) floor(log10(precision));
  if (e >= 0)
    strcpy(format, "%.0f");
  else if (e > -50)
    sprintf(format, "%%.%ldf", -e);
  else
    strcpy(format, "%g");

  sprintf(buffer, format, x);
}
