// ----------------------------------------------------------------------------
// 1-D spectrum data slice display
//
#include <math.h>		// Use floor(), ceil(), pow()

#include "linefit.h"		// use add_fit_peaks()
#include "memalloc.h"		// use new()
#include "num.h"		// Use round()
#include "rectangle.h"		// Use Axis
#include "session.h"		// use Session
#include "spectrum.h"		// Use Spectrum
#include "spoint.h"		// Use IRegion
#include "uiepanel.h"		// Use edge_panel()
#include "uislice.h"		// Use Slice
#include "uiview.h"		// Use View
#include "winsystem.h"		// use ws.add_event_callback()

static double data_min(float *data, int length);
static double data_max(float *data, int length);
static double change_scale(double value, double zero_value, double max_value,
			   double new_max);

// ----------------------------------------------------------------------------
//
Slice::Slice(View *view, double a, double b, int screen_length, bool is_horz)
   : edge_panel(view->session().window_system(), a, b, screen_length, is_horz,
		view->frame(), (is_horz ? "xSlicePanel" : "ySlicePanel"))
{
  this->view = view;
  this->axis = (is_horz ? X : Y);

  if (!view->pointer_position(&(this->center)))
    this->center = view->center();

  this->max_height = view->positive_contours().levels.highest();
  this->min_height = view->negative_contours().levels.highest();

  this->auto_adjust = true;
  this->subtract_peaks = false;
  this->dragging = false;

  ws.add_event_callback(button_1_press_event, widget(), button_press_cb, this);
  ws.add_event_callback(pointer_drag_event, widget(), pointer_drag_cb, this);
  ws.add_event_callback(button_1_release_event,
		     widget(), button_release_cb, this);
}

// ----------------------------------------------------------------------------
//
Slice::~Slice()
{
  ws.remove_event_callback(button_1_press_event, widget(), button_press_cb, this);
  ws.remove_event_callback(pointer_drag_event, widget(), pointer_drag_cb, this);
  ws.remove_event_callback(button_1_release_event,
			widget(), button_release_cb, this);
}

// ----------------------------------------------------------------------------
//
void Slice::button_press_cb(Widget, CB_Data slice, CB_Data event)
{
  Slice *s = (Slice *) slice;
  s->dragging = true;
  s->ws.event_xy(event, &s->last_x, &s->last_y);
}

// ----------------------------------------------------------------------------
//
#define PIXELS_PER_SCALE_DOUBLING 100
void Slice::pointer_drag_cb(Widget, CB_Data slice, CB_Data event)
{
  Slice *s = (Slice *) slice;
  int x, y;
  if (s->dragging && s->ws.event_xy(event, &x, &y))
    {
      int motion = (abs(x - s->last_x) >= abs(y - s->last_y) ?
		    x - s->last_x : y - s->last_y);
      s->last_x = x;
      s->last_y = y;
      if (s->ws.shift_modifier(event))
	{
	  double h = s->thickness();
	  if (h > 0)
	    {
	      double f = motion * s->pixel_size() / h;
	      s->view->shift_slices(f * (s->max_height - s->min_height));
	    }
	}
      else
	{
	  double f = pow(2, (double) motion / PIXELS_PER_SCALE_DOUBLING);
	  s->view->scale_slices(f);
	}
    }
}

// ----------------------------------------------------------------------------
//
void Slice::button_release_cb(Widget, CB_Data slice, CB_Data)
{
  Slice *s = (Slice *) slice;
  s->dragging = false;
}

// ----------------------------------------------------------------------------
//
SPoint Slice::section() const
  { return center; }

// ----------------------------------------------------------------------------
//
void Slice::height(double *hmin, double *hmax) const
{
  *hmin = min_height;
  *hmax = max_height;
}

// ----------------------------------------------------------------------------
//
void Slice::set_height(double hmin, double hmax)
{
  min_height = hmin;
  max_height = hmax;
  erase();
}

// ----------------------------------------------------------------------------
//
bool Slice::auto_scale() const
  { return auto_adjust; }
void Slice::set_auto_scale(bool onoff)
{
  if (onoff != auto_adjust)
    {
      auto_adjust = onoff;
      erase();
    }
}

// ----------------------------------------------------------------------------
//
bool Slice::subtract_fit_peaks() const
  { return subtract_peaks; }
void Slice::show_peaks_subtracted(bool onoff)
{
  if (onoff != subtract_peaks)
    {
      subtract_peaks = onoff;
      erase();
    }
}

// ----------------------------------------------------------------------------
//
void Slice::redraw(double a_clip, double b_clip)
{
  if (auto_scaling_update())
    return;			// Scale change will cause a full redraw.

  int d1, d2;
  float *spect_data, *fit_data, *sel_data, *diff_data;
  get_slice_data(a_clip, b_clip, &d1, &d2,
		 &spect_data, &fit_data, &sel_data, &diff_data);

  if (diff_data)
    draw_curve("green", d1, d2, diff_data);
  if (sel_data)
    draw_curve("cyan", d1, d2, sel_data);
  draw_curve("blue", d1, d2, fit_data);
  draw_curve("black", d1, d2, spect_data);

  delete [] diff_data;
  delete [] fit_data;
  delete [] sel_data;
  delete [] spect_data;
}

// ----------------------------------------------------------------------------
//
bool Slice::auto_scaling_update()
{
  if (!auto_adjust)
    return false;

  double a, b;
  int d1, d2;
  float *spect_data, *fit_data, *sel_data, *diff_data;

  range(&a, &b);
  get_slice_data(a, b, &d1, &d2,
		 &spect_data, &fit_data, &sel_data, &diff_data);

  int length = d2 - d1 + 1;
  double hmin = data_min(spect_data, length);
  double hmax = data_max(spect_data, length);
  hmin = min(hmin, data_min(fit_data, length));
  hmax = max(hmax, data_max(fit_data, length));
  if (sel_data)
    {
      hmin = min(hmin, data_min(sel_data, length));
      hmax = max(hmax, data_max(sel_data, length));
    }
  if (diff_data)
    {
      hmin = min(hmin, data_min(diff_data, length));
      hmax = max(hmax, data_max(diff_data, length));
    }

  double cur_hmin, cur_hmax;
  height(&cur_hmin, &cur_hmax);
  bool changed = (hmin != cur_hmin || hmax != cur_hmax);
  if (changed)
    set_height(hmin, hmax);

  delete [] diff_data;
  delete [] fit_data;
  delete [] sel_data;
  delete [] spect_data;

  return changed;
}

// ----------------------------------------------------------------------------
//
static double data_min(float *data, int length)
{
  double dmin = data[0];

  for (int k = 1 ; k < length ; ++k)
    if (data[k] < dmin)
      dmin = data[k];

  return dmin;
}

// ----------------------------------------------------------------------------
//
static double data_max(float *data, int length)
{
  double dmax = data[0];

  for (int k = 1 ; k < length ; ++k)
    if (data[k] > dmax)
      dmax = data[k];

  return dmax;
}

// ----------------------------------------------------------------------------
//
void Slice::get_slice_data(double p1, double p2, int *d1, int *d2,
			   float **spectrum_data, float **peak_fit,
			   float **selected_peak, float **difference) const
{
  *d1 = round(view->map(p2, axis, PPM, INDEX));
  *d2 = round(view->map(p1, axis, PPM, INDEX));

  IRegion region = slice_region(*d1, *d2);
  int size = region.volume();

  float *spect_data = new float[size];
  Spectrum *sp = view->spectrum();
  sp->heights_for_region(region, spect_data);

  float *fit_data = new float[size];
  for (int k = 0 ; k < size ; ++k)
    fit_data[k] = 0;
  add_fit_peaks(sp->peaklets(), 1.0, region, fit_data);

  float *sel_data = new float[size];
  for (int k = 0 ; k < size ; ++k)
    sel_data[k] = 0;
  add_fit_peaks(sp->selected_peaklets(), 1.0, region, sel_data);

  bool is_zero = true;
  for (int k = 0 ; k < size && is_zero ; ++k)
    if (sel_data[k] != 0)
      is_zero = false;
  if (is_zero)
    {
      delete [] sel_data;
      sel_data = NULL;
    }

  float *diff_data = NULL;
  if (subtract_peaks)
    {
      diff_data = new float[size];
      for (int k = 0 ; k < size ; ++k)
	diff_data[k] = spect_data[k] - fit_data[k];
    }

  *spectrum_data = spect_data;
  *peak_fit = fit_data;
  *selected_peak = sel_data;
  *difference = diff_data;
}

// ----------------------------------------------------------------------------
//
IRegion Slice::slice_region(int d1, int d2) const
{
  int parallel_axis = view->axis(axis);
  int perp_axis = view->axis(axis == X ? Y : X);
  Spectrum *sp = view->spectrum();
  IPoint c = sp->map(center, PPM, INDEX).rounded();
  int f = c[perp_axis];

  return IRegion(parallel_axis, d1, d2, perp_axis, f, f, c);
}

// ----------------------------------------------------------------------------
//
void Slice::draw_curve(const Color &color, int dmin, int dmax, float *data)
{
  set_drawing_color(color);

  double h = thickness();
  int points =  dmax - dmin + 1;
  for (int k = 0 ; k < points-1 ; ++k)
    {
      double d1 = view->map((double) (dmin + k), axis, INDEX, PPM);
      double h1 = change_scale(data[k], min_height, max_height, h);
      double d2 = view->map((double) (dmin + k + 1), axis, INDEX, PPM);
      double h2 = change_scale(data[k+1], min_height, max_height, h);
      if (is_horizontal())
	draw_line(d1, h1, d2, h2);
      else
	draw_line(h-h1, d1, h-h2, d2);
    }
}

// ----------------------------------------------------------------------------
//
static double change_scale(double value, double zero_value, double max_value,
			   double new_max)
{
  return (max_value > zero_value ?
	  new_max * (value - zero_value) / (max_value - zero_value) :
	  0);
}

// ----------------------------------------------------------------------------
//
void Slice::pointer_in(double, double)
{
  update_center();
}

// ----------------------------------------------------------------------------
//
void Slice::pointer_move(double, double)
{
  update_center();
}

// ----------------------------------------------------------------------------
//
void Slice::pointer_out(double, double)
{
}

// ----------------------------------------------------------------------------
//
void Slice::view_region_changed()
{
  update_center();
}

// ----------------------------------------------------------------------------
//
void Slice::update_center()
{
  SPoint p;
  if (view->pointer_position(&p))
    {
      p[this->view->axis(this->axis)] = 0;
      if (p != this->center)
	{
	  this->center = p;
	  erase();
	}
    }
}
