// ----------------------------------------------------------------------------
//

#include "list.h"		// Use List
#include "memalloc.h"		// use new()
#include "notifier.h"		// use Notifier
#include "num.h"		// Use max()
#include "project.h"		// use Project
#include "reporter.h"		// use Reporter
#include "session.h"		// use Session
#include "spectrum.h"		// Use Spectrum
#include "uidialogs.h"		// use help_cb()
#include "uidialog.h"		// use Dialog, Dialog_Table
#include "uiview.h"		// Use View
#include "winsystem.h"		// use WinSys

// ----------------------------------------------------------------------------
//
class sync_dialog : public Dialog
{
public:
  sync_dialog(Session &);
  ~sync_dialog();

  static sync_dialog *the(Session &);

  void show();
  void update();

private:
  Widget table;

  static void view_list_changed_cb(void *sdialog, void *);
  static void synchronize_cb(Widget, CB_Data, CB_Data);
  static void unsynchronize_cb(Widget, CB_Data, CB_Data);

  List selected_view_axes();
};

static int maximum_view_dimension(const List &vlist);
static void create_table_views(WinSys &, const List &vlist, Widget table);
static void label_view_axis_sets(WinSys &, Widget table, const List &views);
static Stringy sync_set_name(const view_axis &va);
static void synchronize(const List &view_axes, Reporter &);
static bool axis_types_match(const view_axis &va1, const view_axis &va2);
static void unsynchronize(const List &view_axes);

// ----------------------------------------------------------------------------
//
void show_sync_dialog(Session &s)
  { sync_dialog::the(s)->show(); }

// ----------------------------------------------------------------------------
//
sync_dialog::sync_dialog(Session &s) : Dialog(s, "syncDialog")
{
  Table_Entry label = TABLE_LABEL;
  table = ws.widget_table(dialog, "viewTable", 1, 5,
		       label, label, label, label, label);

  Widget separator = ws.create_separator(dialog, "separator");

  Widget controls = ws.button_row(dialog, "controls",
			       "synchronize", synchronize_cb, this,
			       "unsynchronize", unsynchronize_cb, this,
			       "close", close_cb, this,
			       "help", help_cb, &s,
			       NULL);

  ws.column_attachments(table, table, separator, controls, END_OF_WIDGETS);

  Notifier &n = session.notifier();
  n.notify_me(nt_added_view_to_project, view_list_changed_cb, this);
  n.notify_me(nt_removed_view_from_project, view_list_changed_cb, this);

  update();
}

// ----------------------------------------------------------------------------
//
sync_dialog::~sync_dialog()
{
  session.dialog_table().delete_dialog("sync_dialog", this);

  Notifier &n = session.notifier();
  n.dont_notify_me(nt_added_view_to_project, view_list_changed_cb, this);
  n.dont_notify_me(nt_removed_view_from_project, view_list_changed_cb, this);
}

// ----------------------------------------------------------------------------
// The default sync_dialog instance.
//
sync_dialog *sync_dialog::the(Session &s)
{
  Stringy name = "sync_dialog";
  Dialog_Table &dt = s.dialog_table();
  if (dt.get_dialog(name) == NULL)
    dt.set_dialog(name, new sync_dialog(s));
  return (sync_dialog *) dt.get_dialog(name);
}

// ----------------------------------------------------------------------------
//
void sync_dialog::view_list_changed_cb(void *sdialog, void *view)
{
  sync_dialog *sd = (sync_dialog *) sdialog;

  if (sd->shown() && ((View *)view)->is_top_level_window())
    sd->update();
}

// ----------------------------------------------------------------------------
//
void sync_dialog::synchronize_cb(Widget, CB_Data client_data, CB_Data)
{
  sync_dialog *sd = (sync_dialog *) client_data;
  List valist = sd->selected_view_axes();
  synchronize(valist, sd->session.reporter());
  free_view_axis_list_entries(valist);
  sd->update();
}

// ----------------------------------------------------------------------------
//
void sync_dialog::unsynchronize_cb(Widget, CB_Data client_data, CB_Data)
{
  sync_dialog *sd = (sync_dialog *) client_data;
  List valist = sd->selected_view_axes();
  unsynchronize(valist);
  free_view_axis_list_entries(valist);
  sd->update();
}

// ----------------------------------------------------------------------------
//
void sync_dialog::show()
{
  update();
  ws.show_dialog(dialog);
  ws.raise_widget(dialog);
}

// ----------------------------------------------------------------------------
//
void sync_dialog::update()
{
  List vlist = session.project().top_level_views();
  create_table_views(ws, vlist, table);
  int rows = vlist.size() + 1;
  int cols = maximum_view_dimension(vlist) + 1;
  ws.manage_table_children(table, rows, cols);
  label_view_axis_sets(ws, table, vlist);
}

// ----------------------------------------------------------------------------
//
static int maximum_view_dimension(const List &vlist)
{
  int max_dim = 2;
  for (int vi = 0 ; vi < vlist.size() ; ++vi)
    max_dim = max(max_dim, ((View *) vlist[vi])->spectrum()->dimension());
  return max_dim;
}

// ----------------------------------------------------------------------------
//
static void create_table_views(WinSys &ws, const List &vlist, Widget table)
{
  for (int k = 0 ; k < vlist.size() ; ++k)
    {
      View *v = (View *) vlist[k];
      Widget view_widget = ws.table_element(table, k+1, 0);
      if (!view_widget)
	{
	  view_widget = ws.create_label(table,
					ws.table_element_name(table, k+1, 0));
	  for (int a = 0 ; a < DIM ; ++a)
	    ws.create_toggle_button(table, 
				    ws.table_element_name(table, k+1, a+1));
	}
      ws.set_label_text(view_widget, v->name());
    }
}

// ----------------------------------------------------------------------------
//
static void label_view_axis_sets(WinSys &ws, Widget table, const List &views)
{
  for (int k = 0 ; k < views.size() ; ++k)
    {
      View *v = (View *) views[k];
      int r = k + 1;

      for (int a = 0 ; a < v->spectrum()->dimension() ; ++a)
	ws.set_toggle_button_text(ws.table_element(table, r, a+1),
				  sync_set_name(view_axis(v, a)));
      for (int a = v->spectrum()->dimension() ; a < DIM ; ++a)
	ws.set_toggle_button_text(ws.table_element(table, r, a+1), "");
    }
}

// ----------------------------------------------------------------------------
//
static Stringy sync_set_name(const view_axis &va)
{
  char set_name = ' ';
  Project &proj = va.view()->session().project();
  const List *set = proj.synchronized_view_axes(va);
  if (set)
    {
      const List &sync_sets = proj.synchronized_view_axis_sets();
      int set_num = sync_sets.find(set);
      set_name =  'a' + set_num;
    }
  Stringy nucleus = va.view()->spectrum()->nucleus_type(va.axis());
  return formatted_string("%s %c", nucleus.cstring(), set_name);
}

// ----------------------------------------------------------------------------
//
List sync_dialog::selected_view_axes()
{
  List sel;
  
  for (int r = 1 ; ; ++r)
    {
      Widget view_widget = ws.table_element(table, r, 0);
      if (!view_widget || !ws.is_viewable(view_widget))
	break;

      Stringy view_name = ws.label_text(view_widget);
      View *v = session.project().find_view(view_name);
      for (int a = 0 ; a < v->spectrum()->dimension() ; ++a)
	{
	  Widget tb = ws.table_element(table, r, a+1);
	  if (ws.toggle_button_state(tb))
	    {
	      sel.append(new view_axis(v, a));
	      ws.set_toggle_button(tb, false, true);
	    }
	}
    }

  return sel;
}

// ----------------------------------------------------------------------------
//
static void synchronize(const List &view_axes, Reporter &rr)
{
  if (view_axes.size() >= 2)
    {
      view_axis *first = (view_axis *) view_axes[0];
      for (int ai = 1 ; ai < view_axes.size() ; ++ai)
	if (!axis_types_match(*first, *(view_axis *) view_axes[ai]))
	  {
	    rr.message("View synchronization: Axis nuclei types don't match.\n");
	    return;
	  }
      Project &proj = first->view()->session().project();
      for (int ai = 1 ; ai < view_axes.size() ; ++ai)
	proj.synchronize_view_axes(*first, *(view_axis *)view_axes[ai]);
    }
}

// ----------------------------------------------------------------------------
//
static bool axis_types_match(const view_axis &va1, const view_axis &va2)
{
  return (va1.view()->spectrum()->nucleus_type(va1.axis())
	  == va2.view()->spectrum()->nucleus_type(va2.axis()));
}

// ----------------------------------------------------------------------------
//
static void unsynchronize(const List &view_axes)
{
  for (int ai = 0 ; ai < view_axes.size() ; ++ai)
    {
      view_axis *va = (view_axis *) view_axes[ai];
      Project &proj = va->view()->session().project();
      proj.unsynchronize_view_axis(*va);
    }
}
