// ----------------------------------------------------------------------------
//

#include "condition.h"		// Use Condition
#include "crosspeak.h"		// Use CrossPeak
#include "list.h"		// Use List
#include "memalloc.h"		// use new()
#include "molecule.h"		// Use Molecule
#include "notifier.h"		// use Notifier
#include "project.h"		// use Project
#include "session.h"		// use Session
#include "spectrum.h"		// Use Spectrum
#include "stringc.h"		// Use Stringy
#include "uiview.h"		// Use View
#include "uidialogs.h"		// use help_cb()
#include "uidialog.h"		// use Dialog, Dialog_Table
#include "winsystem.h"

// ----------------------------------------------------------------------------
//
class hide_dialog : public Dialog
{
public:
  hide_dialog(Session &);
  ~hide_dialog();

  static hide_dialog *the(Session &);

  void show();
  void update();
  void change(View *view);

private:
  Widget vlist;
  List views;

  static void view_shown_cb(void *hdialog, void *view);
  static void view_list_changed_cb(void *hdialog, void *);
  static void toggle_cb(Widget, CB_Data, CB_Data);
  static void show_cb(Widget, CB_Data, CB_Data);
  static void hide_cb(Widget, CB_Data, CB_Data);

  void show_selected_views(bool show);
};

static Stringy view_text(View *view);
static Stringy nucleus_types(Spectrum *sp);
static Stringy spectrum_dimensions(Spectrum *sp);
static int assigned_peaks(Spectrum *sp);

// ----------------------------------------------------------------------------
//
void show_view_list_dialog(Session &s)
  { hide_dialog::the(s)->show(); }

// ----------------------------------------------------------------------------
//
hide_dialog::hide_dialog(Session &s) : Dialog(s, "viewListDialog")
{
  Widget scroller = ws.create_scrolled_list(dialog, "list", &vlist);
  ws.extended_selection_list(vlist);
  ws.add_list_double_click_callback(vlist, toggle_cb, this);

  Widget separator = ws.create_separator(dialog, "separator");

  Widget controls = ws.button_row(dialog, "controls",
			       "show", show_cb, this,
			       "hide", hide_cb, this,
			       "close", close_cb, this,
			       "help", help_cb, &s,
			       NULL);

  ws.column_attachments(scroller, scroller,
			separator, controls, END_OF_WIDGETS);

  Notifier &n = session.notifier();
  n.notify_me(nt_showed_view, view_shown_cb, this);
  n.notify_me(nt_added_view_to_project, view_list_changed_cb, this);
  n.notify_me(nt_removed_view_from_project, view_list_changed_cb, this);
}

// ----------------------------------------------------------------------------
//
hide_dialog::~hide_dialog()
{
  session.dialog_table().delete_dialog("hide_dialog", this);

  Notifier &n = session.notifier();
  n.dont_notify_me(nt_showed_view, view_shown_cb, this);
  n.dont_notify_me(nt_added_view_to_project, view_list_changed_cb, this);
  n.dont_notify_me(nt_removed_view_from_project, view_list_changed_cb, this);
}

// ----------------------------------------------------------------------------
// The default hide_dialog instance.
//
hide_dialog *hide_dialog::the(Session &s)
{
  Stringy name = "hide_dialog";
  Dialog_Table &dt = s.dialog_table();
  if (dt.get_dialog(name) == NULL)
    dt.set_dialog(name, new hide_dialog(s));
  return (hide_dialog *) dt.get_dialog(name);
}

// ----------------------------------------------------------------------------
//
void hide_dialog::view_shown_cb(void *hdialog, void *view)
{
  hide_dialog *hd = (hide_dialog *) hdialog;

  if (hd->shown())
    hd->change((View *) view);
}

// ----------------------------------------------------------------------------
//
void hide_dialog::view_list_changed_cb(void *hdialog, void *)
{
  hide_dialog *hd = (hide_dialog *) hdialog;

  if (hd->shown())
    hd->update();
}

// ----------------------------------------------------------------------------
//
void hide_dialog::toggle_cb(Widget w, CB_Data client_data, CB_Data)
{
  hide_dialog *hd = (hide_dialog *) client_data;
  int pos;
  if (hd->ws.selected_list_position(w, &pos))
    {
      View *view = (View *) hd->views[pos];
      view->show_view(!view->shown());
    }
}

// ----------------------------------------------------------------------------
//
void hide_dialog::show_cb(Widget, CB_Data client_data, CB_Data)
{
  hide_dialog *hd = (hide_dialog *) client_data;
  hd->show_selected_views(true);
}

// ----------------------------------------------------------------------------
//
void hide_dialog::hide_cb(Widget, CB_Data client_data, CB_Data)
{
  hide_dialog *hd = (hide_dialog *) client_data;
  hd->show_selected_views(false);
}

// ----------------------------------------------------------------------------
//
void hide_dialog::show()
{
  update();
  ws.show_dialog(dialog);
  ws.raise_widget(dialog);
}

// ----------------------------------------------------------------------------
//
void hide_dialog::update()
{
  ws.delete_list_items(vlist);
  views = session.project().top_level_views();
  for (int vi = 0 ; vi < views.size() ; ++vi)
    {
      View *view = (View *) views[vi];
      ws.add_list_item(vlist, view_text(view));
    }
  ws.set_visible_list_length(vlist, views.size() + 1);
}

// ----------------------------------------------------------------------------
//
void hide_dialog::change(View *view)
{
  if (view->is_top_level_window())
    {
      int vi = views.find(view);
      if (vi < views.size())
	ws.replace_list_item(vlist, vi, view_text(view));
    }
}

// ----------------------------------------------------------------------------
//
void hide_dialog::show_selected_views(bool show)
{
  int num, *positions;

  ws.selected_list_positions(vlist, &num, &positions);
  for (int c = 0 ; c < num ; ++c)
    ((View *) views[positions[c]])->show_view(show);
  ws.free_list_positions(positions);
}

// ----------------------------------------------------------------------------
//
static Stringy view_text(View *view)
{
  Spectrum *sp = view->spectrum();

  return formatted_string("%1s %16.16s %10.10s %10.10s %12s %14s %8d %8d",
			  (view->shown() ? "+" : "-"),
			  view->name().cstring(),
			  sp->molecule()->name().cstring(),
			  sp->condition()->name().cstring(),
			  nucleus_types(sp).cstring(),
			  spectrum_dimensions(sp).cstring(),
			  sp->crosspeaks().size(),
			  assigned_peaks(sp));
}

// ----------------------------------------------------------------------------
//
static Stringy nucleus_types(Spectrum *sp)
{
  Stringy nucleus_types;

  for (int a = 0 ; a < sp->dimension() ; ++a)
    nucleus_types << (a > 0 ? " " : "") << sp->nucleus_type(a);

  return nucleus_types;
}

// ----------------------------------------------------------------------------
//
static Stringy spectrum_dimensions(Spectrum *sp)
{
  Stringy dimensions;

  for (int a = 0 ; a < sp->dimension() ; ++a)
      dimensions <<
	(a > 0 ? " " : "") <<
	  formatted_string("%d", sp->index_range().size(a));

  return dimensions;
}

// ----------------------------------------------------------------------------
//
static int assigned_peaks(Spectrum *sp)
{
  int assigned = 0;
  const List &peaks = sp->crosspeaks();
  for (int pi = 0 ; pi < peaks.size() ; ++pi)
    if (((CrossPeak *) peaks[pi])->IsAssigned())
      assigned += 1;

  return assigned;
}
