/*
 * Undo.C	- Implementation of Undo
 */
#include "format.h"		// Use ornament_save_for_undo(), ...
#include "list.h"		// use List
#include "memalloc.h"		// use new()
#include "notifier.h"		// use Notifier
#include "reporter.h"		// use Reporter
#include "session.h"		// use Session
#include "undo.h"

#define MAX_LINE_LENGTH		4096

//
// Construct the Undo object
//
// The <doMsg> is printed only if the Undo fails.
// The <undoMsg> is printed when the Undo is successful.
//
Undo::Undo(Session &s, const Stringy &doMsg, const Stringy &undoMsg)
  : session(s)
{
	mDoMsg = doMsg;			// save the <doMsg>
	mUndoMsg = undoMsg;		// save the <undoMsg>
	mCount = 0;			// set ornament count to 0
	mOut = new ostrstream();

	Notifier &n = session.notifier();
	n.notify_me(nt_will_delete_ornament, will_delete_ornament_cb, this);
}

//
// Destruct the Undo object
//
Undo::~Undo()
{
  Notifier &n = session.notifier();
  n.dont_notify_me(nt_will_delete_ornament, will_delete_ornament_cb, this);

  if (mOut)
    delete mOut;
}

void Undo::will_delete_ornament_cb(void *undo, void *ornament)
{
  Undo *u = (Undo *) undo;
  u->ids.delete_ids(ornament);
}

/*
 * Finish the Undo object.
 */
void Undo::finish()
{
	*mOut << ends;

	/*
	 * Print a message to that effect that we are ready to undo.
	 */
	if (!mDoMsg.is_empty()) {
	  Reporter &rr = session.reporter();
	  rr.message("%d ornament%s %s.\n",
		     mCount, mCount == 1 ? "" : "s", mDoMsg.cstring());
	}
}

/*
 * Add Ornament <op> to the undo object.  Saves ornament state.
 */
void Undo::add(Ornament *op)
{
	// Save this ornament's state to file and increment the ornament count
	//
	if (mOut) {
		ornament_save_for_undo(op, *mOut, ids);
		mCount++;
	}
}

/*
 * Undo the operation.
 *
 * Restores the ornament from the string stream.
 */
bool Undo::undo()
{
	if (mOut == NULL)
	  return true;

	char *s = mOut->str();
	delete mOut;
	mOut = NULL;

	istrstream	i(s);
	char		buf[MAX_LINE_LENGTH];

	//
	// Restore from this file.
	//

	bool failed = false;
	while (! failed) {

	  /*
	   * Skip to '< ... >' line, which introduces an ornament.
	   * We skip forward in case some other ornament has not
	   * been properly restored.
	   */
	  for (;;) {
	    buf[0] = '\0';
	    if (i.getline(buf, sizeof buf).eof() || buf[0] == '<')
	      break;
	  }

	  if (buf[0] == '<')
	    failed = (ornament_restore_for_undo(session, buf, i, ids) == NULL);
	  else if (buf[0] == '\0')
	    break;
	  else
	    failed = true;
	}

	//
	// Should the string returned by ostrstream str() be
	// freed? deleted? or left for oststream to clean up?
	//
	// free(s);

	if (!mUndoMsg.is_empty() && mCount > 0) {
	  Reporter &rr = session.reporter();
	  rr.message("%d ornament%s %s%s\n",
		     mCount, (mCount == 1 ? "" : "s"),
		     mUndoMsg.cstring(), (failed ? " **FAILED**" : "."));
	}

	return !failed;
}
