// ----------------------------------------------------------------------------
// Read processed VNMR data and write UCSF format.
//

#include <iostream.h>		// Use cerr
#include <stdlib.h>		// Use exit()

#include "list.h"		// Use List
#include "memalloc.h"		// use new()
#include "memcache.h"		// use Memory_Cache
#include "nmrdata.h"		// Use standard_nucleus_name()
#include "stringc.h"		// Use Stringy
#include "ucsffile.h"		// Use write_ucsf_nmr_data()
#include "varianfile.h"		// Use varian_nmr_data()

#define MEMORY_CACHE_SIZE 8000000

// ----------------------------------------------------------------------------
//
static char *usage = 
"Syntax: vnmr2ucsf <procpar-file> <vnmr-data-file> <ucsf-file>\n"
"		   <nucleus1> <nucleus2> [nucleus3]\n"
"\n"
"The nucleus names for the f1, f2, and f3 axes should be H, C, or N.\n"
"Other parameters are read from the procpar file.  For a 2D experiment\n"
"in directory exp7 the procpar file would be exp7/procpar and the data\n"
"file would be exp7/datdir/phasefile.  To assure that phasefile contains\n"
"the spectrum data you need to use the VNMR flush command after displaying\n"
"the full spectrum with trace='f1'.  For 3D data you first use the\n"
"VNMR getplane command to produce separate files dataf1f3.* for the 2D\n"
"planes.  The vnmr2ucsf file arguments are then extr/procpar3d and\n"
"extr/dataf1f3 (without the plane number suffix).  For more information\n"
"see the Sparky documentation manual/files.html.\n";

// ----------------------------------------------------------------------------
//
int main(int argc, char **argv)
{
  if (argc != 6 && argc != 7)
    { cerr << usage; exit(1); }

  Stringy procpar_path = argv[1];
  Stringy vnmr_data_path = argv[2];
  Stringy ucsf_data_path = argv[3];
  List nucleus_names;
  for (int a = 4 ; a < argc ; ++a)
    nucleus_names.append(new Stringy(standard_nucleus_name(argv[a])));

  Stringy error_msg;
  Memory_Cache mcache(MEMORY_CACHE_SIZE);
  NMR_Data *varian_data = varian_nmr_data(procpar_path, vnmr_data_path,
					  nucleus_names, &error_msg, &mcache);
  free_string_list_entries(nucleus_names);
  if (varian_data == NULL)
    { cerr << "vnmr2ucsf: " << error_msg << endl; exit(1); }

  if (!write_ucsf_nmr_data(varian_data, ucsf_data_path, &error_msg, &mcache))
    { cerr << "vnmr2ucsf: " << error_msg << endl; exit(1); }

  return 0;
}
