// ----------------------------------------------------------------------------
//
#include <tcl.h>
#include <tk.h>

// ----------------------------------------------------------------------------
// Include window system routines portable across all platforms.
//
#include "winsystem-all.cc"

// ----------------------------------------------------------------------------
//
static XImage *rotated_image(Display *display, Window win, XImage *image,
			     bool counter_clockwise);
static Window find_netscape_window(Display *dpy);
static Window find_window_property(Display *d, Window w, Atom a,
				   int max_depth);
static void netscape_command(Display *dpy, Window window, const Stringy &cmd);

// ----------------------------------------------------------------------------
//
static bool x_key(XEvent *event, char *c)
{
  if (event && event->type == KeyPress)
    {
      char string[8];
      int len = XLookupString(&event->xkey, string, sizeof(string), NULL,NULL);

      if (len == 1)
	{
	  *c = string[0];
	  return true;
	}
    }
  return false;
}

// ----------------------------------------------------------------------------
//
static bool x_function_key(XEvent *event, int *f, bool *shifted)
{
  if (event->type == KeyPress)
    {
      KeySym keysym;
      XLookupString(&event->xkey, NULL, 0, &keysym, NULL);

      if (IsFunctionKey(keysym))
	{
	  *f = (int) (keysym - XK_F1 + 1);
	  *shifted = (event->xkey.state & ShiftMask);
	  return true;
	}
    }
  return false;
}

// ----------------------------------------------------------------------------
//
static void set_background_erase(Widget w, GC, unsigned long bg)
{
  Tk_SetWindowBackground(window(w), bg);	// clear_area() uses this
}

// ----------------------------------------------------------------------------
//
static void eventually_redraw(Widget wt, int x, int y, int w, int h)
  { XClearArea(widget_display(wt), x_window(wt), x, y, w, h, True); }
static void draw_background(Display *display, Window win, GC,
			    int x, int y, int w, int h)
  { XClearArea(display, win, x, y, w, h, False); }

// ----------------------------------------------------------------------------
//
static void draw_vertical_text(Widget w, GC gc, Tk_Font font, int x, int y,
			       const Stringy &string, bool up)
{
  if (string.is_empty())
    return;

  int width, ascent, descent;
  text_size(font, string, &width, &ascent, &descent);

  Display *display = widget_display(w);
  Window win = x_window(w);

  int height = ascent + descent;
  int depth = Tk_Depth(window(w));
  Pixmap pixmap = Tk_GetPixmap(display, win, width, height, depth);
  XDrawImageString(display, pixmap, gc, 0, ascent,
		   string.cstring(), string.length());
  XImage *image = XGetImage(display, pixmap,
			    0, 0, width, height, AllPlanes, XYPixmap);
  Tk_FreePixmap(display, pixmap);

  XImage *rot_image = rotated_image(display, win, image, up);
  XDestroyImage(image);

  XPutImage(display, win, gc, rot_image, 0, 0,
	    (up ? x-ascent : x-descent), (up ? y-width : y), height, width);
  XDestroyImage(rot_image);
}

// ----------------------------------------------------------------------------
//
static XImage *rotated_image(Display *display, Window win, XImage *image,
			     bool counter_clockwise)
{
  int w = image->width;
  int h = image->height;
  int d = image->depth;
  Pixmap rotated_pixmap = Tk_GetPixmap(display, win, h, w, d);
  unsigned long planes = (1 << d) - 1;
  XImage *rot_image = XGetImage(display, rotated_pixmap,
				0, 0, h, w, planes, XYPixmap);
  Tk_FreePixmap(display, rotated_pixmap);

  if (counter_clockwise)
    {
      for (int c = 0 ; c < w ; ++c)
	for (int r = 0 ; r < h ; ++r)
	  XPutPixel(rot_image, r, w-c-1, XGetPixel(image, c, r));
    }
  else
    {
      for (int c = 0 ; c < w ; ++c)
	for (int r = 0 ; r < h ; ++r)
	  XPutPixel(rot_image, h-r-1, c, XGetPixel(image, c, r));
    }
  return rot_image;
}

// ----------------------------------------------------------------------------
//
static void translate_window_contents(Widget w, GC gc, int dx, int dy)
{
  Display *display = widget_display(w);
  Window win = x_window(w);

  int width = widget_width(w);
  int height = widget_height(w);
  XCopyArea(display, win, win, gc, 0, 0, width, height, dx, dy);

  if (dx > 0)
    eventually_redraw(w, 0, 0, dx, height);
  else if (dx < 0)
    eventually_redraw(w, width + dx, 0, -dx, height);

  if (dy > 0)
    eventually_redraw(w, 0, 0, width, dy);
  else if (dy < 0)
    eventually_redraw(w, 0, height + dy, width, -dy);
}

// ----------------------------------------------------------------------------
//
static bool platform_specific_show_url(WinSysP *wsp, const Stringy &url)
{
/*
  Stringy command =
    formatted_string("netscape -remote openURL(%s)", url.cstring());
  command = backslashify_characters(command, "()#");
  return run_command(command);
*/

  //
  // Forking the netscape -remote command is very slow.
  // So I instead directly communicate with Netscape via X properties.
  //

  Display *d = widget_display(main_widget(wsp));
  Window w = find_netscape_window(d);
  if (w)
    {
      Stringy cmd = formatted_string("openURL(%s)", url.cstring());
      netscape_command(d, w, cmd);
    }
  return (w != 0);
}

// ----------------------------------------------------------------------------
// Find a top level window running netscape by looking for a property
// netscape sets.
//
static Window find_netscape_window(Display *dpy)
{
  Window root = RootWindowOfScreen (DefaultScreenOfDisplay (dpy));
  Atom xa_mozilla_version = XInternAtom(dpy, "_MOZILLA_VERSION", False);
  int max_depth = 3;	// depth in window tree
  Window w = find_window_property(dpy, root, xa_mozilla_version, max_depth);
  return w;
}

// ----------------------------------------------------------------------------
//
static Window find_window_property(Display *d, Window w, Atom a, int max_depth)
{
  Atom type;
  int format;
  unsigned long nitems, bytesafter;
  unsigned char *version = 0;
  int status = XGetWindowProperty(d, w, a, 0, 1024, False, XA_STRING,
				  &type, &format, &nitems,
				  &bytesafter, &version);
  if (status == Success && type != None)
    {
      XFree(version);
      return w;
    }

  Window aw = 0;
  if (max_depth > 0)
    {
      Window root, parent, *kids;
      unsigned int nkids;
      if (XQueryTree(d, w, &root, &parent, &kids, &nkids))
	{
	  for (int k = nkids-1; k >= 0 && aw == 0 ; --k)
	    aw = find_window_property(d, kids[k], a, max_depth - 1);
	  XFree(kids);
	}
    }

  return aw;
}

// ----------------------------------------------------------------------------
// I don't read response, or set a lock to block other clients.
// This can mess up other applications trying to control
// the same netscape at the same time.  So I'm a bad citizen.
//
static void netscape_command(Display *dpy, Window window, const Stringy &cmd)
{
  Atom xa_mozilla_command = XInternAtom(dpy, "_MOZILLA_COMMAND", False);
  XChangeProperty (dpy, window, xa_mozilla_command, XA_STRING, 8,
		   PropModeReplace, (unsigned char *) cmd.cstring(),
		   cmd.length());
}

// ----------------------------------------------------------------------------
//
static void create_tcl_file_handler(FILE *fp, int mask,
				    Tcl_FileProc *input_cb, ClientData data)
{
  Tcl_CreateFileHandler(fileno(fp), mask, input_cb, data);
}

// ----------------------------------------------------------------------------
//
static void remove_tcl_file_handler(FILE *fp, Tcl_FileProc *, ClientData)
{
  Tcl_DeleteFileHandler(fileno(fp));
}
