// ----------------------------------------------------------------------------
// Window system routines specific to win32 platform
//
#include <windows.h>	// use InvalidateRect(), GetDC(), CreateRectRgn()
			//     InvalidateRgn(), DeleteDC(), DeleteObject()
#undef min
#undef max

#include <tcl.h>
#include <tk.h>
#include <tkPlatDecls.h>	// use Tk_GetHWND()

// ----------------------------------------------------------------------------
// These are Dynamics Data Exchange (DDE) declarations that haven't yet
// made it into the cygwin32 or mingw32 gcc header files.
//

#define     DMLERR_NO_ERROR                    0       /* must be 0 */
#define     TIMEOUT_ASYNC           0xFFFFFFFF

extern "C"
{
  HSZ WINAPI DdeCreateStringHandleA(DWORD idInst, const char *psz,
				    int iCodePage);
  BOOL WINAPI DdeFreeStringHandle(DWORD idInst, HSZ hsz);
  UINT WINAPI DdeInitializeA(LPDWORD pidInst, PFNCALLBACK pfnCallback,
			     DWORD afCmd, DWORD ulRes);
  HCONV WINAPI DdeConnect(DWORD idInst, HSZ hszService, HSZ hszTopic,
			  PCONVCONTEXT pCC);
  HDDEDATA WINAPI DdeCreateDataHandle(DWORD idInst, LPBYTE pSrc, DWORD cb,
				      DWORD cbOff, HSZ hszItem, UINT wFmt,
				      UINT afCmd);
  BOOL WINAPI DdeFreeDataHandle(HDDEDATA hData);
  HDDEDATA WINAPI DdeClientTransaction(LPBYTE pData, DWORD cbData,
				       HCONV hConv, HSZ hszItem, UINT wFmt,
				       UINT wType, DWORD dwTimeout,
				       LPDWORD pdwResult);
  BOOL WINAPI DdeDisconnect(HCONV hConv);
  BOOL WINAPI DdeUninitialize(DWORD idInst);
}
 
// ----------------------------------------------------------------------------
// Used to get foreground, background and mode for drawing in xor mode.
//
int XGetGCValues(Display *d, GC gc, unsigned long mask, XGCValues *values)
{
  if (mask & GCFunction) { values->function = gc->function; }
  if (mask & GCPlaneMask) { values->plane_mask = gc->plane_mask; }
  if (mask & GCForeground) { values->foreground = gc->foreground; }
  if (mask & GCBackground) { values->background = gc->background; }
  if (mask & GCLineWidth) { values->line_width = gc->line_width; }	
  if (mask & GCLineStyle) { values->line_style = gc->line_style; }
  if (mask & GCCapStyle) { values->cap_style = gc->cap_style; }
  if (mask & GCJoinStyle) { values->join_style = gc->join_style; }
  if (mask & GCFillStyle) { values->fill_style = gc->fill_style; }
  if (mask & GCFillRule) { values->fill_rule = gc->fill_rule; }
  if (mask & GCArcMode) { values->arc_mode = gc->arc_mode; }
  if (mask & GCTile) { values->tile = gc->tile; }
  if (mask & GCStipple) { values->stipple = gc->stipple; }
  if (mask & GCTileStipXOrigin) { values->ts_x_origin = gc->ts_x_origin; }
  if (mask & GCTileStipYOrigin) { values->ts_y_origin = gc->ts_y_origin; }
  if (mask & GCFont) { values->font = gc->font; }
  if (mask & GCSubwindowMode) { values->subwindow_mode = gc->subwindow_mode; }
  if (mask & GCGraphicsExposures) { values->graphics_exposures = gc->graphics_exposures; }
  if (mask & GCClipXOrigin) { values->clip_x_origin = gc->clip_x_origin; }
  if (mask & GCClipYOrigin) { values->clip_y_origin = gc->clip_y_origin; }
  if (mask & GCClipMask) { XSetClipMask(d, gc, values->clip_mask); }
  if (mask & GCDashOffset) { values->dash_offset = gc->dash_offset; }
  if (mask & GCDashList) { values->dashes = gc->dashes; }

  return true;
}

// ----------------------------------------------------------------------------
// Used during translations to catch all exposure events
//
void XSync(Display *, Bool) {}

// ----------------------------------------------------------------------------
// Include window system routines portable across all platforms.
//
#include "winsystem-all.cc"

// ----------------------------------------------------------------------------
//
static void ms_draw_background_cb(ClientData gc, XEvent *event);
static void ms_invalidate_rectangle(Window w, int x, int y,
				    unsigned int width, unsigned int height);
static bool ms_netscape_dde_request(DWORD dde_id, const Stringy &topic,
				    const Stringy &arguments);

// ----------------------------------------------------------------------------
//
static bool x_key(XEvent *event, char *c)
{
  if (event && event->type == KeyPress)
    {
      XKeyEvent *e = (XKeyEvent *) event;
      int index = 0;
      if (e->state & ShiftMask)
	index |= 1;
      if (e->state & Mod1Mask)
	index |= 2;

      KeySym keysym = XKeycodeToKeysym(e->display, e->keycode, index);
      if (((keysym != NoSymbol) && (keysym > 0) && (keysym < 256))) 
	*c = (char) keysym;
      else if (keysym == XK_Return)
	*c = (char) 0x0d;
      else if (keysym == XK_Tab)
	*c = (char) 0x09;
      else if (keysym == XK_Delete)
	*c = (char) 0x7f;
      else if (keysym == XK_Escape)
	*c = (char) 0x1b;
      else
	return false;

      return true;
    }

  return false;
}

// ----------------------------------------------------------------------------
//
static bool x_function_key(XEvent *event, int *f, bool *shifted)
{
  if (event->type == KeyPress)
    {
      KeySym keysym;

      keysym = XKeycodeToKeysym(((XKeyEvent *)event)->display,
				((XKeyEvent *)event)->keycode, 0);

      if (IsFunctionKey(keysym))
	{
	  *f = keysym - XK_F1 + 1;
	  *shifted = (event->xkey.state & ShiftMask);
	  return true;
	}
    }
  return false;
}

// ----------------------------------------------------------------------------
//
static void set_background_erase(Widget w, GC gc, unsigned long)
{
  //
  // In Windows Tk Tk_SetWindowBackground() is a noop.  Tk uses the same
  // WindowClass for all child windows and this object contains the background
  // color.  So there is not a simple fix to make Tk_SetWindowBackground()
  // work.  Instead I register an exposure handler to draw the background.
  // Tk event handlers are invoked in the order created so the background
  // will be drawn before other drawing takes place.  The handler will be
  // removed when the window is destroyed.
  //
  add_tk_event_handler(w, ExposureMask, ms_draw_background_cb, gc);
}

// ----------------------------------------------------------------------------
//
static void ms_draw_background_cb(ClientData gc, XEvent *event)
{
  int x, y, w, h;
  if (exposed_region(event, &x, &y, &w, &h))
    draw_background(event->xexpose.display, event->xexpose.window, (GC) gc,
		    x, y, w, h);
}

// ----------------------------------------------------------------------------
//
static void eventually_redraw(Widget wt, int x, int y, int w, int h)
{
  ms_invalidate_rectangle(x_window(wt), x, y, w, h);
}

// ----------------------------------------------------------------------------
// Windows substitute for XClearArea()
//
static void draw_background(Display *display, Window win,
			    GC gc, int x, int y, int w, int h)
{
  XGCValues gc_values;
  XGetGCValues(display, gc, GCForeground | GCBackground, &gc_values);
  XSetForeground(display, gc, gc_values.background);
  XFillRectangle(display, win, gc, x, y, w, h);
  XSetForeground(display, gc, gc_values.foreground);
}

// ----------------------------------------------------------------------------
// Windows substitute for XClearArea()
//
static void ms_invalidate_rectangle(Window w, int x, int y,
				    unsigned int width, unsigned int height)
{
  if (width == 0 || height == 0)
    InvalidateRect(Tk_GetHWND(w), NULL, TRUE);
  else
    {
      RECT r;
      r.left = x;
      r.right = x + width;
      r.top = y;
      r.bottom = y + height;
      InvalidateRect(Tk_GetHWND(w), &r, TRUE);
    }
}

// ----------------------------------------------------------------------------
//
static void draw_vertical_text(Widget w, GC gc, Tk_Font font, int x, int y,
			       const Stringy &string, bool up)
{
  if (string.is_empty())
    return;

  int width, ascent, descent;
  text_size(font, string, &width, &ascent, &descent);

  Display *display = widget_display(w);
  Window win = x_window(w);

  //
  // The Windows Tk 8.0.3 implementation of XGetImage() can only handle
  // depth 1 images.  And, in fact, it has a bug (biCompression set twice
  // and biSizeImage not set) so even this case does not work.  Maybe
  // Tk 8.1 will fix XGetImage().
  //
  // Windows Tk 8.0.3 doesn't have XPutImage() implemented.
  // The implementation has TkPutImage() defined in source file
  // tk8.0.3/win/tkWinDraw.c that works as a substitute.
  //
  // So just stack letters in a column instead of rotating them.
  //

  for (int k = 0 ; k < string.length() ; ++k)
    Tk_DrawChars(display, win, gc, font, string.cstring() + k, 1,
		 x, y + (up ? -k * ascent : (k+1) * ascent));
}

// ----------------------------------------------------------------------------
// Scroll window contents generating exposure events.
// Tk XCopyArea() for MS Windows doesn't do graphics exposures,
// ie. nothing special is done to handle children
// of the scrolled window, or other overlaying windows.
// This routine fixes this.
//
static void translate_window_contents(Widget w, GC gc, int dx, int dy)
{
  HWND hwnd = Tk_GetHWND(x_window(w));
  HDC hdc = GetDC(hwnd);
  HRGN hrgn = CreateRectRgn(1, 1, 0, 0);	// empty region

  if (ScrollDC(hdc, dx, dy, NULL, NULL, hrgn, NULL))
    InvalidateRgn(hwnd, hrgn, TRUE);

  DeleteObject(hrgn);
  DeleteDC(hdc);
}

// ----------------------------------------------------------------------------
//
static bool platform_specific_show_url(WinSysP *, const Stringy &url)
{
  DWORD dde_id = 0;
  UINT status = DdeInitialize(&dde_id, NULL, APPCMD_CLIENTONLY, 0);
  if (status != DMLERR_NO_ERROR)
    return false;
  bool result =
    (ms_netscape_dde_request(dde_id, "WWW_Activate", "-1,0") &&
     ms_netscape_dde_request(dde_id, "WWW_OpenURL", url + ",,-1,0"));
  DdeUninitialize(dde_id);

  return result;
}

// ----------------------------------------------------------------------------
//
static bool ms_netscape_dde_request(DWORD dde_id, const Stringy &topic,
				    const Stringy &arguments)
{
  HSZ service = DdeCreateStringHandle(dde_id, "NETSCAPE", CP_WINANSI);
  HSZ top = DdeCreateStringHandle(dde_id,
				  (char *)topic.cstring(), CP_WINANSI);
  HCONV c = DdeConnect(dde_id, service, top, NULL);
  DdeFreeStringHandle(dde_id, top);
  DdeFreeStringHandle(dde_id, service);
  if (c == 0)
    return false;  // netscape is not running

  HSZ args = DdeCreateStringHandle(dde_id,
				   (char *)arguments.cstring(), CP_WINANSI);

  HDDEDATA result = DdeClientTransaction(NULL, 0, c, args, CF_TEXT,
					 XTYP_REQUEST, TIMEOUT_ASYNC, NULL);

  if (result)
    DdeFreeDataHandle(result);
  DdeFreeStringHandle(dde_id, args);
  DdeDisconnect(c);

  return true;
}

// ----------------------------------------------------------------------------
// These aren't implemented in win32 Tcl.
//
void Tcl_CreateFileHandler(int, int, Tcl_FileProc *, ClientData) {}
void Tcl_DeleteFileHandler(int fd) {}

// ----------------------------------------------------------------------------
// Not implemented.
//
static void create_tcl_file_handler(FILE *, int, Tcl_FileProc *, ClientData)
{
  //
  // The Tcl Tcl_CreateFileHandler() function only works on Unix.
  // I could possibly get the same result with Tcl_CreateChannelHandler()
  // and Tcl_MakeFileChannel().  But I read the data stream via the FILE
  // pointer and so whether the channel is readable may not get updated
  // properly.  Need to test this.  I also will need to squirrel away the
  // channel I create so it can later be closed.
  //

  //  Tcl_Channel channel = Tcl_MakeFileChannel(fileno(fp), mask);
  //  Tcl_CreateChannelHandler(channel, mask, input_cb, data);
}

// ----------------------------------------------------------------------------
//
static void remove_tcl_file_handler(FILE *, Tcl_FileProc *, ClientData)
{
}
