// ----------------------------------------------------------------------------
//
#ifndef WINSYSTEM_HEADER_INCLUDED
#define WINSYSTEM_HEADER_INCLUDED

#include <stdio.h>		// use FILE
#include <stdarg.h>

typedef void *Widget;
typedef void *CB_Data;
typedef void (*CB_Proc)(Widget, CB_Data, CB_Data);
typedef void (*CB_Func)(CB_Data);

class Color;
class DrawWinP;
class List;
class Stringy;
class WinSysP;

enum Event_Type
{
  destroy_query_event,
  destroy_event,
  expose_event,
  resize_event,
  mapping_event,
  button_1_press_event,
  button_1_release_event,
  button_3_press_event,
  pointer_move_event,
  pointer_drag_event,
  pointer_pause_event,
  enter_window_event,
  leave_window_event,
  got_focus_event,
  key_press_event,
  enter_pressed_event
};

enum Table_Entry
{
  TABLE_LABEL,
  TABLE_TEXT_FIELD,
  TABLE_OPTION_MENU,
  TABLE_TOGGLE_BUTTON
};

// ----------------------------------------------------------------------------
//
class WinSys
{
public:
  WinSys(const Stringy &classname, const Stringy &appname,
	 const Stringy &resourcedir, int *argc, char **argv);
  WinSys(const Stringy &classname, const Stringy &appname,
	 const Stringy &resourcedir, void *tcl_interp);
  ~WinSys();

  //
  // Event processing
  //
  void event_loop();
  void exit_event_loop();
  void process_pending_events();
  void process_exposure_events(Widget w);
  bool process_until_modal_reply(Widget w, bool *replied);
  void repaint_widget(Widget w);
  bool more_motion_events();

  //
  // Top level windows (aka dialogs)
  //
  Widget create_dialog(const Stringy &name, bool allow_destroy = false);
  void show_dialog(Widget w);
  void unshow_dialog(Widget w);
  void iconify_dialog(Widget w);
  void unshow_all_dialogs();
  void delete_dialogs_except_main();
  void set_dialog_title(Widget dialog, const Stringy &title);
  void set_icon_name(Widget w, const Stringy &name);
  void set_dialog_position(Widget w, int x, int y);
  void dialog_position(Widget w, int *x, int *y);
  void resize_dialog(Widget w, int width, int height);
  void allow_dialog_resize(Widget w, bool allow);
  void grab_events(Widget dialog);
  void ungrab_events(Widget dialog);
  bool modal_dialog_shown();
  Widget main_widget();
  void set_keyboard_focus_child(Widget w);
  bool show_url(const Stringy &url);

  //
  // Labels
  //
  Widget create_label(Widget parent, const Stringy &name);
  Stringy label_text(Widget label);
  void set_label_text(Widget label, const Stringy &text);

  //
  // Buttons
  //
  Widget button_row(Widget parent, const Stringy &name,
		    char *first_button_name, ...
		    /* char *button_name, CB_Proc button_cb, CB_Data cb_data */);
  Widget push_button(Widget parent, const Stringy &name,
		     CB_Proc cb, CB_Data cb_data);

  //
  // Radio buttons
  //
  Widget create_radio_buttons(Widget parent, const Stringy &name);
  void add_radio_button(Widget rbuttons, const Stringy &name);
  void set_radio_button(Widget rbuttons, const Stringy &name);
  Stringy radio_selection(Widget rbuttons);
  void radio_callback(Widget rbuttons, CB_Proc cb, CB_Data cb_data);

  //
  // Toggle buttons
  //
  Widget create_toggle_button(Widget parent, const Stringy &name);
  void add_toggle_button_callback(Widget tb, CB_Proc cb, CB_Data cb_data);
  bool toggle_button_state(Widget button);
  void set_toggle_button(Widget button, bool state, bool notify);
  void set_toggle_button_text(Widget button, const Stringy &text);

  //
  // Column of toggle buttons
  //
  Widget switches(Widget parent, const Stringy &name, const char *names[],
		  int columns = 1);
  void set_switch(Widget switches, const Stringy &name, bool state);
  bool switch_state(Widget switches, const Stringy &name);
  void switch_callback(Widget switches, const Stringy &name,
		       CB_Proc cb, CB_Data client_data);

  //
  // Option menus
  //
  Widget option_menu(Widget parent, const Stringy &menu_name,
		     const char *entries[]);
  void add_option(Widget option_menu, const Stringy &name);
  void add_options(Widget option_menu, const char *names[]);
  void delete_option(Widget option_menu, const Stringy &name);
  void delete_all_options(Widget option_menu);
  Stringy option_selected(Widget option_menu);
  void set_option(Widget option_menu, const Stringy &name,
		  bool call_callback = true);
  void set_option(Widget option_menu, int num);
  void sensitize_option(Widget option_menu, const Stringy &name, bool sensitive);
  void option_callback(Widget option_menu, CB_Proc cb, CB_Data cb_data);

  //
  // Text
  //
  Widget create_scrolled_text(Widget parent, const Stringy &name, Widget *text);
  int text_size(Widget text);
  void replace_text(Widget t, const Stringy &text);
  void append_text(Widget t, const Stringy &text);
  void add_text_position_callback(Widget t, CB_Proc cb, CB_Data cb_data);
  void remove_text_position_callback(Widget t, CB_Proc cb, CB_Data cb_data);
  void selected_text_position(Widget t, CB_Data tcbdata, int *row, int *column);
  Widget create_text_table(Widget parent, const Stringy &name, Widget *row_heading,
			   Widget *column_heading, Widget *values);

  //
  // Text field routines.
  //
  Widget create_text_field(Widget parent, const Stringy &name);
  Stringy text_field_string(Widget tf);
  void set_text_field(Widget tf, const Stringy &s);
  double numeric_text_field(Widget tf);
  void set_numeric_text_field(Widget tf, const Stringy &format, double value);
  bool text_field_is_empty(Widget tf);
  void add_text_field_changed_callback(Widget tf, CB_Proc cb, CB_Data cb_data);
  void remove_text_field_changed_callback(Widget tf, CB_Proc cb, CB_Data);

  //
  // Text field with a title
  //
  Widget edit_field(Widget parent, const Stringy &name, bool menu = false);
  Widget edit_field_menu(Widget ef);
  void add_edit_field_menu_entry(Widget pane, const Stringy &text, Widget ef);
  Widget edit_field_text(Widget ef);
  Stringy edit_value(Widget ef);
  void set_edit_title(Widget ef, const Stringy &title);
  void set_edit_value(Widget ef, const Stringy &value);
  void edit_field_editable(Widget ef, bool editable);
  double numeric_edit_value(Widget ef);
  void set_numeric_edit_value(Widget ef, const Stringy &format, double value);

  //
  // Lists
  //
  Widget create_scrolled_list(Widget parent, const Stringy &name, Widget *list);
  void add_list_item(Widget list, const Stringy &text);
  void set_list_items(Widget list, const List &strings);
  void replace_list_item(Widget list, int pos, const Stringy &text);
  void delete_list_items(Widget list);
  void delete_list_position(Widget list, int pos);
  bool selected_list_position(Widget list, int *position);
  void selected_list_positions(Widget list, int *num, int **positions);
  void free_list_positions(int *positions);
  Stringy selected_list_item(Widget list);
  List selected_list_items(Widget list);
  void select_list_position(Widget list, int pos, bool select);
  void deselect_list_items(Widget list);
  bool find_list_substring(Widget list, int start_pos, const Stringy &string,
			   int *pos);
  int top_visible_list_position(Widget list);
  void set_top_visible_list_position(Widget list, int position);
  void center_list_item(Widget list, const Stringy &item, bool selected);
  void center_list_position(Widget list, int pos, bool selected);
  void set_visible_list_length(Widget list, int length);
  void multiple_selection_list(Widget list);
  void extended_selection_list(Widget list);
  Widget scrolled_list_heading(Widget slist);
  bool save_list_lines(Widget list, const Stringy &heading, const Stringy &path);
  void add_list_selection_callback(Widget list, CB_Proc cb, CB_Data cb_data);
  void remove_list_selection_callback(Widget list, CB_Proc cb, CB_Data cb_data);
  void add_list_double_click_callback(Widget list, CB_Proc cb, CB_Data cb_data);
  void remove_list_double_click_callback(Widget list, CB_Proc cb, CB_Data);

  //
  // Tables
  //
  Widget widget_table(Widget parent, const Stringy &name, int rows, int columns,
		      ...);  // Table_Entrys
  void manage_table_children(Widget table, int rows, int columns);
  void manage_table_row(Widget table, int r);
  Widget table_element(Widget table, int row, int column);
  Stringy table_element_name(Widget table, int row, int column);

  //
  // Pulldown and popup menus
  //
  Widget create_menu_bar(Widget parent, const Stringy &name);
  Widget add_menu(Widget menubar, const Stringy &button, const Stringy &pane);
  Widget create_menu(Widget parent, const Stringy &button, const Stringy &pane);
  Widget menu_pane(Widget menu);
  void add_menu_button(Widget parent, const Stringy &name,
		       CB_Proc cb, CB_Data cb_data);
  void add_menu_button(Widget parent, const Stringy &name, const Stringy &accel,
		       CB_Proc cb, CB_Data cb_data);
  void add_menu_button_at_top(Widget parent, const Stringy &name,
			      CB_Proc cb, CB_Data cb_data);
  void delete_menu_button(Widget menu, const Stringy &buttonname);
  void delete_menu_buttons(Widget menu);
  void menu_separator(Widget pane, const Stringy &name);
  void menu_label(Widget pane, const Stringy &name);
  Stringy selected_menu_button(Widget, CB_Data call_data);
  Widget cascade_pane(Widget parent, const Stringy &button_name);
  Widget find_cascade_pane(Widget parent, const Stringy &button_name);
  void delete_cascade(Widget parent, const Stringy &button_name);
  bool is_pane_empty(Widget pane);
  void square_up_menu(Widget menu);
  Widget create_menu_pane(Widget parent, const Stringy &name);
  void popup_menu(Widget popup, CB_Data event);

  //
  // Forms
  //
#define END_OF_WIDGETS	((Widget) 0)
  Widget create_form(Widget parent, const Stringy &name);
  void column_attachments(Widget stretch_me, ...);
  void row_attachments(Widget stretch_me, ...);
  void detach_top_and_bottom(Widget w);
  void plot_form_layout(Widget form, Widget center,
			const List &horz_edge_widgets,
			const List &vert_edge_widgets);

  //
  // Scrollbars
  //
  Widget create_scrollbar(Widget parent, const Stringy &name, bool horz,
			  CB_Proc cb, CB_Data cb_data);
  double scrollbar_position(Widget sbar);
  void set_scrollbar(Widget sbar, double v1, double v2,
		     double s1, double s2, double step);

  //
  // Container stuff
  //
  Widget create_frame(Widget parent, const Stringy &name);
  Widget create_separator(Widget parent, const Stringy &name);
  void set_main_window_areas(Widget mw, Widget menubar,
			     Widget workwin, Widget statusline);

  //
  // File selection
  //
  Stringy open_file_dialog(Widget parent, const Stringy &title,
			   const Stringy &directory, const Stringy &file_type,
			   bool prefer_previous);
  Stringy saveas_file_dialog(Widget parent, const Stringy &title,
			     const Stringy &path, const Stringy &file_type,
			     bool prefer_previous);

  //
  // Query dialog (application modal)
  //
  Widget choice_dialog(const Stringy &name, bool allow_destroy,
		       const char *question, ...
		       // const char *choice, CB_Proc, void *
		       );
  void new_choice_question(Widget cd, const Stringy &question);
  void delete_choice_dialog(Widget cd);

  //
  // Event callbacks
  //
  void add_event_callback(Event_Type, Widget, CB_Proc, CB_Data);
  void remove_event_callback(Event_Type, Widget, CB_Proc, CB_Data);

  bool exposure_region(CB_Data event, int *x, int *y, int *w, int *h);
  bool key_pressed(CB_Data event, char *c);
  bool function_key_pressed(CB_Data event, int *f, bool *shifted);
  bool shift_modifier(CB_Data event);
  bool event_xy(CB_Data event, int *x, int *y);

  //
  // Work procedures
  //
  void add_work_procedure(CB_Func, CB_Data, double priority = 0);
  void set_work_procedure_priority(CB_Func, CB_Data, double priority);
  void remove_work_procedure(CB_Func, CB_Data);

  //
  // Timers
  //
  void add_timer(unsigned long millisec, CB_Func, CB_Data);
  void remove_timer(unsigned long millisec, CB_Func, CB_Data);

  void queue_callback(CB_Func, CB_Data);

  //
  // Additional sources for event dispatching
  //
  void add_input_callback(FILE *, CB_Func, CB_Data);
  void remove_input_callback(FILE *, CB_Func, CB_Data);

  //
  // Widget geometry, mapping, hierarchy, general stuff...
  //
  int widget_width(Widget w);
  int widget_height(Widget w);
  int requested_width(Widget w);
  int requested_height(Widget w);
  int widget_x(Widget w);
  int widget_y(Widget w);
  bool is_viewable(Widget w);
  bool is_top_level(Widget w);
  Widget child_widget(Widget parent, const Stringy &childname);
  Stringy widget_name(Widget w);
  Widget named_widget(const Stringy &path);

  void set_widget_size(Widget w, int width, int height);
  void set_widget_width(Widget w, int width);
  void set_widget_height(Widget w, int height);
  void place_widget(Widget w, int x, int y);
  void unplace_widget(Widget w);
  void set_sensitive(Widget w, bool sensitive);

  void raise_widget(Widget w);
  void delete_widget(Widget w);

  Stringy read_application_resource(Widget w,
				    const Stringy &resource_name,
				    const Stringy &class_name,
				    const Stringy &default_value);
  bool color_rgb(const Color &color, double *r, double *g, double *b);

  //
  // Drawing areas
  //
  Widget create_drawing_area(Widget parent, const Stringy &name);

  WinSysP *winsysp();	// Hack to give Drawing_Window access to WinSysP

private:
  WinSysP *wsp;
};

// --------------------------------------------------------------------------
// Draw text and lines in a window
//
class Drawing_Window
{
public:
  Drawing_Window(WinSys &, Widget parent, const Stringy &name);
  virtual ~Drawing_Window();

  Widget widget() const;

  void set_foreground(const Color &color);
  void draw_string(int x, int y, const Stringy &string);
  void draw_vertical_string(int x, int y, const Stringy &string, bool up);
  void draw_line(int x1, int y1, int x2, int y2, bool xor_mode = false);
  void draw_point(int x, int y);
  void draw_rectangle(int x, int y, unsigned int w, unsigned int h,
		      bool xor_mode = false);
  void fill_rectangle(int x, int y, unsigned int w, unsigned int h);
  void fill_triangle(int x1, int y1, int x2, int y2, int x3, int y3);
  void draw_arc(int x, int y, unsigned int w, unsigned int h, int a1, int a2);

  void clear_window();
  void clear_area(int x, int y, int w, int h);
  void draw_background(int x, int y, int w, int h);
  void translate_contents(int dx, int dy);
  void request_backing_store();

  bool set_font_height(int pix_height);
  bool use_label_font();
  void text_size(const Stringy &string, int *w, int *ascent, int *descent);
  int font_height();

private:
  DrawWinP *p;
};

#endif
